/*
 * Copyright (C) 2019 Marvell International Ltd.
 *
 * SPDX-License-Identifier:    GPL-2.0
 * https://spdx.org/licenses
 */

#include <common.h>
#include <command.h>
#include <dm.h>
#include <dm/uclass-internal.h>
#include <net.h>

extern int cgx_intf_set_fec(struct udevice *ethdev, int type);
extern int cgx_intf_get_fec(struct udevice *ethdev);
extern int cgx_intf_get_phy_mod_type(struct udevice *ethdev);
extern int cgx_intf_set_phy_mod_type(struct udevice *ethdev, int type);
extern int cgx_intf_set_mode(struct udevice *ethdev, int mode);
extern void nix_print_mac_info(struct udevice *dev);

static int do_ethlist(cmd_tbl_t *cmdtp, int flag, int argc,
		      char *const argv[])
{
	struct udevice *dev;

	for (uclass_find_first_device(UCLASS_ETH, &dev); dev;
	     uclass_find_next_device(&dev)) {
		printf("eth%d [%s]", dev->seq, dev->name);
		if (!strncmp(dev->name, "rvu_", 4))
			nix_print_mac_info(dev);
		printf("\n");
	}
	return 0;
}

U_BOOT_CMD(
	ethlist, 1, 1, do_ethlist, "Display ethernet interface list",
	"Prints all detected ethernet interfaces with below format\n"
	"ethX [device name] [LMAC info for RVU PF devices]\n"
);

static int do_ethparam_common(cmd_tbl_t *cmdtp, int flag, int argc,
			      char *const argv[])
{
	const char *cmd;
	char *endp;
	const char *devname;
	int ret = -1, type, mode;
	struct udevice *dev;

	if (argc < 2)
		return CMD_RET_USAGE;

	cmd = argv[0];
	devname = argv[1];

	dev = eth_get_dev_by_name(devname);
	if (!dev) {
		printf("device interface %s not found\n", devname);
		return CMD_RET_FAILURE;
	}
	if (strncmp(dev->name, "rvu_", 4)) {
		printf("Invalid eth interface, choose RVU PF device\n");
		return CMD_RET_FAILURE;
	}
	if (strcmp(cmd, "set_fec") == 0) {
		if (argc < 3)
			return CMD_RET_FAILURE;
		type = simple_strtol(argv[2], &endp, 0);
		if (type < 0 || type > 2)
			return CMD_RET_USAGE;
		ret = cgx_intf_set_fec(dev, type);
	} else if (strcmp(cmd, "get_fec") == 0) {
		ret = cgx_intf_get_fec(dev);
	} else if (strcmp(cmd, "set_phymod") == 0) {
		if (argc < 3)
			return CMD_RET_FAILURE;
		type = simple_strtol(argv[2], &endp, 0);
		if (type < 0 || type > 1)
			return CMD_RET_USAGE;
		ret = cgx_intf_set_phy_mod_type(dev, type);
	} else if (strcmp(cmd, "get_phymod") == 0) {
		ret = cgx_intf_get_phy_mod_type(dev);
	} else if (strcmp(cmd, "set_mode") == 0) {
		if (argc < 3)
			return CMD_RET_FAILURE;
		mode = simple_strtol(argv[2], &endp, 0);
		if (mode < 0 || mode > 6)
			return CMD_RET_USAGE;
		ret = cgx_intf_set_mode(dev, mode);
	}
	return (ret == 0) ? CMD_RET_SUCCESS : CMD_RET_FAILURE;
}

U_BOOT_CMD(
	set_fec, 3, 1, do_ethparam_common,
	"Modify fec type for selected ethernet interface",
	"Example - set_fec <ethX> [type]\n"
	"Set FEC type for any of RVU PF based network interfaces\n"
	"- where type - 0 [NO FEC] 1 [BASER_FEC] 2 [RS_FEC]\n"
	"Use 'ethlist' command to display network interface names\n"
);

U_BOOT_CMD(
	get_fec, 2, 1, do_ethparam_common,
	"Display fec type for selected ethernet interface",
	"Example - get_fec <ethX>\n"
	"Get FEC type for any of RVU PF based network interfaces\n"
	"Use 'ethlist' command to display network interface names\n"
);

U_BOOT_CMD(
	set_phymod, 3, 1, do_ethparam_common,
	"Modify line side phy-mod type for selected ethernet interface",
	"Example - set_phymod <ethX> [type]\n"
	"Set PHY MOD type for any of RVU PF based network interfaces\n"
	"Currently, only 50G_R mode supports type 0 [NRZ] or 1 [PAM4]\n"
	"Use 'ethlist' command to display network interface names\n"
);

U_BOOT_CMD(
	get_phymod, 2, 1, do_ethparam_common,
	"Display line side phy-mod type for selected ethernet interface",
	"Example - get_phymod <ethX>\n"
	"Get PHY MOD type for any of RVU PF based network interfaces\n"
	"Use 'ethlist' command to display network interface names\n"
);

U_BOOT_CMD(set_mode, 3, 1, do_ethparam_common,
	   "Modify Interface mode for selected ethernet interface",
	   "Example - set_mode <ethX> [mode]\n"
	   "Change mode of selected network interface\n"
	   "\n"
	   "mode encoding -\n"
	   "	0 - 10G_C2C\n"
	   "	1 - 10G_C2M\n"
	   "	2 - 10G_KR\n"
	   "	3 - 25G_C2C\n"
	   "	4 - 25G_2_C2C\n"
	   "	5 - 50G_C2C\n"
	   "	6 - 50G_4_C2C\n"
	   "Use 'ethlist' command to display network interface names\n"
);

