/*
 * Copyright (C) 2016 Stefan Roese <sr@denx.de>
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */

#include <common.h>
#include <dm.h>
#include <i2c.h>
#include <phy.h>
#include <asm/io.h>
#include <asm/arch/cpu.h>
#include <asm/arch/soc.h>
#include <power/regulator.h>

DECLARE_GLOBAL_DATA_PTR;

/* on Armada3700 rev2 devel-board, IO expander (with I2C address 0x22) bit
 * 14 is used as Serdes Lane 2 muxing, which could be used as SATA PHY or
 * USB3 PHY.
 */
enum COMPHY_LANE2_MUXING {
	COMPHY_LANE2_MUX_USB3,
	COMPHY_LANE2_MUX_SATA
};

/* IO expander I2C device */
#define I2C_IO_EXP_ADDR		0x22
#define I2C_IO_CFG_REG_0	0x6
#define I2C_IO_DATA_OUT_REG_0	0x2
#define I2C_IO_REG_0_SATA_OFF	2
#define I2C_IO_REG_0_USB_H_OFF	1
#define I2C_IO_COMPHY_SATA3_USB_MUX_BIT	14

/* The pin control values are the same for DB and Espressobin */
#define PINCTRL_NB_REG_VALUE	0x000173fa
#define PINCTRL_SB_REG_VALUE	0x00007a23

/* Ethernet switch registers */
/* SMI addresses for multi-chip mode */
#define MVEBU_PORT_CTRL_SMI_ADDR(p)	(16 + (p))
#define MVEBU_SW_G2_SMI_ADDR		(28)

/* Multi-chip mode */
#define MVEBU_SW_SMI_DATA_REG		(1)
#define MVEBU_SW_SMI_CMD_REG		(0)
 #define SW_SMI_CMD_REG_ADDR_OFF	0
 #define SW_SMI_CMD_DEV_ADDR_OFF	5
 #define SW_SMI_CMD_SMI_OP_OFF		10
 #define SW_SMI_CMD_SMI_MODE_OFF	12
 #define SW_SMI_CMD_SMI_BUSY_OFF	15

/* Single-chip mode */
/* Switch Port Registers */
#define MVEBU_SW_LINK_CTRL_REG		(1)
#define MVEBU_SW_PORT_CTRL_REG		(4)

/* Global 2 Registers */
#define MVEBU_G2_SMI_PHY_CMD_REG	(24)
#define MVEBU_G2_SMI_PHY_DATA_REG	(25)


int board_early_init_f(void)
{
	return 0;
}

int board_init(void)
{
	/* adress of boot parameters */
	gd->bd->bi_boot_params = CONFIG_SYS_SDRAM_BASE + 0x100;

	return 0;
}

#ifdef CONFIG_SILEX
static char *sx_bootmode = "normal";

#define GPIO1_BASE		0xD0013800
#define GPIO2_BASE		0xD0018800

#define GPIOBIT_SYSREST		((1 * 32) + 2)
#define GPIOBIT_PREST		((1 * 32) + 3)
#
#define GPIOBIT_PUSHSW		((1 * 32) + 5)
#define GPIOBIT_POWER2_RED	((0 * 32) + 8)
#define GPIOBIT_POWER2_GREEN	((0 * 32) + 9)
#define GPIOBIT_WLAN_RED	((0 * 32) + 0)
#define GPIOBIT_WLAN_GREEN	((0 * 32) + 1)
#define GPIOBIT_STATUS_RED	((1 * 32) + 20)
#define GPIOBIT_STATUS_GREEN	((1 * 32) + 21)

#define GPIOBIT_RJ45_GREEN	((0 * 32) + 2)
#define GPIOBIT_RJ45_RED	((0 * 32) + 3)

#define GPIOBIT_USB_EN1		((1 * 32) + 0)
#define GPIOBIT_USB_OC1		((1 * 32) + 22)
#define GPIOBIT_USB_EN2		((1 * 32) + 1)
#define GPIOBIT_USB_OC2		((1 * 32) + 23)


struct gpio_reg {
	u32 dir;
	u32 reserved1;
	u32 latch;
	u32 reserved2;
	u32 input;
	u32 reserved3;
	u32 output;
	u32 reserved4;
};

static struct gpio_reg* sxgpio_get_base(unsigned gpio)
{
	int base;

	base = (gpio / 32) + 1;

	switch(base) {
	case 1:
		return (struct gpio_reg*)GPIO1_BASE;
	case 2:
		return (struct gpio_reg*)GPIO2_BASE;
	default:
		return NULL;
	}

}
static int sxgpio_set_value(unsigned gpio, int val)
{
	struct gpio_reg *base = sxgpio_get_base(gpio);
	int bit;
	u32 reg;

	bit = (gpio % 32);
	reg = readl(&base->output);

	if (val)
		reg = reg | (1 << bit);
	else
		reg = reg & ~(1 << bit);

	writel(reg, &base->output);

	return 0;
}

static int sxgpio_get_value(unsigned gpio)
{
	struct gpio_reg *base = sxgpio_get_base(gpio);
	u32 reg;
	int bit;

	bit = (gpio % 32);
	reg = readl(&base->input);

	return (reg >> bit) & 1;
}

static int sxgpio_set_dir(unsigned gpio, int out, int val)
{
	struct gpio_reg *base = sxgpio_get_base(gpio);
	int bit;
	u32 reg;

	if (out)
		sxgpio_set_value(gpio, val);

	bit = (gpio % 32);
	reg = readl(&base->dir);

	if (out)
		reg = reg | (1 << bit);
	else
		reg = reg & ~(1 << bit);

	writel(reg, &base->dir);

	return 0;
}

char* get_bootmode(void)
{
	return sx_bootmode;
}

/* Memory test */
static int memtest(ulong *start, ulong *end, ulong pattern)
{
	ulong	*addr;
	ulong	readback;
	int     rcode = 0;
	int 	count = 0;

	printf("start:%p end:%p\n", start, end);
	printf ("\rPattern %08lX  Writing...%12s\b\b\b\b\b\b\b\b\b\b",
								pattern, "");

	for (addr=start; addr<end; addr++) {
		count++;
		if(count == (1024*1024)){
			count = 0;
		}
		*addr = pattern;
	}

	puts ("Reading...");

	count = 0;
	for (addr=start; addr<end; addr++) {
		count++;
		if(count == (1024*1024)){
			count = 0;
		}

		readback = *addr;
		if (readback != pattern) {
			printf ("\nMem error @ 0x%p: "
					"found %lx, expected %08lX\n",
					addr, readback, pattern);
			rcode = 1;
		}
	}

	putc ('\n');

	return rcode;
}

static void memtest_error(void)
{
	/* Turn off Power LED Green */
	sxgpio_set_value(GPIOBIT_RJ45_GREEN, 0);

	for (;;) {
		/* Blink Power LED red */
		sxgpio_set_value(GPIOBIT_RJ45_RED, 0);
		udelay(100000);
		sxgpio_set_value(GPIOBIT_RJ45_RED, 1);
		udelay(100000);
	}
}

/* Select boot mode process */
/* GPIO mapping and specification is defferent for each board */
/* So you need to create funcation for setting sx_bootmode argument */
/*******************************************************************************
* select_bootmode -
*
* DESCRIPTION:
* Select boot mode by status of Push Switch.
*
* INPUT:
*       None.
*
* RETURN: boot mode
*	    BOOTMODE_NORMAL           : Normal boot
*	    BOOTMODE_MANUFACTURE      : Manufacuturing mode
*	    BOOTMODE_FACTORYDEFAULT   : Factory default mode
*	    BOOTMODE_RECOVERY         : Recovery mode
*******************************************************************************/
static int select_bootmode(void)
{
	int i, ret;
	uint32_t *buf = (uint32_t*)CONFIG_SYS_LOAD_ADDR;
	char cmd[128];

	snprintf(cmd, sizeof(cmd), "sf probe; sf read %x %x 4",
		CONFIG_SYS_LOAD_ADDR, KERNEL_OFFS);
	cmd[sizeof(cmd) -1] = '\0';
	run_command(cmd, 0);

	if (*buf != 0xEDFE0DD0) {
		printf("Bad firmware %x\n", *buf);
		return BOOTMODE_MANUFACTURE;
	}

	ret = BOOTMODE_NORMAL;
	/* Sense Push Switch */
	if(!sxgpio_get_value(GPIOBIT_PUSHSW)) {
		/* WLAN LED GREEN */
		sxgpio_set_value(GPIOBIT_WLAN_GREEN, 1);
		sxgpio_set_value(GPIOBIT_WLAN_RED, 0);

		/* Wait to Relase Switch */
		for(i=0; i<=100; i++) {
			if(sxgpio_get_value(GPIOBIT_PUSHSW)) {
				/* Release a push switch */
				break;
			}
			udelay(100000); /* Wait 100ms */

			if(i == 20) {
				/* WLAN LED RED */
				sxgpio_set_value(GPIOBIT_WLAN_GREEN, 0);
				sxgpio_set_value(GPIOBIT_WLAN_RED, 1);
			}
			else if(i == 100) {
				/* WLAN LED ORANGE */
				sxgpio_set_value(GPIOBIT_WLAN_GREEN, 1);
				sxgpio_set_value(GPIOBIT_WLAN_RED, 1);
			}
		}
		if(i <= 20) {
			ret = BOOTMODE_MANUFACTURE;
		}
		else if(i <= 100) {
			ret = BOOTMODE_FACTORYDEFAULT;
		}
		else {
			env_set("update_flg", "1");
		}
	}

	if ((ret != BOOTMODE_MANUFACTURE) && (env_get_yesno("update_flg") == 1)) {
		ret = BOOTMODE_RECOVERY;
	}

	return ret;
}

static void do_memtest(void)
{
	int ret;

	/* There is a reserved memory area from 0x4000000 to 0x5400000
	 * for RT service or TEE region. So we avoid it and do memory test.
	 */
	ret = memtest((ulong*)0x00800000, (ulong*)0x03ffffff, 0x55555555);
	if(ret != 0) {
		memtest_error();
		for(;;);
	}

	ret = memtest((ulong*)0x00800000, (ulong*)0x03ffffff, 0xAAAAAAAA);
	if(ret != 0) {
		memtest_error();
		for(;;);
	}

	ret = memtest((ulong*)0x05400000, (ulong*)0x0f0fffff, 0x55555555);
	if(ret != 0) {
		memtest_error();
		for(;;);
	}

	ret = memtest((ulong*)0x05400000, (ulong*)0x0f0fffff, 0xAAAAAAAA);
	if(ret != 0) {
		memtest_error();
		for(;;);
	}
}

static void bootmode_normal(void)
{
	return;
}

static void bootmode_manufacture(void)
{
	do_memtest();
	env_set("bootcmd", "run bootcmd_net");
	return;
}

static void bootmode_factorydefault(void)
{
	return;
}

static void bootmode_recovery(void)
{
	env_set("bootcmd", "run bootcmd_recovery");
	return;
}

/* bootmode behavier */
static void behave_bootmode(void)
{
	int ret;

	ret = select_bootmode();
	switch(ret){
	case BOOTMODE_FACTORYDEFAULT:
		bootmode_factorydefault();
		sx_bootmode = "setdefault";
		break;
	case BOOTMODE_MANUFACTURE:
		bootmode_manufacture();
		sx_bootmode = "manufacture";
		break;
	case BOOTMODE_NORMAL:
		bootmode_normal();
		sx_bootmode = "normal";
		break;
	case BOOTMODE_RECOVERY:
		bootmode_recovery();
		sx_bootmode = "recovery";
		break;
	}

	printf("bootmode: %s\n", sx_bootmode);

	return;
}

void gpio_sys_reset(void)
{
	sxgpio_set_dir(GPIOBIT_SYSREST, 1, 0);

	udelay(1000);

	sxgpio_set_dir(GPIOBIT_SYSREST, 1, 1);
}

static void gpio_init(void)
{
	u32 reg;

	/* Set MPP from I2C to GPIO */
	/* U-Boot のどこかで MPPが GPIO から I2C に初期化されている
	 * 場所が特定できないので、ここで構成的に初期値の GPIO に
	 * 戻す
	 */
	reg = readl((void*)(GPIO1_BASE + 0x30));
	reg |= 0x400;
	writel(reg, (void*)(GPIO1_BASE + 0x30));

	/* Push Switch and LED */
	sxgpio_set_dir(GPIOBIT_PUSHSW, 0, 0);
	sxgpio_set_dir(GPIOBIT_POWER2_RED, 1, 0);
	sxgpio_set_dir(GPIOBIT_POWER2_GREEN, 1, 1);
	sxgpio_set_dir(GPIOBIT_WLAN_RED, 1, 0);
	sxgpio_set_dir(GPIOBIT_WLAN_GREEN, 1, 0);
	sxgpio_set_dir(GPIOBIT_STATUS_RED, 1, 0);
	sxgpio_set_dir(GPIOBIT_STATUS_GREEN, 1, 0);
	sxgpio_set_dir(GPIOBIT_RJ45_RED, 1, 0);
	sxgpio_set_dir(GPIOBIT_RJ45_GREEN, 1, 0);

	/* USB */
	sxgpio_set_dir(GPIOBIT_USB_EN1, 1, 0);
	sxgpio_set_dir(GPIOBIT_USB_OC1, 0, 0);
	sxgpio_set_dir(GPIOBIT_USB_EN2, 1, 0);
	sxgpio_set_dir(GPIOBIT_USB_OC2, 0, 0);

	/* Set Unused PIN to OUTPUT/LOW */
	/* < North Bridge GPIO1 Pin Output Low > */
	reg = readl((void*)(GPIO1_BASE + 0x18));
	reg &= ~0xF80C0410;
	writel(reg, (void*)(GPIO1_BASE + 0x18));
	/* < North Bridge GPIO1 Pin Output Enable Low > */
	reg = readl((void*)(GPIO1_BASE + 0x00));
	reg |= 0xF80C0410;
	writel(reg, (void*)(GPIO1_BASE + 0x00));

	/* < North Bridge GPIO1 Pin Output High > */
	reg = readl((void*)(GPIO1_BASE + 0x1C));
	reg &= ~0x0F;
	writel(reg, (void*)(GPIO1_BASE + 0x1C));
	/* < North Bridge GPIO1 Pin Output Enable High > */
	reg = readl((void*)(GPIO1_BASE + 0x04));
	reg |= 0x0F;
	writel(reg, (void*)(GPIO1_BASE + 0x04));

	/* < South Bridge GPIO2 Pin Output > */
	reg = readl((void*)(GPIO2_BASE + 0x18));
	reg &= ~0x3F000010;
	writel(reg, (void*)(GPIO2_BASE + 0x18));
	/* < South Bridge GPIO2 Pin Output Enable > */
	reg = readl((void*)(GPIO2_BASE + 0x00));
	reg |= 0x3F000010;
	writel(reg, (void*)(GPIO2_BASE + 0x00));

	return;
}

void sx_late_init(void)
{
	char *env;
	char bootargs[512];

	memset(bootargs, 0, sizeof(bootargs));

	env = env_get("bootargs");
	if (env != NULL) {
		snprintf(bootargs, sizeof(bootargs) -1,
			"%s ldrver=%s bootmode=%s",
			 env, SX_SILEX_VERSION, get_bootmode());
		bootargs[sizeof(bootargs) -1] = '\0';
		env_set("bootargs", bootargs);
	}

	return;
}

#if defined(CONFIG_MISC_INIT_R)
int misc_init_r(void)
{
	printf("silex version: %s\n", SX_SILEX_VERSION);
	gpio_init();
	behave_bootmode();
	return 0;

}
#endif
#endif /* CONFIG_SILEX */
