/**
 * @file
 *
 * silex standard GPIO control API
 *
 * Copyright (C) 2008 - 2010 silex technology, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#ifndef _SXGPIO_H
#define _SXGPIO_H 1

#include <linux/ioctl.h>
#include <asm/byteorder.h>

struct sxgpio_readreq;
struct sxgpio_ptnreq;
struct sxgpio_waitreq;
struct sxgpio_dirreq;

#define SXGPIO_GPIOS                    128 /* The number of GPIOs */

#define SXGPIO_MAJOR                    234    /* Get "UNASSIGNED" section in Documentation/devices.txt */
#define SXGPIO_MINOR                    1
#define SXGPIO_DEV                      MKDEV(SXGPIO_MAJOR, SXGPIO_MINOR)

#define SXGPIOMAGIC                     'p'
#define SXGPIO_IOCS_EXEPTN              _IOW(SXGPIOMAGIC, 0, struct sxgpio_ptnreq)   /* Execute blink a LED */
#define SXGPIO_IOCR_GET                 _IOR(SXGPIOMAGIC, 1, struct sxgpio_readreq)  /* Get status of a LED */
#define SXGPIO_IOCS_WAIT                _IOW(SXGPIOMAGIC, 2, struct sxgpio_waitreq)  /* Wait GPIO change */
#define SXGPIO_IOCS_DIR                 _IOW(SXGPIOMAGIC, 3, struct sxgpio_dirreq)   /* Set GPIO direction */

/* Continue types ----------------------------------------------------------- */
enum tag_SXGPIO_CTRL_CTYPE {
	SXGPIO_CTRL_CTYPE_NONE = 0,
	SXGPIO_CTRL_CTYPE_TIME,         /* Time         */
	SXGPIO_CTRL_CTYPE_COUNT,        /* Count        */
	SXGPIO_CTRL_CTYPE_PERMANENT     /* Parmanet     */
};

/* LED Control options ----------------------------------------------------- */
#define SXGPIO_OPT_LOCK                 0x0001        /* Locked LED State         */
#define SXGPIO_OPT_FORCE                0x0002        /* Forcibly set LED pattern */


/* LED Control pattern parameters  ----------------------------------------- */
struct sxgpio_ctrl_pattern {
	uint32_t interval;              /* exec blink Interval         */
	uint32_t pattern;               /* blink pattern               */
	uint32_t pattern_bitlen;        /* blink pattern bit length    */
	uint32_t cont_type;             /* Continue type (Times/msec)  */
	int32_t cont_volume;            /* Continue volume (Times/msec */
	uint32_t after_ctrl;            /* After control (ON/OFF)      */
};

struct sxgpio_ctrl_wait {
	uint32_t state;                 /* Wait state( 1 or 0) of gpio bit */
	uint32_t timeout;               /* Wait timeout */
};

struct sxgpio_waitreq {
	uint32_t num;                   /* GPIO bit number */
	struct sxgpio_ctrl_wait waiti;  /* Wait info */
};

struct sxgpio_ptnreq {
	uint32_t num;                   /* GPIO Number */
	uint32_t option;                /* Lock flag */
	struct sxgpio_ctrl_pattern pattern;
};

struct sxgpio_readreq {
	uint32_t num;
	uint32_t *status;
};

struct sxgpio_dirreq {
	uint32_t num;                   /* GPIO Number */
	uint32_t dir;                   /* GPIO Direction */
	uint32_t bit;                   /* Default bit */
};


#ifdef __KERNEL__

enum {
	SXGPIO_INTSTATE_NONE = 0,
	SXGPIO_INITSTATE_WAITING, 
	SXGPIO_INTSTATE_PENDING,
};

struct sxgpio_ctrl {
	int num;                        /* gpio number */
	int locked;                     /* GPIO locked(1)/unlocked(0)   */
	struct timer_list *timer;       /* GPIO Blinking timer          */
	int shift_cnt;                  /* Shift Pattern count          */
	int status;                     /* Current gpio status          */
	int intstat;                    /* Interrupt status             */
	wait_queue_head_t intwaitq;     /* Wait event of GPIO interrupt */
	struct sxgpio_ctrl_pattern ptn; /* Current pattern              */

#ifdef CONFIG_SXLED_PROCFS
	struct proc_dir_entry *dirp;    /* Proc fs information          */
#endif /* CONFIG_SXLED_PROCFS */
};


int sxgpio_set(unsigned gpio, int value, u32 option);
int sxgpio_ctl_pattern(unsigned gpio, struct sxgpio_ctrl_pattern *ptn, u32 option);
int sxgpio_get(unsigned gpio);
int sxgpio_wait(unsigned gpio, struct sxgpio_ctrl_wait *waitip);

int sxgpio_ethernet_link_event(int up, int speed);
int sxgpio_ethernet_recv_event(void);

#endif /* __KERNEL__ */
#endif /* _SXGPIO_H */
