/**
 * @file
 *
 * procfs interface for silex
 *
 * Copyright (C) 2008 silex technology, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/types.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/timer.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/workqueue.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>

#include <linux/sxproc.h>

static struct proc_dir_entry *silex_proc_dir;

/* GPIO の制御をカーネルにするか、ユーザランドにするかのスイッチ
 * 0:User 1:Kernel
 */
static int g_gpio_kernel_ctrl;

static int gpio_proc_read(struct seq_file *m, void *v)
{
	int *ctrl = m->private;

	seq_printf(m, "%d\n", *ctrl);

	return 0;
}

static ssize_t gpio_proc_write(struct file *file, const char __user *buffer,
				size_t count, loff_t *ppos)
{
	int *ctrl = PDE_DATA(file_inode(file));
	char c;
	int ret;

	ret = get_user(c, buffer);
	if (ret != 0)
		return -EFAULT;

	switch (c) {
	case '0':
		*ctrl = 0;
		break;
	case '1':
		*ctrl = 1;
		break;
	default:
		printk(KERN_ERR "%s:%d invalid data\n", __func__, __LINE__);
		break;
	}

	return count;
}

static int gpio_proc_open(struct inode *inode, struct file *file)
{
	return single_open(file, gpio_proc_read, PDE_DATA(inode));
}

static const struct file_operations gpio_proc_fops = {
	.owner		= THIS_MODULE,
	.open		= gpio_proc_open,
	.read		= seq_read,
	.llseek		= seq_lseek,
	.release	= single_release,
	.write		= gpio_proc_write,
};

int sxproc_gpio_kernel_ctl(void)
{
	return g_gpio_kernel_ctrl;
}
EXPORT_SYMBOL(sxproc_gpio_kernel_ctl);

static int __init sxproc_init(void)
{
	struct proc_dir_entry *entry;
	int ret;

	/* 親ディレクトリの作成 */
	silex_proc_dir = proc_mkdir(SILEX_PROCFS_DIRECTORY, NULL);
	if (!silex_proc_dir) {
		printk(KERN_ERR "%s:%d proc_mkdir() error\n", __func__, __LINE__);
		ret = -EINVAL;
		goto error;
	}

	/* 親ディレクトリ内にファイルを作成 */
	g_gpio_kernel_ctrl = 1; /* Enable Kernel Control */
	entry = proc_create_data("gpio_kernel_ctrl", 0660,
			silex_proc_dir, &gpio_proc_fops, &g_gpio_kernel_ctrl);
	if (!entry) {
		printk(KERN_ERR "%s:%d proc_createy() error\n", __func__, __LINE__);
		ret = -EINVAL;
		goto error;
	}

	ret = 0;
error:
	return ret;
}

static void __exit sxproc_cleanup(void)
{
	/* [TODO] Implement cleanup this module */
	printk(KERN_INFO "%s:%d cleanup nothing to do\n", __func__, __LINE__);
}

module_init(sxproc_init);
module_exit(sxproc_cleanup);
