/**
 * @file
 *
 * silex standard GPIO control API
 *
 * Copyright (C) 2008 -2010 silex technology, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/types.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/kernel.h>
#include <linux/timer.h>
#include <linux/spinlock.h>
#include <linux/jiffies.h>
#include <linux/module.h>
#include <linux/uaccess.h>
#include <linux/cdev.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/wait.h>
#include <linux/device.h>
#include <linux/irq.h>
#include <linux/sched.h>

#include <linux/sxgpio.h>
#include <linux/sxproc.h>

#define MODULE_NAME     "sxgpio"
#define CYCLE           (1000 / HZ)             /* 1 Tick time(msec) */

static spinlock_t sxgpio_spnlock;

int sxgpio_ctrl_items;
struct sxgpio_ctrl *sxgpio_ctrl_table;

static struct cdev sxgpio_cdev;

#define GPP_GROUP(gpio) gpio/32
#define GPP_ID(gpio)	gpio%32
#define GPP_BIT(gpio)	0x1 << GPP_ID(gpio)

/**
 * GPIO lock.
 *
 * @param Target GPIO
 *
 */
static void sxgpio_lock(struct sxgpio_ctrl *gpioc)
{
	if(gpioc != NULL)	gpioc->locked = 1;
}

/**
 * GPIO unlock.
 *
 * @param Target GPIO
 *
 */
static void sxgpio_unlock(struct sxgpio_ctrl *gpioc)
{
	if(gpioc != NULL)	gpioc->locked = 0;
}

/**
 * Set GPIO (depend on products)
 *
 * @param Target gpio
 * @parm on_off
 *
 */
static void sxgpio_set_value(unsigned gpio, int value)
{
	gpio_set_value(gpio, value);
}

/**
 * Get GPIO status (depend on products)
 *
 * @param Target gpio
 *
 * @return gpio status
 */
static int sxgpio_get_value(unsigned gpio)
{
	/* ARMADA 16x SDK return bit mask gpio_get_value()
	 * sxgpio_get_valu shoud return bit status (0/1),
	 * so convert the data format
	 */
	return gpio_get_value(gpio) ? 1 : 0 ;
}

/**
 * Finish the GPIO pattern control.
 *
 * @param gpio Target gpio control structure.
 *
 * @return 0 success
 * @return <0 error
 */
static int sxgpio_finish_pattern(struct sxgpio_ctrl *gpioc)
{
	/* After control process start! */
	sxgpio_unlock(gpioc);

	return 0;
}

/**
 * sxgpio Timer function.
 * This function called by timer.
 *
 * @parm arg  timer argument (Target sxgpio info)
 *
 */
static void sxgpio_timer_fn(unsigned long arg)
{
	int value;
	int shift_cnt;
	unsigned long j = jiffies;
	struct sxgpio_ctrl *gpioc = (struct sxgpio_ctrl*)arg;
	struct sxgpio_ctrl_pattern *ptn = &gpioc->ptn;
	unsigned long flags;

	spin_lock_irqsave(&sxgpio_spnlock, flags);

	/* Blink process */
	shift_cnt = gpioc->shift_cnt;
	if (shift_cnt >= ptn->pattern_bitlen)
		shift_cnt = 0;

	gpioc->shift_cnt = shift_cnt + 1;

	value = ((ptn->pattern >> shift_cnt) & 0x1);
	sxgpio_set_value(gpioc->num, value);

	/* cut down cont_volume according to cont_type */
	switch(ptn->cont_type) {
	case SXGPIO_CTRL_CTYPE_TIME:
		ptn->cont_volume -= ptn->interval;
		break;

	case SXGPIO_CTRL_CTYPE_COUNT:
		if(gpioc->shift_cnt >= ptn->pattern_bitlen) {
			/* revolve pattern */
			ptn->cont_volume--;
		}
		break;
	}

	/* Check finish state */
	if(ptn->cont_volume <= 0) {
		/* Stop timer and set value */
		sxgpio_finish_pattern(gpioc);

		sxgpio_set_value(gpioc->num, ptn->after_ctrl);

		spin_unlock_irqrestore(&sxgpio_spnlock, flags);
		return;
	}

	/* continue pattern */
	mod_timer(gpioc->timer, j + ptn->interval);
	spin_unlock_irqrestore(&sxgpio_spnlock, flags);
}

/**
 * Execute blinking according to control pattern.
 *
 * This function startup timer.
 *
 * @parm gpioc Target gpio control structure
 *
 * @return 0  Success
 * @return <0 Error
 */
static int sxgpio_exec_pattern(struct sxgpio_ctrl *gpioc)
{
	gpioc->shift_cnt = 0;

	/* Immediate start up Timer */
	mod_timer(gpioc->timer, jiffies);

	return 0;
}

/**
 * Set value to gpio.
 *
 * @parm gpio Target gpio pin number
 * @parm value (1 or 0)
 * @parm option Mode
 *
 * @return 0  Success
 * @return <0 Error
 */
int sxgpio_set(unsigned gpio, int value, u32 option)
{
	struct sxgpio_ctrl *gpioc;
	struct sxgpio_ctrl_pattern *ptn;
	unsigned long flags;

	if(gpio < 0 || gpio >= sxgpio_ctrl_items)
		return -ENODEV;

	gpioc = &sxgpio_ctrl_table[gpio];
	ptn = &gpioc->ptn;

	spin_lock_irqsave(&sxgpio_spnlock, flags);

	/* if the gpio had already locked ;-( */
	if(gpioc->locked && !(option & SXGPIO_OPT_FORCE)) {
		if (ptn->cont_type != SXGPIO_CTRL_CTYPE_PERMANENT) {
			/* Illegal specification!! */
			/* Overwrite Next pattern setting */
			ptn->after_ctrl = value;
			spin_unlock_irqrestore(&sxgpio_spnlock, flags);
			return -1;
		}
		else {
			spin_unlock_irqrestore(&sxgpio_spnlock, flags);
			return -EBUSY;
		}
	}

	sxgpio_finish_pattern(gpioc);

	ptn->interval        = 0;
	ptn->pattern         = (value != 0) ? 1 : 0;
	ptn->pattern_bitlen  = 1;
	ptn->cont_type       = SXGPIO_CTRL_CTYPE_PERMANENT;
	ptn->cont_volume     = 0x0FFFFFFF; /* Dummy Data */
	ptn->after_ctrl      = value;

	/* Change SXGPIO Turn on / off */
	sxgpio_set_value(gpioc->num, value);

	if(option & SXGPIO_OPT_LOCK)
		sxgpio_lock(gpioc);

	spin_unlock_irqrestore(&sxgpio_spnlock, flags);

	return 0;
}
EXPORT_SYMBOL(sxgpio_set);

#define CONFIG_DS700AC
#ifdef CONFIG_DS700AC
#define GPIO_LED_RJ45_GREEN             2 /* 1000BASE-T */
#define GPIO_LED_RJ45_ORANGE            3 /* 10/100BASE-TX */

/* Turn on for 100msec */
struct sxgpio_ctrl_pattern ptn_eth = {
    .interval =100,
    .pattern = 0b10,
    .pattern_bitlen = 2,
    .cont_type = SXGPIO_CTRL_CTYPE_COUNT,
    .cont_volume = 2,
    .after_ctrl = 1,
};

static int g_speed = 0;
int sxgpio_ethernet_link_event(int up, int speed)
{
	if (!sxproc_gpio_kernel_ctl())
		return 0;

	sxgpio_set(GPIO_LED_RJ45_GREEN, 0, SXGPIO_OPT_FORCE);
	sxgpio_set(GPIO_LED_RJ45_ORANGE, 0, SXGPIO_OPT_FORCE);

	if (up) {
		if (speed == 1000) {
			sxgpio_set(GPIO_LED_RJ45_GREEN, 1, SXGPIO_OPT_FORCE);
		}
		else {
			sxgpio_set(GPIO_LED_RJ45_ORANGE, 1, SXGPIO_OPT_FORCE);
		}
		g_speed = speed;
		return 0;
	}
	else {
		/* Nothing to do */
		g_speed = 0;
		return 0;
	}
}

int sxgpio_ethernet_recv_event(void)
{
	if (!sxproc_gpio_kernel_ctl())
		return 0;

	if (g_speed == 0) {
		printk(KERN_ERR "Recv a packet from ethernet during linkdown\n");
		return 0;
	}

	if (g_speed == 1000) {
		sxgpio_ctl_pattern(GPIO_LED_RJ45_GREEN, &ptn_eth,
				SXGPIO_OPT_LOCK);
	}
	else {
		sxgpio_ctl_pattern(GPIO_LED_RJ45_ORANGE, &ptn_eth,
				SXGPIO_OPT_LOCK);
	}

	return 0;
}
#endif /* CONFIG_DS700AC */

/**
 * Control GPIO turn on/off according to control pattern.
 *
 * @parm gpio Target gpio number
 * @parm ptn Control pattern info
 * @parm option Mode
 *
 * @return 0  Success
 * @return <0 Error
 */
int sxgpio_ctl_pattern(unsigned gpio, struct sxgpio_ctrl_pattern *ptn, u32 option)
{
	struct sxgpio_ctrl *gpioc;
	u32 interval, volume;
	unsigned long flags;

	if(gpio < 0 || gpio >= sxgpio_ctrl_items)
		return -ENODEV;

	gpioc = &sxgpio_ctrl_table[gpio];

	/* MAX Check */
	if((ptn->pattern_bitlen == 0) ||
			 (ptn->pattern_bitlen > sizeof(ptn->pattern) * 8)) {
		return -EINVAL;
	}

	if((ptn->after_ctrl != 0) && (ptn->after_ctrl != 1)) {
		return -EINVAL;
	}

	/* interval (msec) -> jiffies */
	interval = ptn->interval / CYCLE;
	if(interval == 0)
		return -EINVAL;

	/* Change cont_volume according to continue type */
	switch(ptn->cont_type) {
	case SXGPIO_CTRL_CTYPE_TIME:
		volume = ptn->cont_volume / CYCLE;
		break;

	case SXGPIO_CTRL_CTYPE_COUNT:
		volume = ptn->cont_volume;
		break;

	case SXGPIO_CTRL_CTYPE_PERMANENT:
		volume = 0x0FFFFFFF; /* Dummy data */
		break;

	default:
		return -EINVAL;
	}

	spin_lock_irqsave(&sxgpio_spnlock, flags);

	/* if GPIO had already locked ;-( */
	if((gpioc->locked) && !(option & SXGPIO_OPT_FORCE)) {
		spin_unlock_irqrestore(&sxgpio_spnlock, flags);
		return -EBUSY;
	}

	sxgpio_finish_pattern(gpioc);

	/* Current pattern setting */
	gpioc->ptn.interval = interval;
	gpioc->ptn.pattern = ptn->pattern;
	gpioc->ptn.pattern_bitlen = ptn->pattern_bitlen;
	gpioc->ptn.cont_type = ptn->cont_type;
	gpioc->ptn.cont_volume = volume;
	gpioc->ptn.after_ctrl = ptn->after_ctrl;

	/* !!!! Execute !!!! */
	sxgpio_exec_pattern(gpioc);

	/* If this opration request LOCKing gpio, reserved gpio resource. */
	if(option & SXGPIO_OPT_LOCK)
		sxgpio_lock(gpioc);

	spin_unlock_irqrestore(&sxgpio_spnlock, flags);

	return 0;
}
EXPORT_SYMBOL(sxgpio_ctl_pattern);

/**
 * Get gpio status.
 *
 * @parm gpio Target gpio number
 *
 * @return 0 Turn on
 * @return 1 Turn off
 */
int sxgpio_get(unsigned gpio)
{
	int ret;
	unsigned long flags;

	if(gpio < 0 || gpio >= sxgpio_ctrl_items)
		return -ENODEV;

	spin_lock_irqsave(&sxgpio_spnlock, flags);
	ret = sxgpio_get_value(gpio);
	spin_unlock_irqrestore(&sxgpio_spnlock, flags);

	return ret;
}
EXPORT_SYMBOL(sxgpio_get);

static irqreturn_t sxgpio_interrupt(int irq, void *data)
{
	struct sxgpio_ctrl *gpioc;
	int value;

	if (data == NULL) {
		printk(KERN_ERR MODULE_NAME
			":Illegal gpio interrupt received\n");
		return IRQ_NONE;
	}

	gpioc = (struct sxgpio_ctrl*)data;
	if(gpioc->intstat != SXGPIO_INITSTATE_WAITING) {
		return IRQ_NONE;
	}

	/* Change IRQ type to stop the interrupt loop */
	value = gpio_get_value(gpioc->num) ? 1 : 0;
	if (value)
		irq_set_irq_type(irq, IRQ_TYPE_LEVEL_LOW);
	else
		irq_set_irq_type(irq, IRQ_TYPE_LEVEL_HIGH);

	/* Receive IRQ success */
	gpioc->intstat = SXGPIO_INTSTATE_PENDING;
	wake_up_interruptible(&gpioc->intwaitq);

	return IRQ_HANDLED;
}

int sxgpio_wait(unsigned gpio, struct sxgpio_ctrl_wait *waitip)
{
	int ret;
	struct sxgpio_ctrl *gpioc;
	char bitname[8];
	unsigned long flags;
	unsigned long irqflags;

	if(gpio < 0 || gpio >= sxgpio_ctrl_items)
		return -ENODEV;

	if(waitip->state != 0 && waitip->state != 1)
		return -EINVAL;

	snprintf(bitname, sizeof(bitname), "gpio%d", gpio);
	ret = gpio_request_one(gpio, GPIOF_IN, bitname);
	if (ret != 0) {
		return ret;
	}

	spin_lock_irqsave(&sxgpio_spnlock, flags);
	if(waitip->state == 0)
		irqflags = IRQ_TYPE_LEVEL_LOW;
	else
		irqflags = IRQ_TYPE_LEVEL_HIGH;

	gpioc = &sxgpio_ctrl_table[gpio];
	gpioc->intstat = SXGPIO_INITSTATE_WAITING;
	spin_unlock_irqrestore(&sxgpio_spnlock, flags);

	ret = request_irq(gpio_to_irq(gpio), &sxgpio_interrupt,
				irqflags, bitname, gpioc);
	if(ret < 0) {
		gpio_free(gpio);
		return ret;
	}

	if(waitip->timeout == 0) {
		ret = wait_event_interruptible(
				gpioc->intwaitq,
				gpioc->intstat != SXGPIO_INITSTATE_WAITING);
	}
	else {
		/* wait_event_interruptible_timeout() return rest of the time(jiffy).
		 * If the time out error occur, jiffy is zero.
		 */
		ret = wait_event_interruptible_timeout(
				gpioc->intwaitq,
				gpioc->intstat != SXGPIO_INITSTATE_WAITING,
				msecs_to_jiffies(waitip->timeout));
		if(ret == 0)
			ret = -ETIME;
		else if(ret >= 0)
			ret = 0;
	}
	free_irq(gpio_to_irq(gpio), gpioc);
	gpio_free(gpio);

	spin_lock_irqsave(&sxgpio_spnlock, flags);
	gpioc->intstat = SXGPIO_INTSTATE_NONE;
	spin_unlock_irqrestore(&sxgpio_spnlock, flags);

	return ret;
}
EXPORT_SYMBOL(sxgpio_wait);

int sxgpio_set_dir(int gpio, int dir, int bit)
{
	struct sxgpio_ctrl *gpioc;
	unsigned long flags;
	int ret;

	if(gpio < 0 || gpio >= sxgpio_ctrl_items)
		return -ENODEV;

	gpioc = &sxgpio_ctrl_table[gpio];

	spin_lock_irqsave(&sxgpio_spnlock, flags);

	ret = gpio_request(gpio, "sxgpio");
	if (ret < 0) {
		printk(KERN_INFO MODULE_NAME ":gpio request error\n");
		goto end;
	}

	sxgpio_finish_pattern(gpioc);

	if (dir) {
		/* OUT */
		ret = gpio_direction_output(gpio, bit);
	}
	else {
		/* IN */
		ret = gpio_direction_input(gpio);
	}
	if (ret < 0) {
		printk(KERN_INFO MODULE_NAME ":gpio direction %s error\n",
			dir?"output":"input");
	}
	gpio_free(gpio);

end:
	spin_unlock_irqrestore(&sxgpio_spnlock, flags);

	return ret;
}

/* -------------------------------------------------------------------
 * sxgpio charcter device driver
 * ------------------------------------------------------------------- */
static int sxgpio_ioctl(struct inode *inode, struct file *file,
			 		unsigned int cmd, unsigned long arg)
{
	int ret = 0;

	switch(cmd) {
	case SXGPIO_IOCS_EXEPTN:
	{
		struct sxgpio_ptnreq req;

		ret = copy_from_user(&req, (void*)arg, sizeof(struct sxgpio_ptnreq));
		if(ret){
			ret = -EFAULT;
			break;
		}

		if(req.pattern.pattern_bitlen == 1 &&
			req.pattern.cont_type == SXGPIO_CTRL_CTYPE_PERMANENT) {
			switch(req.pattern.pattern) {
			case 0:
			case 1:
				ret = sxgpio_set(req.num,
						req.pattern.pattern, req.option);
				break;
			default:
				ret = -EINVAL;
			}
		}
		/* Pattern execution */
		else {
			ret = sxgpio_ctl_pattern(req.num, &req.pattern,
								req.option);
		}
		break;
	}
	case SXGPIO_IOCR_GET:
	{
		struct sxgpio_readreq req;
		int val;

		ret = copy_from_user(&req, (void*)arg, sizeof(struct sxgpio_readreq));
		if(ret){
			ret = -EFAULT;
			break;
		}
		val = sxgpio_get(req.num);
		if(val < 0) {
			ret = val;
			break;
		}
		ret = copy_to_user(req.status, &val, sizeof(uint32_t));
		if(ret) {
			ret = -EFAULT;
			break;
		}
		ret = 0;
		break;
	}

	case SXGPIO_IOCS_WAIT:
	{
		struct sxgpio_waitreq req;
		ret = copy_from_user(&req, (void*)arg, sizeof(struct sxgpio_waitreq));
		if(ret) {
			ret = -EFAULT;
			break;
		}
		ret = sxgpio_wait(req.num, &req.waiti);
		break;
	}
	case SXGPIO_IOCS_DIR:
	{
		struct sxgpio_dirreq req;

		ret = copy_from_user(&req, (void*)arg, sizeof(struct sxgpio_dirreq));
		if(ret) {
			ret = -EFAULT;
			break;
		}
		ret = sxgpio_set_dir(req.num, req.dir, req.bit);
		break;
	}

	default:
		ret = -ENOIOCTLCMD;
		break;
	}

	return ret;
}

static long sxgpio_compioctl(struct file *file, unsigned int cmd,
							unsigned long arg)
{
	return (long)sxgpio_ioctl(NULL, file, cmd, arg);
}

static ssize_t sxgpio_read(struct file *file, char __user * buffer,
				size_t count, loff_t * offset)
{
	return 0;
}

static ssize_t sxgpio_write(struct file *file, const char __user * buffer,
				size_t count, loff_t * offset_is_ignored)
{
	return 0;
}

static int sxgpio_open(struct inode *inode, struct file *file)
{
	return 0;
}

static int sxgpio_release(struct inode *inode, struct file *file)
{
	return 0;
}

static const struct file_operations sxgpio_fops = {
	.owner = THIS_MODULE,
	.read = sxgpio_read,
	.write = sxgpio_write,
	.unlocked_ioctl = sxgpio_compioctl,
	.compat_ioctl = sxgpio_compioctl,
	.open = sxgpio_open,
	.release = sxgpio_release,
};

static char *sxgpio_devnode(struct device *dev, umode_t *mode)
{
	*mode = 0666;
	return NULL;
}

/**
 * Initial sxgpio control.
 *
 *
 * @return 0 success
 * @return <0 error
 */
int __init sxgpio_init(void)
{
	int i, ret = 0;
	struct sxgpio_ctrl *gpioc;
	static struct class *sxgpio_class;

	sxgpio_ctrl_items = SXGPIO_GPIOS;

	sxgpio_ctrl_table = kmalloc((sizeof(struct sxgpio_ctrl) * sxgpio_ctrl_items),
								GFP_KERNEL);
	if(sxgpio_ctrl_table == NULL) {
		ret = -ENOMEM;
		goto error;
	}

	gpioc = sxgpio_ctrl_table;
	for(i = 0; i < sxgpio_ctrl_items; i++, gpioc++) {
		gpioc->num = i;

		gpioc->timer = kmalloc(sizeof(struct timer_list), GFP_KERNEL);
		if(gpioc->timer == NULL) {
			printk(KERN_ERR MODULE_NAME ":No mem in %s()\n", __func__);
			ret = -ENOMEM;
			goto error;
		}

		init_timer(gpioc->timer);
		gpioc->timer->data = (unsigned long)gpioc;
		gpioc->timer->function = sxgpio_timer_fn;

		/* Initialize gpio parameter */
		memset(&gpioc->ptn, 0, sizeof(struct sxgpio_ctrl_pattern));

		/* Initialize event for GPIO interrupt */
		gpioc->intstat = SXGPIO_INTSTATE_NONE;
		init_waitqueue_head(&gpioc->intwaitq);

		sxgpio_unlock(gpioc);
	}

	/* lock for sxgpio_ctrl_table */
	spin_lock_init(&sxgpio_spnlock);

	/* Init cdev structure */
	cdev_init(&sxgpio_cdev, &sxgpio_fops);
	sxgpio_cdev.owner = THIS_MODULE;
	kobject_set_name(&sxgpio_cdev.kobj, MODULE_NAME);

	ret = cdev_add(&sxgpio_cdev, SXGPIO_DEV, 1);
	if (ret) {
		printk(KERN_ERR MODULE_NAME ":failed to register minor device block\n");
		goto error;
	}

	/* Device create event to udev */
	sxgpio_class = class_create(THIS_MODULE, MODULE_NAME);
	if (IS_ERR(sxgpio_class)) {
		printk(KERN_ERR MODULE_NAME ":failed to create class\n");
		ret = -EINVAL;
		goto error;
	}
	sxgpio_class->devnode = sxgpio_devnode;
	device_create(sxgpio_class, NULL, SXGPIO_DEV, NULL, MODULE_NAME);

	printk(KERN_INFO MODULE_NAME ":add %d gpios success\n",
						sxgpio_ctrl_items);

	return 0;

error:

	if(sxgpio_ctrl_table != NULL) {
		for(i = 0; i < sxgpio_ctrl_items; i++) {
			gpioc = &sxgpio_ctrl_table[i];
			if(gpioc->timer->function != NULL)
				del_timer_sync(gpioc->timer);
			if(gpioc->timer != NULL)
				kfree(gpioc->timer);
		}
		kfree(sxgpio_ctrl_table);
	}
	cdev_del(&sxgpio_cdev);

	return ret;
}
EXPORT_SYMBOL(sxgpio_init);
module_init(sxgpio_init);

/**
 * Cleanup sxgpio control.
 *
 */
void sxgpio_cleanup(void)
{
	int i;
	struct sxgpio_ctrl *gpioc;

	if(sxgpio_ctrl_table != NULL) {
		for(i = 0; i < sxgpio_ctrl_items; i++) {
			gpioc = &sxgpio_ctrl_table[i];
			if(gpioc->timer->function != NULL)
				del_timer_sync(gpioc->timer);
			if(gpioc->timer != NULL)
				kfree(gpioc->timer);
		}
		kfree(sxgpio_ctrl_table);
	}
	cdev_del(&sxgpio_cdev);
}
EXPORT_SYMBOL(sxgpio_cleanup);
