/* Linux kernel driver for resistive touch panel on some Quatro reference boards
 *
 * Quasar resistive touch panel kernel driver
 * 
 * Copyright (c) 2014, 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2022/01/06 10:11:28 $
//  $Change: 56812 $
//
// =========================================================
#include <linux/io.h>
#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/input/mt.h>
#include <linux/irq.h>
#include <asm/unaligned.h>
#include "core.h"
#include "bsp.h"
#include "util.h"

#include <linux/delay.h>

static int wdt8913a_get_touch_cnt(struct wdt8913a_dev *wdev)
{
	struct i2c_client *client;
	u8  cmd;
	u8  data[3];
	int rv;

	/* 
	 * Command for retrieving touch information
	 */
	cmd    = 0x90; 
	client = wdev->client;

	rv = i2c_master_send(client, &cmd, sizeof(cmd));
	if (rv != sizeof(cmd)) 
	{
		dev_err(&client->dev, 
				"%s: fail to send command to device, err %d\n", __func__, rv);
		return rv;
	}

	rv = i2c_master_recv(client, data, sizeof(data));
	if (rv == sizeof(cmd)) 
	{
		dev_err(&client->dev, 
				"%s: fail to retrive info from device, err %d\n", __func__, rv);
		return rv;
	}

	/*
	 * data[0] contains the number of touch points 
	 */
	return data[0];
}

static void wdt8913a_in_report(struct wdt8913a_dev *wdev, u8 *data)
{
	int flag;
	int id;
	int x;
	int y;

	flag = data[0];

	/*
	 * Normally, the `flag' should not be zero. But somehow if it happens to 
	 * be zero, we should return it immediately, or the calculation of id will
	 * be larger than `data' size, which would cause unexpected memory access.
	 */
	if (flag == 0)
	{
		dev_err(&wdev->client->dev,
				"%s: invalid flag value, ignored\n", __func__);
		return;
	}

	id = (flag >> 3) - 1;

	/*
	 * Do not trust the data provided by device, or it will corrupt our
	 * precious system.
	 */
	if (!WDT8913A_VALID_TOUCH_ID(id))
	{
		dev_err(&wdev->client->dev,
				"%s: invalided touch ID %d, ignored\n", __func__, id);
		return;
	}

	flag = flag & 0x01;
	x = get_unaligned_le16(&(data[1]));
	y = get_unaligned_le16(&(data[3]));

	if (flag == 0 && wdev->flags[id])
	{
		input_mt_slot(wdev->input, id);
		input_mt_report_slot_state(wdev->input, MT_TOOL_FINGER, false);
		wdev->flags[id] = false;
	}
	else
	{
		input_mt_slot(wdev->input, id);
		input_mt_report_slot_state(wdev->input, MT_TOOL_FINGER, true);
		input_report_abs(wdev->input, ABS_MT_POSITION_X, x);
		input_report_abs(wdev->input, ABS_MT_POSITION_Y, y);
		wdev->flags[id] = true;
	}
}

#ifdef DEBUG
static void wdt8913a_dump_packets(
		struct device *dev,
		const u8 n_touches, const u8 n_attr_pt, u8 *data)
{
	int i;
	u8 offset = 0;
	char buf[80] = {0};

	for (i=0; i<n_touches*n_attr_pt; i++)
	{
		offset = strlen(buf);
		scnprintf(buf+offset, sizeof(buf)-offset, "%x ", data[i]);
	}

	dev_dbg(dev, "%s: dump raw data: <<%s>>\n", __func__, buf);
}
#endif /* DEBUG */

static int wdt8913a_in_report_touches(struct wdt8913a_dev *wdev, u8 n_touches)
{
	struct i2c_client *client;
	u8    cmd;
	u8    i;
	int   rv;

	/*
	 * Each point has 5 attributes 
	 */
	const u8 n_attr_pt = 5;
	u8 data[n_attr_pt*WDT8913A_MAX_TOUCH_POINTS];

	/* 
	 * Command for retrieving touches
	 */
	cmd    = 0x91;
	client = wdev->client;

	rv = i2c_master_send(client, &cmd, sizeof(cmd));
	if (rv != sizeof(cmd))
	{
		dev_err(&client->dev,
				"%s: fail to send command to device, err %d\n", __func__, rv);
		return rv;
	}

	rv = i2c_master_recv(client, data, n_touches*n_attr_pt);
	if (rv != n_touches*n_attr_pt)
	{
		dev_err(&client->dev, 
				"%s: fail to retrive info from device, err %d\n", __func__, rv);
		return rv;
	}

#ifdef DEBUG
	wdt8913a_dump_packets(&client->dev, n_touches, n_attr_pt, data);
#endif

	/*
	 * Process each touch info and report input 
	 */
	for(i=0; i<n_touches; i++)
	{
		wdt8913a_in_report(wdev, data+i*n_attr_pt);
	}

	input_mt_sync_frame(wdev->input);
	input_sync(wdev->input);

	return 0;
}

irqreturn_t __wdt8913a_isr(struct wdt8913a_dev *wdev)
{
	struct i2c_client *client = wdev->client;
	u8  n_touches;
	int rv;

	rv = wdt8913a_get_touch_cnt(wdev);
	if (rv > 0)
	{
		/*
		 * Do not trust device, or it may crash our system.
		 * Need to check data integrity
		 */
		n_touches = rv;
		if (n_touches <= WDT8913A_MAX_TOUCH_POINTS)
		{
			dev_dbg(&client->dev,
					"%s: touch count: %d\n", __func__, n_touches);

			wdt8913a_in_report_touches(wdev, n_touches);
		}
		else
		{
			dev_err(&client->dev,
					"%s: too many touch points %d\n", __func__, n_touches);
		}
	}
	else
	{
		dev_err(&client->dev, 
				"%s: fail to get touch count, err %d\n", __func__, rv);
	}

	return IRQ_HANDLED;
}

irqreturn_t wdt8913a_isr(int this_irq, void *dev_id)
{
	struct wdt8913a_dev *wdev = dev_id;

	if (!wdt8913a_int_chk_rst(wdev))
	{
		return IRQ_HANDLED;
	}

	return __wdt8913a_isr(wdev);
}

static int wdt8913a_input_init(const char *name, 
		struct device *parent, struct input_dev **out_input)
{
	struct input_dev *input;

	input = devm_input_allocate_device(parent);
	if (!input)
	{
		return -EINVAL;
	}
	input->name       = name;
	input->id.bustype = BUS_I2C;
	input->dev.parent = parent;

	input_set_abs_params(input, ABS_MT_POSITION_X, 0, 0x7FFF, 0, 0);
	input_set_abs_params(input, ABS_MT_POSITION_Y, 0, 0x7FFF, 0, 0);
	input_set_abs_params(input, ABS_MT_TRACKING_ID, 0, 9, 0, 0);

	set_bit(ABS_MT_POSITION_X, input->absbit);
	set_bit(ABS_MT_POSITION_Y, input->absbit);
	set_bit(EV_ABS, input->evbit);
	set_bit(EV_SYN,	input->evbit);

	input_mt_init_slots(input, 
			WDT8913A_MAX_TOUCH_POINTS, INPUT_MT_DIRECT | INPUT_MT_DROP_UNUSED);

	input_register_device(input);

	*out_input = input;
	return 0;
}

static int wdt8913a_dev_init(struct i2c_client *client, 
		struct input_dev *input, struct wdt8913a_dev **out_wdev)
{
	struct wdt8913a_dev *wdev;

	wdev = devm_kzalloc(&client->dev, sizeof(*wdev), GFP_KERNEL);
	if (!wdev)
	{
		return -EINVAL;
	}
	memset(wdev->flags, 0, sizeof(wdev->flags));
	wdev->client = client;
	wdev->input  = input;

	input_set_drvdata(input, wdev);

	*out_wdev = wdev;
	return 0;
}

static int wdt8913a_probe(struct i2c_client *client,
		const struct i2c_device_id *id)
{
	int    ret;
	struct wdt8913a_dev *wdev;
	struct input_dev    *input;

	dev_dbg(&client->dev, 
			"%s: i2c slave addr 0x%02x\n", __func__, client->addr);

	ret = i2c_check_functionality(client->adapter, I2C_FUNC_I2C);
	if (!ret)
	{
		dev_err(&client->dev, 
				"%s: I2C_FUNC_I2C is not supported\n", __func__);
		return ret;
	}

	ret = wdt8913a_input_init(client->name, &client->dev, &input);
	if (ret < 0)
	{
		dev_err(&client->dev, 
				"%s: fail to allocate input device, error %d\n", __func__, ret);
		return ret;
	}

	ret = wdt8913a_dev_init(client, input, &wdev);
	if (ret < 0)
	{
		dev_err(&client->dev, 
				"%s: fail to allocate wdt8913a device, error %d\n", __func__, ret);
		return ret;
	}
	i2c_set_clientdata(client, wdev);

	ret = wdt8913a_setup_board(wdev);
	if (ret < 0)
	{
		dev_err(&client->dev, 
				"%s: fail to setup board, error %d\n", __func__, ret);
		return ret;
	}

	ret = wdt8913a_setup_irq(wdev);
	if (ret < 0)
	{
		dev_err(&client->dev, 
				"%s: fail to setup wdt8913a interrupt, error %d\n", __func__, ret);
		return ret;
	}

	return 0;
}

static int wdt8913a_remove(struct i2c_client *client)
{
	struct wdt8913a_dev *wdev = i2c_get_clientdata(client);

	if (wdev->int_reg_cause)
	{
		iounmap(wdev->int_reg_cause);
	}

	if (wdev->int_reg_dat)
	{
		iounmap(wdev->int_reg_dat);
	}

	dev_dbg(&client->dev, "%s: device removed\n", __func__);
	return 0;
}

static const struct i2c_device_id wdt8913a_id[]={
	{ "wdt8913a", 0 },
	{}
};
MODULE_DEVICE_TABLE(i2c,wdt8913a_id);

static const struct of_device_id wdt8913_of_id[]={
	{.compatible = "wdt,wdt8913a",},
	{}
};

static struct i2c_driver wdt8913a_driver={
	.driver = {
		.owner          = THIS_MODULE,
		.name           = "wdt8913a",
		.of_match_table = of_match_ptr(wdt8913_of_id),
	},
	.probe    = wdt8913a_probe,
	.remove   = wdt8913a_remove,
	.id_table = wdt8913a_id,
};
module_i2c_driver(wdt8913a_driver);

MODULE_DESCRIPTION("Weida Hi-Tech WDT8913A Capacitive Touch Screen Driver");
MODULE_LICENSE("GPL");
