/* Linux kernel driver for resistive touch panel on some Quatro reference boards
 *
 * Quasar resistive touch panel kernel driver
 * 
 * Copyright (c) 2014, 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2022/01/06 10:11:28 $
//  $Change: 56812 $
//
// =========================================================
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/irq.h>
#include <quasar/qbsocregs.h>
#include "core.h"
#include "bsp.h"
#include "util.h"

#define WDT8913A_OF_SET_REG(client, name) do { \
	int ret; \
	ret = wdt8913a_of_setup_reg(client, name); \
	if (!ret) return ret; \
} while (0)

#define WDT8913A_OF_READ_U64(of_node, name, out) do { \
	int ret; \
	ret = of_property_read_u64(of_node, name, out); \
	if (ret != 0) return ret; \
} while (0)

#define WDT8913A_OF_READ_U8(of_node, name, out) do { \
	int ret; \
	ret = of_property_read_u8(of_node, name, out); \
	if (ret != 0) return ret; \
} while (0)


/**
 * Fetch 64-bit size from where `value' point to and with `index' 
 * as offset from it. The data pointed by `value' was stored as 
 * big-endian but it will be converted to current cpu endian 
 * format before returning the result.
 */
static u64 wdt8913a_of_read_array_u64(const __be32 *value, int index)
{
	value = value + index * 2;
	return of_read_number(value, 2);
}

/**
 * The customized properties of gpio setup was defined as six cells,
 * such as:
 *      <0 0x05632E20 0xFFFFFFFF 0xFFFFFFFF 0 0xFFF30000>
 *      
 * The first two cells represented as address to control the pin, and 
 * the middle two is given as the mask for the last two cells, and
 * the last two is given as value we are going to write.
 * 
 * The address and value are defined as 64-bit length for maybe future
 * extension.
 */
static int wdt8913a_of_setup_reg(struct i2c_client *client, const char *name)
{
	const __be32 *value;
	int lenp;
	u64 addr, mask, regv;
	u64 o_val;

	value = of_get_property(client->dev.of_node, name, &lenp);
	if (value)
	{
		lenp /= sizeof(*value);
		if (lenp != 6)
		{
			dev_err(&client->dev, 
					"%s: Length should be 6 cells, but %d cells found\n", 
					__func__, lenp);

			return -EINVAL;
		}

		/*
		 * The properties are 64-bit pairs 
		 */
		addr = wdt8913a_of_read_array_u64(value, 0);
		mask = wdt8913a_of_read_array_u64(value, 1);
		regv = wdt8913a_of_read_array_u64(value, 2);
	}
	else
	{
		dev_err(&client->dev, "%s: %s not found in DTS\n", __func__, name);
		return -EINVAL;
	}

	o_val = wdt8913a_rread(addr); 
	o_val = (o_val & ~mask) | (regv & mask);

	dev_dbg(&client->dev, 
			"%s: %s addr %llx with mask/value/writen %llx/%llx/%llx\n", 
			__func__, name, addr, mask, regv, o_val);
	
	wdt8913a_rwrite(addr, o_val);
	return 1;
}

// TBD: for Q6600
#if defined(Q6300EVM) || defined(Q6600) || defined(Q6300EVMNAND)

int wdt8913a_setup_board(struct wdt8913a_dev *wdev)
{
	struct i2c_client *client = wdev->client;

	/*
	 * Pins setup for the current board is defined in .dts
	 * file, and were brought here.
	 */
	WDT8913A_OF_SET_REG(client, "quasar,gpiosrc");
	WDT8913A_OF_SET_REG(client, "quasar,gpiodir");
	WDT8913A_OF_SET_REG(client, "quasar,gpioout");
	WDT8913A_OF_SET_REG(client, "quasar,gpioienf");

	return 0;
}

int wdt8913a_setup_irq(struct wdt8913a_dev *wdev)
{
	struct i2c_client *client = wdev->client;
	devm_request_threaded_irq(&client->dev, client->irq, 
			NULL, wdt8913a_isr, IRQF_ONESHOT, client->name, wdev);

	return 0;
}

int wdt8913a_int_chk_rst(struct wdt8913a_dev *wdev)
{
	return 1;
}

#elif defined(Q6300RDK)

int wdt8913a_setup_board(struct wdt8913a_dev *wdev)
{
	struct i2c_client *client = wdev->client;
	u64 int_reg_cause, int_reg_dat;

	WDT8913A_OF_SET_REG(client, "quasar,gpiodir");
	WDT8913A_OF_SET_REG(client, "quasar,gpiomsk");

	WDT8913A_OF_READ_U64(client->dev.of_node, "quasar,intcause", &int_reg_cause);
	WDT8913A_OF_READ_U64(client->dev.of_node, "quasar,intdat", &int_reg_dat);
	WDT8913A_OF_READ_U8(client->dev.of_node, "quasar,intoff", &wdev->int_off);

	/*
	 * Setup io mapping
	 */
	wdev->int_reg_cause = ioremap(int_reg_cause, 4);
	wdev->int_reg_dat   = ioremap(int_reg_dat, 4);

	/*
	 * Clear interrupt first
	 */
	writel(~wdev->int_off, wdev->int_reg_cause);
	return 0;
}

int wdt8913a_setup_irq(struct wdt8913a_dev *wdev)
{
	struct i2c_client *client = wdev->client;

	devm_request_threaded_irq(&client->dev, client->irq, 
			NULL, wdt8913a_isr, IRQF_ONESHOT, client->name, wdev);

	return 0;
}

/**
 * Check if we need to handle this interrupt and reset it.
 * 
 * No matter the interrupt is changed from High to Low or 
 * in reverse order, the interrupt handler will always be
 * called. However, we only need to handle the falling.
 * 
 * Also, the interrupt is combined type, so it would be 
 * trigger by GPIOs. We need to check if this interrupt
 * is the one we need.
 * 
 * Return 1 if the interrupt needs to be handled. Otherwise,
 * return 0.
 */
int wdt8913a_int_chk_rst(struct wdt8913a_dev *wdev)
{
	size_t rv;

	/*
	 * Check if the interrupt belongs to us
	 */ 
	rv = readl(wdev->int_reg_cause);
	if (!(rv & wdev->int_off))
	{
		dev_dbg(&wdev->client->dev,
				"%s: not our interrupt, ignored\n", __func__);
		return 0;
	}

	/*
	 * Clear interrupt cause register. Be careful not 
	 * to reset others.
	 * Fortunately, set 1 has no effect, so we can 
	 * only set our pin to zero.
	 */
	writel(~wdev->int_off, wdev->int_reg_cause);

	/*
	 * Check if the interrupt is high or low. We need 
	 * to handle low, and ignore high
	 */
	rv = readl(wdev->int_reg_dat);
	if (rv & wdev->int_off)
	{
		dev_dbg(&wdev->client->dev,
				"%s: ignore interrupt with HIGH value\n", __func__);
		return 0;
	}

	return 1;
}

#else // !Q6300EVM && !Q6300RDK

int wdt8913a_setup_board(struct wdt8913a_dev *wdev)
{
	dev_err(&client->dev, "%s: Not impl for this board\n", __func__);
	return -EINVAL;
}

int wdt8913a_setup_irq(struct wdt8913a_dev *wdev)
{
	dev_err(&client->dev, "%s: Not impl for this board\n", __func__);
	return -EINVAL;
}

int wdt8913a_int_chk_rst(struct wdt8913a_dev *wdev)
{
	dev_err(&client->dev, "%s: Not impl for this board\n", __func__);
	return -EINVAL;
}

#endif // Q6300EVM || Q6300RDK
