/*=================================================================================================
Filename:    vos_sem.c
Description: abstraction functions of Linux kernel 
Data:        Initial revision. Bob@2019-12-05
==================================================================================================*/
#ifdef __cplusplus 
    extern "C" { 
#endif

#include "vos_if.h"
#include "vos_innr.h"

HVOSDLK g_hSemDlk = 0;

//build-bundle\poky\build\tmp\work-shared\simva32\kernel-source\kernel\locking\seamphore.c
//build-bundle\poky\build\tmp\work-shared\simva32\kernel-source\include\linux\seamphore.h
void vos_semInit(void)
{
	g_hSemDlk = vos_dlkCreate(sizeof(Sem_ITEM_S), 1, NULL);
	VOS_ASSERT(g_hSemDlk != 0);
	return;
}

HVOSSEM vos_semCreate(char * strSemName, unsigned long nSemCnt)
{
	struct semaphore * pSem = NULL;
	HVOSSEM vSemId;
	Sem_ITEM_S sSemItem = {0,};
	Sem_ITEM_S *pSemItem = &sSemItem;

	pSem = (struct semaphore *)vos_malloc(sizeof(struct semaphore));
	VOS_ASSERT(pSem != NULL);
	
	sema_init(pSem, nSemCnt);

	pSemItem->pSem = pSem;
	pSemItem->vosType = VOS_SEM_TYPE;
	if(strSemName != NULL)
		strncpy(pSemItem->sName, strSemName, 11);
	else
		strcpy(pSemItem->sName, "NULL");
	
	vSemId = (HVOSSEM)vos_dlkInsertData(g_hSemDlk, (void *)pSemItem);	;
	vosPrintf("vos_semCreate: SemId=0x%x(0x%x), nSemCnt=%d\r\n", pSem, vSemId, nSemCnt);
	
	return vSemId;

}

int vos_semTake(HVOSSEM hVsem, unsigned long nWaitForMSec)
{
	int nRet = 0;
	struct semaphore * pSem = NULL;
	Sem_ITEM_S *pSemItem;

	pSemItem = (Sem_ITEM_S *)hVsem;
	VOS_ASSERT(pSemItem->vosType == VOS_SEM_TYPE);
	pSem = pSemItem->pSem;
	VOS_ASSERT(NULL != pSem);
	
	if(VOS_WAIT_FOREVER == nWaitForMSec)
	{
		down(pSem);
		//nRet = down_interruptible(pSem);
		VOS_ERROR_RPT(nRet);
		return nRet;
	}

	nRet = down_timeout(pSem, nWaitForMSec);
	// VOS_ERROR_RPT(nRet);

	return nRet;
}

int vos_semGive(HVOSSEM hVsem)
{
	int nRet = 0;
	struct semaphore *pSem = NULL;
	Sem_ITEM_S *pSemItem;

	pSemItem = (Sem_ITEM_S *)hVsem;
	VOS_ASSERT(pSemItem->vosType == VOS_SEM_TYPE);
	pSem = pSemItem->pSem;
	VOS_ASSERT(NULL != pSem);

	up(pSem);
	VOS_ERROR_RPT(nRet);
	
	return nRet;
}

int vos_semGetCnt(HVOSSEM hVsem)
{
	int nCnt=0;
	struct semaphore *pSem = NULL;
	Sem_ITEM_S *pSemItem;

	pSemItem = (Sem_ITEM_S *)hVsem;
	VOS_ASSERT(pSemItem->vosType == VOS_SEM_TYPE);
	pSem = pSemItem->pSem;
	VOS_ASSERT(NULL != pSem);

	nCnt = pSem->count;
	return nCnt;
}

int vos_semShowAll()
{
	int i, nRet=0, nCnt=0;
	struct semaphore *pSem = NULL;
	Sem_ITEM_S *pSemItem;
	DLK_NODE *pShow, *pHead;

	vosPrintf("%-4s %-10s   %-10s   %-12s %-8s\r\n", \
			"No.", "SemId", "vSemId", "Name", "Counter");
	vosPrintf("----------------------------------------------------\r\n");
	
	pHead = vos_dlkGetHead(g_hSemDlk);
	DLK_WALK_LOOP_START(pShow, pHead)
	{
		pSemItem = (Sem_ITEM_S *)(pShow + 1);
		pSem = pSemItem->pSem;
		if(pSem != 0)
		{
			nCnt = pSem->count;
			VOS_ERROR_RPT(nRet);
			vosPrintf("%-4d 0x%-10x 0x%-10x %-12s %-6d\r\n", \
				i, pSem, pSemItem, pSemItem->sName, nCnt);
		}
	}
	DLK_WALK_LOOP_END(pShow, pHead)
	return 0;
}

EXPORT_SYMBOL(vos_semCreate);
EXPORT_SYMBOL(vos_semTake);
EXPORT_SYMBOL(vos_semGive);
EXPORT_SYMBOL(vos_semGetCnt);

int test_sem(int nFlag, HVOSSEM hVsem, unsigned long nMs)
{
    HVOSSEM hVsemId = (HVOSSEM)NULL;
	int nRet=0, nVal;
	

	switch(nFlag)
	{
		case 0:
			nRet = vos_semShowAll();
			break;
			
		case 1:
			hVsemId = vos_semCreate("sem1", 0);
			vosPrintf("vos_semCreate1 ret=0x%x\r\n", hVsemId);
			hVsemId = vos_semCreate("sem2", 1);
			vosPrintf("vos_semCreate2 ret=0x%x\r\n", hVsemId);
			break;

		case 2:
			if(nMs == 0)nMs = VOS_WAIT_FOREVER;
			nVal = vos_GetSysCntMs();	
			nRet = vos_semTake(hVsem, nMs);
			vosPrintf("vos_semTake ret=0x%x, take %dms\r\n", nRet, vos_GetSysCntMs()-nVal);
			break;

		case 3:
			nRet = vos_semGive(hVsem);
			break;

		default:
			break;
	}

    return nRet;
}


#ifdef __cplusplus 
    }
#endif

