/*=================================================================================================
Filename:    vos_dlq.c
Description: API of vos ring buffer
Data:         Initial revision. Bob@2020-03-30
==================================================================================================*/
#ifdef __cplusplus 
    extern "C" { 
#endif

#include "vos_if.h"
#include "vos_innr.h"

//mutexFlag: 0-support 1 reading task and 1 writting task at same time, but not support more than 1 writting tasks or readding tasks at same time, 1-support all
HRBUF vos_rbufCreate(unsigned long nUnitSize, unsigned long nMaxCnt, int mutexFlag)
{
	VosRbuf_S *pVosRbuf = NULL;
	void *pBuff = NULL;
	char dlqName[8] = {0,};
	static int g_nRbufCnt = 0;

	pVosRbuf = (VosRbuf_S *)vos_malloc(sizeof(VosRbuf_S));
	VOS_ASSERT(pVosRbuf != NULL);
	pBuff = vos_malloc(nUnitSize*(nMaxCnt+1));
	VOS_ASSERT(pBuff != NULL);
	pVosRbuf->nMagicId = VOS_RBUF_MAGIC_ID;
	pVosRbuf->nUnitSize = nUnitSize;
	pVosRbuf->nMaxCnt = nMaxCnt+1;
	pVosRbuf->pBuff = pBuff;
	pVosRbuf->nReadIndex = 0;
	pVosRbuf->nWriteIndex = 0;
	pVosRbuf->hWtMutex = 0;
	pVosRbuf->hRdMutex = 0;	
	
	if(mutexFlag)
	{
		sprintf(dlqName, "rbuf_%d", g_nRbufCnt);g_nRbufCnt++;
		pVosRbuf->hWtMutex = vos_mutexCreate(dlqName, 0);
		VOS_ASSERT(pVosRbuf->hWtMutex != 0);
		sprintf(dlqName, "rbuf_%d", g_nRbufCnt);g_nRbufCnt++;
		pVosRbuf->hRdMutex = vos_mutexCreate(dlqName, 0);
		VOS_ASSERT(pVosRbuf->hRdMutex != 0);
	}

	return (HRBUF)pVosRbuf;
}

int vos_rbufDelete(HRBUF hRbuf)
{
	VosRbuf_S *pVosRbuf = NULL;
	
	pVosRbuf = (VosRbuf_S *)hRbuf;
	VOS_ASSERT(pVosRbuf != NULL);
	VOS_ASSERT(pVosRbuf->nMagicId == VOS_RBUF_MAGIC_ID);
	VOS_ASSERT(pVosRbuf->pBuff != NULL);

	pVosRbuf->nMagicId = VOS_RBUF_MAGIC_ID+1;
	pVosRbuf->nReadIndex = 0;
	pVosRbuf->nWriteIndex = 0;
	vos_free(pVosRbuf->pBuff);
	vos_free(pVosRbuf);
	return 0;
}

int vos_rbufClean(HRBUF hRbuf)
{
	VosRbuf_S *pVosRbuf = NULL;
	
	pVosRbuf = (VosRbuf_S *)hRbuf;
	VOS_ASSERT(pVosRbuf != NULL);
	VOS_ASSERT(pVosRbuf->nMagicId == VOS_RBUF_MAGIC_ID);
	VOS_ASSERT(pVosRbuf->pBuff != NULL);

	if(pVosRbuf->hWtMutex)vos_mutexLock(pVosRbuf->hWtMutex);
	if(pVosRbuf->hRdMutex)vos_mutexLock(pVosRbuf->hRdMutex);
	
	pVosRbuf->nReadIndex = 0;
	pVosRbuf->nWriteIndex = 0;

	if(pVosRbuf->hRdMutex)vos_mutexUnlock(pVosRbuf->hRdMutex);
	if(pVosRbuf->hWtMutex)vos_mutexUnlock(pVosRbuf->hWtMutex);
	
	return 0;
}

//flag: 1-for trace buffer string
int vos_rbufShow(HRBUF hRbuf, int flag)
{
	VosRbuf_S *pVosRbuf = NULL;
	char *pBuff = NULL;
	unsigned long nDataCnt, i;

	pVosRbuf = (VosRbuf_S *)hRbuf;
	VOS_ASSERT(pVosRbuf != NULL);
	VOS_ASSERT(pVosRbuf->nMagicId == VOS_RBUF_MAGIC_ID);
	VOS_ASSERT(pVosRbuf->pBuff != NULL);

	nDataCnt = (pVosRbuf->nWriteIndex + pVosRbuf->nMaxCnt - pVosRbuf->nReadIndex) % pVosRbuf->nMaxCnt;
	vosPrintf("ring buf(0x%x): nMaxCnt=%d, nUnitSize=%d, nReadIndex=%d, nWriteIndex=%d, nDataCnt=%d\r\n", pVosRbuf, pVosRbuf->nMaxCnt-1, pVosRbuf->nUnitSize, pVosRbuf->nReadIndex, pVosRbuf->nWriteIndex, nDataCnt);
	for(i=0; i<nDataCnt; i++)
	{
		pBuff = pVosRbuf->pBuff + ((pVosRbuf->nReadIndex+i) % pVosRbuf->nMaxCnt) * pVosRbuf->nUnitSize;
		if(flag == 0)
		{
			vosPrintf("0x%x 0x%x\r\n", *(unsigned long *)pBuff, *((unsigned long *)pBuff + 1));
		}
		else
		{
			vosPrintf("%d: %s\r\n", i, (char *)pBuff);
		}
	}
	return nDataCnt;	
}

int vos_rbufWrite(HRBUF hRbuf, void *pData, int nSize, int nWaitForMSec)
{
	VosRbuf_S *pVosRbuf = NULL;
	char *pDesBuff = NULL;
	unsigned long nDataCnt, nFreeCnt, nWritCnt, nWritCnt1, nWritCnt2, nWriteSize, nSleepMs =1, nSleepTotal=0;
	
	pVosRbuf = (VosRbuf_S *)hRbuf;
	VOS_ASSERT(pVosRbuf != NULL);
	VOS_ASSERT(pVosRbuf->nMagicId == VOS_RBUF_MAGIC_ID);
	VOS_ASSERT(pVosRbuf->pBuff != NULL);

	if(nSize == 0) //write one unit data
	{
		nWritCnt = 1;
		nWriteSize = pVosRbuf->nUnitSize;
	}
	else
	{
		VOS_ASSERT((nSize % pVosRbuf->nUnitSize) == 0); //must to be multi size
		nWritCnt = (nSize / pVosRbuf->nUnitSize);
		nWriteSize = nSize;
	}

	if(pVosRbuf->hWtMutex)vos_mutexLock(pVosRbuf->hWtMutex);
	
	nDataCnt = (pVosRbuf->nWriteIndex + pVosRbuf->nMaxCnt - pVosRbuf->nReadIndex) % pVosRbuf->nMaxCnt;
	nFreeCnt = pVosRbuf->nMaxCnt - nDataCnt - 1;
	if((nWaitForMSec == 0) && (nFreeCnt < nWritCnt))
	{
		if(pVosRbuf->hWtMutex)vos_mutexUnlock(pVosRbuf->hWtMutex);
		return 0;
	}

	while(nFreeCnt < nWritCnt)
	{
		if((VOS_WAIT_FOREVER != nWaitForMSec) && (nSleepTotal >= nWaitForMSec))
		{
			if(pVosRbuf->hWtMutex)vos_mutexUnlock(pVosRbuf->hWtMutex);
			return 0;
		}
		
		vos_taskSleep(nSleepMs);
		nSleepTotal += nSleepMs;
		if(nSleepMs < 100)nSleepMs++;
		
		nDataCnt = (pVosRbuf->nWriteIndex + pVosRbuf->nMaxCnt - pVosRbuf->nReadIndex) % pVosRbuf->nMaxCnt;
		nFreeCnt = pVosRbuf->nMaxCnt - nDataCnt - 1;		
	}
	
	pDesBuff = pVosRbuf->pBuff + (pVosRbuf->nWriteIndex % pVosRbuf->nMaxCnt) * pVosRbuf->nUnitSize;
	if((nSize == 0) || ((pVosRbuf->nWriteIndex + nWritCnt) < pVosRbuf->nMaxCnt))
	{
		memcpy(pDesBuff, pData, nWriteSize);
	}
	else
	{
		nWritCnt1 = pVosRbuf->nMaxCnt - pVosRbuf->nWriteIndex;
		nWritCnt2 = nWritCnt - nWritCnt1;
		memcpy(pDesBuff, pData, nWritCnt1*pVosRbuf->nUnitSize);
		memcpy(pVosRbuf->pBuff, (char *)pData + nWritCnt1*pVosRbuf->nUnitSize, nWritCnt2*pVosRbuf->nUnitSize);
	}
	
	pVosRbuf->nWriteIndex = (pVosRbuf->nWriteIndex + nWritCnt) % pVosRbuf->nMaxCnt;
	if(pVosRbuf->hWtMutex)vos_mutexUnlock(pVosRbuf->hWtMutex);
	return nWriteSize;
}

int vos_rbufRead(HRBUF hRbuf, void *pData, int nSize, int nWaitForMSec)
{
	VosRbuf_S *pVosRbuf = NULL;
	char *pReadBuff = NULL;
	unsigned long nDataCnt, nReadCnt, nReadCnt1, nReadCnt2, nReadSize, nSleepMs =1, nSleepTotal=0;
	
	pVosRbuf = (VosRbuf_S *)hRbuf;
	VOS_ASSERT(pVosRbuf != NULL);
	VOS_ASSERT(pVosRbuf->nMagicId == VOS_RBUF_MAGIC_ID);
	VOS_ASSERT(pVosRbuf->pBuff != NULL);

	if(nSize == 0) //read one unit data
	{
		nReadCnt = 1;
		nReadSize = pVosRbuf->nUnitSize;
	}
	else
	{
		VOS_ASSERT((nSize % pVosRbuf->nUnitSize) == 0); //must to be multi size
		nReadCnt = (nSize / pVosRbuf->nUnitSize);
		nReadSize = nSize;
	}

	if(pVosRbuf->hRdMutex)vos_mutexLock(pVosRbuf->hRdMutex);

	nDataCnt = (pVosRbuf->nWriteIndex + pVosRbuf->nMaxCnt - pVosRbuf->nReadIndex) % pVosRbuf->nMaxCnt;
	if((nWaitForMSec == 0) && (nDataCnt < nReadCnt))
	{
		if(pVosRbuf->hRdMutex)vos_mutexUnlock(pVosRbuf->hRdMutex);
		return 0;
	}

	while(nDataCnt < nReadCnt)
	{
		if((VOS_WAIT_FOREVER != nWaitForMSec) && (nSleepTotal >= nWaitForMSec))
		{
			if(pVosRbuf->hRdMutex)vos_mutexUnlock(pVosRbuf->hRdMutex);
			return 0;
		}
		
		vos_taskSleep(nSleepMs);
		nSleepTotal += nSleepMs;
		if(nSleepMs < 100)nSleepMs++;
		
		nDataCnt = (pVosRbuf->nWriteIndex + pVosRbuf->nMaxCnt - pVosRbuf->nReadIndex) % pVosRbuf->nMaxCnt;
	}
	
	pReadBuff = pVosRbuf->pBuff + (pVosRbuf->nReadIndex % pVosRbuf->nMaxCnt) * pVosRbuf->nUnitSize;
	if((nSize == 0) || ((pVosRbuf->nReadIndex + nReadCnt) < pVosRbuf->nMaxCnt))
	{
		memcpy(pData, pReadBuff, nReadSize);
	}
	else
	{
		nReadCnt1 = pVosRbuf->nMaxCnt - pVosRbuf->nReadIndex;
		nReadCnt2 = nReadCnt - nReadCnt1;
		memcpy(pData, pReadBuff, nReadCnt1*pVosRbuf->nUnitSize);
		memcpy((char *)pData + nReadCnt1*pVosRbuf->nUnitSize, pVosRbuf->pBuff, nReadCnt2*pVosRbuf->nUnitSize);
	}
	
	pVosRbuf->nReadIndex = (pVosRbuf->nReadIndex + nReadCnt) % pVosRbuf->nMaxCnt;
	if(pVosRbuf->hRdMutex)vos_mutexUnlock(pVosRbuf->hRdMutex);
	return nReadSize;
}

int vos_rbufWriteTrace(HRBUF hRbuf, void *pData)
{
	VosRbuf_S *pVosRbuf = NULL;
	char *pDesBuff = NULL;
	unsigned long nDataCnt, nFreeCnt, nWritCnt, nWriteSize;
	
	pVosRbuf = (VosRbuf_S *)hRbuf;
	VOS_ASSERT(pVosRbuf != NULL);
	VOS_ASSERT(pVosRbuf->nMagicId == VOS_RBUF_MAGIC_ID);
	VOS_ASSERT(pVosRbuf->pBuff != NULL);

	//write one unit data
	nWritCnt = 1;
	nWriteSize = pVosRbuf->nUnitSize;

	if(pVosRbuf->hWtMutex)vos_mutexLock(pVosRbuf->hWtMutex);
	
	nDataCnt = (pVosRbuf->nWriteIndex + pVosRbuf->nMaxCnt - pVosRbuf->nReadIndex) % pVosRbuf->nMaxCnt;
	nFreeCnt = pVosRbuf->nMaxCnt - nDataCnt - 1;
	if(nFreeCnt < nWritCnt)
	{
		pVosRbuf->nReadIndex = (pVosRbuf->nReadIndex + 1) % pVosRbuf->nMaxCnt; //discard this one
	}

	pDesBuff = pVosRbuf->pBuff + (pVosRbuf->nWriteIndex % pVosRbuf->nMaxCnt) * pVosRbuf->nUnitSize;
	memcpy(pDesBuff, pData, nWriteSize);
	
	pVosRbuf->nWriteIndex = (pVosRbuf->nWriteIndex + nWritCnt) % pVosRbuf->nMaxCnt;
	if(pVosRbuf->hWtMutex)vos_mutexUnlock(pVosRbuf->hWtMutex);
	return nWriteSize;
}

unsigned long vos_rbufDataSizeGet(HRBUF hRbuf)
{
	VosRbuf_S *pVosRbuf = NULL;
	unsigned long nDataCnt;
	
	pVosRbuf = (VosRbuf_S *)hRbuf;
	VOS_ASSERT(pVosRbuf != NULL);
	VOS_ASSERT(pVosRbuf->nMagicId == VOS_RBUF_MAGIC_ID);
	VOS_ASSERT(pVosRbuf->pBuff != NULL);

	nDataCnt = (pVosRbuf->nWriteIndex + pVosRbuf->nMaxCnt - pVosRbuf->nReadIndex) % pVosRbuf->nMaxCnt;
	return nDataCnt;
}

unsigned long vos_rbufFreeSizeGet(HRBUF hRbuf)
{
	VosRbuf_S *pVosRbuf = NULL;
	unsigned long nDataCnt;
	
	pVosRbuf = (VosRbuf_S *)hRbuf;
	VOS_ASSERT(pVosRbuf != NULL);
	VOS_ASSERT(pVosRbuf->nMagicId == VOS_RBUF_MAGIC_ID);
	VOS_ASSERT(pVosRbuf->pBuff != NULL);

	nDataCnt = (pVosRbuf->nReadIndex + pVosRbuf->nMaxCnt - pVosRbuf->nWriteIndex-1) % pVosRbuf->nMaxCnt;
	return nDataCnt;
}

unsigned long vos_rbufBuffSizeGet(HRBUF hRbuf)
{
	VosRbuf_S *pVosRbuf = NULL;
	
	pVosRbuf = (VosRbuf_S *)hRbuf;
	VOS_ASSERT(pVosRbuf != NULL);
	VOS_ASSERT(pVosRbuf->nMagicId == VOS_RBUF_MAGIC_ID);
	VOS_ASSERT(pVosRbuf->pBuff != NULL);

	return pVosRbuf->nMaxCnt*pVosRbuf->nUnitSize;
}

int vos_rbufRdCheck(HRBUF hRbuf, void *pData, int nSize, int nWaitForMSec)
{
	VosRbuf_S *pVosRbuf = NULL;
	char *pReadBuff = NULL;
	unsigned long nDataCnt, nReadCnt, nReadCnt1, nReadCnt2, nReadSize, nSleepMs =1, nSleepTotal=0;
	
	pVosRbuf = (VosRbuf_S *)hRbuf;
	VOS_ASSERT(pVosRbuf != NULL);
	VOS_ASSERT(pVosRbuf->nMagicId == VOS_RBUF_MAGIC_ID);
	VOS_ASSERT(pVosRbuf->pBuff != NULL);

	if(nSize == 0) //read one unit data
	{
		nReadCnt = 1;
		nReadSize = pVosRbuf->nUnitSize;
	}
	else
	{
		VOS_ASSERT((nSize % pVosRbuf->nUnitSize) == 0); //must to be multi size
		nReadCnt = (nSize / pVosRbuf->nUnitSize);
		nReadSize = nSize;
	}

	if(pVosRbuf->hRdMutex)vos_mutexLock(pVosRbuf->hRdMutex);

	nDataCnt = (pVosRbuf->nWriteIndex + pVosRbuf->nMaxCnt - pVosRbuf->nReadIndex) % pVosRbuf->nMaxCnt;
	if((nWaitForMSec == 0) && (nDataCnt < nReadCnt))
	{
		if(pVosRbuf->hRdMutex)vos_mutexUnlock(pVosRbuf->hRdMutex);
		return 0;
	}

	while(nDataCnt < nReadCnt)
	{
		if((VOS_WAIT_FOREVER != nWaitForMSec) && (nSleepTotal >= nWaitForMSec))
		{
			if(pVosRbuf->hRdMutex)vos_mutexUnlock(pVosRbuf->hRdMutex);
			return 0;
		}
		
		vos_taskSleep(nSleepMs);
		nSleepTotal += nSleepMs;
		if(nSleepMs < 100)nSleepMs++;
		
		nDataCnt = (pVosRbuf->nWriteIndex + pVosRbuf->nMaxCnt - pVosRbuf->nReadIndex) % pVosRbuf->nMaxCnt;
	}
	
	pReadBuff = pVosRbuf->pBuff + (pVosRbuf->nReadIndex % pVosRbuf->nMaxCnt) * pVosRbuf->nUnitSize;
	if((nSize == 0) || ((pVosRbuf->nReadIndex + nReadCnt) < pVosRbuf->nMaxCnt))
	{
		memcpy(pData, pReadBuff, nReadSize);
	}
	else
	{
		nReadCnt1 = pVosRbuf->nMaxCnt - pVosRbuf->nReadIndex;
		nReadCnt2 = nReadCnt - nReadCnt1;
		memcpy(pData, pReadBuff, nReadCnt1*pVosRbuf->nUnitSize);
		memcpy((char *)pData + nReadCnt1*pVosRbuf->nUnitSize, pVosRbuf->pBuff, nReadCnt2*pVosRbuf->nUnitSize);
	}
	
	//pVosRbuf->nReadIndex = (pVosRbuf->nReadIndex + nReadCnt) % pVosRbuf->nMaxCnt;
	if(pVosRbuf->hRdMutex)vos_mutexUnlock(pVosRbuf->hRdMutex);
	return nReadSize;
}

unsigned char* vos_MainQBegin(HRBUF hRbuf)
{	
	VosRbuf_S *pVosRbuf = NULL;
	pVosRbuf = (VosRbuf_S *)hRbuf;
	return pVosRbuf->pBuff + pVosRbuf->nReadIndex;
}
unsigned char* vos_MainQEnd(HRBUF hRbuf)
{
	VosRbuf_S *pVosRbuf = NULL;
	pVosRbuf = (VosRbuf_S *)hRbuf;
	return pVosRbuf->pBuff + pVosRbuf->nMaxCnt;
}
unsigned char* vos_MainQBufBegin(HRBUF hRbuf)
{
	VosRbuf_S *pVosRbuf = NULL;
	pVosRbuf = (VosRbuf_S *)hRbuf;
	return pVosRbuf->pBuff;
}
unsigned char* vos_MainQBufEnd(HRBUF hRbuf)
{
	VosRbuf_S *pVosRbuf = NULL;
	pVosRbuf = (VosRbuf_S *)hRbuf;
	return pVosRbuf->pBuff + pVosRbuf->nWriteIndex;
}


int test_rbuf(int nFlag, int nPar1, int nPar2)
{
	unsigned long nDataAddr;
	int nRet, i;
	HRBUF hRbuf;
	int buff[64] = {0,};

	switch(nFlag)
	{
		case 0:
			nRet = vos_rbufShow((HRBUF)nPar1, 0);
			break;
			
		case 1:
			nRet = (HRBUF)vos_rbufCreate(8, 10, 1);
			break;

		case 2:
			nRet = vos_rbufDelete((HRBUF)nPar1);
			break;

		case 3:
			buff[0] = nPar2;
			buff[1] = ~nPar2;
			nRet = vos_rbufWrite((HRBUF)nPar1, (void *)buff, nPar2, 2000);
			break;

		case 4:
			nRet = vos_rbufRead((HRBUF)nPar1, (void *)buff, 0, 2000);
			vosPrintf("vos_dlqRead: size=%d, 0x%x 0x%x\r\n", nRet, buff[0], buff[1]);
			break;

		case 10:
			hRbuf = (HRBUF)vos_rbufCreate(8, 10, 1);
			for(i=0; i<8; i++){buff[0] = i;buff[1] = ~buff[0];(void)vos_rbufWrite(hRbuf, (void *)buff, 0, 0);}
			for(i=0; i<3; i++)(void)vos_rbufRead(hRbuf, (void *)buff, 0, 0);
			vos_rbufShow(hRbuf, 0);

			for(i=0; i<64; i++)buff[i] = (0x20+i);
			(void)vos_rbufWrite(hRbuf, (void *)buff, 8*4, 0);
			vos_rbufShow(hRbuf, 0);
			break;

		default:
			break;
	}

    return nRet;
}

#ifdef __cplusplus 
    }
#endif

