/*=================================================================================================
Filename:    vos_mutex.c
Description: abstraction functions of Linux kernel 
Data:        Initial revision. Bob@2019-12-05
==================================================================================================*/
#ifdef __cplusplus 
    extern "C" { 
#endif

#include "vos_if.h"
#include "vos_innr.h"

HVOSDLK g_hMutexDlk = 0;

//build-bundle/poky/build/tmp/work-shared/simva32/kernel-source/kernel/locking/mutex.c
//build-bundle\poky\build\tmp\work-shared\simva32\kernel-source\include\linux\mutex.h
void vos_mutexInit(void)
{
	g_hMutexDlk = vos_dlkCreate(sizeof(Mutex_ITEM_S), 0, NULL);
	VOS_ASSERT(g_hMutexDlk != 0);
	return;
}

HVOSMUTEX vos_mutexCreate(char * pName, int bLockMutex)
{
	HVOSMUTEX hMutexId = (HVOSMUTEX)NULL;
	struct mutex *pMutex = NULL;
	Mutex_ITEM_S sMutexItem = {0,};
	Mutex_ITEM_S * pMutexItem = &sMutexItem;
	char sName[16]={'N', 'U', 'L', 'L',0};

	if(pName!=NULL)
	{
		strncpy(sName, pName, 12);
	}
	vos_dbgPrint(2, "%s: pName=%s\r\n", __func__, sName);
	
	pMutex = vos_malloc(sizeof(struct mutex));
	if(NULL == pMutex)
	{
		vosPrintf("Mem is not enough: %s @%d\r\n", __func__, __LINE__);
		return (HVOSMUTEX)NULL;
	}

	mutex_init(pMutex);

	pMutexItem->pMutex = pMutex;
	pMutexItem->nCnt = 0;
	pMutexItem->nUsedThread = 0;
	strncpy(pMutexItem->sName, sName, 11);
	pMutexItem->sName[11] = 0;
	pMutexItem->vosType = VOS_MUTEX_TYPE;

	hMutexId = (HVOSSEM)vos_dlkInsertData(g_hMutexDlk, (void *)pMutexItem);
	vosPrintf("vos_CreateMutex: MutexId=0x%x(0x%x)\r\n", pMutex, hMutexId);

	if(bLockMutex)
	{
		(void)vos_mutexLock(hMutexId);
	}

	return hMutexId;

}

int vos_mutexLock(HVOSMUTEX hMutexId)
{
	struct mutex *pMutex = NULL;
	Mutex_ITEM_S * pMutexItem;
	int nRet = 0;

	vos_dbgPrint(1, "%s: hMutexId=0x%x\r\n", __func__, hMutexId);
	
	pMutexItem = (Mutex_ITEM_S *)hMutexId;
	VOS_ASSERT(pMutexItem->vosType == VOS_MUTEX_TYPE);
	pMutex = pMutexItem->pMutex;
	VOS_ASSERT(NULL != pMutex);
	
	mutex_lock(pMutex);
	pMutexItem->nCnt = 1;
	
	return nRet;
}

int vos_mutexUnlock(HVOSMUTEX hMutexId)
{
	struct mutex *pMutex = NULL;
	Mutex_ITEM_S * pMutexItem;
	int nRet = 0;

	vos_dbgPrint(1, "%s: hMutexId=0x%x\r\n", __func__, hMutexId);
	
	pMutexItem = (Mutex_ITEM_S *)hMutexId;
	VOS_ASSERT(pMutexItem->vosType == VOS_MUTEX_TYPE);
	pMutex = pMutexItem->pMutex;
	VOS_ASSERT(NULL != pMutex);
	
	mutex_unlock(pMutex);
	pMutexItem->nCnt = 0;

	return nRet;
}

int vos_mutexClose(HVOSMUTEX hMutexId)
{
	struct mutex *pMutex = NULL;
	Mutex_ITEM_S * pMutexItem;

	pMutexItem = (Mutex_ITEM_S *)hMutexId;
	VOS_ASSERT(pMutexItem->vosType == VOS_MUTEX_TYPE);
	pMutex = pMutexItem->pMutex;
	VOS_ASSERT(NULL != pMutex);

	vos_dlkDelete(g_hMutexDlk, (DLK_NODE *)hMutexId-1);
	vos_free(pMutex);
	return 0;
}

int vos_mutexShowAll(void)
{
	int i;
	struct mutex *pMutex = NULL;
	Mutex_ITEM_S * pMutexItem;
	int nRet = 0;
	int nIndex = 1;
	char sName[32] = {0,};
	char *pName;
	DLK_NODE *pShow, *pHead;

	vosPrintf("%-4s %-14s %-8s    %-12s   %-12s %-8s\r\n", \
			"No.", "Name", "nVMutID", "nMutID", "Curtask", "Counter");
    vosPrintf("-------------------------------------------------------------------\r\n");
	
	pHead = vos_dlkGetHead(g_hMutexDlk);
	DLK_WALK_LOOP_START(pShow, pHead)
	{
		pMutexItem = (Mutex_ITEM_S *)(pShow + 1);
		pMutex = pMutexItem->pMutex;
		if(pMutex != NULL)
		{
			pName = sName;
			sprintf(sName, "NULL");
							
			vosPrintf("%-4d %-14s 0x%-8x  0x%-12x %-12s %-8d\r\n", \
				nIndex, pMutexItem->sName, pMutexItem, pMutex, pName, pMutexItem->nCnt);
			nIndex++;
		}
	}
	DLK_WALK_LOOP_END(pShow, pHead)
	return nIndex;
}

EXPORT_SYMBOL(vos_mutexCreate);
EXPORT_SYMBOL(vos_mutexLock);
EXPORT_SYMBOL(vos_mutexUnlock);
EXPORT_SYMBOL(vos_mutexClose);

int test_mut(int nFlag, HVOSMUTEX vMutexId)
{
    HVOSMUTEX hMutexId = (HVOSMUTEX)NULL;
	HVOSTASK nVid;
	void *arg = 0;
	int nRet=0;

	switch(nFlag)
	{
		case 0:
			nRet = vos_mutexShowAll();
			break;
			
		case 1:
			hMutexId = vos_mutexCreate("mutex1", 0);
			vosPrintf("vos_mutexCreate1 ret=0x%x\r\n", hMutexId);
			hMutexId = vos_mutexCreate("mutex2", 0);
			vosPrintf("vos_mutexCreate2 ret=0x%x\r\n", hMutexId);
			break;

		case 2:
			nRet = vos_mutexLock(vMutexId);
			break;

		case 3:
			nRet = vos_mutexUnlock(vMutexId);
			break;

		case 4:
			nRet = vos_mutexClose(vMutexId);
			break;

		default:
			break;
	}

    return nRet;
}


#ifdef __cplusplus 
    }
#endif

