/*=================================================================================================
Filename:	 vos_mpool.c
Description: memory pool management 
Data:		 Initial revision. Bob@2020-4-24
==================================================================================================*/
#ifdef __cplusplus 
		extern "C" { 
#endif
#include "vos_if.h"
#include "vos_innr.h"

#define MPL_ALIGN_RIGHT(a,b)        (((a)+(b)-1)/(b)*(b))

HVOSDLK g_hMpoolDlk = 0;


//type: 0-no phyAddr, no USING_EXTERNAL_NODE; 1-have phyAddr, no USING_EXTERNAL_NODE; 2-have phyAddr, have USING_EXTERNAL_NODE
  HVOSMPL vos_mplCreatePoolPhy(const char * poolName, int type, void * pPoolBuf, int nSize, unsigned int nPhyAddrStart, int mutxFlag)
{
	unsigned char *pBuffBase;
	unsigned int nPhyAddrBase, nPhyAddr;
	PS_MPL_POOL pPool;

	nPhyAddr = nPhyAddrStart;
	if((0 == pPoolBuf) && (0 == nPhyAddr))return NULL;

	if(0 == nPhyAddr)nPhyAddr = (unsigned int)pPoolBuf;
	nPhyAddrBase =MPL_ALIGN_RIGHT(nPhyAddr, 32); 
	nSize = nSize - (nPhyAddrBase - nPhyAddr);

	if(2 == type) //using external node
	{
		pPool = (PS_MPL_POOL)vos_malloc(sizeof(S_MPL_POOL));
	}
	else
	{
		pBuffBase = (unsigned char *)pPoolBuf + (nPhyAddrBase - nPhyAddr);
		pPool = (PS_MPL_POOL)pBuffBase;
	}
	VOS_ASSERT(pPool != NULL);

	if((nSize - sizeof(S_MPL_POOL) - sizeof(S_MPL_NODE)) < 256)
	{
		vosPrintf("%s() error: %s, size:%d@%d\r\n", __FUNCTION__, poolName, nSize, __LINE__);
		return NULL;
	}
	memset(pPool, 0, sizeof(S_MPL_POOL));
	pPool->nMagicId = VOS_MPL_MAGIC_ID;
	pPool->type = type;
	pPool->nPoolAddrStart = (unsigned int)pPoolBuf;
	pPool->nPhyAddrStart = nPhyAddrStart;
#if MPL_USING_IN_KENERL	
	pPool->nameList = vos_dlkCreate(VOS_MPL_NAME_MAX_SIZE, 0, NULL);
	VOS_ASSERT(pPool->nameList != 0);	
#endif	
	strncpy(pPool->szPoolName, poolName, 11);
	if(mutxFlag)
	{
		pPool->hMtx = vos_mutexCreate((char *)poolName, 0);
		VOS_ASSERT(0 != pPool->hMtx);
	}
	
	if(2 == type) //using external node
	{
		pPool->nTotalSize = nSize;
		pPool->nPoolAddr = 0;
	}
	else
	{
		pPool->nTotalSize = nSize - sizeof(S_MPL_POOL) - 2*sizeof(S_MPL_NODE); //header&tail node
		pPool->nPoolAddr = (unsigned int)pBuffBase;
	}
	pPool->nRemainSize = pPool->nTotalSize;
	pPool->nPhyAddr = nPhyAddrBase;
	pPool->nPeakUsed = 0;

	if(2 == type) //using external node
	{
		pPool->pHeadNode = (PS_MPL_NODE)vos_malloc(sizeof(S_MPL_NODE));VOS_ASSERT(pPool->pHeadNode != NULL);
		pPool->pTailNode = (PS_MPL_NODE)vos_malloc(sizeof(S_MPL_NODE));VOS_ASSERT(pPool->pTailNode != NULL);
	}
	else
	{
		pPool->pHeadNode = (void *)(pBuffBase + sizeof(S_MPL_POOL));
		pPool->pTailNode = (void *)(pBuffBase + nSize - sizeof(S_MPL_NODE));
	}
	
	memset(pPool->pHeadNode, 0, sizeof(S_MPL_NODE));
	pPool->pHeadNode->pNext = pPool->pTailNode;
	pPool->pHeadNode->nMagicId = MPL_MEM_MAGIC_ID;
	pPool->pHeadNode->nStartAddr = nPhyAddrBase + sizeof(S_MPL_POOL);
	pPool->pHeadNode->nUsedSize = sizeof(S_MPL_NODE);
	
	memset(pPool->pTailNode, 0, sizeof(S_MPL_NODE));
	pPool->pTailNode->pPrev = pPool->pHeadNode;
	pPool->pTailNode->nMagicId = MPL_MEM_MAGIC_ID;
	pPool->pTailNode->nStartAddr = nPhyAddrBase + nSize - sizeof(S_MPL_NODE);
	pPool->pTailNode->nUsedSize = sizeof(S_MPL_NODE);

	if(2 == type) //using external node
	{
		pPool->pHeadNode->nStartAddr = nPhyAddrBase;
		pPool->pHeadNode->nUsedSize = 0;

		pPool->pTailNode->nStartAddr = nPhyAddrBase + nSize;
		pPool->pTailNode->nUsedSize = 0;
	}

	if(g_hMpoolDlk == 0)g_hMpoolDlk = vos_dlkCreate(sizeof(HVOSMPL), 0, NULL);
	VOS_ASSERT(g_hMpoolDlk != 0);
	vos_dlkInsertData(g_hMpoolDlk, (void *)&pPool);
	
	return (HVOSMPL)pPool;
}

HVOSMPL vos_mplCreatePool(const char * szDesc, void * pBaseBuf, int nSize)
{
	return vos_mplCreatePoolPhy(szDesc, 0, pBaseBuf, nSize, 0, 1);
}

void * vos_mplAlloc(HVOSMPL hPool, int nSize, int align, const char * szFile, int nLine)
{
	void * pAddr = NULL;
	PS_MPL_NODE pNode = NULL;
	PS_MPL_NODE pNextNode = NULL;
	PS_MPL_NODE pFoundNode = NULL;
	PS_MPL_POOL pPool = (PS_MPL_POOL)hPool;
	int nUsedSize = 0, align_size;
	unsigned int nFoundStartAddr = 0, malloc_addr;
	char *pName;
	char sName[VOS_MPL_NAME_MAX_SIZE] = {0,};

	if((NULL == pPool) || (nSize <= 0))return NULL;
	if(align < 4)align = 4;
	
	if(pPool->hMtx)vos_mutexLock(pPool->hMtx);
	
	if(2 == pPool->type) //using external node
		nUsedSize = MPL_ALIGN_RIGHT(nSize, align); //external node should use same align size
	else
		nUsedSize = MPL_ALIGN_RIGHT((nSize+sizeof(S_MPL_NODE)+align), align); 

	if( pPool->nRemainSize < nUsedSize )
	{
		if(pPool->hMtx)vos_mutexUnlock(pPool->hMtx);
		return NULL;
	}

	pNode = pPool->pHeadNode;
	VOS_ASSERT(pNode != NULL);
	
	while(pNode != NULL)
	{
		pNextNode = pNode->pNext;
		if(pNextNode == NULL)break;
		
		if((pNextNode->nStartAddr - pNode->nStartAddr - pNode->nUsedSize) >= nUsedSize)
		{
			nFoundStartAddr = pNode->nStartAddr + pNode->nUsedSize;			
			if(2 == pPool->type) //using external node
			{
				malloc_addr = nFoundStartAddr;
				malloc_addr = MPL_ALIGN_RIGHT(malloc_addr, align);
				pFoundNode = (PS_MPL_NODE)vos_malloc(sizeof(S_MPL_NODE));
				VOS_ASSERT(pFoundNode != NULL);
			}
			else
			{
				malloc_addr = nFoundStartAddr + sizeof(S_MPL_NODE);
				malloc_addr = MPL_ALIGN_RIGHT(malloc_addr, align);
				pFoundNode = (PS_MPL_NODE)(malloc_addr - sizeof(S_MPL_NODE));
				//pFoundNode = (PS_MPL_NODE)((unsigned int)pNode + pNode->nUsedSize + (malloc_addr-nFoundStartAddr) - sizeof(S_MPL_NODE));
			}
			
			pNode->pNext = pFoundNode;
			pFoundNode->pPrev = pNode;
			pFoundNode->pNext = pNextNode;
			pNextNode->pPrev = pFoundNode;
			break;
		}
		pNode = pNextNode;
	}

	if(pFoundNode == NULL)
	{
		if(pPool->hMtx)vos_mutexUnlock(pPool->hMtx);
		return NULL;
	}

#if MPL_USING_IN_KENERL
	strncpy(sName, szFile, VOS_MPL_NAME_MAX_SIZE-1);
	pName = vos_dlkInsertData(pPool->nameList, (void *)sName);
#else
	pName = (char *)szFile;
#endif
	pFoundNode->szFile = pName;
	pFoundNode->nLine = nLine;
	pFoundNode->nMagicId = MPL_MEM_MAGIC_ID;
	pFoundNode->nStartAddr = nFoundStartAddr;
	pFoundNode->nPhyAddr = malloc_addr;
	pFoundNode->nUsedSize = nUsedSize;
	pFoundNode->hPool = hPool;
	
	pPool->nRemainSize -= nUsedSize;
	nUsedSize = pPool->nTotalSize - pPool->nRemainSize;
	if(nUsedSize > pPool->nPeakUsed)pPool->nPeakUsed = nUsedSize;
	
	if(pPool->hMtx)vos_mutexUnlock(pPool->hMtx);
	pAddr = (unsigned char *)(pFoundNode+1);
	return pAddr;
}

void * vos_mplRealloc(HVOSMPL hPool, void * pOrgMem, int nSize, int align, const char * szFile, int nLine)
{
	void * pAddr = NULL;
	PS_MPL_POOL pPool = (PS_MPL_POOL)hPool;
	PS_MPL_NODE pNode = NULL;
	PS_MPL_NODE pNextNode = NULL;

	if(NULL == pPool)
	{
		return NULL;
	}
	if(NULL == pOrgMem)
	{
		return vos_mplAlloc(hPool, nSize, align, szFile, nLine);
	}

	pNode = (PS_MPL_NODE)(pOrgMem)-1;
	if(nSize <= (pNode->nUsedSize))
	{
		return pOrgMem;
	}
	
	pAddr = vos_mplAlloc(hPool, nSize, align, szFile, nLine);
	if(NULL == pAddr)return pOrgMem;

	memcpy(pAddr, pOrgMem, nSize);
	vos_mplFree(hPool, pOrgMem);
	
	return pAddr;
}

void vos_mplFree(HVOSMPL hPool, void * pMem)
{
	PS_MPL_POOL pPool = (PS_MPL_POOL)hPool;
	PS_MPL_NODE pNode = NULL;
	PS_MPL_NODE pPrevNode = NULL;
	PS_MPL_NODE pNextNode = NULL;
	int nUsedSize;

	if((NULL == pMem) || (NULL == pPool))return;

	if(2 != pPool->type) //not using external node
	{
		//VOS_ASSERT((unsigned char *)pMem < ((unsigned char *)(pPool) + (pPool->nTotalSize-sizeof(S_MPL_NODE))));
		VOS_ASSERT((unsigned char *)pMem < pPool->pTailNode);
	}

	if(pPool->hMtx)vos_mutexLock(pPool->hMtx);

	pNode = (PS_MPL_NODE)(pMem)-1;
	VOS_ASSERT(pNode->nMagicId == MPL_MEM_MAGIC_ID);

	pPrevNode = pNode->pPrev;
	pNextNode = pNode->pNext;
	VOS_ASSERT(pNextNode != NULL);

	pPool->nRemainSize += pNode->nUsedSize;
	nUsedSize = pPool->nTotalSize - pPool->nRemainSize;
	if(nUsedSize > pPool->nPeakUsed)pPool->nPeakUsed = nUsedSize;

	// memset(pNode, 0, sizeof(S_MPL_NODE));
	VOS_ASSERT(pPrevNode != NULL);
	pPrevNode->pNext = pNextNode;
	pNextNode->pPrev = pPrevNode;
	pNode->nMagicId = MPL_MEM_MAGIC_ID+1;
#if MPL_USING_IN_KENERL
	vos_dlkDelete(pPool->nameList, (DLK_NODE *)pNode->szFile - 1);
#endif
	if(2 == pPool->type) //using external node
	{
		vos_free((void *)pNode);
	}

	if(pPool->hMtx)vos_mutexUnlock(pPool->hMtx);
}

void vos_mplFreeAll(HVOSMPL hPool)
{
	PS_MPL_POOL pPool = (PS_MPL_POOL)hPool;		
	PS_MPL_NODE pHead = NULL;
	PS_MPL_NODE pNode = NULL;
	PS_MPL_NODE pNextNode = NULL;
	
	if(pPool->hMtx)vos_mutexLock(pPool->hMtx);

	if (2 == pPool->type)//using external node
	{
		pHead = pPool->pHeadNode;
		pNextNode = pHead->pNext;
	
		while(pNextNode != pPool->pTailNode)
		{
			pNode = pNextNode;
			pNextNode = pNode->pNext;
			
			pNode->nMagicId = MPL_MEM_MAGIC_ID+1;
			vos_free((void *)pNode);
		}
	}

	pPool->nRemainSize = pPool->nTotalSize;
	pPool->nPeakUsed = 0;

	pPool->pHeadNode->pNext = pPool->pTailNode;
	pPool->pHeadNode->nUsedSize = sizeof(S_MPL_NODE);
	if(2 == pPool->type) //using external node
	{
		pPool->pHeadNode->nUsedSize = 0;
	}
	
	pPool->pTailNode->pPrev = pPool->pHeadNode;
#if MPL_USING_IN_KENERL	
	vos_dlkClean(pPool->nameList);
#endif
	if(pPool->hMtx)vos_mutexUnlock(pPool->hMtx);
}

void vos_mplDestroy(HVOSMPL hPool)
{
	PS_MPL_POOL pPool = (PS_MPL_POOL)hPool;		
	
	vos_mplFreeAll(hPool);
	
#if MPL_USING_IN_KENERL	
	vos_dlkDestroy(pPool->nameList);
#endif	

	if(pPool->hMtx)vos_mutexClose(pPool->hMtx);
	
	if(2 == pPool->type) //using external node
	{
		vos_free(pPool->pHeadNode);
		vos_free(pPool->pTailNode);
		vos_free(pPool);
	}	
}

unsigned int vos_mplGetPoolAddrStart(HVOSMPL hPool)
{
	PS_MPL_POOL pPool = (PS_MPL_POOL)hPool;		

	return pPool->nPoolAddrStart;
}

HVOSMPL vos_mplGetPoolByMem(void * pMem)
{
	PS_MPL_NODE pNode = NULL;
	
	pNode = (PS_MPL_NODE)(pMem)-1;	
	VOS_ASSERT(pNode->nMagicId == MPL_MEM_MAGIC_ID);
	
	return pNode->hPool;
}


void vos_mplGetSize(HVOSMPL hPool, unsigned long *pTotal, unsigned long *pUsed, unsigned long *pPeakUsed)
{
	int nSize = 0;
	PS_MPL_POOL pPool = (PS_MPL_POOL)hPool;

	if(NULL == hPool)return;	
	VOS_ASSERT(pPool->nMagicId == VOS_MPL_MAGIC_ID);
	
	if(pTotal)*pTotal = pPool->nTotalSize;
	if(pUsed)*pUsed = pPool->nTotalSize - pPool->nRemainSize;
	if(pPeakUsed)*pPeakUsed = pPool->nPeakUsed;	
	
	return;
}

void *vos_mplGetPhyAddr(HVOSMPL hPool, void * pMem)
{
	PS_MPL_POOL pPool = (PS_MPL_POOL)hPool;
	PS_MPL_NODE pNode = NULL;
	
	if((NULL == pPool) || (NULL == pMem))return NULL;
	
	pNode = (PS_MPL_NODE)(pMem)-1;	
	VOS_ASSERT(pNode->nMagicId == MPL_MEM_MAGIC_ID);

	return (void *)pNode->nPhyAddr;
}

//it's only use for type1
void * vos_mplGetKvirAddr(HVOSMPL hPool, void *pPhyAddr)
{
	PS_MPL_POOL pPool = (PS_MPL_POOL)hPool;
	PS_MPL_NODE pHead = NULL;
	PS_MPL_NODE pNode = NULL;
	void *pKvirAddr = NULL;

	if(NULL == hPool)return;	
	VOS_ASSERT(pPool->nMagicId == VOS_MPL_MAGIC_ID);

	if(pPool->hMtx)vos_mutexLock(pPool->hMtx);

	pHead = pPool->pHeadNode;
	pNode = pHead->pNext;	
	while(pNode != pPool->pTailNode)
	{
		if(pNode->nPhyAddr == (unsigned int)pPhyAddr)
		{
			pKvirAddr = (void *)(pNode + 1);
			break;
		}
		
		pNode = pNode->pNext;
	}
	if(pPool->hMtx)vos_mutexUnlock(pPool->hMtx);

	return pKvirAddr;	
}

int vos_mplCheck(HVOSMPL hPool, void * pMem)
{
	PS_MPL_NODE pNode = NULL;
	PS_MPL_NODE pNextNode = NULL;
	PS_MPL_POOL pPool = (PS_MPL_POOL)hPool;
	unsigned int i=0, nTotalUsedSize=0, nFreeSize, nMaxFreeBlock=0;
	PS_MPL_NODE pNodeCheck = NULL;
	int nCheckResult = 0; //not find

	if(NULL == pPool)return 0;
	
	if(pPool->hMtx)vos_mutexLock(pPool->hMtx);

	pNodeCheck = (PS_MPL_NODE)(pMem) - 1;

	pNode = pPool->pHeadNode;

	do
	{
		pNextNode = pNode->pNext;
		if(pNode == pNodeCheck)
		{
			nCheckResult = 1; //find it
			break;		
		}
		pNode = pNextNode;
	}while(pNextNode != (pPool->pTailNode));

	if(pPool->hMtx)vos_mutexUnlock(pPool->hMtx);

	return nCheckResult;
}


unsigned int vos_mplInfo(HVOSMPL hPool, unsigned int *pMaxFreeBlock, int nPrtFlag)
{
	PS_MPL_NODE pNode = NULL;
	PS_MPL_NODE pNextNode = NULL;
	PS_MPL_POOL pPool = (PS_MPL_POOL)hPool;
	unsigned int i=0, nTotalUsedSize=0, nFreeSize, nMaxFreeBlock=0;
	unsigned int nResvHead, nResvTail;

	if((NULL == pPool) || (pPool->pHeadNode == NULL))return 0;
	
	if(pPool->hMtx)vos_mutexLock(pPool->hMtx);

	pNode = pPool->pHeadNode;
	while((pNode != NULL) && (pNode->pNext != NULL))
	{
		VOS_ASSERT(MPL_MEM_MAGIC_ID == pNode->nMagicId);
		i++;
		pNextNode = pNode->pNext;

		if(nPrtFlag == 2)
		{
			if(pNode->nUsedSize != 0) //not include head
				vosPrintf("%03d:[usrAddr:0x%x, phyAddr=0x%x, usedSize:0x%x], [%s, Line:%d]\r\n", i-1, (unsigned int)pNode + sizeof(S_MPL_NODE), pNode->nPhyAddr, pNode->nUsedSize, pNode->szFile, pNode->nLine);
		}
			
		nFreeSize = pNextNode->nStartAddr - pNode->nStartAddr - pNode->nUsedSize;
		if(nFreeSize > nMaxFreeBlock)nMaxFreeBlock = nFreeSize;
		pNode = pNextNode;
	}

	nTotalUsedSize = pPool->nTotalSize - pPool->nRemainSize;
	if(pPool->hMtx)vos_mutexUnlock(pPool->hMtx);
	if(pMaxFreeBlock)*pMaxFreeBlock = nMaxFreeBlock;
	if(nPrtFlag)
	{
		vosPrintf("Total=0x%x, Used=0x%x, Peak=0x%x, Free=0x%x, MaxFreeBlock=0x%x\r\n", pPool->nTotalSize, nTotalUsedSize, pPool->nPeakUsed, pPool->nRemainSize, nMaxFreeBlock);
		vosPrintf("----[above]hPool(0x%x)-%s, nPhyAddr:0x%x, nPoolAddr:0x%x\r\n", pPool, pPool->szPoolName, pPool->nPhyAddr, pPool->nPoolAddr);
	}

	return nTotalUsedSize;
}

unsigned int vos_mplInfoAll(int nPrtFlag)
{
	HVOSMPL hPool;
	DLK_NODE *pShow, *pHead;

	if(g_hMpoolDlk == 0)return 1;
	
	pHead = vos_dlkGetHead(g_hMpoolDlk);
	DLK_WALK_LOOP_START(pShow, pHead)
	{
		hPool = (HVOSMPL)*(unsigned long *)(pShow + 1);
		vos_mplInfo(hPool, NULL, nPrtFlag);
	}
	DLK_WALK_LOOP_END(pShow, pHead)
}

int test_mpl(int nFlag, int nPar1, int nPar2)
{
	static HVOSMPL test_hPool = NULL;
	void *pBaseBuf, *pBaseBuf2;
	int nRet = 0;
	unsigned int nTotalUsedSize, MaxFreeBlock = 0;

	switch(nFlag)
	{
		case 0:
			if(test_hPool == NULL)break;
			nTotalUsedSize = vos_mplInfo(test_hPool, &MaxFreeBlock, nPar1);
			vosPrintf("nTotalUsedSize=%u, MaxFreeBlock=%u\r\n", nTotalUsedSize, MaxFreeBlock);
			break;

		case 1:
			pBaseBuf = vos_malloc(nPar1);
			test_hPool = vos_mplCreatePool("test_mpl", pBaseBuf, nPar1);
			nRet = (int)test_hPool;
			break;
			
		case 2:
			if(test_hPool == NULL)break;
			nRet = (int)vos_mplAlloc(test_hPool, nPar1, nPar2, __FILE__, __LINE__);
			break;

		case 3:
			vos_mplFree(test_hPool, (void *)nPar1);
			break;

		case 4:
			vos_mplFreeAll(test_hPool);
			break;

		case 5:
			vos_mplDestroy(test_hPool);
			test_hPool = NULL;
			break;

		case 10:
			pBaseBuf = vos_malloc(nPar1);
			pBaseBuf2 = vos_malloc(nPar1);
			test_hPool = vos_mplCreatePoolPhy("test_mpl", 1, pBaseBuf, nPar1, pBaseBuf2, 1);
			vosPrintf("pBaseBuf:0x%x, 0x%x\r\n", pBaseBuf, pBaseBuf2);
			nRet = (int)test_hPool;
			break;

		case 11:
			pBaseBuf = vos_malloc(nPar1);
			test_hPool = vos_mplCreatePoolPhy("test_mpl", 2, NULL, nPar1, pBaseBuf, 1);
			nRet = (int)test_hPool;
			break;
			
	#if MPL_USING_IN_KENERL 
		case 12:
			pBaseBuf = VOS_KMALLOC(nPar1); //nPar2: type=1 or 2
			pBaseBuf2 = (void *)vos_memVir2Phy((unsigned long)pBaseBuf);

			test_hPool = vos_mplCreatePoolPhy("test_mpl", nPar2, pBaseBuf, nPar1, pBaseBuf2, 1);
			vosPrintf("pBaseBuf:0x%x, 0x%x\r\n", pBaseBuf, pBaseBuf2);
			nRet = (int)test_hPool;
			break;
			
		case 13:
			pBaseBuf = vos_kmalloc_coherent(nPar1, (void *)&pBaseBuf2);
			
			test_hPool = vos_mplCreatePoolPhy("test_mpl", nPar2, pBaseBuf, nPar1, pBaseBuf2, 1);
			vosPrintf("pBaseBuf:0x%x, 0x%x\r\n", pBaseBuf, pBaseBuf2);
			nRet = (int)test_hPool;
			break;
	#endif
	
		case 14:
			nRet = (int)vos_mplGetPhyAddr(test_hPool, (void *)nPar1);
			break;
		
		case 15:
			nRet = (int)vos_mplGetKvirAddr(test_hPool, (void *)nPar1);
			break;

		default:
			break;
	}

	return nRet;
}

#if MPL_USING_IN_KENERL	
EXPORT_SYMBOL(vos_mplGetPoolAddrStart);
EXPORT_SYMBOL(vos_mplCreatePoolPhy);
EXPORT_SYMBOL(vos_mplGetSize);
EXPORT_SYMBOL(vos_mplGetKvirAddr);
EXPORT_SYMBOL(vos_mplAlloc);
EXPORT_SYMBOL(vos_mplDestroy);
EXPORT_SYMBOL(vos_mplFreeAll);
EXPORT_SYMBOL(vos_mplInfo);
EXPORT_SYMBOL(vos_mplFree);
#endif


#ifdef __cplusplus
}
#endif // __cplusplus


