/*
 * qxspi.c --  A driver for the xspi in Quatro SOCs
 *
 * Copyright (c) 2014, 2015, The Linux Foundation. All rights reserved.
 *
 * Copyright (c) 2010-2011 Abc Incorporated
 *
 * Copyright (c) 2016, QBit Semiconductor LTD.
 * Quasar XSPI kernel driver
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
// =========================================================
//
//  $DateTime: 2022/01/19 08:48:00 $
//  $Change: 57110 $
//
// =========================================================

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/icu.h>
#include <quasar/qioctl.h>

static int dbg_ints = 0;
module_param(dbg_ints, int, 0644);
MODULE_PARM_DESC(dbg_ints, "print mbox interrupt info");

static int xspi_intenable=0;
module_param(xspi_intenable, int, 0644);
MODULE_PARM_DESC(xspi_intenable, "XSPI Interrupt enable");

#define XSPI_DEVNAME	"qxspi"

#define MAX_XSPIS 			2 /* get this from a header! */
#define MAX_XSPIIRQS   		2
#define MAX_XSPIIOREGS		1

// convert from regset to spi core index, each core have 3 regset (cs0~cs2)
#define REGSET_TO_SPICORE(regset)   (regset / Spi1RegSet0)

struct quasar_io_region {
	u32						start;
	u32						end;
	volatile u8	__iomem		*base;
};

struct quasar_xspi {
	struct cdev				cdev;
	struct device			*frdev;

	struct quasar_io_region	ioregs[MAX_XSPIIOREGS];
	int						nioregs;
	
	unsigned long			irqs[MAX_XSPIIRQS];
	int						nirqs;
	
	int						refs;				/* reference count */
	int						minor;				/* device minor number */
    wait_queue_head_t	    mbq;				/* wait-queue for mailbox interrupt */
	
	spinlock_t				lock;				/* lock for m3 mail ring buffer access */
    struct semaphore        donesem[MAX_XSPIS];	/* interrupt counting for XSPIxD*/     
    struct semaphore        errsem[MAX_XSPIS];	/* interrupt counting for XSPIxE*/     
};

static int g_nxspis = 0;

static volatile u8 __iomem* xspi_addr(struct quasar_xspi* qxspi, u32 reg)
{
	volatile u8 __iomem* addr = 0;
 	int      i;

	/* check for existing mapping */
	for (i = 0; i < qxspi->nioregs; i++) {
		if(reg >= qxspi->ioregs[i].start && reg < qxspi->ioregs[i].end) {
			return(qxspi->ioregs[i].base + (reg - qxspi->ioregs[i].start));
		}
	}
	printk(KERN_WARNING XSPI_DEVNAME "xspi_addr - could not map reg %08x\n", reg);
	return addr;
}

static int xspi_readl(struct quasar_xspi* qxspi, u32 reg, unsigned* val)
{
	volatile u32 rv = 0;
	int ret = 0;
	volatile u8 __iomem* addr = xspi_addr(qxspi, reg);
	
	if(addr == NULL)
		return -1;
	rv = readl(addr);
	/*printk("readl %08X [%08X] -> %08X\n", reg, addr, rv);*/
	*val = (unsigned long)rv;
	return ret;
}

static int xspi_writel(struct quasar_xspi* qxspi, u32 reg, u32 val)
{
	volatile u8 __iomem* addr = xspi_addr(qxspi, reg);
	if(addr == NULL)
		return -1;
	/* printk("writel %08X [%08X] <= %08x\n", reg, addr, val);*/
	writel(val, addr);
	return 0;
}

static int qxspi_int_wait(struct quasar_xspi *qxspi, u32 timeoutms)
{
    long jiffies;
    struct timespec tspec;
    
    tspec.tv_sec = timeoutms / 1000;
    tspec.tv_nsec = (timeoutms - tspec.tv_sec * 1000) * 1000000;
    jiffies = timespec_to_jiffies(&tspec);
    return down_timeout(&qxspi->donesem[qxspi->minor], jiffies);
}

//////////////////////////////////////////////////////////////////////////////////////
//
// SPI access routines
//
/*
**      Functions to configure the SPI registers.
**
**      The SPI Registers, selected by the eSPI_REGSETx enum.
*/
#if defined(Q6600)
static u32 ERRRegs[] = {    SPI0ERR,   SPI1ERR,   SPI2ERR,   
                            SPI3ERR,   SPI4ERR,   SPI5ERR,   
                            SPI6ERR,   SPI7ERR,   SPI8ERR   };
static u32 CFGRegs[] = {    SPI0CFG0,  SPI0CFG1,  SPI0CFG2,  
                            SPI1CFG0,  SPI1CFG1,  SPI1CFG2,  
                            SPI2CFG0,  SPI2CFG1,  SPI2CFG2,  
                            SPI3CFG0,  SPI3CFG1,  SPI3CFG2,  
                            SPI4CFG0,  SPI4CFG1,  SPI4CFG2,  
                            SPI5CFG0,  SPI5CFG1,  SPI5CFG2,  
                            SPI6CFG0,  SPI6CFG1,  SPI6CFG2,  
                            SPI7CFG0,  SPI7CFG1,  SPI7CFG2,  
                            SPI8CFG0,  SPI8CFG1,  SPI8CFG2  };
static u32 STARegs[] = {    SPI0STAT0, SPI0STAT1, SPI0STAT2, 
                            SPI1STAT0, SPI1STAT1, SPI1STAT2, 
                            SPI2STAT0, SPI2STAT1, SPI2STAT2, 
                            SPI3STAT0, SPI3STAT1, SPI3STAT2, 
                            SPI4STAT0, SPI4STAT1, SPI4STAT2, 
                            SPI5STAT0, SPI5STAT1, SPI5STAT2, 
                            SPI6STAT0, SPI6STAT1, SPI6STAT2, 
                            SPI7STAT0, SPI7STAT1, SPI7STAT2, 
                            SPI8STAT0, SPI8STAT1, SPI8STAT2 };
static u32 DMCRegs[] = {    SPI0CMD0,  SPI0CMD1,  SPI0CMD2,  
                            SPI1CMD0,  SPI1CMD1,  SPI1CMD2,  
                            SPI2CMD0,  SPI2CMD1,  SPI2CMD2,  
                            SPI3CMD0,  SPI3CMD1,  SPI3CMD2,  
                            SPI4CMD0,  SPI4CMD1,  SPI4CMD2,  
                            SPI5CMD0,  SPI5CMD1,  SPI5CMD2,  
                            SPI6CMD0,  SPI6CMD1,  SPI6CMD2,  
                            SPI7CMD0,  SPI7CMD1,  SPI7CMD2,  
                            SPI8CMD0,  SPI8CMD1,  SPI8CMD2  };
static u32 LMCRegs[] = {    SPI0LIT0,  SPI0LIT1,  SPI0LIT2,  
                            SPI1LIT0,  SPI1LIT1,  SPI1LIT2,  
                            SPI2LIT0,  SPI2LIT1,  SPI2LIT2,  
                            SPI3LIT0,  SPI3LIT1,  SPI3LIT2,  
                            SPI4LIT0,  SPI4LIT1,  SPI4LIT2,  
                            SPI5LIT0,  SPI5LIT1,  SPI5LIT2,  
                            SPI6LIT0,  SPI6LIT1,  SPI6LIT2,  
                            SPI7LIT0,  SPI7LIT1,  SPI7LIT2,  
                            SPI8LIT0,  SPI8LIT1,  SPI8LIT2  };
static u32 DMARegs[] = {    SPI0DCTL,  SPI1DCTL,  SPI2DCTL,  
                            SPI3DCTL,  SPI4DCTL,  SPI5DCTL,  
                            SPI6DCTL,  SPI7DCTL,  SPI8DCTL  };
static u32 DWRRegs[] = {    SPI0DADDR, SPI1DADDR, SPI2DADDR, 
                            SPI3DADDR, SPI4DADDR, SPI5DADDR, 
                            SPI6DADDR, SPI7DADDR, SPI8DADDR };
static u32 DRDRegs[] = {    SPI0RADDR, SPI1RADDR, SPI2RADDR, 
                            SPI3RADDR, SPI4RADDR, SPI5RADDR, 
                            SPI6RADDR, SPI7RADDR, SPI8RADDR };
#else
static u32 ERRRegs[] = { SPI0ERR,   SPI1ERR   };
static u32 CFGRegs[] = { SPI0CFG0,  SPI0CFG1,  SPI0CFG2,  SPI1CFG0,  SPI1CFG1,  SPI1CFG2  };
static u32 STARegs[] = { SPI0STAT0, SPI0STAT1, SPI0STAT2, SPI1STAT0, SPI1STAT1, SPI1STAT2 };
static u32 DMCRegs[] = { SPI0CMD0,  SPI0CMD1,  SPI0CMD2,  SPI1CMD0,  SPI1CMD1,  SPI1CMD2  };
static u32 LMCRegs[] = { SPI0LIT0,  SPI0LIT1,  SPI0LIT2,  SPI1LIT0,  SPI1LIT1,  SPI1LIT2  };
static u32 DMARegs[] = { SPI0DCTL,  SPI1DCTL  };
static u32 DWRRegs[] = { SPI0DADDR, SPI1DADDR };
static u32 DRDRegs[] = { SPI0RADDR, SPI1RADDR };
#endif
    
/*      This is the default literal mode readbits setting.   
**      (Defined here in order to avoid using magic numbers.)
*/
#define DEFAULT_READBITS   7
#define DEFAULT_WRITEBITS  7

/*
**      The default for initial re-try counter.
*/
#define WAIT_CYCLES         0x80000

/*
**      SPI Interrupts that are monitored, selected by the eSPI_REGSETx enum.
*/
#define SPI0_ERRS           GICD_SPISR0__XSPI0E__MASK
#define SPI1_ERRS           GICD_SPISR1__XSPI1E__MASK

#define SPI0_DONE           GICD_SPISR0__XSPI0D__MASK
#define SPI1_DONE           GICD_SPISR1__XSPI1D__MASK

#define SPI0_INTS           (SPI0_ERRS | SPI0_DONE)
#define SPI1_INTS           (SPI1_DONE | SPI1_ERRS)

#define SPI0SPINT_MASK      (SPI0SPIINT__D__MASK | SPI0SPIINT__E__MASK)
#define SPI1SPINT_MASK      (SPI1SPIINT__D__MASK | SPI1SPIINT__E__MASK)

static u32 INT_SPIint[] =  { SPI0SPIINT, SPI0SPIINT, SPI0SPIINT, SPI1SPIINT, SPI1SPIINT, SPI1SPIINT};
static u32 SPI_SPitMask[] =  { SPI0SPINT_MASK, SPI0SPINT_MASK, SPI0SPINT_MASK, SPI1SPINT_MASK, SPI1SPINT_MASK, SPI1SPINT_MASK};

static u32 ErrsInts[] = { SPI0_ERRS, SPI0_ERRS, SPI0_ERRS, SPI1_ERRS, SPI1_ERRS, SPI1_ERRS };
static u32 DoneInts[] = { SPI0_DONE, SPI0_DONE, SPI0_DONE, SPI1_DONE, SPI1_DONE, SPI1_DONE };
static u32 SPI_Ints[] = { SPI0_INTS, SPI0_INTS, SPI0_INTS, SPI1_INTS, SPI1_INTS, SPI1_INTS };
static u32 GICD_SPISR[] = { GICD_SPISR0, GICD_SPISR0, GICD_SPISR0, GICD_SPISR1, GICD_SPISR1, GICD_SPISR1 };

#define IS_DONE_INTx(i,r)    (i & DoneInts[r]) ? 1 : 0
#define IS_ERRS_INTx(i,r)    (i & ErrsInts[r]) ? 1 : 0

/*
**  Clears the spi interrupts
*/
static int SPI_ClearINTSx(struct quasar_xspi *qxspi, eSPI_REGSETx reg_set)
{
    xspi_writel(qxspi, INT_SPIint[reg_set], SPI_SPitMask[reg_set]);
    return 0;
}

/*
**  Configures: SPI0CMD0, SPI0CMD1, SPI0CMD2, SPI1CMD0, SPI1CMD1, SPI1CMD2
*/
int SPI_ConfigureDM_REGx (struct quasar_xspi *qxspi, eSPI_REGSETx reg_set, sSPI_DMWORD *pCfg)
{
    unsigned int t, v = 0;
    
    t = (pCfg->OP << SPI0CMD0__OP__SHIFT) & SPI0CMD0__OP__MASK,  v |= t; 
    t = (pCfg->WB << SPI0CMD0__WB__SHIFT) & SPI0CMD0__WB__MASK,  v |= t;
    t = (pCfg->NI << SPI0CMD0__NI__SHIFT) & SPI0CMD0__NI__MASK,  v |= t; 
    t = (pCfg->DA << SPI0CMD0__DA__SHIFT) & SPI0CMD0__DA__MASK,  v |= t; 
    t = (pCfg->RB << SPI0CMD0__RB__SHIFT) & SPI0CMD0__RB__MASK,  v |= t;

    // write without the enable
    xspi_writel(qxspi, DMCRegs[reg_set], v);
    // write again with the enable
    xspi_writel(qxspi, DMCRegs[reg_set], v | SPI0CMD0__S__MASK);
    return 0;
}

/*
**  Configures: SPI0LIT0, SPI0LIT1, SPI0LIT2, SPI1LIT0, SPI1LIT1, SPI1LIT2
*/
int SPI_ConfigureLM_REGx (struct quasar_xspi *qxspi, eSPI_REGSETx reg_set, sSPI_LMWORD *pCfg)
{
    unsigned int t, v = 0;
    
    t = (pCfg->OP  << SPI0LIT0__OP__SHIFT)  & SPI0LIT0__OP__MASK,  v |= t;
    t = (pCfg->DL  << SPI0LIT0__DL__SHIFT)  & SPI0LIT0__DL__MASK,  v |= t; 
    t = (pCfg->ARG << SPI0LIT0__ARG__SHIFT) & SPI0LIT0__ARG__MASK, v |= t; 
    t = (pCfg->WD  << SPI0LIT0__WD__SHIFT)  & SPI0LIT0__WD__MASK,  v |= t;

    // write without the enable
    xspi_writel(qxspi, LMCRegs[reg_set], v);
    // write again with the enable
    xspi_writel(qxspi, LMCRegs[reg_set], v | SPI0LIT0__S__MASK);
    return 0;
}

/*
**  Configures: SPI0DCTL, SPI1DCTL
*/
int SPI_ConfigureDMA_REGx (struct quasar_xspi *qxspi, eSPI_REGSETx reg_set, sSPI_DMACTL *pCfg)
{
    unsigned int t, v = 0;
    int spicore=REGSET_TO_SPICORE(reg_set);
    
    t = (pCfg->PC << SPI0DCTL__PC__SHIFT) & SPI0DCTL__PC__MASK,  v |= t;
    t = (pCfg->RB << SPI0DCTL__RB__SHIFT) & SPI0DCTL__RB__MASK,  v |= t; 
    t = (pCfg->NW << SPI0DCTL__NW__SHIFT) & SPI0DCTL__NW__MASK,  v |= t; 
    t = (pCfg->NA << SPI0DCTL__NA__SHIFT) & SPI0DCTL__NA__MASK,  v |= t;
    t = (pCfg->I  << SPI0DCTL__I__SHIFT)  & SPI0DCTL__I__MASK,   v |= t;

    xspi_writel(qxspi, DMARegs[spicore], v);
    return 0;
}

/*
**  Enables Command Mode (as opposed to ROM or PROG modes)
*/
int SPI_EnableCmdMode(struct quasar_xspi *qxspi, eSPI_REGSETx reg_set, int useReverseEndian)
{
    u32 cfst;
    
    /*  Read the current settings in the configuration reg
    */
    xspi_readl(qxspi, CFGRegs[reg_set], (unsigned*)&cfst);

    /*  Clear or set the endian bit
    */
    if (useReverseEndian == 0)
        cfst &= ~SPI0CFG0__EN__MASK;
    else
        cfst |= SPI0CFG0__EN__MASK;

    /*  Set the spi interface enable bit
    */
    cfst |= SPI0CFG0__E__MASK;

    /*  Write these changes back
    */
    xspi_writel(qxspi, CFGRegs[reg_set], cfst);
    
    return 0;
}

/*
**  Reset the internal FIFO pointer.
*/
void SPI_FifoClrCnt (struct quasar_xspi *qxspi, eSPI_REGSETx reg_set)
{
    u32 val;
    int spicore=REGSET_TO_SPICORE(reg_set);
    
    xspi_readl(qxspi, DMARegs[spicore], &val);
    xspi_writel(qxspi, DMARegs[spicore], val | SPI0DCTL__I__MASK);
}

/*
**  Clear any FIFO errors.
*/
void SPI_FifoClrErr (struct quasar_xspi *qxspi, eSPI_REGSETx reg_set)
{
    u32 val;
    int spicore=REGSET_TO_SPICORE(reg_set);
    
    xspi_readl(qxspi, ERRRegs[spicore], &val);
    xspi_writel(qxspi, ERRRegs[spicore], val & ~SPI0ERR__ER__MASK);
}

/*
**  Return -1 if FIFO error is found.
*/
int SPI_CheckFIFOError (struct quasar_xspi *qxspi, eSPI_REGSETx reg_set)
{
    u32 val;
    int spicore=REGSET_TO_SPICORE(reg_set);
    
    xspi_readl(qxspi, ERRRegs[spicore], (unsigned*)&val);
    
    if (val & SPI0ERR__ER__MASK)
        return -1;
    else
        return 0;
}

/**  Reads the spi interrupts in the main exception cause register.
*/
static int SPI_CheckINTSx(struct quasar_xspi *qxspi, eSPI_REGSETx reg_set)
{
    u32 val;
    
    //printk("SPI_CheckINTSx: reg_set=%d, reg=0x%08X\n", reg_set, GICD_SPISR[reg_set]);
    xspi_readl(qxspi, GICD_SPISR[reg_set], (unsigned*)&val);
    
    return (val & SPI_Ints[reg_set]);
}

static int SPI_WaitForINTSxPool(struct quasar_xspi *qxspi, eSPI_REGSETx reg_set) 
{
    u32 retry = WAIT_CYCLES;
    u32 sta = 0;

    while (retry--)
    {
        sta = SPI_CheckINTSx(qxspi, reg_set);

        /*  Break if DONE,
        *   Return fail for ERRS or retries exceeded
        */
        if ( IS_DONE_INTx(sta, reg_set))
                break;

        if ( IS_ERRS_INTx(sta, reg_set))
                return -1;

        if (retry == 0)
            return -1;
    }
    return 0;
}

/*
**  Return 0 if a done interrupt is read.
**  Return -1 if an error is found or retries exceeded.
*/
static int SPI_WaitForINTSx(struct quasar_xspi *qxspi, eSPI_REGSETx reg_set) 
{
    if (xspi_intenable)
    {
        u32 timeoutms=100;
        
        return qxspi_int_wait(qxspi, timeoutms);
    }
    else
    {
        return SPI_WaitForINTSxPool(qxspi, reg_set);
    }
}


/******************************************************************************
* Name:         qxspi_directmodecmd
*
* Description:  Send an SPI Direct Mode Command to the attached  
*               serial device.                            
*
* Parameters:   Pointer to sSPI_DMCMD structure.
*
* Return:        0   - Operation completed.
*               -1   - Operation failed.
*               Failure sources include timeouts, register 
*               operations, and fifo use. 
*
***/
static int qxspi_directmodecmd(struct quasar_xspi *qxspi, sSPI_DMCMD *pCmd)
{
    sSPI_DMWORD     cmdWord;
    sSPI_DMACTL     ctlWord;
    int spicore=REGSET_TO_SPICORE(pCmd->reg_set);

    /*  Zero out the command words.
    */
    memset(&cmdWord, 0x00, sizeof(sSPI_DMWORD));
    memset(&ctlWord, 0x00, sizeof(sSPI_DMACTL));

    /*  Enable the interface for command use
    */
    SPI_EnableCmdMode(qxspi, pCmd->reg_set, 0);

    /*  Clear any fifo errors
    */
    SPI_FifoClrErr(qxspi, pCmd->reg_set);

    /*  Clear any fifo counts
    */
    SPI_FifoClrCnt(qxspi, pCmd->reg_set);

    /*  Clear any spi bits in the exception clear reg by
    *   writing them to it..
    */
    SPI_ClearINTSx(qxspi, pCmd->reg_set);

    /*  Set up the dma address registers
    */
    if (pCmd->rdCount)
        xspi_writel(qxspi, DRDRegs[spicore], (u32)(size_t)pCmd->pRdBuf);
    if (pCmd->wrCount)
        xspi_writel(qxspi, DWRRegs[spicore], (u32)(size_t)pCmd->pWrBuf);

    /*  If the addresses are separate, then set the dma control
    *   to different addresses and invalidate the dma buffer.
    */
    if (pCmd->pRdBuf != pCmd->pWrBuf)
    {
        ctlWord.RB = 1;
        ctlWord.I  = 1;
    }

    /*  Set the DM Operation
    */
    cmdWord.OP =  pCmd->OP;
    
    /*  no data alignment
    */
    cmdWord.DA = 0;

    /*  Allow the SPI to interrupt
    */
    cmdWord.NI = 0;

    /*  Act based on OP type.  
    */
    if (pCmd->OP != BusyCheck)
    {
        /*  All operations but the busy check involve 
         *  writing something, so if not checking a busy,
         *  set the number of bits to write.
        */
        cmdWord.WB = (pCmd->wrCount > 0) ? (pCmd->wrCount*8)-1 : 0;
    }

    /*  The one direct mode read command needs it's
    *   Read bits count set
    */
    if (pCmd->OP == WriteThenWaitBeforeRead)
    {
        cmdWord.RB =  (pCmd->rdCount > 0) ? (pCmd->rdCount*8)-1 : 0;
    }

    /*  Write the dma control word first
    */
    SPI_ConfigureDMA_REGx(qxspi, pCmd->reg_set, &ctlWord);

    /*  Write the command word.  This triggers the spi command.
    */
    SPI_ConfigureDM_REGx(qxspi, pCmd->reg_set, &cmdWord);

    /*  return failed if done is not found in the main int cause register...
    */
    if (SPI_WaitForINTSx(qxspi, pCmd->reg_set))
        return -1;
    
    /*  Check the fifo status...
    */
    if (SPI_CheckFIFOError(qxspi, pCmd->reg_set))
        return -1;
    
    /*  Clear any fifo errors
    */
    SPI_FifoClrErr(qxspi, pCmd->reg_set);

    /*  Clear any fifo counts
    */
    SPI_FifoClrCnt(qxspi, pCmd->reg_set);
    
    return 0;
} 

/******************************************************************************
* Name:         qxspi_literalmodecmd
*
* Description:  Send an SPI Literal Mode Command to the attached  
*               serial device.                            
*
* Parameters:   Pointer to sSPI_LMCMD structure.
*
* Return:        0 - Operation completed.
*               -1 - Operation failed.
*
*               Failure sources include timeouts, register 
*               operations, and fifo use. 
*
***/
static int qxspi_literalmodecmd (struct quasar_xspi *qxspi, sSPI_LMCMD *pCmd) 
{
    sSPI_LMWORD cmdWord;
    u32      readStat = 0;
    
    /*  Initialize the Literal Mode command word.
    */
    memset(&cmdWord, 0x00, sizeof(sSPI_LMWORD));
    
    /*  Enable the interface for command use
    */
    SPI_EnableCmdMode(qxspi, pCmd->reg_set, 0);

    /*  Get the LM operation type from the cmd structure
    */
    cmdWord.OP  =  pCmd->OP;

    /*  If the operation type has a read ONLY involved, set the
    *   read data length for the status register, and
    *   set the read flag for later...
    */
    if (pCmd->OP == GetReadDataOnly)
    {
        cmdWord.DL = DEFAULT_READBITS;
        readStat = 1;
    }
    
    else
    {
        /*  All the other modes use a command type.
        */
        cmdWord.ARG = pCmd->CMD;

        /*  If operation type has read data, set the data length
         *  like above, and set the read flag for later
        */
        if (pCmd->OP == SendCmdOnly_GetReadData)
        {
            cmdWord.DL = DEFAULT_READBITS;
            readStat = 1;
        }

        /*  If command type has send data, copy that into the 
         *  command word
        */
        if (pCmd->OP == SendCmdandDataOnly)
        {
            cmdWord.WD = pCmd->DATA;
            cmdWord.DL = DEFAULT_WRITEBITS;
        }
    }
    
    /*  Clear any spi interrupt bits.
    */
    SPI_ClearINTSx(qxspi, pCmd->reg_set);

    /*  Write the command word which initiates the LM function.
    */
    SPI_ConfigureLM_REGx(qxspi, pCmd->reg_set, &cmdWord);

    /*  return failed if done is not found in the main int cause register...
    */
    if (SPI_WaitForINTSx(qxspi, pCmd->reg_set))
        return -1;

    /*  And if the read flag is set, read the returned byte
     *  from the spi status register
    */
    if (readStat == 1)
    {
        u32 val;
        
        xspi_readl(qxspi, STARegs[pCmd->reg_set], &val); 
        pCmd->DATA = val & SPI0STAT0__STAT__MASK;
    }

    /*  Return 0 errors 
    */
    return 0;
}

static irqreturn_t qxspi_interrupt(int irq, void *dev_id)
{
	struct quasar_xspi *qxspi = (struct quasar_xspi *)dev_id;
	int ret=IRQ_HANDLED;

    //printk("<%d-%d>", qxspi->minor, irq); 
	if (irq == qxspi->irqs[0])
    {
        up(&qxspi->errsem[qxspi->minor]);
    }
    else if (irq == qxspi->irqs[1])
    {
        up(&qxspi->donesem[qxspi->minor]);
    }
    else
    {
        printk(XSPI_DEVNAME "Spurious irq %d\n", irq);
    }

    wake_up_interruptible(&qxspi->mbq);
    
	return ret;
}

static int qxspi_open(struct inode* inode, struct file* filp)
{
	struct quasar_xspi *qxspi;
	int minor;
	
    qxspi = container_of(inode->i_cdev, struct quasar_xspi, cdev);
	minor = iminor(inode);
	if(minor < 0 || minor >= g_nxspis)
		return -ENODEV;
	if(qxspi->refs > 0)
		return -EBUSY;
	qxspi->refs++;

	filp->private_data = qxspi;
	return 0;
}

static int qxspi_release(struct inode* inode, struct file* filp)
{
	struct quasar_xspi* qxspi;
	
	qxspi = (struct quasar_xspi*)filp->private_data;
	if(qxspi->refs <= 0)
		return -EBADF;
	qxspi->refs--;
	filp->private_data = NULL;
	return 0;
}

static unsigned int qxspi_poll(struct file *filp, poll_table *wait)
{
	struct quasar_xspi* qxspi;
	unsigned int mask = 0;
	unsigned long flags;

	qxspi = (struct quasar_xspi*)filp->private_data;
    poll_wait(filp, &qxspi->mbq, wait);
	spin_lock_irqsave(&qxspi->lock, flags);
	if(qxspi->donesem[qxspi->minor].count > 0) {
		mask |= POLLIN | POLLRDNORM; /* readable */
	}
	spin_unlock_irqrestore(&qxspi->lock, flags);
	return mask;
}

static long qxspi_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct quasar_xspi* qxspi;
	struct q_regio ioval;
    sSPI_DMCMD  dmcmd;
    sSPI_LMCMD  lmcmd;
	int ret = 0;
	
	qxspi = (struct quasar_xspi*)filp->private_data;
	if(! qxspi)
		return -ENODEV;

	switch(cmd)
	{
	case QSETREG:
		ret = copy_from_user(&ioval, (void*)arg, sizeof(struct q_regio));      
		ret = xspi_writel(qxspi, ioval.reg, (unsigned long)ioval.val);
		break;
	case QGETREG:
		ret = copy_from_user(&ioval, (void*)arg, sizeof(struct q_regio));
		ret = xspi_readl(qxspi, ioval.reg, (unsigned*)&ioval.val);
		ret = copy_to_user((void*)arg, &ioval, sizeof(struct q_regio));
		break;
	case QXSPI_WAIT_INTERRUPT:
		ret = qxspi_int_wait(qxspi, arg);
		break;
    case QXSPI_LITERALMODE_CMD:
        ret = copy_from_user(&lmcmd, (void*)arg, sizeof(sSPI_LMCMD));
		ret = qxspi_literalmodecmd(qxspi, &lmcmd);
		ret = copy_to_user((void*)arg, &lmcmd, sizeof(sSPI_LMCMD));
        break;        
    case QXSPI_DIRECTMODE_CMD:
        ret = copy_from_user(&dmcmd, (void*)arg, sizeof(sSPI_DMCMD));
		ret = qxspi_directmodecmd(qxspi, &dmcmd);
		ret = copy_to_user((void*)arg, &dmcmd, sizeof(sSPI_DMCMD));
        break;
	default:
		printk(KERN_WARNING XSPI_DEVNAME "Bad ioctl %d\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static struct file_operations qxspi_ops = {
	.owner		= THIS_MODULE,
	.open		= qxspi_open,
	.poll		= qxspi_poll,    
	.release	= qxspi_release,
	.unlocked_ioctl	= qxspi_ioctl,
	.compat_ioctl	= qxspi_ioctl 
};

static int __init qxspi_probe(struct platform_device *pdev)
{
	struct resource	*ioreg, *irqs;
	struct quasar_xspi *qxspi;
	dev_t  qxspin;
	int irq;
	int ret = 0;
	int irqrecs = 0;
	
	if(g_nxspis >= MAX_XSPIS)
		return -ENXIO;

	qxspi = kzalloc(sizeof(struct quasar_xspi), GFP_KERNEL);
	if(! qxspi) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	qxspi->frdev = &pdev->dev;
	qxspi->nioregs = 0;
	qxspi->nirqs = 0;
	
    printk(KERN_WARNING XSPI_DEVNAME "init qxspi_probe, xspi_intenable=%d ...\n", xspi_intenable); 
	do {
		ioreg = platform_get_resource(pdev, IORESOURCE_MEM, qxspi->nioregs);
		if (ioreg) {
			qxspi->ioregs[qxspi->nioregs].start = ioreg->start;
			qxspi->ioregs[qxspi->nioregs].end = ioreg->end;
			qxspi->ioregs[qxspi->nioregs].base =
				ioremap(ioreg->start, ioreg->end - ioreg->start + 4);
            if (!qxspi->ioregs[qxspi->nioregs].base) {
				ret = -ENOMEM;
				dev_dbg(&pdev->dev, "could not map I/O memory %0d\n", qxspi->nioregs);
				goto out_abort;
			}
			/*
			printk("qxspi - mapped qxspi %px %x to %x at %x\n", 
					qxspi, ioreg->start, ioreg->end, qxspi->ioregs[qxspi->nioregs].base);
			*/
			//printk(KERN_EMERG "qxspi - register %d  reg: 0x%x - 0x%x\n", qxspi->nioregs, qxspi->ioregs[qxspi->nioregs].start, qxspi->ioregs[qxspi->nioregs].end);
			qxspi->nioregs++;
		}
	} while (ioreg && (qxspi->nioregs < MAX_XSPIIOREGS));

	if (qxspi->ioregs[0].start != 0x400c000)
	{	//SPI0 driver for fax move to FW.
	    if (xspi_intenable)
	    {
	        /* init wait queue for mailbox */
	        init_waitqueue_head(&qxspi->mbq);
	        do {
	            irqs = platform_get_resource(pdev, IORESOURCE_IRQ, irqrecs++);
	            if (irqs) {
					//printk(KERN_EMERG "qxspi - interrupt %d  irq: %d - %d\n", qxspi->nirqs, irqs->start, irqs->end);
	                for(irq = irqs->start; irq <= irqs->end; irq++) {
	                    qxspi->irqs[qxspi->nirqs++] = irq;
	                    ret = request_irq(irq, qxspi_interrupt, 0, "qxspi", qxspi);
	                    if (ret) {
	                        dev_dbg(&pdev->dev,
	                                  "could not request irq %d\n", irq);
	                        goto out_abort;
	                    }
	                }
	            }
	        } while (irqs && (qxspi->nirqs < MAX_XSPIIRQS));
	    }
	}
		
	spin_lock_init(&qxspi->lock);

	cdev_init(&qxspi->cdev, &qxspi_ops);
	qxspin = MKDEV(QXSPI_MAJOR, g_nxspis);
	ret = cdev_add(&qxspi->cdev, qxspin, 1);
	if (ret) {
		printk(KERN_WARNING "qxspi - could not create char dev %d\n", g_nxspis);
		ret = -ENODEV;
		goto out_abort;
	}
	/*printk("qxspi - adding char dev %d:%d\n", MAJOR(qxspin), MINOR(qxspin));*/
	qxspi->minor = MINOR(qxspin);
    sema_init(&qxspi->donesem[qxspi->minor], 0);
    sema_init(&qxspi->errsem[qxspi->minor], 0);
	g_nxspis++;
	ret = 0;

	platform_set_drvdata(pdev, qxspi);
	device_init_wakeup(&pdev->dev, 1);
	dev_info(&pdev->dev, "Quasar qbspi\n");
	return 0;

out_abort:
	while (qxspi->nioregs > 0) {
		qxspi->nioregs--;
		iounmap(qxspi->ioregs[qxspi->nioregs].base);
	}
	for(irq = 0; irq < qxspi->nirqs; irq++)
		free_irq(qxspi->irqs[irq], qxspi);
	kfree(qxspi);
	return ret;
}

static int __exit qxspi_remove(struct platform_device *pdev)
{
	struct quasar_xspi *qxspi = platform_get_drvdata(pdev);
	int irq;

	cdev_del(&qxspi->cdev);
	while (qxspi->nioregs > 0) {
		qxspi->nioregs--;
		iounmap(qxspi->ioregs[qxspi->nioregs].base);
	}
	for(irq = 0; irq < qxspi->nirqs; irq++)
		free_irq(qxspi->irqs[irq], qxspi);
	kfree(qxspi);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

MODULE_ALIAS("platform:qxspi");

static const struct of_device_id  qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-qbspi" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver qxspi_driver_ops = {
	.probe		= qxspi_probe,
	.remove		= qxspi_remove,
	.driver		= {
		.name	= "quasar-qbspi",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init qxspi_init(void)
{
	int ret;

		ret = platform_driver_register(&qxspi_driver_ops);
	return ret;
}
module_init(qxspi_init);

static void __exit qxspi_exit(void)
{
	platform_driver_unregister(&qxspi_driver_ops);
}
module_exit(qxspi_exit);

MODULE_DESCRIPTION("Quasar qbspi Driver");
MODULE_LICENSE("Dual BSD/GPL");

