/*
 * quasardev.c -- quasar-device - a bulk endpoint device for 
 *                composite usb gadget driver under qcomposite
 *
 * Copyright (c) 2014, 2015, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
//#define DEBUG 1
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/utsname.h>
#include <linux/device.h>
#include <linux/moduleparam.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/types.h>
#include <linux/ctype.h>
#include <linux/cdev.h>

#include <asm/byteorder.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/uaccess.h>
#include <asm/unaligned.h>

#include "quasargadget.h"
#include <linux/usb/ch9.h>
#include <linux/usb/gadget.h>
#include <linux/usb/composite.h>
#if LINUX_VERSION_CODE <= KERNEL_VERSION(4,1,0)
#include "../drivers/usb/gadget/gadget_chips.h"
#else
#include "gadget_chips.h"
#endif
#include "quasardev.h"
#include "../qdwc3/core.h"

uint dbgOn = 0;
module_param(dbgOn, uint, 0644);

struct usb_request *qdev_req_alloc(struct usb_ep *ep, unsigned len, gfp_t gfp_flags, u8 num)
{
	struct usb_request	*req;

	req = usb_ep_alloc_request(ep, gfp_flags);

	if (req != NULL) {
		if (len > USB_BUFSIZE) {
			printk("buffer size too large\n");
			return 0;
		}
		req->length = len;

		req->buf = kmalloc(len, gfp_flags);
		if (req->buf == NULL) {
			usb_ep_free_request(ep, req);
			return NULL;
		}
	}

	return req;
}

void qdev_req_free(struct usb_ep *ep, struct usb_request *req)
{
	if (ep != NULL && req != NULL) {
		kfree(req->buf);
		usb_ep_free_request(ep, req);
	}
}

static void rx_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct quasar_dev	*dev = ep->driver_data;
	struct usb_composite_dev *cdev = dev->cdev;
	int			status = req->status;
	unsigned long flags;

	spin_lock_irqsave(&dev->lock_io, flags);

	list_del_init(&req->list);	/* Remode from Active List */

	switch (status) {

	/* normal completion */
	case 0:
		if (req->actual > 0) {
			list_add_tail(&req->list, &dev->rx_buffers);
			DBG(cdev, "G_qdev : rx length %d\n", req->actual);
		} else {
			list_add(&req->list, &dev->rx_reqs);
		}
		break;

	/* software-driven interface shutdown */
	case -ECONNRESET:		/* unlink */
	case -ESHUTDOWN:		/* disconnect etc */
		VDBG(cdev, "rx shutdown, code %d\n", status);
		list_add(&req->list, &dev->rx_reqs);
		break;

	/* for hardware automagic (such as pxa) */
	case -ECONNABORTED:		/* endpoint reset */
		DBG(cdev, "rx %s reset\n", ep->name);
		list_add(&req->list, &dev->rx_reqs);
		break;

	/* data overrun */
	case -EOVERFLOW:
		/* FALLTHROUGH */

	default:
		DBG(cdev, "rx status %d\n", status);
		list_add(&req->list, &dev->rx_reqs);
		break;
	}
	spin_unlock_irqrestore(&dev->lock_io, flags);

	wake_up_interruptible(&dev->rx_wait);
}

static void tx_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct quasar_dev	*dev = ep->driver_data;
	unsigned long flags;
	
	switch (req->status) {
	default:
		printk("tx err %d\n", req->status);
		/* FALLTHROUGH */
	case -ECONNRESET:		/* unlink */
	case -ESHUTDOWN:		/* disconnect etc */
		break;
	case 0:
		break;
	}
	spin_lock_irqsave(&dev->lock_io, flags);
	/* Take the request struct off the active list and put it on the
	 * free list.
	 */
	list_del_init(&req->list);
	list_add(&req->list, &dev->tx_reqs);

	spin_unlock_irqrestore(&dev->lock_io, flags);

	wake_up_interruptible(&dev->tx_wait);
	if (likely(list_empty(&dev->tx_reqs_active)))
		wake_up_interruptible(&dev->tx_flush_wait);

}

/* This function must be called with NOT lock_io */
static void setup_rx_reqs(struct quasar_dev *dev)
{
	struct usb_request              *req;
	unsigned long flags;

	/* Prevent from kernel warning by kernel 4.19 when this ep is not
	 * enabled yet while USBDserver starts asking transfers.
	 * Kernel 4.4 doesn't implement the warning message */
	if (!dev->out_ep->enabled)
		return;

	if (dev->gadget->connected == 0)
		return;

	spin_lock_irqsave(&dev->lock_io, flags);
	while (likely(!list_empty(&dev->rx_reqs))) {
		int error;

		req = container_of(dev->rx_reqs.next,
				struct usb_request, list);
		list_del_init(&req->list);

		/* The USB Host sends us whatever amount of data it wants to
		 * so we always set the length field to the full USB_BUFSIZE.
		 * If the amount of data is more than the read() caller asked
		 * for it will be stored in the request buffer until it is
		 * asked for by read().
		 */
		req->length = USB_BUFSIZE;
		req->complete = rx_complete;

		/* add to active list */
		list_add(&req->list, &dev->rx_reqs_active);

		/* don't hold lock now, since ep_queue will most likely
		 * lock the controller which is also locked in controller irq
		 * and irq might call rx_complete which locks out lock_io
		 * which is a deadlock.  This is completely safe to unlock
		 * here, as the req is properly listed.
		 */
		spin_unlock_irqrestore(&dev->lock_io, flags);
		/* Prevent from kernel warning by kernel 4.19 when this ep is not
		 * enabled yet while USBDserver starts asking transfers.
		 * Kernel 4.4 doesn't implement the warning message */
		if ((dev->out_ep->enabled) && (dev->gadget->connected))		
			error = usb_ep_queue(dev->out_ep, req, GFP_ATOMIC);
		else
			error = -ESHUTDOWN;
		
		spin_lock_irqsave(&dev->lock_io, flags);

		if (error) {
			/* couldn't submit so hand remove from active list
			 * and add back to reqs
			 */
			DBG(dev->cdev, "rx submit --> %d\n", error);
			list_del_init(&req->list);
			list_add(&req->list, &dev->rx_reqs);
			break;
		}
	}
	spin_unlock_irqrestore(&dev->lock_io, flags);
}

ssize_t qdev_read(struct file *fd, char __user *buf, size_t len, loff_t *ptr)
{
	struct quasar_dev		*dev = fd->private_data;
	unsigned long			flags;
	size_t				size;
	size_t				bytes_copied;
	struct usb_request		*req;
	/* This is a pointer to the current USB rx request. */
	struct usb_request		*current_rx_req;
	/* This is the number of bytes in the current rx buffer. */
	size_t				current_rx_bytes;
	/* This is a pointer to the current rx buffer. */
	u8				*current_rx_buf;

	if (len == 0)
		return -EINVAL;

	if (! dev)
		return -ESHUTDOWN;

	/* Until OUT ep is enabled, don't do read */
	if (!dev->out_ep->enabled)
		return 0 /*no bytes_copied*/;

	//DBG(dev->cdev, "qdev_read trying to read %d bytes\n", (int)len);

	/* We will use this flag later to check if a reset happened
	 * after we turn interrupts back on.
	 */
	bytes_copied = 0;

	if (down_interruptible(&dev->sem)) {
		return -EINTR;
	}
	setup_rx_reqs(dev);

	dev->reset_request = 0;
	current_rx_req = dev->current_rx_req;
	current_rx_bytes = dev->current_rx_bytes;
	current_rx_buf = dev->current_rx_buf;
	dev->current_rx_req = NULL;
	dev->current_rx_bytes = 0;
	dev->current_rx_buf = NULL;

	/* Check if there is any data in the read buffers. Please note that
	 * current_rx_bytes is the number of bytes in the current rx buffer.
	 * If it is zero then check if there are any other rx_buffers that
	 * are on the completed list. We are only out of data if all rx
	 * buffers are empty.
	 */
	if ((current_rx_bytes == 0) &&
			(likely(list_empty(&dev->rx_buffers)))) {
		/*
		 * If no data is available check if this is a NON-Blocking
		 * call or not.
		 */
		if (fd->f_flags & (O_NONBLOCK|O_NDELAY)) {
			up(&dev->sem);
			return -EAGAIN;
		}
		/* Sleep until data is available */
		wait_event_interruptible(dev->rx_wait,
				(likely(!list_empty(&dev->rx_buffers))));
	}

	/* We have data to return then copy it to the caller's buffer.*/
	while ((current_rx_bytes || likely(!list_empty(&dev->rx_buffers)))
			&& len) {
			
		spin_lock_irqsave(&dev->lock_io, flags);
		if (current_rx_bytes == 0) {
			req = container_of(dev->rx_buffers.next,
					struct usb_request, list);
			list_del_init(&req->list);

			if (req->actual && req->buf) {
				current_rx_req = req;
				current_rx_bytes = req->actual;
				current_rx_buf = req->buf;
			} else {
				list_add(&req->list, &dev->rx_reqs);
				spin_unlock_irqrestore(&dev->lock_io, flags);
				printk("read loop\n");
				continue;
			}
		}
		spin_unlock_irqrestore(&dev->lock_io, flags);

		if (len > current_rx_bytes)
			size = current_rx_bytes;
		else
			size = len;

		size -= copy_to_user(buf, current_rx_buf, size);
		
		bytes_copied += size;
		len -= size;
		buf += size;

		/* We've disconnected or reset so return. */
		if (dev->reset_request) {
			spin_lock_irqsave(&dev->lock_io, flags);
			list_add(&current_rx_req->list, &dev->rx_reqs);
			spin_unlock_irqrestore(&dev->lock_io, flags);
			up(&dev->sem);
			return -EAGAIN;
		}

		/* If we not returning all the data left in this RX request
		 * buffer then adjust the amount of data left in the buffer.
		 * Othewise if we are done with this RX request buffer then
		 * requeue it to get any incoming data from the USB host.
		 */
		if (size < current_rx_bytes) {
			current_rx_bytes -= size;
			current_rx_buf += size;
		} else {
			spin_lock_irqsave(&dev->lock_io, flags);
			list_add(&current_rx_req->list, &dev->rx_reqs);
			spin_unlock_irqrestore(&dev->lock_io, flags);
			current_rx_bytes = 0;
			current_rx_buf = NULL;
			current_rx_req = NULL;
		}
	}

	dev->current_rx_req = current_rx_req;
	dev->current_rx_bytes = current_rx_bytes;
	dev->current_rx_buf = current_rx_buf;

	up(&dev->sem);

	if (dbgOn == 1)
		DBG(dev->cdev, "qdev_read returned %d bytes\n", (int)bytes_copied);

	if (bytes_copied)
		return bytes_copied;
	else
		return -EAGAIN;
}

ssize_t qdev_write(struct file *fd, const char __user *buf, size_t len, loff_t *ptr)
{
	struct quasar_dev	*dev = fd->private_data;
	unsigned long		flags;
	size_t			size;	/* Amount of data in a TX request. */
	size_t			bytes_copied = 0;
	struct usb_request	*req;
	int				rc;

	//DBG(dev->cdev, "qdev_write trying to send %d bytes\n", (int)len);

	if (! dev)
		return -ESHUTDOWN;

	if (len == 0)
		return -EINVAL;

	if (down_interruptible(&dev->sem)) {
		return -EINTR;
	}

	/* Check if a reset happens while we have interrupts on */
	dev->reset_request = 0;

	while (bytes_copied < len) {

		/* Check if there is any available write buffers */
		if (likely(list_empty(&dev->tx_reqs))) {
			/*
			 * write buffers aren't available, check if this is
			 * a NON-Blocking call or not.
			 */
			if (fd->f_flags & (O_NONBLOCK|O_NDELAY)) {
				//printk("wr %d of %d\n", bytes_copied, len);
				/*
				if (bytes_copied == 0) {
					extern void dump_usbsts(void);
					dump_usbsts();
				}
				*/
				up(&dev->sem);
				return bytes_copied ? bytes_copied : -EAGAIN;
			}
			/* Sleep until a write buffer is available */
			wait_event_interruptible(dev->tx_wait,
					(likely(!list_empty(&dev->tx_reqs))));
		}
		size = len - bytes_copied;
		if (size > USB_BUFSIZE)
			size = USB_BUFSIZE;

		spin_lock_irqsave(&dev->lock_io, flags);

		req = container_of(dev->tx_reqs.next, struct usb_request,
				list);
		list_del_init(&req->list);

		req->complete = tx_complete;
		req->length = size;

		/* Check if we need to send a zero length packet. */
		if (len > (bytes_copied + size))
			/* there will be more TX requests so no yet. */
			req->zero = 0;
		else
			/* If the data amount is not a multple of the
			 * maxpacket size then send a zero length packet.
			 */
			req->zero = ((len % dev->in_ep->maxpacket) != 0);

		/* unlock io lists during copy to avoid blocking complete interrupts */
		spin_unlock_irqrestore(&dev->lock_io, flags);
		
		/* copy out buf to the req buf */
		if (copy_from_user(req->buf, buf, size)) {
			DBG(dev->cdev, "tx copy from user failed\n");
			spin_lock_irqsave(&dev->lock_io, flags);
			list_add(&req->list, &dev->tx_reqs);
			spin_unlock_irqrestore(&dev->lock_io, flags);
			return -EINVAL;
		}
		spin_lock_irqsave(&dev->lock_io, flags);

		bytes_copied += size;
		buf += size;
	
		/* We've disconnected or reset so free the req and buffer */
		if (dev->reset_request) {
			list_add(&req->list, &dev->tx_reqs);
			spin_unlock_irqrestore(&dev->lock_io, flags);
			return bytes_copied ? bytes_copied : -EAGAIN;
		}
		/* add it to active list */
		list_add(&req->list, &dev->tx_reqs_active);
		
		/* unlock us now since we're calling into the controller driver
		 * which might be trying to lock us in tx/rx_complete from irq
		 * This is safe, since reqs from a single thread are single
		 * threaded anyway, and its ok to mingle reqs to diff. eps
		 */
		spin_unlock_irqrestore(&dev->lock_io, flags);
		/* Prevent from kernel warning by kernel 4.19 when this ep is not
		 * enabled yet while USBDserver starts asking transfers.
		 * Kernel 4.4 doesn't implement the warning message */
		if ((dev->in_ep->enabled) && (dev->gadget->connected))		
			rc = usb_ep_queue(dev->in_ep, req, GFP_ATOMIC);
		else
			rc = -ESHUTDOWN;
		if (rc) {
			DBG(dev->cdev, "tx submit --> %d\n", rc);
			printk("tx submit --> %d\n", rc);
			spin_lock_irqsave(&dev->lock_io, flags);
			list_del_init(&req->list);
			list_add(&req->list, &dev->tx_reqs);
			spin_unlock_irqrestore(&dev->lock_io, flags);
			up(&dev->sem);
			//return bytes_copied ? bytes_copied : rc;
			return rc;
		}
	}
	up(&dev->sem);
	if (dbgOn == 1)
		DBG(dev->cdev, "qdev_write sent %d bytes\n", (int)bytes_copied);

	return bytes_copied ? bytes_copied : -EAGAIN;
}

int qdev_fsync(struct file *fd, int datasync)
{
	struct quasar_dev	*dev = fd->private_data;
	int			tx_list_empty;

	if (! dev)
		return -ESHUTDOWN;

	tx_list_empty = (likely(list_empty(&dev->tx_reqs)));

	if (!tx_list_empty) {
		/* Sleep until all data has been sent */
		wait_event_interruptible(dev->tx_flush_wait,
				(likely(list_empty(&dev->tx_reqs_active))));
	}

	return 0;
}

unsigned int qdev_poll(struct file *fd, poll_table *wait)
{
	struct quasar_dev	*dev = fd->private_data;
	int			status = 0;

	if (! dev)
		return -ESHUTDOWN;
	if (dev->gadget->connected != 0) {

	setup_rx_reqs(dev);

	poll_wait(fd, &dev->rx_wait, wait);
	poll_wait(fd, &dev->tx_wait, wait);

	if (likely(!list_empty(&dev->tx_reqs)))
		status |= POLLOUT | POLLWRNORM;

	if (likely(dev->current_rx_bytes) ||
			likely(!list_empty(&dev->rx_buffers)))
		status |= POLLIN | POLLRDNORM;
	} else {
		/* If the caller doesn't use Quasar USBD service scheme,
		 * return POLLPRI to update exceptset for USB disconnection */
		status = POLLPRI;
	}

	return status;
}

int pmxdev_buf_flush(struct quasar_dev *dev, int rx)
{
	struct usb_request	*req;
	unsigned long flags;
	int sts = 0;

	//INFO(dev, "Received Buffer Flush Request.\n");

	if (! dev)
		return -ESHUTDOWN;

	/* handle ioctls */
	spin_lock_irqsave(&dev->lock_io, flags);

#if 1 // do not flush RX
	// tony only flush received req
	if(rx) {
		printk("flush rx\n");
		if (dev->current_rx_req != NULL) {
			list_add(&dev->current_rx_req->list, &dev->rx_reqs);
			dev->current_rx_req = NULL;
		}
		dev->current_rx_bytes = 0;
		dev->current_rx_buf = NULL;

		while (likely(!(list_empty(&dev->rx_buffers)))) {
			req = container_of(dev->rx_buffers.next, struct usb_request,
					list);
			list_del_init(&req->list);
			list_add(&req->list, &dev->rx_reqs);
			INFO(dev, "Flush RX\n");
		}
	}
#endif

	// flush tx req
	while (likely(!(list_empty(&dev->tx_reqs_active)))) {
		req = container_of(dev->tx_reqs_active.next,
				struct usb_request, list);
		spin_unlock_irqrestore(&dev->lock_io, flags);
		sts = usb_ep_dequeue(dev->in_ep, req);
		spin_lock_irqsave(&dev->lock_io, flags);
		if(sts) {
			INFO(dev, "usb_ep_dequeue(dev->in_ep, req): %d.\n", sts);
			list_del_init(&req->list);
			list_add(&req->list, &dev->tx_reqs);
		}
	}

	spin_unlock_irqrestore(&dev->lock_io, flags);

	return sts;
}

