/*
 * f_umstorage.c -- usb mass storage gadget function driver delegate
 *                  for Quasar composite device - uses kernel driver
 *                  f_mass_storage to provide file-backed storage
 *                  on top of a block device or file on the board
 *
 * Copyright (c) 2014, 2015, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/version.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/timer.h>
#include <linux/list.h>
#include <linux/interrupt.h>
#include <linux/utsname.h>
#include <linux/device.h>
#include <linux/moduleparam.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <linux/types.h>
#include <linux/ctype.h>
#include <linux/cdev.h>

#include <asm/byteorder.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <linux/uaccess.h>
#include <asm/unaligned.h>

/* include first here so DUALSPEED is defined for composite, etc.
*/
#include "quasargadget.h"
#include <linux/usb/ch9.h>
#include <linux/usb/gadget.h>
#include <linux/usb/composite.h>
#include <asm/ioctls.h>		/* for FIONBIO */
#include "quasar/qbsocregs.h"
#if LINUX_VERSION_CODE <= KERNEL_VERSION(4,1,0)
#include "../drivers/usb/gadget/gadget_chips.h"
#else
#include "gadget_chips.h"
#endif

#define NOT_SUPPORT_AUTOMOUNT   0

#define FSG_MAX_DEVICES		2
#define FSG_MAX_PARTITIONS	2
#define SUPPORTED_LUN_NUMBER	2

/* Defined as 1 to be forcedly read-only for MSC pass-through feature */
#define FSG_READ_ONLY 0

void quasar_automount(void *);
void quasar_set_capacities(unsigned int lun, u32 *last_sector, u32 *sector_size);

#include "f_mass_storage.c"

static struct fsg_module_parameters mod_data = {
	.file[0] = "/root/fsg_backing",
	.ro[0] = 0,
	.removable[0] = 0,
	.cdrom[0] = 0,
	
	.file_count = 1,
	.ro_count = 1,
	.removable_count = 1,
	.cdrom_count = 1,
	.luns = 1,
	.stall = 0
};
FSG_MODULE_PARAMETERS(/* no prefix */, mod_data);

#define DEVICE_PATH_LENGTH 64

struct fsg_mount_info {
	char	device[FSG_MAX_DEVICES][FSG_MAX_PARTITIONS][DEVICE_PATH_LENGTH];
	struct fsg_config	*fsg_cfg;
};

static struct fsg_config *fsg_cfg;
static struct fsg_common *fsg_common;

void quasar_automount(void *common_)
{
#if !NOT_SUPPORT_AUTOMOUNT
	struct file		*filp = NULL;
	char		devicepath[DEVICE_PATH_LENGTH];
	int			devno, partno, mounted, ro;
	struct fsg_common	*common = common_;
	struct fsg_mount_info	*mount_info = common->private_data;
	struct fsg_config	*fsg_cfg = mount_info->fsg_cfg;
	u32			i;

	mounted = 0;

	if (fsg_cfg->luns[common->lun].filename) {
		filp = filp_open(fsg_cfg->luns[common->lun].filename,
				O_RDWR | O_LARGEFILE, 0);
		if (IS_ERR(filp)) {
			filp = filp_open(fsg_cfg->luns[common->lun].filename,
					O_RDONLY | O_LARGEFILE, 0);
			if (IS_ERR(filp)) {
				/* The drive is disconnected */
				printk(KERN_INFO "%s: %s is removed\n", __func__, fsg_cfg->luns[common->lun].filename);
				common->luns[common->lun].unit_attention_data = SS_MEDIUM_NOT_PRESENT;
				fsg_cfg->luns[common->lun].filename = 0;
				fsg_lun_close(&common->luns[common->lun]);
				return;
			}
		}
		/* Must close filp otherwise sda will not be released
		 * for the next usage
		 */
		if (filp) {
			filp_close(filp, NULL);
			filp = NULL;
		}
		return;
	}

	for (devno = FSG_MAX_DEVICES-1; devno >= 0; devno--) {
		for (partno = FSG_MAX_PARTITIONS-1; partno >= 0; partno--) {
			ro = -1;
			if (partno > 0) {
				snprintf(devicepath, sizeof(devicepath), "/dev/sd%c%d", devno + 'a', partno);
			} else {
				snprintf(devicepath, sizeof(devicepath), "/dev/sd%c", devno + 'a');
			}

			filp = filp_open(devicepath, O_RDWR | O_LARGEFILE, 0);
			if (IS_ERR(filp)) {
				filp = filp_open(devicepath, O_RDONLY | O_LARGEFILE, 0);
				if (IS_ERR(filp)) {
#ifdef CONFIG_MMC
					if (partno > 0) {
						snprintf(devicepath, sizeof(devicepath), "/dev/mmc%d", partno);
					} else {
						snprintf(devicepath, sizeof(devicepath), "/dev/mmc");
					}

					filp = filp_open(devicepath, O_RDWR | O_LARGEFILE, 0);
					if (IS_ERR(filp)) {
						filp = filp_open(devicepath, O_RDONLY | O_LARGEFILE, 0);
						if (IS_ERR(filp)) {
							continue;
						} else {
							ro = 1;
						}
					} else {
						ro = 0;
					}
					if (ro < 0)
#endif
					continue;
				} else {
					ro = 1;
				}
			} else {
				ro = 0;
			}

			mounted = 0;

			/* Check if mounted */
			for (i = 0; i < fsg_cfg->nluns; i++) {
				if ((fsg_cfg->luns[i].filename) && (strcmp(fsg_cfg->luns[i].filename, devicepath) == 0)) {
					mounted = 1;
					break;
				}
			}

			/* Mounted to current lun */
			if (!mounted) {
				strlcpy(mount_info->device[devno][partno], devicepath,
					DEVICE_PATH_LENGTH);
				fsg_cfg->luns[common->lun].filename = mount_info->device[devno][partno];
				common->luns[common->lun].filp = filp;
				printk(KERN_INFO "%s: %s is mounted\n", __func__, fsg_cfg->luns[common->lun].filename);

				if ((ro) || (FSG_READ_ONLY))
					common->luns[common->lun].initially_ro = 1;
				else
					common->luns[common->lun].initially_ro = 0;

				if (fsg_lun_open(&common->luns[common->lun], fsg_cfg->luns[common->lun].filename) == 0)
					common->luns[common->lun].unit_attention_data = SS_NOT_READY_TO_READY_TRANSITION;
				else
					printk(KERN_WARNING "%s: Mount %s failed\n", __func__, fsg_cfg->luns[common->lun].filename);

				filp_close(filp, NULL);
				filp = NULL;
				return;
			}

			if (filp) {
				filp_close(filp, NULL);
				filp = NULL;
			}
			if (partno == 1)
				break;
		}
	}
#endif
}

void quasar_set_capacities(unsigned int lun, u32 *last_sector, u32 *sector_size)
{
}

int __init quasar_storage_add(struct usb_configuration *c)
{
	int ret = 0;

	ret = fsg_bind_config(c->cdev, c, fsg_common);

	return ret;
}

int __init quasar_storage_init(struct usb_composite_dev *cdev)
{
	int status = 0;
	int i;
	struct fsg_mount_info *mount_info;

	fsg_cfg = kzalloc(sizeof(struct fsg_config), GFP_KERNEL);
	if (!fsg_cfg) {
		printk(KERN_WARNING "%s: Unable to allocate fsg_config\n", __func__);
		return -ENOMEM;
	}

	fsg_cfg->nluns = SUPPORTED_LUN_NUMBER;

	for (i = 0; i < fsg_cfg->nluns; i++) {
		fsg_cfg->luns[i].removable = 1;
		fsg_cfg->luns[i].ro = 0;
		fsg_cfg->luns[i].cdrom = 0;
		fsg_cfg->luns[i].filename = 0;
	}

#if NOT_SUPPORT_AUTOMOUNT
	msleep(1000);
	fsg_cfg->luns[0].filename = "/dev/sda";
#endif

	fsg_cfg->vendor_name = "Quasar";
	fsg_cfg->product_name = "USB Storage";

	fsg_cfg->private_data = 0;
	fsg_cfg->can_stall = 0;

	mount_info = kzalloc(sizeof(struct fsg_mount_info), GFP_KERNEL);
	if (!mount_info) {
		printk(KERN_WARNING "%s: Unable to allocate fsg_mount_info\n", __func__);
		return -ENOMEM;
	}
	mount_info->fsg_cfg = fsg_cfg;
	fsg_cfg->private_data = mount_info;

	fsg_common = fsg_common_init(NULL, cdev, fsg_cfg);
	if (IS_ERR(fsg_common))
		status = PTR_ERR(fsg_common);

	return status;
}

int __init quasar_storage_put(struct usb_composite_dev *cdev)
{
	fsg_common_put(fsg_common);
	return 0;
}

void quasar_storage_unmap(void)
{
}
