/*
 * A driver for the USB device controller in the Quasar processors
 *
 * Copyright (c) 2014, 2015, The Linux Foundation. All rights reserved.
 *
 * Based on core.c copyright (C) 2010-2011 Texas Instruments Incorporated
 * - http://www.ti.com by Felipe Balbi <balbi@ti.com> and
 *                        Sebastian Andrzej Siewior <bigeasy@linutronix.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/io.h>
#include <linux/list.h>
#include <linux/delay.h>
#include <linux/dma-mapping.h>
#include <linux/of.h>
#include <linux/version.h>

#include <linux/usb/ch9.h>
#include <linux/usb/gadget.h>
#include <linux/usb/of.h>
#include <linux/usb/otg.h>

#include <quasar/qbsocregs.h>

#include "platform_data.h"
#include "core.h"
#include "gadget.h"
#include "io.h"
#include "debug.h"

#if defined(BOARD_TYPE_RC4) || defined(BOARD_TYPE_RC3L)
static int usb3 = 0;
#else
static int usb3 = 1;
#endif
module_param(usb3, int, 0);
MODULE_PARM_DESC(usb3, "USB3");

u32 usbd_ss = 0;   /* 1 when this driver is for USDR */

int disable_superspeed = 0;   // 1 to disable USB SS even if this is USB 3.0 controller

/* -------------------------------------------------------------------------- */

void dwc3_set_mode(struct dwc3 *dwc, u32 mode)
{
	u32 reg;

	reg = dwc3_readl(dwc->regs, DWC3_GCTL);
	reg &= ~(DWC3_GCTL_PRTCAPDIR(DWC3_GCTL_PRTCAP_OTG));
	reg |= DWC3_GCTL_PRTCAPDIR(mode);
	dwc3_writel(dwc->regs, DWC3_GCTL, reg);
}

/**
 * dwc3_free_one_event_buffer - Frees one event buffer
 * @dwc: Pointer to our controller context structure
 * @evt: Pointer to event buffer to be freed
 */
static void dwc3_free_one_event_buffer(struct dwc3 *dwc,
		struct dwc3_event_buffer *evt)
{
#if FIXED_MEMORY_FOR_EVT
	iounmap(evt->buf);
	iounmap(evt->lpos);
	evt->buf = NULL;
	evt->dma = 0;
	evt->lpos = 0;
#else
#ifdef Q6300
	dma_free_coherent(dwc->dev, evt->length, evt->buf, evt->dma);
#else
	dma_free_coherent(dwc->dev->parent, evt->length, evt->buf, evt->dma);
#endif
#endif
}

/**
 * dwc3_alloc_one_event_buffer - Allocates one event buffer structure
 * @dwc: Pointer to our controller context structure
 * @length: size of the event buffer
 *
 * Returns a pointer to the allocated event buffer structure on success
 * otherwise ERR_PTR(errno).
 */
static struct dwc3_event_buffer *dwc3_alloc_one_event_buffer(struct dwc3 *dwc,
		unsigned length)
{
	struct dwc3_event_buffer	*evt;

	evt = devm_kzalloc(dwc->dev, sizeof(*evt), GFP_KERNEL);
	if (!evt)
		return ERR_PTR(-ENOMEM);

	evt->dwc	= dwc;
	evt->length	= length;
	/** S/W workaround based on 63xx Errata 1.8,
	 *  cannot use USBMEM due to Simultaneous read and write issue
	 */
#if FIXED_MEMORY_FOR_EVT
	evt->dma = (dma_addr_t)USBD_EVT_BUF_START;
	evt->buf = ioremap_nocache(USBD_EVT_BUF_START, length);
	evt->lpos = ioremap_nocache(USBD_EVT_LPOS_PTR, 4);
#else
#ifdef Q6300
	evt->buf	= dma_alloc_coherent(dwc->dev, length,
			&evt->dma, GFP_KERNEL);
#else
	evt->buf	= dma_alloc_coherent(dwc->dev->parent, length,
			&evt->dma, GFP_KERNEL);
#endif
	if (!evt->buf)
		return ERR_PTR(-ENOMEM);
#endif

	return evt;
}

/**
 * dwc3_free_event_buffers - frees all allocated event buffers
 * @dwc: Pointer to our controller context structure
 */
static void dwc3_free_event_buffers(struct dwc3 *dwc)
{
	struct dwc3_event_buffer	*evt;
	int i;

	for (i = 0; i < dwc->num_event_buffers; i++) {
		evt = dwc->ev_buffs[i];
		if (evt)
			dwc3_free_one_event_buffer(dwc, evt);
	}
}

/**
 * dwc3_alloc_event_buffers - Allocates @num event buffers of size @length
 * @dwc: pointer to our controller context structure
 * @length: size of event buffer
 *
 * Returns 0 on success otherwise negative errno. In the error case, dwc
 * may contain some buffers allocated but not all which were requested.
 */
static int dwc3_alloc_event_buffers(struct dwc3 *dwc, unsigned length)
{
	int			num;
	int			i;

	num = DWC3_NUM_INT(dwc->hwparams.hwparams1);
	dwc->num_event_buffers = num;

	dwc->ev_buffs = devm_kzalloc(dwc->dev, sizeof(*dwc->ev_buffs) * num,
			GFP_KERNEL);
	if (!dwc->ev_buffs) {
		dev_err(dwc->dev, "can't allocate event buffers array\n");
		return -ENOMEM;
	}

	for (i = 0; i < num; i++) {
		struct dwc3_event_buffer	*evt;

		evt = dwc3_alloc_one_event_buffer(dwc, length);
		if (IS_ERR(evt)) {
			dev_err(dwc->dev, "can't allocate event buffer\n");
			return PTR_ERR(evt);
		}
		dwc->ev_buffs[i] = evt;
	}

	return 0;
}

/**
 * dwc3_event_buffers_setup - setup our allocated event buffers
 * @dwc: pointer to our controller context structure
 *
 * Returns 0 on success otherwise negative errno.
 */
static int dwc3_event_buffers_setup(struct dwc3 *dwc)
{
	struct dwc3_event_buffer	*evt;
	int				n;

	for (n = 0; n < dwc->num_event_buffers; n++) {
		evt = dwc->ev_buffs[n];
		dev_dbg(dwc->dev, "Event buf %px dma %08llx length %d\n",
				evt->buf, (unsigned long long) evt->dma,
				evt->length);

#if FIXED_MEMORY_FOR_EVT
		*(u32 *)evt->lpos = 0;
#else
		evt->lpos = 0;
#endif

		dwc3_writel(dwc->regs, DWC3_GEVNTADRLO(n),
				lower_32_bits(evt->dma));
		dwc3_writel(dwc->regs, DWC3_GEVNTADRHI(n),
				upper_32_bits(evt->dma));
		dwc3_writel(dwc->regs, DWC3_GEVNTSIZ(n),
				DWC3_GEVNTSIZ_SIZE(evt->length));
		dwc3_writel(dwc->regs, DWC3_GEVNTCOUNT(n), 0);
	}

	return 0;
}

static void dwc3_event_buffers_cleanup(struct dwc3 *dwc)
{
	struct dwc3_event_buffer	*evt;
	int				n;

	for (n = 0; n < dwc->num_event_buffers; n++) {
		evt = dwc->ev_buffs[n];

#if FIXED_MEMORY_FOR_EVT
		*(uint_port *)evt->lpos = 0;
#else
		evt->lpos = 0;
#endif

		dwc3_writel(dwc->regs, DWC3_GEVNTADRLO(n), 0);
		dwc3_writel(dwc->regs, DWC3_GEVNTADRHI(n), 0);
		dwc3_writel(dwc->regs, DWC3_GEVNTSIZ(n), DWC3_GEVNTSIZ_INTMASK
				| DWC3_GEVNTSIZ_SIZE(0));
		dwc3_writel(dwc->regs, DWC3_GEVNTCOUNT(n), 0);
	}
}

static int dwc3_alloc_scratch_buffers(struct dwc3 *dwc)
{
	if (!dwc->has_hibernation)
		return 0;

	if (!dwc->nr_scratch)
		return 0;

	dwc->scratchbuf = kmalloc_array(dwc->nr_scratch,
			DWC3_SCRATCHBUF_SIZE, GFP_KERNEL);
	if (!dwc->scratchbuf)
		return -ENOMEM;

	return 0;
}

static int dwc3_setup_scratch_buffers(struct dwc3 *dwc)
{
	dma_addr_t scratch_addr;
	u32 param;
	int ret;

	if (!dwc->has_hibernation)
		return 0;

	if (!dwc->nr_scratch)
		return 0;

	 /* should never fall here */
	if (!WARN_ON(dwc->scratchbuf))
		return 0;

#ifdef Q6300
	scratch_addr = dma_map_single(dwc->dev, dwc->scratchbuf,
			dwc->nr_scratch * DWC3_SCRATCHBUF_SIZE,
			DMA_BIDIRECTIONAL);
	if (dma_mapping_error(dwc->dev, scratch_addr)) {
		dev_err(dwc->dev, "failed to map scratch buffer\n");
		ret = -EFAULT;
		goto err0;
	}
#else
	scratch_addr = dma_map_single(dwc->dev->parent, dwc->scratchbuf,
			dwc->nr_scratch * DWC3_SCRATCHBUF_SIZE,
			DMA_BIDIRECTIONAL);
	if (dma_mapping_error(dwc->dev->parent, scratch_addr)) {
		dev_err(dwc->dev->parent, "failed to map scratch buffer\n");
		ret = -EFAULT;
		goto err0;
	}
#endif

	dwc->scratch_addr = scratch_addr;

	param = lower_32_bits(scratch_addr);

	ret = dwc3_send_gadget_generic_command(dwc,
			DWC3_DGCMD_SET_SCRATCHPAD_ADDR_LO, param);
	if (ret < 0)
		goto err1;

	param = upper_32_bits(scratch_addr);

	ret = dwc3_send_gadget_generic_command(dwc,
			DWC3_DGCMD_SET_SCRATCHPAD_ADDR_HI, param);
	if (ret < 0)
		goto err1;

	return 0;

err1:
#ifdef Q6300
	dma_unmap_single(dwc->dev, dwc->scratch_addr, dwc->nr_scratch *
			DWC3_SCRATCHBUF_SIZE, DMA_BIDIRECTIONAL);
#else
	dma_unmap_single(dwc->dev->parent, dwc->scratch_addr, dwc->nr_scratch *
			DWC3_SCRATCHBUF_SIZE, DMA_BIDIRECTIONAL);
#endif

err0:
	return ret;
}

static void dwc3_free_scratch_buffers(struct dwc3 *dwc)
{
	if (!dwc->has_hibernation)
		return;

	if (!dwc->nr_scratch)
		return;

	 /* should never fall here */
	if (!WARN_ON(dwc->scratchbuf))
		return;

#ifdef Q6300
	dma_unmap_single(dwc->dev, dwc->scratch_addr, dwc->nr_scratch *
			DWC3_SCRATCHBUF_SIZE, DMA_BIDIRECTIONAL);
#else
	dma_unmap_single(dwc->dev->parent, dwc->scratch_addr, dwc->nr_scratch *
			DWC3_SCRATCHBUF_SIZE, DMA_BIDIRECTIONAL);
#endif
	kfree(dwc->scratchbuf);
}

static void dwc3_core_num_eps(struct dwc3 *dwc)
{
	struct dwc3_hwparams	*parms = &dwc->hwparams;

	dwc->num_in_eps = DWC3_NUM_IN_EPS(parms);
	dwc->num_out_eps = DWC3_NUM_EPS(parms) - dwc->num_in_eps;

	dev_vdbg(dwc->dev, "found %d IN and %d OUT endpoints\n",
			dwc->num_in_eps, dwc->num_out_eps);
}

static void dwc3_cache_hwparams(struct dwc3 *dwc)
{
	struct dwc3_hwparams	*parms = &dwc->hwparams;

	parms->hwparams0 = dwc3_readl(dwc->regs, DWC3_GHWPARAMS0);
	parms->hwparams1 = dwc3_readl(dwc->regs, DWC3_GHWPARAMS1);
	parms->hwparams2 = dwc3_readl(dwc->regs, DWC3_GHWPARAMS2);
	parms->hwparams3 = dwc3_readl(dwc->regs, DWC3_GHWPARAMS3);
	parms->hwparams4 = dwc3_readl(dwc->regs, DWC3_GHWPARAMS4);
	parms->hwparams5 = dwc3_readl(dwc->regs, DWC3_GHWPARAMS5);
	parms->hwparams6 = dwc3_readl(dwc->regs, DWC3_GHWPARAMS6);
	parms->hwparams7 = dwc3_readl(dwc->regs, DWC3_GHWPARAMS7);
	parms->hwparams8 = dwc3_readl(dwc->regs, DWC3_GHWPARAMS8);
}

/**
 * dwc3_core_init - Low-level initialization of DWC3 Core
 * @dwc: Pointer to our controller context structure
 *
 * Returns 0 on success otherwise negative errno.
 */
static int dwc3_core_init(struct dwc3 *dwc)
{
	u32			hwparams4 = dwc->hwparams.hwparams4;
	u32			reg;
	int			ret;

	reg = dwc3_readl(dwc->regs, DWC3_GSNPSID);
	/* This should read as U3 followed by revision number */
	if ((reg & DWC3_GSNPSID_MASK) != 0x55330000) {
		dev_err(dwc->dev, "this is not a DesignWare USB3 DRD Core\n");
		ret = -ENODEV;
		goto err0;
	}
	dwc->revision = reg;

	// IMPORTANT: disable bit 0: INCRBRSTENA
	dwc3_writel(dwc->regs, DWC3_GSBUSCFG0, 4);

	reg = dwc3_readl(dwc->regs, DWC3_GCTL);
	reg &= ~DWC3_GCTL_SCALEDOWN_MASK;
	reg &= ~DWC3_GCTL_DISSCRAMBLE;

	switch (DWC3_GHWPARAMS1_EN_PWROPT(dwc->hwparams.hwparams1)) {
	case DWC3_GHWPARAMS1_EN_PWROPT_CLK:
		/**
		 * WORKAROUND: DWC3 revisions between 2.10a and 2.50a have an
		 * issue which would cause xHCI compliance tests to fail.
		 *
		 * Because of that we cannot enable clock gating on such
		 * configurations.
		 *
		 * Refers to:
		 *
		 * STAR#9000588375: Clock Gating, SOF Issues when ref_clk-Based
		 * SOF/ITP Mode Used
		 */
		if ((dwc->dr_mode == USB_DR_MODE_HOST ||
				dwc->dr_mode == USB_DR_MODE_OTG) &&
				(dwc->revision >= DWC3_REVISION_210A &&
				dwc->revision <= DWC3_REVISION_250A))
			reg |= DWC3_GCTL_DSBLCLKGTNG | DWC3_GCTL_SOFITPSYNC;
		else
			reg &= ~DWC3_GCTL_DSBLCLKGTNG;
		break;
	case DWC3_GHWPARAMS1_EN_PWROPT_HIB:
		/* enable hibernation here */
		dwc->nr_scratch = DWC3_GHWPARAMS4_HIBER_SCRATCHBUFS(hwparams4);
		break;
	default:
		dev_dbg(dwc->dev, "No power optimization available\n");
	}

	/*
	 * WORKAROUND: DWC3 revisions <1.90a have a bug
	 * where the device can fail to connect at SuperSpeed
	 * and falls back to high-speed mode which causes
	 * the device to enter a Connect/Disconnect loop
	 */
	if (dwc->revision < DWC3_REVISION_190A)
		reg |= DWC3_GCTL_U2RSTECN;
	else
		reg &= ~DWC3_GCTL_U2RSTECN;   // Resolve link speed issue for later revisions

	dwc3_core_num_eps(dwc);

	dwc3_writel(dwc->regs, DWC3_GCTL, reg);

#if !PERIPHERAL_L1_TEST   /* hardware approach */
	if (disable_superspeed == 0) {
		/* Support USB 2.0 EXT(LPM) function in USB 3.x device */
		reg = dwc3_readl(dwc->regs, DWC3_GUCTL1);
		reg |= USDR_GUCTL1__DEV_L1_EXIT_BY_HW__MASK;
		dwc3_writel(dwc->regs, DWC3_GUCTL1, reg);

		reg = dwc3_readl(dwc->regs, DWC3_DCFG);
		reg |= USDR_DCFG__LPMCAP__MASK;
		dwc3_writel(dwc->regs, DWC3_DCFG, reg);
	}
#endif
#ifdef Q6300A0
	reg = dwc3_readl(dwc->regs, DWC3_GUCTL2);
	reg &= 0xfc07ffff;   /* mask [25:19] */
    reg |= (0x2e << 19);
	dwc3_writel(dwc->regs, DWC3_GUCTL2, reg);
#endif
#ifdef DISABLE_BURST_MODE
    reg = dwc3_readl(dwc->regs, DWC3_GRXTHRCFG);
    reg &= USDR_GRXTHRCFG__UsbRxPktCntSel__INV_MASK;
    dwc3_writel(dwc->regs, DWC3_GRXTHRCFG, reg);
#else
	if (usbd_ss == 1) {
		reg = dwc3_readl(dwc->regs, DWC3_GRXTHRCFG);
		reg &= USDR_GRXTHRCFG__UsbMaxRxBurstSize__INV_MASK;
		reg |= (6 << USDR_GRXTHRCFG__UsbMaxRxBurstSize__SHIFT);
		dwc3_writel(dwc->regs, DWC3_GRXTHRCFG, reg);
	}
#endif
	ret = dwc3_alloc_scratch_buffers(dwc);
	if (ret)
		goto err1;

	ret = dwc3_setup_scratch_buffers(dwc);
	if (ret)
		goto err2;

	return 0;

err2:
	dwc3_free_scratch_buffers(dwc);

err1:
	usb_phy_shutdown(dwc->usb2_phy);
	usb_phy_shutdown(dwc->usb3_phy);
	phy_exit(dwc->usb2_generic_phy);
	phy_exit(dwc->usb3_generic_phy);

err0:
	return ret;
}

static void dwc3_core_exit(struct dwc3 *dwc)
{
	dwc3_free_scratch_buffers(dwc);
	usb_phy_shutdown(dwc->usb2_phy);
	if (usbd_ss)
		usb_phy_shutdown(dwc->usb3_phy);

	phy_exit(dwc->usb2_generic_phy);
	if (usbd_ss)
		phy_exit(dwc->usb3_generic_phy);
}

static int dwc3_core_get_phy(struct dwc3 *dwc)
{
	struct device		*dev = dwc->dev;
	struct device_node	*node = dev->of_node;
	int ret;

	if (node) {
		dwc->usb2_phy = devm_usb_get_phy_by_phandle(dev, "usb-phy", 0);
		if (usbd_ss)
			dwc->usb3_phy = devm_usb_get_phy_by_phandle(dev, "usb-phy", 1);
	} else {
		dwc->usb2_phy = devm_usb_get_phy(dev, USB_PHY_TYPE_USB2);
		if (usbd_ss)
			dwc->usb3_phy = devm_usb_get_phy(dev, USB_PHY_TYPE_USB3);
	}

	if (IS_ERR(dwc->usb2_phy)) {
		ret = PTR_ERR(dwc->usb2_phy);
		if (ret == -ENXIO || ret == -ENODEV) {
			dwc->usb2_phy = NULL;
		} else if (ret == -EPROBE_DEFER) {
			return ret;
		} else {
			dev_err(dev, "no usb2 phy configured\n");
			return ret;
		}
	}

	if (IS_ERR(dwc->usb3_phy) && usbd_ss) {
		ret = PTR_ERR(dwc->usb3_phy);
		if (ret == -ENXIO || ret == -ENODEV) {
			dwc->usb3_phy = NULL;
		} else if (ret == -EPROBE_DEFER) {
			return ret;
		} else {
			dev_err(dev, "no usb3 phy configured\n");
			return ret;
		}
	}

	dwc->usb2_generic_phy = devm_phy_get(dev, "usb2-phy");
	if (IS_ERR(dwc->usb2_generic_phy)) {
		ret = PTR_ERR(dwc->usb2_generic_phy);
		if (ret == -ENOSYS || ret == -ENODEV) {
			dwc->usb2_generic_phy = NULL;
		} else if (ret == -EPROBE_DEFER) {
			return ret;
		} else {
			dev_err(dev, "no usb2 phy configured\n");
			return ret;
		}
	}

	if (usbd_ss) {
		dwc->usb3_generic_phy = devm_phy_get(dev, "usb3-phy");
		if (IS_ERR(dwc->usb3_generic_phy)) {
			ret = PTR_ERR(dwc->usb3_generic_phy);
			if (ret == -ENOSYS || ret == -ENODEV) {
				dwc->usb3_generic_phy = NULL;
			} else if (ret == -EPROBE_DEFER) {
				return ret;
			} else {
				dev_err(dev, "no usb3 phy configured\n");
				return ret;
			}
		}
	}

	return 0;
}

static int dwc3_core_init_mode(struct dwc3 *dwc)
{
	struct device *dev = dwc->dev;
	int ret;

	switch (dwc->dr_mode) {
	case USB_DR_MODE_PERIPHERAL:
		dwc3_set_mode(dwc, DWC3_GCTL_PRTCAP_DEVICE);
		ret = dwc3_gadget_init(dwc);
		if (ret) {
			dev_err(dev, "failed to initialize gadget\n");
			return ret;
		}
		break;
	case USB_DR_MODE_HOST:
		dwc3_set_mode(dwc, DWC3_GCTL_PRTCAP_HOST);
		ret = dwc3_host_init(dwc);
		if (ret) {
			dev_err(dev, "failed to initialize host\n");
			return ret;
		}
		break;
	case USB_DR_MODE_OTG:
		dwc3_set_mode(dwc, DWC3_GCTL_PRTCAP_OTG);
		ret = dwc3_host_init(dwc);
		if (ret) {
			dev_err(dev, "failed to initialize host\n");
			return ret;
		}

		ret = dwc3_gadget_init(dwc);
		if (ret) {
			dev_err(dev, "failed to initialize gadget\n");
			return ret;
		}
		break;
	default:
		dev_err(dev, "Unsupported mode of operation %d\n", dwc->dr_mode);
		return -EINVAL;
	}

	return 0;
}

static void dwc3_core_exit_mode(struct dwc3 *dwc)
{
	switch (dwc->dr_mode) {
	case USB_DR_MODE_PERIPHERAL:
		dwc3_gadget_exit(dwc);
		break;
	case USB_DR_MODE_HOST:
		dwc3_host_exit(dwc);
		break;
	case USB_DR_MODE_OTG:
		dwc3_host_exit(dwc);
		dwc3_gadget_exit(dwc);
		break;
	default:
		/* do nothing */
		break;
	}
}

/*static*/
u8 phy_ss_test_read(struct dwc3 *dwc, u8 addr)
{
	u32 reg = 0;

	if (addr > 0x1f) {
		printk("wrong addr 0x%02X\n", addr);
	} else {
		reg = readl(dwc->regs_gpf + USDR_CSR2_OFF);
		reg &= USDR_CSR2__TEST_ADDR__INV_MASK;
		reg |= 0x01;   // set TEST_I[0] for read
		reg |= ((u32)addr << USDR_CSR2__TEST_ADDR__SHIFT);
		writel(reg, dwc->regs_gpf + USDR_CSR2_OFF);
		udelay(1);
		reg = (readl(dwc->regs_gpf + USDR_CSR2_OFF) >>
				USDR_CSR2__TEST_O__SHIFT);
	}
	return (u8)reg;
}

/*static*/
u8 phy_ss_test_read_status(struct dwc3 *dwc, u8 addr)
{
	u32 reg = 0;

	reg = readl(dwc->regs_gpf + USDR_CSR2_OFF);
	reg &= USDR_CSR2__TEST_ADDR__INV_MASK;
	reg &= ~0x01;   /* clear TEST_I[0] to read address 0x02 */
	reg |= ((u32)addr << USDR_CSR2__TEST_ADDR__SHIFT);
	writel(reg, dwc->regs_gpf + USDR_CSR2_OFF);
	udelay(1);
	reg = (readl(dwc->regs_gpf + USDR_CSR2_OFF) >>
			USDR_CSR2__TEST_O__SHIFT);
	return (u8)reg;
}

/*static*/
void phy_ss_test_write(struct dwc3 *dwc, u8 addr, u8 value)
{
	u32 reg = 0;

	if (addr > 0x1f) {
		printk("wrong addr 0x%02X\n", addr);
	} else {
		reg = readl(dwc->regs_gpf + USDR_CSR2_OFF);
		reg &= (USDR_CSR2__TEST_ADDR__INV_MASK &
				USDR_CSR2__TEST_I__INV_MASK &
				USDR_CSR2__TEST_WRITE__INV_MASK);
		reg |= ((u32)addr << USDR_CSR2__TEST_ADDR__SHIFT);
		reg |= ((u32)value << USDR_CSR2__TEST_I__SHIFT);
		writel(reg, dwc->regs_gpf + USDR_CSR2_OFF);
		udelay(1);
		reg |= USDR_CSR2__TEST_WRITE__MASK;     /* assert TEST_WRITE */
		writel(reg, dwc->regs_gpf + USDR_CSR2_OFF);
		udelay(1);
		reg &= USDR_CSR2__TEST_WRITE__INV_MASK; /* de-assert TEST_WRITE */
		writel(reg, dwc->regs_gpf + USDR_CSR2_OFF);
	}
}

/*static*/
void phy_ss_tx_swing_manual(struct dwc3 *dwc, u8 swing_by_ctrl, u8 swing)
{
	u32 reg1 = 0, reg2 = 0;

	if (swing > 1) {
		printk("wrong swing %d\n", swing);
	} else {
		reg1 = readl(dwc->regs_gpf + USDR_CSR1_OFF);
		reg1 &= USDR_CSR1__SS_PHY_TX_SWING_SEL__INV_MASK;
		if (swing_by_ctrl) {
			reg2 = dwc3_readl(dwc->regs, DWC3_GUSB3PIPECTL(0));
			reg2 &= USDR_GUSB3PIPECTL_0__TX_SWING__INV_MASK;
			reg2 |= (swing << USDR_GUSB3PIPECTL_0__TX_SWING__SHIFT);
			dwc3_writel(dwc->regs, DWC3_GUSB3PIPECTL(0), reg2);
		} else {
			reg1 |= USDR_CSR1__SS_PHY_TX_SWING_SEL__MASK;
			reg1 &= USDR_CSR1__SS_PHY_TX_SWING__INV_MASK;
			reg1 |= (swing << USDR_CSR1__SS_PHY_TX_SWING__SHIFT);
		}
		writel(reg1, dwc->regs_gpf + USDR_CSR1_OFF);
	}
}

/*static*/
void phy_ss_tx_deemp_manual(struct dwc3 *dwc, u8 deemp_by_ctrl, u8 deemp)
{
	u32 reg1 = 0, reg2 = 0;

	if (deemp > 2) {
		printk("wrong deemp %d\n", deemp);
	} else {
		reg1 = readl(dwc->regs_gpf + USDR_CSR1_OFF);
		reg1 &= USDR_CSR1__SS_PHY_TX_DEEMP_SEL__INV_MASK;
		if (deemp_by_ctrl) {
			reg2 = dwc3_readl(dwc->regs, DWC3_GUSB3PIPECTL(0));
			reg2 &= USDR_GUSB3PIPECTL_0__SS_TX_DE_EMPHASIS__INV_MASK;
			reg2 |= (deemp << USDR_GUSB3PIPECTL_0__SS_TX_DE_EMPHASIS__SHIFT);
			dwc3_writel(dwc->regs, DWC3_GUSB3PIPECTL(0), reg2);
		} else {
			reg1 |= USDR_CSR1__SS_PHY_TX_DEEMP_SEL__MASK;
			reg1 &= USDR_CSR1__SS_PHY_TX_DEEMP__INV_MASK;
			reg1 |= (deemp << USDR_CSR1__SS_PHY_TX_DEEMP__SHIFT);
		}
		writel(reg1, dwc->regs_gpf + USDR_CSR1_OFF);
	}
}

/*static*/
void phy_ss_tx_margin_manual(struct dwc3 *dwc, u8 margin_by_ctrl, u8 margin)
{
	u32 reg1 = 0, reg2 = 0;

	if (margin > 7) {
		printk("wrong margin %d\n", margin);
	} else {
		reg1 = readl(dwc->regs_gpf + USDR_CSR1_OFF);
		reg1 &= USDR_CSR1__SS_PHY_TX_MARGIN_SEL__INV_MASK;
		if (margin_by_ctrl) {
			reg2 = dwc3_readl(dwc->regs, DWC3_GUSB3PIPECTL(0));
			reg2 &= USDR_GUSB3PIPECTL_0__TX_MARGIN__INV_MASK;
			reg2 |= (margin << USDR_GUSB3PIPECTL_0__TX_MARGIN__SHIFT);
			dwc3_writel(dwc->regs, DWC3_GUSB3PIPECTL(0), reg2);
		} else {
			reg1 |= USDR_CSR1__SS_PHY_TX_MARGIN_SEL__MASK;
			reg1 &= USDR_CSR1__SS_PHY_TX_MARGIN__INV_MASK;
			reg1 |= (margin << USDR_CSR1__SS_PHY_TX_MARGIN__SHIFT);
		}
		writel(reg1, dwc->regs_gpf + USDR_CSR1_OFF);
	}
}

/*static*/
void phy_ss_tx_amp(struct dwc3 *dwc)
{
	u32 reg = 0;

	/* Increase overall swing level */
	reg = (u32)phy_ss_test_read(dwc, SS_PHY_PARAM06_ADDR);
	reg &= ~0xc0;
	reg |= 0xc0;
	phy_ss_test_write(dwc, SS_PHY_PARAM06_ADDR, (u8)reg);

	/* Increase pre-emphasis */
	reg = (u32)phy_ss_test_read(dwc, SS_PHY_PARAM10_ADDR);
	reg &= ~0x70;
	reg |= 0x30;
	phy_ss_test_write(dwc, SS_PHY_PARAM10_ADDR, (u8)reg);

	/* Increase pre-emphasis */
	reg = (u32)phy_ss_test_read(dwc, SS_PHY_PARAM24_ADDR);
	reg &= ~0x80;
	reg |= 0x80;
	phy_ss_test_write(dwc, SS_PHY_PARAM24_ADDR, (u8)reg);
}

/*static*/
void phy_ss_rx_eq(struct dwc3 *dwc, u8 auto_eq, u8 eq_val)
{
	u32 reg = 0;

	if (eq_val > 0x3f) {
		printk("wrong eq_val 0x%02x\n", eq_val);
		return;
	}

	reg = (u32)phy_ss_test_read(dwc, SS_PHY_PARAM06_ADDR);
	reg &= ~0x3f;
	reg |= eq_val;
	phy_ss_test_write(dwc, SS_PHY_PARAM06_ADDR, (u8)reg);

	reg = (u32)phy_ss_test_read(dwc, SS_PHY_PARAM00_ADDR);
	if (auto_eq == 1) {
		reg &= ~0x08;
	} else {
		reg |= 0x08;
	}
	phy_ss_test_write(dwc, SS_PHY_PARAM00_ADDR, (u8)reg);
}

#if !PHY_RX_MANUAL_EQ
/*static*/
u8 phy_ss_get_auto_eq(struct dwc3 *dwc)
{
	u32 reg = 0;

	if (usbd_ss == 0)
		return 0;

	reg = phy_ss_test_read_status(dwc, SS_PHY_PARAM07_ADDR);
	reg &= 0x3f;   /* autoEQ value */

	return (reg);
}

/*static*/
void phy_ss_auto_to_manual_eq(struct dwc3 *dwc, u8 eq_val)
{
	u32 reg = 0;

	if (eq_val > 0x3f) {
		printk("wrong eq_val 0x%02x\n", eq_val);
		return;
	}

	reg = phy_ss_test_read(dwc, SS_PHY_PARAM06_ADDR);
	reg &= ~0x3f;
	reg |= eq_val;
	phy_ss_test_write(dwc, SS_PHY_PARAM06_ADDR, (u8)reg);

	reg = phy_ss_test_read(dwc, SS_PHY_PARAM00_ADDR);
	reg |= 0x08;   /* manual EQ */
	phy_ss_test_write(dwc, SS_PHY_PARAM00_ADDR, (u8)reg);
}
#endif

static void phy_ss_cdr_cpd_trim(struct dwc3 *dwc, u8 val)
{
	u32 reg = 0;

	if (val > 0x0f) {
		printk("wrong CDR_CPD_TRIM value %d\n", val);
		return;
	}
	reg = (u32)phy_ss_test_read(dwc, SS_PHY_PARAM07_ADDR);
	reg &= ~0x0f;
	reg |= val;
	phy_ss_test_write(dwc, SS_PHY_PARAM07_ADDR, (u8)reg);
}

static void phy_ss_cdr_trim(struct dwc3 *dwc, u8 val)
{
	u32 reg = 0;

	if (val > 0x03) {
		printk("wrong CDR_TRIM value %d\n", val);
		return;
	}
	reg = (u32)phy_ss_test_read(dwc, SS_PHY_PARAM13_ADDR);
	reg &= ~0x60;
	reg |= (val << 5);
	phy_ss_test_write(dwc, SS_PHY_PARAM13_ADDR, (u8)reg);
}

static void phy_ss_cdr_cpf_trim(struct dwc3 *dwc, u8 val)
{
	u32 reg = 0;

	if (val > 0x0f) {
		printk("wrong CDR_CPF_TRIM value %d\n", val);
		return;
	}
	reg = (u32)phy_ss_test_read(dwc, SS_PHY_PARAM08_ADDR);
	reg &= ~0x0f;
	reg |= val;
	phy_ss_test_write(dwc, SS_PHY_PARAM08_ADDR, (u8)reg);
}

/*static*/
void phy_ss_bgr_trim(struct dwc3 *dwc, u8 val)
{
	u32 reg = 0;

	if (val > 0x0f) {
		printk("wrong BGR_TRIM value %d\n", val);
		return;
	}
	reg = (u32)phy_ss_test_read(dwc, SS_PHY_PARAM11_ADDR);
	reg &= ~0x0f;
	reg |= val;
	phy_ss_test_write(dwc, SS_PHY_PARAM11_ADDR, (u8)reg);
}

#if CDR_RESET_COUNTERMEASURE
void phy_ss_cdr_reset(struct dwc3 *dwc)
{
#if CDR_RESET_COUNTERMEASURE_DEBUG
	u8 status0_before_1, status0_after_4, status0_after_5, status9_before_1, status9_after_4, status9_after_5;
#endif
	u8 reg = 0;
	u32 i = 0, j = 0;

#if CDR_RESET_COUNTERMEASURE_DEBUG
	printk("\nsppd 0x%X\n", dwc3_readl(dwc->regs, DWC3_DSTS) & USDR_DSTS__CONNECTSPD__MASK);
	printk("lnkst 0x%X\n", (dwc3_readl(dwc->regs, DWC3_DSTS) & USDR_DSTS__USBLNKST__MASK) >> USDR_DSTS__USBLNKST__SHIFT);
#endif
	if (((dwc3_readl(dwc->regs, DWC3_DSTS) & USDR_DSTS__USBLNKST__MASK)
		>> USDR_DSTS__USBLNKST__SHIFT) == 7 /*POLL*/) {  // only apply to SuperSpeed

		while (1) {
			reg = phy_ss_test_read_status(dwc, SS_PHY_PARAM09_ADDR);
			if (0 == (reg & 0x08)) {
				j++;
				udelay(300);   // at least 224us
				continue;
			} else {
#if CDR_RESET_COUNTERMEASURE_DEBUG
				printk("\nSTS9=8\n");
#endif
			}
			udelay(300);   // at least 224us
#if CDR_RESET_COUNTERMEASURE_DEBUG
			status0_before_1 = phy_ss_test_read_status(dwc, SS_PHY_PARAM00_ADDR);
			status9_before_1 = phy_ss_test_read_status(dwc, SS_PHY_PARAM09_ADDR);
#endif
			phy_ss_test_write(dwc, SS_PHY_PARAM14_ADDR, 0x80);
			mb();
			phy_ss_test_write(dwc, SS_PHY_PARAM17_ADDR, 0xCF);
			mb();
			phy_ss_test_write(dwc, SS_PHY_PARAM01_ADDR, 0x07);
			mb();
			phy_ss_test_write(dwc, SS_PHY_PARAM01_ADDR, 0x04);
			mb();
#if CDR_RESET_COUNTERMEASURE_DEBUG
			status0_after_4 = phy_ss_test_read_status(dwc, SS_PHY_PARAM00_ADDR);
			status9_after_4 = phy_ss_test_read_status(dwc, SS_PHY_PARAM09_ADDR);
			mb();
#endif
			phy_ss_test_write(dwc, SS_PHY_PARAM17_ADDR, 0xCE);
			mb();
#if CDR_RESET_COUNTERMEASURE_DEBUG
			status0_after_5 = phy_ss_test_read_status(dwc, SS_PHY_PARAM00_ADDR);
			status9_after_5 = phy_ss_test_read_status(dwc, SS_PHY_PARAM09_ADDR);
#endif
			reg = phy_ss_test_read_status(dwc, SS_PHY_PARAM09_ADDR);
			if (reg & 0x08) {
#if CDR_RESET_COUNTERMEASURE_DEBUG
				printk("CDR reset %d-time done, 0x%X, wait %d00 us before resetting CDR\n", (i+1), reg, j);
				printk("before stpe 1: STATUS0=0x%X, STATUS9=0x%X\n", status0_before_1, status9_before_1);
				printk("after  stpe 4: STATUS0=0x%X, STATUS9=0x%X\n", status0_after_4, status9_after_4);
				printk("after  stpe 5: STATUS0=0x%X, STATUS9=0x%X\n", status0_after_5, status9_after_5);
#endif
				break;
			} else {
				if (20 < i++) {
					printk("CDR reset failed, 0x%X\n", reg);
					break;
				} else {
					continue;
				}
			}
		}
	} else {
#if CDR_RESET_COUNTERMEASURE_DEBUG
		printk("lnkst 0x%X\n", (dwc3_readl(dwc->regs, DWC3_DSTS) & USDR_DSTS__USBLNKST__MASK) >> USDR_DSTS__USBLNKST__SHIFT);
#endif
	}
}
#endif

#define NEW_DDR3_B2_EVM 0   /* setting for new DDR3 B2 EVM */

/*static*/
void phy_init_ss(struct dwc3 *dwc)
{
	// wr USDR_CSR1[0] or USDR_GUSB3PIPECTL_0[6]
	// rd USDR_CSR1[2]
	// 0 full swing
	// 1 low swing
	//phy_ss_tx_swing_manual(dwc, 1, 0);

	// wr USDR_CSR1[5:4] or USDR_GUSB3PIPECTL_0[2:1]
	// rd USDR_CSR1[9:8]
	// 00 -6dB
	// 01 -3.5dB
	// 10 0dB
	// 11 rreserved
	phy_ss_tx_deemp_manual(dwc, 1, 0);

	// wr USDR_CSR1[14:12] or USDR_GUSB3PIPECTL_0[5:3]
	// rd USDR_CSR1[18:16]
	// 000 1000(mV) (default)
	// 001 1100
	// 010  900
	// 011  800
	// 100  700
	// 101  600
	// 110  500
	// 111  400
#if NEW_DDR3_B2_EVM
	phy_ss_tx_margin_manual(dwc, 1, 1);
#else
	phy_ss_tx_margin_manual(dwc, 1, 2);
#endif

	//phy_ss_tx_amp(dwc);   // PHY's own adjustment for Tx

	// Enlage Tx
#if !NEW_DDR3_B2_EVM
	phy_ss_test_write(dwc, SS_PHY_PARAM06_ADDR, 0xE0);
	phy_ss_test_write(dwc, SS_PHY_PARAM10_ADDR, 0x3F);
	phy_ss_test_write(dwc, SS_PHY_PARAM24_ADDR, 0x80);
#endif

#if PHY_RX_MANUAL_EQ
	phy_ss_rx_eq(dwc, 0 /*0=manual EQ*/, PHY_RX_MANUAL_EQ_VALUE);
#endif
#ifdef Q6300A0
	phy_ss_test_write(dwc, SS_PHY_PARAM01_ADDR, 0x84);   // per PHY vendor for U1.exit issue for ES0
#endif

	// addr 0x07[3:0]
	phy_ss_cdr_cpd_trim(dwc, 0x6);

	// addr 0x0d[6:5]
	phy_ss_cdr_trim(dwc, 0x02);
	// addr 0x08[3:0]
	phy_ss_cdr_cpf_trim(dwc, 0x01);

	// addr 0x0b[3:0]
#if NEW_DDR3_B2_EVM
	phy_ss_bgr_trim(dwc, 0x0f);
#endif
}

/*static*/
u8 phy_hs_test_read(struct dwc3 *dwc, u16 addr)
{
	u32 reg = 0, val = 0;

	if (addr > 0xfff) {
		printk("wrong addr 0x%x\n", addr);
	} else {
		reg = readl(dwc->regs_gpf + USDR_CSR4_OFF);
		reg &= 0x8000ffff;   // mask PHY_CONFIG_MONITOR[62:48]
		reg |= (addr << 16);   // prepare address
		writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
		udelay(1);
		reg |= 0x10000000;   // assert addr strobe
		writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
		udelay(1);
		reg &= ~0x10000000;   // de-assert addr strobe
		writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
		udelay(1);
		reg |= 0x40000000;   // assert read strobe
		writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
		udelay(1);
		val = readl(dwc->regs_gpf + USDR_CSR5_OFF);
		val &= 0xff;   // get data
		reg &= ~0x40000000;   // de-assert read strobe
		writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
	}
	return (u8)val;
}

/*static*/
void phy_hs_test_write(struct dwc3 *dwc, u16 addr, u8 value)
{
	u32 reg = 0;

	if (addr > 0xfff) {
		printk("wrong addr 0x%x\n", addr);
	} else {
		reg = readl(dwc->regs_gpf + USDR_CSR4_OFF);
		reg &= 0x8000ffff;   // mask PHY_CONFIG_MONITOR[62:48]
		reg |= (addr << 16);   // prepare address
		writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
		udelay(1);
		reg |= 0x10000000;   // assert addr strobe
		writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
		udelay(1);
		reg &= ~0x10000000;   // de-assert addr strobe
		writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
		udelay(1);
		reg &= 0xFF00FFFF;
		reg |= (value << 16);   // prepare write data
		writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
		udelay(1);
		reg |= 0x20000000;   // assert write strobe
		writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
		udelay(1);
		reg &= ~0x20000000;   // de-assert write strobe
		writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
	}
}

void phy_dump_settings(struct dwc3 *dwc)
{
	printk("********************************\n");
	printk("*** CDR reset approach is %s\n", CDR_RESET_COUNTERMEASURE ? "on" : "off");
	printk("*** SS_PHY_PARAM00 (0x00) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM00_ADDR));
	printk("*** SS_PHY_PARAM01 (0x01) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM01_ADDR));
	printk("*** SS_PHY_PARAM02 (0x02) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM02_ADDR));
	printk("*** SS_PHY_PARAM02'(0x02) = 0x%02X\n", phy_ss_test_read_status(dwc, SS_PHY_PARAM02_ADDR));
	printk("*** SS_PHY_PARAM03 (0x03) = 0x%02X (deskew)\n", phy_ss_test_read(dwc, SS_PHY_PARAM03_ADDR));
	printk("*** SS_PHY_PARAM04 (0x04) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM04_ADDR));
	printk("*** SS_PHY_PARAM05 (0x05) = 0x%02X ([3:0]=EQ bit)\n", phy_ss_test_read(dwc, SS_PHY_PARAM05_ADDR));
	printk("*** SS_PHY_PARAM06 (0x06) = 0x%02X ([5:0]=initial EQ)\n", phy_ss_test_read(dwc, SS_PHY_PARAM06_ADDR));
	printk("*** SS_PHY_PARAM07 (0x07) = 0x%02X ([3:0]=cpd)\n", phy_ss_test_read(dwc, SS_PHY_PARAM07_ADDR));
	printk("*** SS_PHY_PARAM07'(auto eq) = 0x%02X\n", phy_ss_test_read_status(dwc, SS_PHY_PARAM07_ADDR));
	printk("*** SS_PHY_PARAM08 (0x08) = 0x%02X ([3:0]=cpf)\n", phy_ss_test_read(dwc, SS_PHY_PARAM08_ADDR));
	printk("*** SS_PHY_PARAM09 (0x09) = 0x%02X (txpll)\n", phy_ss_test_read(dwc, SS_PHY_PARAM09_ADDR));
	printk("*** SS_PHY_PARAM09'(0x09) = 0x%02X ([3]=EQ training requesting)\n", phy_ss_test_read_status(dwc, SS_PHY_PARAM09_ADDR));
	printk("*** SS_PHY_PARAM10 (0x0A) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM10_ADDR));
	printk("*** SS_PHY_PARAM11 (0x0B) = 0x%02X ([3:0]=BGR)\n", phy_ss_test_read(dwc, SS_PHY_PARAM11_ADDR));
	printk("*** SS_PHY_PARAM12 (0x0C) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM12_ADDR));
	printk("*** SS_PHY_PARAM13 (0x0D) = 0x%02X ([6:5]=cdr)\n", phy_ss_test_read(dwc, SS_PHY_PARAM13_ADDR));
	printk("*** SS_PHY_PARAM14 (0x0E) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM14_ADDR));
	printk("*** SS_PHY_PARAM15 (0x0F) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM15_ADDR));
	printk("*** SS_PHY_PARAM16 (0x10) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM16_ADDR));
	printk("*** SS_PHY_PARAM17 (0x11) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM17_ADDR));
	printk("*** SS_PHY_PARAM18 (0x12) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM18_ADDR));
	printk("*** SS_PHY_PARAM19 (0x13) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM19_ADDR));
	printk("*** SS_PHY_PARAM20 (0x14) = 0x%02X ([3:0]=rxterm)\n", phy_ss_test_read(dwc, SS_PHY_PARAM20_ADDR));
	printk("*** SS_PHY_PARAM21 (0x15) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM21_ADDR));
	printk("*** SS_PHY_PARAM22 (0x16) = 0x%02X ([3:0]=CDRVCO)\n", phy_ss_test_read(dwc, SS_PHY_PARAM22_ADDR));
	printk("*** SS_PHY_PARAM23 (0x17) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM23_ADDR));
	printk("*** SS_PHY_PARAM24 (0x18) = 0x%02X ([5:4]=OFFSETCNT)\n", phy_ss_test_read(dwc, SS_PHY_PARAM24_ADDR));
	printk("*** SS_PHY_PARAM25 (0x19) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM25_ADDR));
	printk("*** SS_PHY_PARAM26 (0x1A) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM26_ADDR));
	printk("*** SS_PHY_PARAM27 (0x1B) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM27_ADDR));
	printk("*** SS_PHY_PARAM28 (0x1C) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM28_ADDR));
	printk("*** SS_PHY_PARAM29 (0x1D) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM29_ADDR));
	printk("*** SS_PHY_PARAM30 (0x1E) = 0x%02X\n", phy_ss_test_read(dwc, SS_PHY_PARAM30_ADDR));
	printk("*** SS_PHY_PARAM31 (0x1F) = 0x%02X ([7:4]=RXDET_FILTER)\n", phy_ss_test_read(dwc, SS_PHY_PARAM31_ADDR));
	printk("*** DWC3_GUSB3PIPECTL     = 0x%02X\n", dwc3_readl(dwc->regs, DWC3_GUSB3PIPECTL(0)));
	printk("***             Tx margin = 0x%02X\n", (dwc3_readl(dwc->regs, DWC3_GUSB3PIPECTL(0)) >> 3) & 0x7);
	printk("***             Tx deemp  = 0x%02X\n", (dwc3_readl(dwc->regs, DWC3_GUSB3PIPECTL(0)) >> 1) & 0x3);
	printk("*** HS_PHY_PARAM00        = 0x%02X\n", phy_hs_test_read(dwc, 0));
	printk("*** HS_PHY_PARAM01        = 0x%02X\n", phy_hs_test_read(dwc, 1));
}

/*static*/
void phy_hs_default(struct dwc3 *dwc, u8 use_efuse,
			u32 hi, u32 lo)
{
	u32 reg = 0;
#ifndef Q6300
	u32 trimr, trimi;
#endif

	if (use_efuse)
	{
		reg = readl(dwc->regs_gpf + USDR_CSR5_OFF);
#ifdef Q6300
		reg |= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__MASK;
#else
		reg &= (USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__INV_MASK &
				USDR_CSR5__HS_PHY_EFUSE_TRIMI_SEL__INV_MASK);
#endif
		writel(reg, dwc->regs_gpf + USDR_CSR5_OFF);
		return;
	} else	{
		reg = readl(dwc->regs_gpf + USDR_CSR5_OFF);
#ifdef Q6300
		reg &= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__INV_MASK;
#else
		reg |= (USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__MASK |
				USDR_CSR5__HS_PHY_EFUSE_TRIMI_SEL__MASK);
		reg &= (USDR_CSR5__HS_PHY_TRIMR__INV_MASK &
				USDR_CSR5__HS_PHY_TRIMI__INV_MASK);

		/* TRIMR & TRIMI */
		trimr = ((lo & 0x000000C0) >> 2) + ((lo & 0x0000F000) >> 12);
		trimi = ((lo & 0xF0000000) >> 28);
		reg |= (trimr << USDR_CSR5__HS_PHY_TRIMR__SHIFT) +
			(trimi << USDR_CSR5__HS_PHY_TRIMI__SHIFT);
#endif
		writel(reg, dwc->regs_gpf + USDR_CSR5_OFF);

		writel(lo, dwc->regs_gpf + USDR_CSR3_OFF);
		writel(hi, dwc->regs_gpf + USDR_CSR4_OFF);
	}
}

// [7:6]
/*static*/
void phy_hs_rterm_trim(struct dwc3 *dwc, u8 use_efuse, u8 rterm)
{
	u32 reg = 0;
#ifndef Q6300
	u32 trimr;
#endif

	if (rterm > 0x03) {
		printk("wrong rterm 0x%02X\n", rterm);
		return;
	}
	if (use_efuse) {
		reg = readl(dwc->regs_gpf + USDR_CSR5_OFF);
#ifdef Q6300
		reg |= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__MASK;
#else
		reg &= USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__INV_MASK;
#endif
		writel(reg, dwc->regs_gpf + USDR_CSR5_OFF);
		return;
	} else {
		reg = readl(dwc->regs_gpf + USDR_CSR5_OFF);
#ifdef Q6300
		reg &= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__INV_MASK;
#else
		reg |= USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__MASK;
		reg &= USDR_CSR5__HS_PHY_TRIMR__INV_MASK;

		/* TRIMR & TRIMI */
		trimr = (rterm << 4) + ((readl(dwc->regs_gpf + USDR_CSR3_OFF) & 0x0000F000) >> 12);
		reg |= (trimr << USDR_CSR5__HS_PHY_TRIMR__SHIFT);
#endif
		writel(reg, dwc->regs_gpf + USDR_CSR5_OFF);

		reg = readl(dwc->regs_gpf + USDR_CSR3_OFF);
		reg &= ~0x000000c0;
		reg |= (rterm << 6);
		writel(reg, dwc->regs_gpf + USDR_CSR3_OFF);
	}
}

// [15:12]
/*static*/
void phy_hs_term_resistor_trim(struct dwc3 *dwc, u8 use_efuse, u8 term_resistor)
{
	u32 reg = 0;
#ifndef Q6300
	u32 trimr;
#endif

	if (term_resistor > 0x0f) {
		printk("wrong term_resistor 0x%02X\n", term_resistor);
		return;
	}
	if (use_efuse) {
		reg = readl(dwc->regs_gpf + USDR_CSR5_OFF);
#ifdef Q6300
		reg |= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__MASK;
#else
		reg &= USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__INV_MASK;
#endif
		writel(reg, dwc->regs_gpf + USDR_CSR5_OFF);
		return;
	} else {
		reg = readl(dwc->regs_gpf + USDR_CSR5_OFF);
#ifdef Q6300
		reg &= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__INV_MASK;
#else
		reg |= USDR_CSR5__HS_PHY_EFUSE_TRIMR_SEL__MASK;
		reg &= USDR_CSR5__HS_PHY_TRIMR__INV_MASK;

		/* TRIMR & TRIMI */
		trimr = ((readl(dwc->regs_gpf + USDR_CSR3_OFF) & 0x000000C0) >> 2) + term_resistor;
		reg |= (trimr << USDR_CSR5__HS_PHY_TRIMR__SHIFT);
#endif
		writel(reg, dwc->regs_gpf + USDR_CSR5_OFF);

		reg = readl(dwc->regs_gpf + USDR_CSR3_OFF);
		reg &= ~0x0000f000;
		reg |= (term_resistor << 12);
		writel(reg, dwc->regs_gpf + USDR_CSR3_OFF);
	}
}

// [17:16]
/*static*/
void phy_hs_swing(struct dwc3 *dwc, u8 swing)
{
	u32 reg = 0;

	if (swing > 0x03) {
		printk("wrong swing 0x%02X\n", swing);
		return;
	}
	reg = readl(dwc->regs_gpf + USDR_CSR3_OFF);
	reg &= ~0x00030000;
	reg |= (swing << 16);
	writel(reg, dwc->regs_gpf + USDR_CSR3_OFF);
}

// [25:23]
/*static*/
void phy_hs_band_gap(struct dwc3 *dwc, u8 bandgap)
{
	u32 reg = 0;

	if (bandgap > 0x07) {
		printk("wrong bandgap 0x%02X\n", bandgap);
		return;
	}
	reg = readl(dwc->regs_gpf + USDR_CSR3_OFF);
	reg &= ~0x03800000;
	reg |= (bandgap << 23);
	writel(reg, dwc->regs_gpf + USDR_CSR3_OFF);
}

// [31:28]
/*static*/
void phy_hs_current_trim(struct dwc3 *dwc, u8 use_efuse, u8 curr)
{
	u32 reg = 0;

	if (curr > 0x0f) {
		printk("wrong current 0x%02X\n", curr);
		return;
	}
	if (use_efuse) {
		reg = readl(dwc->regs_gpf + USDR_CSR5_OFF);
#ifdef Q6300
		reg |= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__MASK;
#else
		reg &= USDR_CSR5__HS_PHY_EFUSE_TRIMI_SEL__INV_MASK;
#endif
		writel(reg, dwc->regs_gpf + USDR_CSR5_OFF);
		return;
	} else {
		reg = readl(dwc->regs_gpf + USDR_CSR5_OFF);
#ifdef Q6300
		reg &= USDR_CSR5__HS_PHY_P0_EFUSE_SEL__INV_MASK;
#else
		reg |= USDR_CSR5__HS_PHY_EFUSE_TRIMI_SEL__MASK;
		reg &= USDR_CSR5__HS_PHY_TRIMI__INV_MASK;

		/* TRIMI */
		reg |= (curr << USDR_CSR5__HS_PHY_TRIMI__SHIFT);
#endif
		writel(reg, dwc->regs_gpf + USDR_CSR5_OFF);

		reg = readl(dwc->regs_gpf + USDR_CSR3_OFF);
		reg &= ~0xf0000000;
		reg |= (curr << 28);
		writel(reg, dwc->regs_gpf + USDR_CSR3_OFF);
	}
}

// [36]
/*static*/
void phy_hs_driver_slew(struct dwc3 *dwc, u8 high_slew)
{
	u32 reg = 0;

	if (high_slew > 1) {
		printk("wrong current 0x%02X\n", high_slew);
		return;
	}
	reg = readl(dwc->regs_gpf + USDR_CSR4_OFF);
	reg &= ~0x00000010;
	reg |= (high_slew << 4);
	writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
}

// [40:39]
/*static*/
void phy_hs_sq_trim(struct dwc3 *dwc, u8 squelch)
{
	u32 reg = 0;

	if (squelch > 3) {
		printk("wrong current 0x%02X\n", squelch);
		return;
	}
	reg = readl(dwc->regs_gpf + USDR_CSR4_OFF);
	reg &= ~0x00000180;
	reg |= (squelch << 7);
	writel(reg, dwc->regs_gpf + USDR_CSR4_OFF);
}

/*static*/
void phy_init_hs(struct dwc3 *dwc)
{
#ifdef Q6600B0
    phy_hs_default(dwc, 0, 0x00000706, 0x92316680);   /* 0000_0706_9231_6680 */
#else
	phy_hs_default(dwc, 0, 0x00000106, 0x92316680);   /* 0000_0106_9231_6680 */
#endif

	// [7:6]
	//phy_hs_rterm_trim(dwc, 0, 0);

	// [15:12]
	// 0001 Resistance large
	// 0110 Resistance (default)
	// 1111 Resistance small
	//phy_hs_term_resistor_trim(dwc, 0, 6);   // for FS

	// [17:16]
	// 00 = 125%
	// 01 = 100% (recommended)
	// 10 = 110%
	// 11 =  90%
	phy_hs_swing(dwc, 2);

	// [25:23]
	// 000 =  80%
	// 001 =  85%
	// 010 =  90%
	// 011 =  95%
	// 100 = 100% (recommended)
	// 101 = 105%
	// 110 = 110%
	// 111 = 115%
	//phy_hs_band_gap(dwc, 4);

	// [31:28]
	//phy_hs_current_trim(dwc, 0, 0x9);   // if device HS eye too small or too large, 0xb is good for socket board US HS

	// [36]
	// 0 = high slew rate
	// 1 = low slew rate
	//phy_hs_driver_slew(dwc, 1);

	// [40:39]
	// 00 = 110mV
	// 01 = 122mV
	// 10 = 134mV (recommended)
	// 11 = 146mV
	//phy_hs_sq_trim(dwc, 2);

#ifdef Q6300   // QB66xx USB HS PHY has default settings for it
	phy_hs_test_write(dwc, 0, 0x20);   // RX_CHK_SYNC=1
	phy_hs_test_write(dwc, 1, 0x01);   // RX_SYNC_SEL=00b=6 bits; 01b=5 bits; 10b=4 bits
#endif
}

#define DWC3_ALIGN_MASK		(64 - 1) // aligned to 64 bytes, ori (16 - 1)

/*static*/
void phy_reset_ss(struct dwc3 *dwc, int reset)
{
	u32 reg;

	if (usbd_ss == 0)
		return;

	udelay(5);
	if (reset) {
		reg = readl(dwc->regs_gpf + USDR_RST_OFF);
		reg &= (USDR_RST__SS_PHY_RESET_N_SEL__INV_MASK &
			USDR_RST__SS_PHY_RESET_N__INV_MASK);
		writel(reg, dwc->regs_gpf + USDR_RST_OFF);

		reg = readl(dwc->regs_gpf + USDR_CSR2_OFF);
		reg &= USDR_CSR2__TEST_RESET_N__INV_MASK;
		writel(reg, dwc->regs_gpf + USDR_CSR2_OFF);
	} else {
		reg = readl(dwc->regs_gpf + USDR_CSR2_OFF);
		reg |= USDR_CSR2__TEST_RESET_N__MASK;
		writel(reg, dwc->regs_gpf + USDR_CSR2_OFF);
		udelay(5);

		phy_init_ss(dwc);   /* must set after TEST_RESET_N is high */
		udelay(5);

		reg = readl(dwc->regs_gpf + USDR_RST_OFF);
		reg |= USDR_RST__SS_PHY_RESET_N__MASK;
		writel(reg, dwc->regs_gpf + USDR_RST_OFF);
	}
	udelay(5);
}

/*static*/
void phy_reset_hs(struct dwc3 *dwc, int reset)
{
	u32 reg;

	udelay(5);
	if (reset) {
		reg = readl(dwc->regs_gpf + USDR_CSR6_OFF);
		reg &= (USDR_CSR6__HS_PHY_P0_SUSPENDM_SEL__INV_MASK &
			USDR_CSR6__HS_PHY_P0_SUSPENDM__INV_MASK);
		reg |= (2 << USDR_CSR6__HS_PHY_P0_SUSPENDM_SEL__SHIFT);
		writel(reg, dwc->regs_gpf + USDR_CSR6_OFF);

		udelay(5);

		reg = readl(dwc->regs_gpf + USDR_RST_OFF);
		reg |=  USDR_RST__HS_PHY_P0_RESET__MASK;
		reg &= (USDR_RST__HS_PHY_P0_RESET_SEL__INV_MASK &
			USDR_RST__HS_PHY_PWON_RESETN__INV_MASK);
		writel(reg, dwc->regs_gpf + USDR_RST_OFF);
	} else {
		reg = readl(dwc->regs_gpf + USDR_RST_OFF);
		reg |= USDR_RST__HS_PHY_PWON_RESETN__MASK;
		writel(reg, dwc->regs_gpf + USDR_RST_OFF);
		udelay(5);

		phy_init_hs(dwc);   /* must set after PWON_RESETN is high */
		udelay(5);

		reg = readl(dwc->regs_gpf + USDR_RST_OFF);
		reg &= USDR_RST__HS_PHY_P0_RESET__INV_MASK;
		writel(reg, dwc->regs_gpf + USDR_RST_OFF);
		udelay(5);

		reg = readl(dwc->regs_gpf + USDR_CSR6_OFF);
		reg |= USDR_CSR6__HS_PHY_P0_SUSPENDM__MASK;
		writel(reg, dwc->regs_gpf + USDR_CSR6_OFF);
	}
	udelay(5);
}

static void usbd_hw_reset(struct dwc3 *dwc)
{
	volatile u8	__iomem	*rstgenswrststatic2;
	u32 reg;

	rstgenswrststatic2 = ioremap(RSTGEN_SWRSTSTATIC2, 4);

	reg  = readl(rstgenswrststatic2);
#if defined(BOARD_TYPE_RC3L)
	reg |= RSTGEN_SWRSTSTATIC2__SYS_UHDR0__MASK;
#elif defined(BOARD_TYPE_RC4)
	reg |= RSTGEN_SWRSTSTATIC2__SYS_UHDR1__MASK;
#else
	reg |= RSTGEN_SWRSTSTATIC2__SYS_USDR__MASK;
#endif
	writel(reg, rstgenswrststatic2);
	udelay(5);
	reg  = readl(rstgenswrststatic2);
#if defined(BOARD_TYPE_RC3L)
	reg &= RSTGEN_SWRSTSTATIC2__SYS_UHDR0__INV_MASK;
#elif defined(BOARD_TYPE_RC4)
	reg &= RSTGEN_SWRSTSTATIC2__SYS_UHDR1__INV_MASK;
#else
	reg &= RSTGEN_SWRSTSTATIC2__SYS_USDR__INV_MASK;
#endif
	writel(reg, rstgenswrststatic2);
	udelay(5);

#ifdef Q6300A0   /* For 63XX A0 only, B0 doesn't have VAUX */
	reg = readl(dwc->regs_gpf + USDR_RST_OFF);
	reg &= (USDR_RST__CNTRL_VAUX_RESET_N__INV_MASK &
			USDR_RST__CNTRL_VCC_RESET_N__INV_MASK);
	writel(reg, dwc->regs_gpf + USDR_RST_OFF);
	udelay(5);
	reg = readl(dwc->regs_gpf + USDR_RST_OFF);
	reg |= (USDR_RST__CNTRL_VAUX_RESET_N__MASK |
			USDR_RST__CNTRL_VCC_RESET_N__MASK);
	writel(reg, dwc->regs_gpf + USDR_RST_OFF);
	udelay(5);
#else   /* For B0 */
	reg = readl(dwc->regs_gpf + USDR_RST_OFF);
	reg &= USDR_RST__CNTRL_VCC_RESET_N__INV_MASK;
	writel(reg, dwc->regs_gpf + USDR_RST_OFF);
	udelay(5);
	reg = readl(dwc->regs_gpf + USDR_RST_OFF);
	reg |= USDR_RST__CNTRL_VCC_RESET_N__MASK;
	writel(reg, dwc->regs_gpf + USDR_RST_OFF);
	udelay(5);
#endif

	if (usbd_ss == 1) {
		phy_reset_ss(dwc, 1);
		phy_reset_ss(dwc, 0);
	}
	phy_reset_hs(dwc, 1);
	phy_reset_hs(dwc, 0);

	iounmap(rstgenswrststatic2);
}

static int dwc3_probe(struct platform_device *pdev)
{
	struct device		*dev = &pdev->dev;
	struct dwc3_platform_data *pdata = dev_get_platdata(dev);
	struct device_node	*node = dev->of_node;
	struct resource		*res;
	struct dwc3		*dwc;

	int			ret;

	void __iomem		*regs, *regs_gpf;
	void			*mem;
	int			i;

	/** S/W workaround based on 63xx Errata 1.14,
	 *  USB 3.2 Gen 1x1 SuperSpeed (5 Gbit/s) not supported in 14L & 14P packages
	 */
	if ((usb3) && (!DISABLE_SUPERSPEED))
		disable_superspeed = 0;
	else
		disable_superspeed = 1;

	mem = devm_kzalloc(dev, sizeof(*dwc) + DWC3_ALIGN_MASK, GFP_KERNEL);
	if (!mem) {
		dev_err(dev, "not enough memory\n");
		return -ENOMEM;
	}
	dwc = PTR_ALIGN(mem, DWC3_ALIGN_MASK + 1);
	dwc->mem = mem;
	dwc->dev = dev;

	res = platform_get_resource(pdev, IORESOURCE_IRQ, 0);
	if (!res) {
		dev_err(dev, "missing IRQ\n");
		return -ENODEV;
	}
	dwc->xhci_resources[1].start = res->start;
	dwc->xhci_resources[1].end = res->end;
	dwc->xhci_resources[1].flags = res->flags;
	dwc->xhci_resources[1].name = res->name;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	if (!res) {
		dev_err(dev, "missing memory resource\n");
		return -ENODEV;
	}

	if ((u32)res->start == USDRGP_BASE)
		usbd_ss = 1;
	else
		usbd_ss = 0;

	regs_gpf = devm_ioremap_resource(dev, res);
	if (IS_ERR(regs_gpf))
		return PTR_ERR(regs_gpf);

	dwc->regs_gpf	= regs_gpf;
	dwc->regs_size_gpf	= resource_size(res);

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(dev, "missing memory resource\n");
		return -ENODEV;
	}

	if (node) {
#if LINUX_VERSION_CODE > KERNEL_VERSION(4,3,0)
		dwc->maximum_speed = usb_get_maximum_speed(dev);
#else
		dwc->maximum_speed = of_usb_get_maximum_speed(node);
#endif

		dwc->needs_fifo_resize = of_property_read_bool(node, "tx-fifo-resize");
#if LINUX_VERSION_CODE > KERNEL_VERSION(4,3,0)
		dwc->dr_mode = usb_get_dr_mode(dev);
#else
		dwc->dr_mode = of_usb_get_dr_mode(node);
#endif
	} else if (pdata) {
		dwc->maximum_speed = pdata->maximum_speed;

		dwc->needs_fifo_resize = pdata->tx_fifo_resize;
		dwc->dr_mode = pdata->dr_mode;
	}

	/* default to superspeed if no maximum_speed passed */
	if (dwc->maximum_speed == USB_SPEED_UNKNOWN) {
		if (disable_superspeed)
			dwc->maximum_speed = USB_SPEED_HIGH;
		else
			dwc->maximum_speed = USB_SPEED_SUPER;
	}

	ret = dwc3_core_get_phy(dwc);
	if (ret)
		return ret;

	dwc->xhci_resources[0].start = res->start;
	dwc->xhci_resources[0].end = dwc->xhci_resources[0].start +
					DWC3_XHCI_REGS_END;
	dwc->xhci_resources[0].flags = res->flags;
	dwc->xhci_resources[0].name = res->name;

	/*
	 * Request memory region but exclude xHCI regs,
	 * since it will be requested by the xhci-plat driver.
	 */
	regs = devm_ioremap_resource(dev, res);
	if (IS_ERR(regs))
		return PTR_ERR(regs);

	spin_lock_init(&dwc->lock);
	platform_set_drvdata(pdev, dwc);

	dwc->regs	= (void __iomem *)((uint_port)regs + DWC3_GLOBALS_REGS_START);
	dwc->regs_size	= resource_size(res);

	dev->dma_mask	= dev->parent->dma_mask;
	dev->dma_parms	= dev->parent->dma_parms;
	dma_set_coherent_mask(dev, dev->parent->coherent_dma_mask);

	pm_runtime_enable(dev);
	pm_runtime_get_sync(dev);
	pm_runtime_forbid(dev);

	usbd_hw_reset(dwc);

	dwc3_cache_hwparams(dwc);

	ret = dwc3_alloc_event_buffers(dwc, DWC3_EVENT_BUFFERS_SIZE);
	if (ret) {
		dev_err(dwc->dev, "failed to allocate event buffers\n");
		ret = -ENOMEM;
		goto err0;
	}

	if (IS_ENABLED(CONFIG_USB_DWC3_HOST))
		dwc->dr_mode = USB_DR_MODE_HOST;
	else if (IS_ENABLED(CONFIG_USB_DWC3_GADGET))
		dwc->dr_mode = USB_DR_MODE_PERIPHERAL;

	if (dwc->dr_mode == USB_DR_MODE_UNKNOWN)
		dwc->dr_mode = USB_DR_MODE_OTG;

	ret = dwc3_core_init(dwc);
	if (ret) {
		dev_err(dev, "failed to initialize core\n");
		goto err0;
	}

	/* for low-power share data */
	dwc->lp_last_active_trb_index
		= ioremap_nocache(USBD_LP_LAST_ACTIVE_TRB_INDEX, DWC3_ENDPOINTS_NUM*4);
	dwc->lp_last_active_trb_dma_addr_lo
		= ioremap_nocache(USBD_LP_LAST_ACTIVE_TRB_ADDR_LO, DWC3_ENDPOINTS_NUM*4);
	/* first store TRB address for ep_index=2 (PRN_RECV) */
	writel((USBD_TRB_POOL_START + (USBD_TRB_POOL_SIZE_EP*2)), (void *)((uint_port)dwc->lp_last_active_trb_dma_addr_lo + (2*4)));
	dwc->lp_last_active_trb_dma_addr_hi
		= ioremap_nocache(USBD_LP_LAST_ACTIVE_TRB_ADDR_HI, DWC3_ENDPOINTS_NUM*4);
	dwc->lp_last_active_buf_dma_addr_lo
		= ioremap_nocache(USBD_LP_LAST_ACTIVE_BUF_ADDR_LO, DWC3_ENDPOINTS_NUM*4);
	/* first store buffer address for ep_index=2 (PRN_RECV) */
	dwc->lp_last_active_buf_dma_addr_hi
		= ioremap_nocache(USBD_LP_LAST_ACTIVE_BUF_ADDR_HI, DWC3_ENDPOINTS_NUM*4);
	dwc->lp_transfer_received_size
		= ioremap_nocache(USBD_LP_TRANSFER_RECEIVED_SIZE, DWC3_ENDPOINTS_NUM*4);

	usb_phy_set_suspend(dwc->usb2_phy, 0);
	if (usbd_ss)
		usb_phy_set_suspend(dwc->usb3_phy, 0);

	ret = phy_power_on(dwc->usb2_generic_phy);
	if (ret < 0)
		goto err1;

	if (usbd_ss) {
		ret = phy_power_on(dwc->usb3_generic_phy);
		if (ret < 0)
			goto err_usb2phy_power;
	}

	ret = dwc3_event_buffers_setup(dwc);
	if (ret) {
		dev_err(dwc->dev, "failed to setup event buffers\n");
		goto err_usb3phy_power;
	}

	ret = dwc3_core_init_mode(dwc);
	if (ret)
		goto err2;

	ret = dwc3_debugfs_init(dwc);
	if (ret) {
		dev_err(dev, "failed to initialize debugfs\n");
		goto err3;
	}

	pm_runtime_allow(dev);

	/* Clear backup print buffer for low-power-mode use */
	dwc->lp_backup_buf = ioremap_nocache(USBD_PRN_RECV_BUF, 1024);
	*(u32 *)(uint_port)dwc->lp_backup_buf = 0xBE01BE02;
	*(u32 *)((uint_port)dwc->lp_backup_buf + 508) = 0xBE03BE04;
	*(u32 *)((uint_port)dwc->lp_backup_buf + 1020) = 0xBE05BE06;
	for (i = 0; i < DWC3_ENDPOINTS_NUM; i++)
		dwc->lp_last_active_buf_addr[i] = 0;
	dwc->lp_back = 0;

	return 0;

err3:
	dwc3_core_exit_mode(dwc);

err2:
	dwc3_event_buffers_cleanup(dwc);

err_usb3phy_power:
	phy_power_off(dwc->usb3_generic_phy);

err_usb2phy_power:
	phy_power_off(dwc->usb2_generic_phy);

err1:
	usb_phy_set_suspend(dwc->usb2_phy, 1);
	usb_phy_set_suspend(dwc->usb3_phy, 1);
	dwc3_core_exit(dwc);

err0:
	dwc3_free_event_buffers(dwc);

	return ret;
}

static int dwc3_remove(struct platform_device *pdev)
{
	struct dwc3	*dwc = platform_get_drvdata(pdev);

	usb_phy_set_suspend(dwc->usb2_phy, 1);
	usb_phy_set_suspend(dwc->usb3_phy, 1);
	phy_power_off(dwc->usb2_generic_phy);
	phy_power_off(dwc->usb3_generic_phy);

	pm_runtime_put_sync(&pdev->dev);
	pm_runtime_disable(&pdev->dev);

	dwc3_debugfs_exit(dwc);
	dwc3_core_exit_mode(dwc);
	dwc3_event_buffers_cleanup(dwc);
	dwc3_free_event_buffers(dwc);
	dwc3_core_exit(dwc);

	iounmap(dwc->lp_last_active_trb_index);
	iounmap(dwc->lp_last_active_trb_dma_addr_lo);
	iounmap(dwc->lp_last_active_trb_dma_addr_hi);
	iounmap(dwc->lp_last_active_buf_dma_addr_lo);
	iounmap(dwc->lp_last_active_buf_dma_addr_hi);
	iounmap(dwc->lp_transfer_received_size);

	return 0;
}

#ifdef CONFIG_PM_SLEEP
static int dwc3_prepare(struct device *dev)
{
	struct dwc3	*dwc = dev_get_drvdata(dev);
	unsigned long	flags;

	spin_lock_irqsave(&dwc->lock, flags);

	switch (dwc->dr_mode) {
	case USB_DR_MODE_PERIPHERAL:
	case USB_DR_MODE_OTG:
		/* FALLTHROUGH */
	case USB_DR_MODE_HOST:
	default:
		break;
	}

	spin_unlock_irqrestore(&dwc->lock, flags);

	return 0;
}

static void dwc3_complete(struct device *dev)
{
	struct dwc3	*dwc = dev_get_drvdata(dev);
	unsigned long	flags;

	spin_lock_irqsave(&dwc->lock, flags);

	switch (dwc->dr_mode) {
	case USB_DR_MODE_PERIPHERAL:
	case USB_DR_MODE_OTG:
		/* FALLTHROUGH */
	case USB_DR_MODE_HOST:
	default:
		break;
	}

	spin_unlock_irqrestore(&dwc->lock, flags);
}

static int dwc3_suspend(struct device *dev)
{
	struct dwc3	*dwc = dev_get_drvdata(dev);
	unsigned long	flags;
	volatile u8	__iomem	*connected;   // QTracker 2544
	u32 i;

	connected = ioremap_nocache(USBD_LP_SW_CONNECT_STATUS, 4);   // QTracker 2544
	writel(dwc->gadget.connected, connected);   // QTracker 2544
	iounmap(connected);   // QTracker 2544

	/* Reset unused PHYs */
	switch (dwc->gadget.speed) {
	case USB_SPEED_UNKNOWN:
		if (usbd_ss == 1)
			phy_reset_ss(dwc, 1);
		phy_reset_hs(dwc, 1);
		break;
	case USB_SPEED_SUPER:
		phy_reset_hs(dwc, 1);
		break;
	case USB_SPEED_HIGH:
	case USB_SPEED_FULL:
	case USB_SPEED_LOW:
		if (usbd_ss == 1)
			phy_reset_ss(dwc, 1);
		break;
	default:
		break;
	}

	spin_lock_irqsave(&dwc->lock, flags);

	switch (dwc->dr_mode) {
	case USB_DR_MODE_PERIPHERAL:
	case USB_DR_MODE_OTG:
		dwc3_gadget_suspend(dwc);
		/* FALLTHROUGH */
	case USB_DR_MODE_HOST:
	default:
		/* do nothing */
		break;
	}

	dwc->gctl = dwc3_readl(dwc->regs, DWC3_GCTL);
	spin_unlock_irqrestore(&dwc->lock, flags);
	/* Clear backup print buffer for low-power-mode use */
	for (i = 0; i < 1024; i+=4)
		*(u32 *)((uint_port)dwc->lp_backup_buf + i) = 0;
	*(u32 *)(uint_port)dwc->lp_backup_buf = 0xBE01BE02;
	*(u32 *)((uint_port)dwc->lp_backup_buf + 508) = 0xBE03BE04;
	*(u32 *)((uint_port)dwc->lp_backup_buf + 1020) = 0xBE05BE06;
	/* if a new transfer was queued, clear the first packet area */
	if (dwc->lp_last_active_buf_addr[2])
		for (i = 0; i < 1024; i+=4)
			*(u32 *)(dwc->lp_last_active_buf_addr[2] + i) = 0;
	dwc->lp_back = 1;
	return 0;
}

static int dwc3_resume(struct device *dev)
{
	struct dwc3	*dwc = dev_get_drvdata(dev);
	unsigned long	flags;
	volatile u8	__iomem	*evt;
	u32 resume_evt;
	u32 i;

	/* copy the 1st print packet data received during LP from backup
	   share memory to the real print buffer */
	if ((*(u32 *)(uint_port)dwc->lp_backup_buf != 0xBE01BE02) ||
		(*(u32 *)((uint_port)dwc->lp_backup_buf + 508) != 0xBE03BE04) ||
		(*(u32 *)((uint_port)dwc->lp_backup_buf + 1020) != 0xBE05BE06)) {
		for (i = 0; i < 1024; i+=4)
			*(u32 *)(dwc->lp_last_active_buf_addr[2] + i) = 
					*(u32 *)((uint_port)dwc->lp_backup_buf + i);
		*(u32 *)(uint_port)dwc->lp_backup_buf = 0xBE01BE02;
		*(u32 *)((uint_port)dwc->lp_backup_buf + 508) = 0xBE03BE04;
		*(u32 *)((uint_port)dwc->lp_backup_buf + 1020) = 0xBE05BE06;
		dwc->lp_last_active_buf_addr[2] = 0;
	}
	dwc->lp_back = 0;

	evt = ioremap_nocache(USBD_LP_SHARE_RESUME_EVT, 4);
	resume_evt = readl(evt);
	switch(resume_evt) {
	case USBD_LP_RESUME_EVT_DISCONNECT:
	case USBD_LP_RESUME_EVT_CONNECT:
		if (usbd_ss == 1) {
			phy_reset_ss(dwc, 1);
			phy_reset_ss(dwc, 0);
		}
		phy_reset_hs(dwc, 1);
		phy_reset_hs(dwc, 0);
		break;
	default:
		if (dwc->gadget.speed == USB_SPEED_UNKNOWN) {
			if (usbd_ss == 1) {
				phy_reset_ss(dwc, 1);
				phy_reset_ss(dwc, 0);
			}
			phy_reset_hs(dwc, 1);
			phy_reset_hs(dwc, 0);
		}
		break;
	}

	spin_lock_irqsave(&dwc->lock, flags);

	dwc3_writel(dwc->regs, DWC3_GCTL, dwc->gctl);

	switch (dwc->dr_mode) {
	case USB_DR_MODE_PERIPHERAL:
	case USB_DR_MODE_OTG:
		dwc3_gadget_resume(dwc, resume_evt);
		/* FALLTHROUGH */
	case USB_DR_MODE_HOST:
	default:
		/* do nothing */
		break;
	}

	spin_unlock_irqrestore(&dwc->lock, flags);

	writel(0, evt);
	iounmap(evt);

	pm_runtime_disable(dev);
	pm_runtime_set_active(dev);
	pm_runtime_enable(dev);

	return 0;
}

static const struct dev_pm_ops dwc3_dev_pm_ops = {
	.prepare	= dwc3_prepare,
	.complete	= dwc3_complete,

	SET_SYSTEM_SLEEP_PM_OPS(dwc3_suspend, dwc3_resume)
};

#define DWC3_PM_OPS	&(dwc3_dev_pm_ops)
#else
#define DWC3_PM_OPS	NULL
#endif

#ifdef CONFIG_OF
static const struct of_device_id of_dwc3_match[] = {
	{
		.compatible = "snps,dwc3"
	},
	{
		.compatible = "synopsys,dwc3"
	},
#ifdef CONFIG_SOC_QUASAR6300   /* for kernel 4.4 */
	{
		.compatible = "qbit,quasar-usbd"
	},
#elif defined(BOARD_TYPE_EP4) || defined(BOARD_TYPE_EP3L)
	{
		.compatible = "qbit,quasar-usdr",
	},
#elif defined(BOARD_TYPE_RC3L)
	{
		.compatible = "qbit,quasar-uhdr0",
	},
#elif defined(BOARD_TYPE_RC4)
	{
		.compatible = "qbit,quasar-uhdr1",
	},
#endif
	{ },
};
MODULE_DEVICE_TABLE(of, of_dwc3_match);
#endif

static struct platform_driver dwc3_driver = {
	.probe		= dwc3_probe,
	.remove		= dwc3_remove,
	.driver		= {
		.name	= "dwc3",
		.of_match_table	= of_match_ptr(of_dwc3_match),
		.pm	= DWC3_PM_OPS,
	},
};

module_platform_driver(dwc3_driver);

MODULE_ALIAS("platform:dwc3");
MODULE_AUTHOR("Felipe Balbi <balbi@ti.com>");
MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("DesignWare USB3 DRD Controller Driver");
