/*
 * This file contains definitions for ioctls and device numbers for the peripherals
 * and drivers used for Linux running on Quatro processors
 *
 * Copyright (c) 2016, QBit Semiconductor LTD.
 * 
 *  Copyright (c) 2014, 2015, The Linux Foundation.
 *  All rights reserved.
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2021/12/16 14:40:02 $
//  $Change: 56289 $
//
// =========================================================

#ifndef QIOCTL_H
#define QIOCTL_H 1

/* structure used to pass info for reg get/set and dsp pc get/set dsp pc get/set
*/
struct q_regio {
	unsigned reg;		/* register to set/get, use actual physical address */
	unsigned val;		/* val to set or val gotten */
};

/* structure used to pass co-processor (DSP/TCC/SBE/etc.) completion info
*  back to a user application
*/
struct q_doneinfo {
	unsigned msg;			/* return message/value      	       */
	unsigned detail;		/* detail for message      		 */
	unsigned long long endtime;	/* finish time absolute in HRT microseconds */
	unsigned long long cycles;	/* cycle count where appropriate   	 */
};

/* Generic get co-processor done info
 *
 * struct q_doneinfo doneinfo;
 * int    rc;
 *
 * rc = ioctl(fs, QGETDONE, &doneinfo); 
 */
#define QGETDONE			0x9D00

/* generic messages in .msg member of doneinfo
*/
#define QMSG_DONE		0xff000000
#define QMSG_MSG		0x01000000
#define QMSG_ABORT		0x10000000

/* Generic wait for co-processor complete
 *
 * rc = ioctl(fs, QWAITDONE, &doneinfo); 
 */
#define QWAITDONE		0x9D01

/* Generic enable irq (if disabled) */
#define QENABLEIRQ		0x9D02

/*
 * Generic reset co-processor
 */
#define QRESET  		0x9D03

/* Generic read/write a single 32 bit register
 *
 * struct q_regio regio;
 * int rc;
 *
 * regio.reg = REGADDRESS
 * regio.val = val
 *
 * rc = ioctl(fs, QSETREG, &regio); 
 */
#define QSETREG			0x9A01
#define QGETREG			0x9A02

/* Multiple register write, for efficiency
*/
#define QSETREGS		0x9A03
#define QGETREGS		0x9A04

/* ---------------------------------------------------------------------
 * GPDMA driver
 */
#define QGPDMA_MAJOR		208

#define QVIRT2PHYS		0x9E00
#define QPHYS2VIRT		0x9E01

struct q_gpdma_xfer {
	void*	virt_addr;
	void*	phys_addr;
	unsigned bytes;
};

/* ---------------------------------------------------------------------
 * DSP Driver
 */
#define QDSP_MAJOR		209

/* -- done messages from dsp to arm
*/
#define QDSP_HALTED		QMSG_DONE	/* dsp has halted */

#define QDSP_MSG_ABT		QMSG_ABORT	/* dsp has aborted (trap) */
#define QDSP_MSG		QMSG_MSG	/* dsp has message, detail in detail */
#define QDSP_MSG_REG		0x10000000	/* dsp has a "print register" message */
#define QDSP_MSG_STR		0x11000000	/* dsp has a "print string" message */

/* DSP Start Message: start the DSP running at address.  Passes the 
 * text and data segment params as well, assumes the DSP driver has
 * access (cached) to those segments already.  The text and data
 * addresses are the *user-mode* addresses as known by the application
 * and are used only as a lookup tag, not as an address
 */
struct q_dsp_start {
	unsigned start_addr;
	void*    text_addr;
	unsigned text_size;
	void*    data_addr;
	unsigned data_size;
	unsigned nparams;
	unsigned params[32];
};

/*****  DSP ioctls **************
*/
/* set the DSP's code base register to a physical address after copying the code
 * and data as specified in a q_dsp_start structure
 */
#define QDSPLOADMODULE		0x9A10

/* get the DSP instruction at PC in regio.reg  (offset from base code) into regio.val
 */
#define QDSPGETINST		0x9A11

/* disable IRQ for DSP Halt
 */
#define QDSPDISABLEINTHALT		0x9A13

/* enable IRQ for DSP Halt
 */
#define QDSPENABLEINTHALT		0x9A14

/* set a register with value, uses struct q_regio* as parameter
 * set reg to actual physical address of register
 */
#define QDSPSETREG		QSETREG

/* get a value from a register, uses struct qscan_regio* as parameter
 */
#define QDSPGETREG		QGETREG

/* start the DSP with params (see struct q_dsp_start above) after loaded
 * (or typically setting from cache) text and data segments as per the
 * q_dsp_start struct
 *
 * int ret;
 * struct q_dsp_start strt;
 * <fill in strt>
 * ret = ioctl(fd, QDSPEXECUTE, &strt);
 */
#define QDSPEXECUTE		0x9A03

/* get the least recent mailbox entry (done info)
 */
#define QDSPGETDONEINFO		0x9A04

/* ---------------------------------------------------------------------
 * Scanner Driver 
 */
#define QSCAN_MAJOR		210

#define QSCAN_MAX_NUM_XRES	32
#define QSCAN_MAX_NUM_YRES	32

#define QSCAN_MAX_NUM_CHANS	3

/* Do one scan-op (see qscan.h for list of opcodes). The scan device driver delegates the
 * operation to the underlying actual scanner device drive
 */
#define QSCANSSCANOP		0x9B00

/* Get number of bands available to read
*/
#define QSCANGSCANBANDSPEND	0x9B01

/* set a register with value, uses struct q_regio* as parameter
 * set reg to actual physical address of register
 */
#define QSCANSETREG		QSETREG

/* get a value from a register, uses struct q_regio* as parameter
 */
#define QSCANGETREG		QGETREG

/* get the physical address of the shading (gain/offset) table
 * allocated in the scan driver
 */
#define QSCANGSHADETABLE	0x9B20

/* Size of the shade table allocation (e.g for appropriate mmapping)
*/
#define QSCAN_SHADE_TABLE_SIZE	(65536+16384)

/* ---------------------------------------------------------------------
 * Printer band-dma Driver
 */
#define QPRINT_MAJOR	211

/* Submit a band to a page
*/
#define QPRINT_SUBMIT_BAND		0x9B90
#define QPRINT_RETRIEVE_BAND		0x9B91
#define QPRINT_BLOCKING_RETRIEVE_BAND	0x9B92

#define QPRINT_BIG_ENDIAN	0x0001
#define QPRINT_BIT_SWAPPED	0x0002
#define QPRINT_JBIG_COMPRESSED	0x0004

/* used to submit a band
*/
struct q_print_bandspec {
	unsigned int  plane;		/* plane band belongs to */
	unsigned int  width;		/* in pixels */
	unsigned int  height;		/* in lines */
	unsigned int  depth;		/* in bits per pixel */
	unsigned int  stride;		/* bytes per one line in band, including padding */
	unsigned int  y;		/* y coordinate from top, in line, the top of this band represents */
	int           flags;		/* byte/bit swapped, compressed, etc. */
	unsigned int  compr_size;   	/* bytes of compressed data */
	void*         compr_pa;		/* physical address of compressed band data */
	unsigned int  raster_size; 	/* bytes of data in pixel data */
	void*         raster_pa;	/* physical address of band pixel data */
};

/* Print the page (start print DMA). If arg is non-zero it is
 * a pointer to a pagespec (see below), otherwise band dimension
 * are used internally to figure page dimension
 */
#define QPRINT_PRINT_PAGE	0x9B94

struct q_print_pagespec {
	unsigned int  width;		/* page width in pixels */
	unsigned int  height;		/* page height in lines */
	unsigned int  lmarg;		/* left margin width in pixels */
	unsigned int  tmarg;		/* top margin height in lines */
	unsigned int  resx;		/* x resolution */
	unsigned int  resy;		/* y resolution */
};

/* Set Calibration Offsets
*/
#define QPRINT_SETCALOFFSETS 0x9B95

/* used to set calibration offset (margins) per plane
*/
struct q_print_caloffsets {
	unsigned int  cyanTopOffset;
	unsigned int  magentaTopOffset;
	unsigned int  yellowTopOffset;
	unsigned int  blackTopOffset;
	unsigned int  cyanLeftOffset;
	unsigned int  magentaLeftOffset;
	unsigned int  yellowLeftOffset;
	unsigned int  blackLeftOffset;
};

/* Setup Video PLL
*/
#define QPRINT_SETVIDEOPLL 0x9B96

/* Setup Video PLL
*/
#define QPRINT_SETVIDEOPLL 0x9B96

struct q_print_pllvalues {
    unsigned int  pllclock;     /* value to use for video PLL clock setting */
    unsigned int  pllfilter;    /* value to use for video PLL filter setting */
};

/* ---------------------------------------------------------------------
 * DMA and BAND Memory Driver 
 */
#define QMEM_MAJOR		212

/* structure to pass back allocation data
*/
struct q_mem_alloc {
	unsigned int req_size;	/* requested size in bytes      */
	unsigned int ret_size;	/* actual alloced size in bytes */
	void*        ret_addr;	/* returned physical address    */
	int          cached;	/* should the allocated memory be cacheable */
				/* (for high mem, ignored on in, high 32 of phys on out */
							
	char strFunc[32];
	int numLine;
};

/* Allocate a block of physically contiguous (kernel) memory.  Pass in the size
 * in bytes and get back a physical address pointer
 *
 * struct q_mem_alloc val = { size, 0, NULL, 0 };
 * ioctl(fd, QMEMALLOC, (void*)&val);
 */
#define QMEMALLOC		0x9C21

/* Allocate a block of physically contiguous (kernel) memory from 
 * high-memory (in systems with PAE), uses the same q_mem_alloc
 * structure, but the cached member is used to return the high
 * 32 bits of physical address, and ignored on input
 */
#define QMEMALLOCHIGH		0x9C81

/* Allocates from coherent "DMA" uncached area
*/
#define QMEMALLOCDMA		0x9C22

/* Allocates from on chip scratch pad area (will cause it to be mapped the first alloc)
*/
#define QMEMALLOCSPAD		0x9C23
#define QMEMALLOCJBSP		0x9C24

/* frees the mem.  Pass in the physical address returned from QMEMALLOC
*/
#define QMEMFREE		0x9C26

#define QMEMGETMAXALLOCSIZE	0x9C30
#define QMEMGETMAXALLOCS	0x9C31
#define QMEMGETCURALLOCS	0x9C32

#define QMEMGETAVAILABLE	0x9C34
#define QMEMGETSPADAVAILABLE	0x9C35
#define QMEMGETTOTALAVAILABLE	0x9C36
#define QMEMGETLOWAVAILABLE     0x9C37
#define QMEMGETHIGHAVAILABLE  0x9C38

/* for cached buffers that have to be seen by hardware */

#define QMEMFLUSH       	0x9C40
#define QMEMINVALIDATE  	0x9C41
#define QMEMFLUSHINVALIDATE     0x9C42

/* ---------------------------------------------------------------------
 * TCC driver
 */
#define QTCC_MAJOR		213

#define QTCCSETIM    0x9F00
#define QTCCSETOM    0x9F01
#define QTCCSETSTATE 0x9F02
#define QTCCGETSTATE 0x9F03

struct q_tcc_set {
    int data0;
    int data1;
    int data2;
    int data3;
    int cb;
};

/* ---------------------------------------------------------------------
 * SBE driver
 */
#define QSBE_MAJOR		214

/* ---------------------------------------------------------------------
 * JBIG driver
 */
#define QJBIG_MAJOR		215

/* ---------------------------------------------------------------------
 * FILTER h/w block
 */
#define QFIR_MAJOR		217

#define QFIRSETIM    0x9F02
#define QFIRSETOM    0x9F03

struct q_fir_set {
    int data0;
    int data1;
    int data2;
    int data3;
    int mon;
    int cb;
};

/* ---------------------------------------------------------------------
 * SCALER h/w block
 */
#define QSCAL_MAJOR		218

/* ---------------------------------------------------------------------
 * SCREENER h/w block
 */
#define QSCRN_MAJOR		219

#define QSCRNSETIM    0x9F04
#define QSCRNSETOM    0x9F05

struct q_scrn_set {
    int data0;
    int data1;
    int data2;
    int data3;
    int cb;
};

/* ---------------------------------------------------------------------
 * Scrubber DMA driver
 */
#define QSDMA_MAJOR			223

#define SCRUB_PHYS			2230
#define SCRUB_VIRT			2231

struct q_sdma_xfer {
	unsigned int ch;
	void*	virt_addr;
	void*	phys_addr;
	unsigned int bytes;
};

/* ---------------------------------------------------------------------
 * Cortex M3 driver
 */
#define QM3_MAJOR		227

#define QM3_MAX_ARGS	30
/* ---------------------------------------------------------------------
 * ADC driver
 */
 #define QADC_MAJOR	242
 
 /* ---------------------------------------------------------------------
 * LCD driver
 */
 #define QLCD_MAJOR	243

/* parameters to Cortex M3 for command and back for response/notification
*/
struct q_m3_io {
	unsigned int  msg;					/* message, command, response, etc. code */
	int           args;					/* count of valid parms/results in args  */
	unsigned int  val[QM3_MAX_ARGS];	/* values/parameters from/to Cortex M3   */
};

/* m3 ioctls 
*
* struct g_m3_io cmd;
* int rc;
*
* cmd.msg = cmdtosend
* cmd.args = 0
* rc = ioctl(hm3, QM3_SEND_COMMAND, &cmd);
* rc = ioctl(hm3, QM3_GET_RESPONSE, &cmd);
*/
#define QM3_SEND_COMMAND	0x9A04
#define QM3_GET_RESPONSE	0x9A05
#define QM3_GET_NOTIFICATION	0x9A06

/* check/wait for a response (like select)
*
* int block = 0; // 1 for blocking wait
* int rc;
*
* rc = ioctl(hm3, QM3_WAIT_RESPONSE,     &block);
* rc = ioctl(hm3, QM3_WAIT_NOTIFICATION, &block);
*/
#define QM3_WAIT_RESPONSE	0x9A08
#define QM3_WAIT_NOTIFICATION	0x9A09

/* Start/stop the M3 executing
*
* int start = 1; // 1 = run, use 0 for stop
* int rc = ioctl(hm3, QM3_EXECUTE, &start);
*/
#define QM3_EXECUTE		0x9A03

/* ---------------------------------------------------------------------
 * Security Protocol Accelerator driver
 */
#define QSECUR_MAJOR	228

/* ------------------------------------------------------------------
 * System bus access driver
 */
#define	QSYSBUS_MAJOR	231

#define QSETREG8		0x9C01
#define QGETREG8		0x9C02

struct q_regio8 {
	unsigned reg;	/* register to set/get, use actual physical address */
	unsigned char val;	/* val to set or val gotten */
};

#define QSYSBUS_MAP		0x9C03

struct q_sysbus_map {
	unsigned int phys;	/* physical address to map */
	unsigned int size;	/* size in bytes to map */
};

/* ------------------------------------------------------------------
 * Vopu driver (appears as /dev/fb0, i.e.: c 29 0
 */
#define QVOPU2_GET_TOUCH	0x9D04
#define QVOPU2_WAIT_TOUCH	0x9D05

/* ------------------------------------------------------------------
 * Mali (GPU) driver
 */
#define QMALI_MAJOR   232
/* ------------------------------------------------------------------
 * JPEG driver
 */
#define QJPEG_MAJOR   233

/* ------------------------------------------------------------------
 * XSPI driver
 */
#define QXSPI_MAJOR   234

/* ------------------------------------------------------------------
 * GPIO driver 
 */
#define QGPIO_MAJOR   236

/*          
**          To specify an SPI register set. 
*/
typedef enum
{
    Spi0RegSet0 = 0,///< SPI0-cs0
                    // SPI0ERR,SPI0CLK0,SPI0CFG0,SPI0XCFG,SPI0CMD0,SPI0LIT0,SPI0STAT0,SPI0DCTL,SPI0DADDR,SPI0RADDR
    Spi0RegSet1,    ///< SPI0-cs1
                    // SPI0ERR,SPI0CLK1,SPI0CFG1,SPI0XCFG,SPI0CMD1,SPI0LIT1,SPI0STAT1,SPI0DCTL,SPI0DADDR,SPI0RADDR
    Spi0RegSet2,    ///< SPI0-cs2
                    // SPI0ERR,SPI0CLK2,SPI0CFG2,SPI0XCFG,SPI0CMD2,SPI0LIT2,SPI0STAT2,SPI0DCTL,SPI0DADDR,SPI0RADDR
    Spi1RegSet0,    ///< SPI1-cs0
                    // SPI1ERR,SPI1CLK0,SPI1CFG0,SPI1XCFG,SPI1CMD0,SPI1LIT0,SPI1STAT0,SPI1DCTL,SPI1DADDR,SPI1RADDR     
    Spi1RegSet1,    ///< SPI1-cs1
                    // SPI1ERR,SPI1CLK1,SPI1CFG1,SPI1XCFG,SPI1CMD1,SPI1LIT1,SPI1STAT1,SPI1DCTL,SPI1DADDR,SPI1RADDR 
    Spi1RegSet2,    ///< SPI1-cs2
                    // SPI1ERR,SPI1CLK2,SPI1CFG2,SPI1XCFG,SPI1CMD2,SPI1LIT2,SPI1STAT2,SPI1DCTL,SPI1DADDR,SPI1RADDR 
    Spi2RegSet0,    ///< SPI2-cs0
                    // SPI2ERR,SPI2CLK0,SPI2CFG0,SPI2XCFG,SPI2CMD0,SPI2LIT0,SPI2STAT0,SPI2DCTL,SPI2DADDR,SPI2RADDR     
    Spi2RegSet1,    ///< SPI2-cs1
                    // SPI2ERR,SPI2CLK1,SPI2CFG1,SPI2XCFG,SPI2CMD1,SPI2LIT1,SPI2STAT1,SPI2DCTL,SPI2DADDR,SPI2RADDR 
    Spi2RegSet2,    ///< SPI2-cs2
                    // SPI2ERR,SPI2CLK2,SPI2CFG2,SPI2XCFG,SPI2CMD2,SPI2LIT2,SPI2STAT2,SPI2DCTL,SPI2DADDR,SPI2RADDR 
    Spi3RegSet0,    ///< SPI3-cs0
                    // SPI3ERR,SPI3CLK0,SPI3CFG0,SPI3XCFG,SPI3CMD0,SPI3LIT0,SPI3STAT0,SPI3DCTL,SPI3DADDR,SPI3RADDR     
    Spi3RegSet1,    ///< SPI3-cs1
                    // SPI3ERR,SPI3CLK1,SPI3CFG1,SPI3XCFG,SPI3CMD1,SPI3LIT1,SPI3STAT1,SPI3DCTL,SPI3DADDR,SPI3RADDR 
    Spi3RegSet2,    ///< SPI3-cs2
                    // SPI3ERR,SPI3CLK2,SPI3CFG2,SPI3XCFG,SPI3CMD2,SPI3LIT2,SPI3STAT2,SPI3DCTL,SPI3DADDR,SPI3RADDR 
    Spi4RegSet0,    ///< SPI4-cs0
                    // SPI4ERR,SPI4CLK0,SPI4CFG0,SPI4XCFG,SPI4CMD0,SPI4LIT0,SPI4STAT0,SPI4DCTL,SPI4DADDR,SPI4RADDR     
    Spi4RegSet1,    ///< SPI4-cs1
                    // SPI4ERR,SPI4CLK1,SPI4CFG1,SPI4XCFG,SPI4CMD1,SPI4LIT1,SPI4STAT1,SPI4DCTL,SPI4DADDR,SPI4RADDR 
    Spi4RegSet2,    ///< SPI4-cs2
                    // SPI4ERR,SPI4CLK2,SPI4CFG2,SPI4XCFG,SPI4CMD2,SPI4LIT2,SPI4STAT2,SPI4DCTL,SPI4DADDR,SPI4RADDR 
    Spi5RegSet0,    ///< SPI5-cs0
                    // SPI5ERR,SPI5CLK0,SPI5CFG0,SPI5XCFG,SPI5CMD0,SPI5LIT0,SPI5STAT0,SPI5DCTL,SPI5DADDR,SPI5RADDR     
    Spi5RegSet1,    ///< SPI5-cs1
                    // SPI5ERR,SPI5CLK1,SPI5CFG1,SPI5XCFG,SPI5CMD1,SPI5LIT1,SPI5STAT1,SPI5DCTL,SPI5DADDR,SPI5RADDR 
    Spi5RegSet2,    ///< SPI5-cs2
                    // SPI5ERR,SPI5CLK2,SPI5CFG2,SPI5XCFG,SPI5CMD2,SPI5LIT2,SPI5STAT2,SPI5DCTL,SPI5DADDR,SPI5RADDR 
    Spi6RegSet0,    ///< SPI6-cs0
                    // SPI6ERR,SPI6CLK0,SPI6CFG0,SPI6XCFG,SPI6CMD0,SPI6LIT0,SPI6STAT0,SPI6DCTL,SPI6DADDR,SPI6RADDR     
    Spi6RegSet1,    ///< SPI6-cs1
                    // SPI6ERR,SPI6CLK1,SPI6CFG1,SPI6XCFG,SPI6CMD1,SPI6LIT1,SPI6STAT1,SPI6DCTL,SPI6DADDR,SPI6RADDR 
    Spi6RegSet2,    ///< SPI6-cs2
                    // SPI6ERR,SPI6CLK2,SPI6CFG2,SPI6XCFG,SPI6CMD2,SPI6LIT2,SPI6STAT2,SPI6DCTL,SPI6DADDR,SPI6RADDR 
    Spi7RegSet0,    ///< SPI7-cs0
                    // SPI7ERR,SPI7CLK0,SPI7CFG0,SPI7XCFG,SPI7CMD0,SPI7LIT0,SPI7STAT0,SPI7DCTL,SPI7DADDR,SPI7RADDR     
    Spi7RegSet1,    ///< SPI7-cs1
                    // SPI7ERR,SPI7CLK1,SPI7CFG1,SPI7XCFG,SPI7CMD1,SPI7LIT1,SPI7STAT1,SPI7DCTL,SPI7DADDR,SPI7RADDR 
    Spi7RegSet2,    ///< SPI7-cs2
                    // SPI7ERR,SPI7CLK2,SPI7CFG2,SPI7XCFG,SPI7CMD2,SPI7LIT2,SPI7STAT2,SPI7DCTL,SPI7DADDR,SPI7RADDR 
    Spi8RegSet0,    ///< SPI8-cs0
                    // SPI8ERR,SPI8CLK0,SPI8CFG0,SPI8XCFG,SPI8CMD0,SPI8LIT0,SPI8STAT0,SPI8DCTL,SPI8DADDR,SPI8RADDR     
    Spi8RegSet1,    ///< SPI8-cs1
                    // SPI8ERR,SPI8CLK1,SPI8CFG1,SPI8XCFG,SPI8CMD1,SPI8LIT1,SPI8STAT1,SPI8DCTL,SPI8DADDR,SPI8RADDR 
    Spi8RegSet2     ///< SPI8-cs2
                    // SPI8ERR,SPI8CLK2,SPI8CFG2,SPI8XCFG,SPI8CMD2,SPI8LIT2,SPI8STAT2,SPI8DCTL,SPI8DADDR,SPI8RADDR                                                                                                                         
} eSPI_REGSETx;

/*          
**          SPI0CFGx and SPI1CFGx Register configuration.
*/
typedef struct
{
    unsigned int E;   // Enable
    unsigned int CP;  // SPI Clock Select 0 or 1
    unsigned int CSP; // Chip Select Setup/Host Time
    unsigned int EN;  // Endian Mode
    unsigned int SM;  // SPI Special Mode
    unsigned int CMD; // ROM/PROG Command
    unsigned int LD;  // SPI Data Input Latch Disable
    unsigned int LAT; // Read Latency
    unsigned int CL;  // Command Length
    unsigned int AL;  // Address Length
                
} sSPI_CFGx;

/*          
**          SPI0CLKx and SPI1CLKx Clock Register configuration.
*/
typedef struct
{
    unsigned int M;   // SPI Clock Mode
    unsigned int P;   // SPI Clock Prescaler
    unsigned int CH;  // SPI Clock High-phase
    unsigned int CL;  // SPI Clock Low-phase
                
} sSPI_CLKx;

/*          
**          SPI0XCFG and SPI1XCFG EXTENDED Register configuration.
*/
typedef struct
{
    unsigned int BT;  // Boot ROM type
    unsigned int W;   // Program Data Width
    unsigned int PTO; // Program Timout
                
} sSPI_XCFGx;

/*
**                      L I T E R A L    M O D E
*/

/*          
**          The Literal Mode SPI0LITx and SPI1LITx register command word 
*/
typedef struct
{
    unsigned int S;   // Command Start
    unsigned int OP;  // Operation to be executed
    unsigned int DL;  // Data Length (DL+1 is written from WD or is captured)
    unsigned int ARG; // Argument (command plus address)
    unsigned int WD;  // Write Data 
} sSPI_LMWORD;

/*          
**          Literal Mode Operation Types
**
**  Note: There are 4 literal mode operation SPI0LITx.OP types.
**  Also, LAT is specified in the corresponding SPI Configuration Register SPI0CFGx
*/
typedef enum
{ 
    SendCmdOnly = 0,        // ARG is asserted only, no read

    SendCmdandDataOnly,     // ARG + WD are asserted, no read

    GetReadDataOnly,        // ARG and WD are not asserted, wait LAT 
                            // cycles, capture read data in SPI0STATx.STAT
    SendCmdOnly_GetReadData // ARG is asserted but no WD, wait LAT cycles,
                            // capture read data in SPI0STATx.STAT
} eSPI_LMOP;

/*          
**          A Literal Mode (LM) function consists of the operation
**          type and CMD and DATA which are specific to the SPI device
*           connected.   
*/
typedef struct
{
    unsigned int    CMD;        /*  The command that is sent.   */
    unsigned int    DATA;       /*  returned data               */
    eSPI_LMOP       OP;         /*  eSPI_LMOP                   */
    eSPI_REGSETx    reg_set;    /*  The spi register set to use */
} sSPI_LMCMD;

/*
**                      D I R E C T    M O D E
*/

/*          
**          The DM SPI0CMDx and SPI1CMDx register command word 
*/
typedef struct
{
    unsigned int S;   // Command Start
    unsigned int OP;  // Operation to be executed
    unsigned int WB;  // Number of write bits (WB+1 will be written)
    unsigned int NI;  // No Done Interrupt
    unsigned int DA;  // Data Alignment
    unsigned int RB;  // Number of read bits (RB+1 will be read)

} sSPI_DMWORD;

/*          
**          Direct Mode Operation Types
**
**  Note: There are 4 possible Direct Mode operation types.
**  Also, LAT is specified in the corresponding SPI Configuration Register SPI0CFGx
*/
typedef enum
{
    BusyCheck = 0,          // Checks if device is busy by asserting the select
                            // and polling the data line, updates SPI0STATx.DB.
                            // Only applicable if SPI0CFGx.NB = 0.
                            // This feature is supported by selected SPI devices only.
    WriteImmediate,         // Write <WB+1> bits only
    WaitBeforeRead,         // Wait SPI0CFGx.LAT cycles, read <RB+1> data bits
    WriteThenWaitBeforeRead // Write <WB+1> bits, wait SPI0CFGx.LAT cycles, read <RB+1> data bits
                             
} eSPI_DMOP;

/*          
**          The DMA SPI0DCTL and SPI1DCTL control register word 
*/
typedef struct
{
    unsigned int PC;  // Page Count (for ROM and PROG functions)
    unsigned int RB;  // Separate DMA addresses (read and write)
    unsigned int NW;  // No DMA for write data (1 for SPI0CMDx.OP=3)
    unsigned int NA;  // No Automatic FIFO reload or flush.
    unsigned int I;   // 1 to invalidate the DMA buffer.
    unsigned int N;   // Number of valid entries in DMA buffer.

} sSPI_DMACTL;

/*          A Direct Mode function consists of the operation type, along
**          with pointers to buffers and buffer sizes. 
*/
typedef struct
{
    eSPI_REGSETx    reg_set;    /*  The spi register set to use     */
    eSPI_DMOP       OP;         /*  eSPI_DMOP                       */
    int             wrCount;    /*  Number of 32 bit words to write */
    int             rdCount;    /*  Number of 32 bit words to read  */
    unsigned int    *pWrBuf;    /*  Pointer to a write buffer       */
    unsigned int    *pRdBuf;    /*  Pointer to a read buffer        */

} sSPI_DMCMD;


#define QXSPI_WAIT_INTERRUPT    0x9A05 
#define QXSPI_LITERALMODE_CMD   0x9A07 
#define QXSPI_DIRECTMODE_CMD    0x9A08 

/* ------------------------------------------------------------------
 * OTP driver 
 */
#define QOTP_MAJOR   237

/* parameters to otp for read/write command
*/
struct q_otp_io {
    unsigned int  byteOffset;   /* byte index */
    unsigned int  bitindex;     /* bit index for write */
    unsigned int  count;        /* byte counts for read, should be in range between 1~4 */
    unsigned int  val;          /* read/write value  */
};

#define QOTP_READ               0x9A02 
#define QOTP_WRITE_BIT          0x9A03 

/* ------------------------------------------------------------------
 * Hardware semaphore driver 
 */
#define QSEM_MAJOR    238

/** To specify hardware semaphore index 
*/
typedef enum
{
    HWSEM_0 = 0,        ///< hardware semaphore index 0
    HWSEM_1,            ///< hardware semaphore index 1
    HWSEM_2,            ///< hardware semaphore index 2
    HWSEM_3,            ///< hardware semaphore index 3
    HWSEM_4,            ///< hardware semaphore index 4
    HWSEM_5,            ///< hardware semaphore index 5
    HWSEM_6,            ///< hardware semaphore index 6
    HWSEM_7,            ///< hardware semaphore index 7
    HWSEM_8,            ///< hardware semaphore index 8
    HWSEM_9,            ///< hardware semaphore index 9
    HWSEM_10,           ///< hardware semaphore index 10
    HWSEM_11,           ///< hardware semaphore index 11
    HWSEM_12,           ///< hardware semaphore index 12
    HWSEM_13,           ///< hardware semaphore index 13
    HWSEM_14,           ///< hardware semaphore index 14
    HWSEM_15,           ///< hardware semaphore index 15
    HWSEM_16,           ///< hardware semaphore index 16
    HWSEM_17,           ///< hardware semaphore index 17
    HWSEM_18,           ///< hardware semaphore index 18
    HWSEM_19,           ///< hardware semaphore index 19
    HWSEM_MAX
} HWSEM_INDEX;

/* parameters for semaphore get/release command
*/
struct q_sem_io {
    HWSEM_INDEX     sem;        /* semaphore index */
    unsigned int    secs;       /* seconds to wait for get */
    unsigned int    usecs;      /* microseconds to wait for get */
    unsigned int    val;        /* return value  */
};

#define QHWSEM_GET              0x9A22 
#define QHWSEM_RELEASE          0x9A23 

#endif
