/*
 * A driver for TCC found on Quatro processors
 *
 * Quasar TCC kernel driver
 *
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/01/27 08:19:40 $
//  $Change: 57409 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#define QTCC_DEVNAME	"qtcc"

#define MAX_MAILBOX 64

#ifdef TCC0TCC_CST_OFF
#define TCC_CST_OFF                TCC0TCC_CST_OFF
#define TCC_CST__CLR_INT__MASK     TCC0TCC_CST__CLR_INT__MASK
#define TCC_DBG_OFF                TCC0TCC_DBG_OFF
#define TCC_DBG__RESP_ERR__MASK    TCC0TCC_DBG__RESP_ERR__MASK
#endif

struct tcc_quasar {
	int			ref;
	int			minor;
	struct cdev		cdev;
	volatile u8	__iomem	*regs;
	unsigned long		ioreg_start;
	unsigned long		ioreg_end;
	int			irq;

	wait_queue_head_t	mbq;	/* wait-queue for dma-doneinterrupt events */
	struct q_doneinfo	doneinfo[MAX_MAILBOX];
	int			di_head, di_tail;
	spinlock_t		lock;
	int			is_tcclut_rst;
};

static int qtcc_open(struct inode* inode, struct file* filp)
{
	struct tcc_quasar *tcc;
    volatile u8 __iomem *reg;
    volatile u8 __iomem *sts;
	
	tcc = container_of(inode->i_cdev, struct tcc_quasar, cdev);
	if(tcc->ref > 0)
		return -EBUSY;
	tcc->ref++;
	tcc->minor = iminor(inode);
	filp->private_data = tcc;
	tcc->di_head = tcc->di_tail = 0;
	
// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
    reg = ioremap(IPM1CLKDISCTRL, 4);
    sts = ioremap(IPM1CLKDISSTAT, 4);
    writel(readl(reg) & IPM1CLKDISCTRL__IPM1_TCC0__INV_MASK, reg);
    while (readl(sts) & IPM1CLKDISSTAT__IPM1_TCC0__MASK) {;}
	iounmap(reg);
	iounmap(sts);
	reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
    writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_TCC0__INV_MASK, reg);
    printk(KERN_INFO "QTCC %d Reset\n", tcc->minor);
    iounmap(reg);
#endif
	return 0;
}

static int qtcc_release(struct inode* inode, struct file* filp)
{
	struct tcc_quasar *tcc;
	
	tcc = container_of(inode->i_cdev, struct tcc_quasar, cdev);
	if(tcc->ref <= 0)
		return -EFAULT;
	tcc->ref--;
	filp->private_data = NULL;
	return 0;
}

static ssize_t qtcc_read(struct file* filp, char __user *buffer, size_t length, loff_t* offset)
{
	struct tcc_quasar *tcc;	
	
	tcc = (struct tcc_quasar*)filp->private_data;
	return -EINVAL;
}

static ssize_t qtcc_write(struct file* filp, const char __user *buffer, size_t length, loff_t* offset)
{
	struct tcc_quasar *tcc;	
	
	tcc = (struct tcc_quasar*)filp->private_data;
	return -EINVAL;
}

static unsigned int qtcc_poll(struct file *filp, poll_table *wait)
{
	struct tcc_quasar* tcc;	
	unsigned int mask = 0;
	unsigned long flags;
	
	tcc = (struct tcc_quasar*)filp->private_data;

	poll_wait(filp, &tcc->mbq, wait);
	spin_lock_irqsave(&tcc->lock, flags);
	if(tcc->di_head != tcc->di_tail)
		mask |= POLLIN | POLLRDNORM; /* readable */
	spin_unlock_irqrestore(&tcc->lock, flags);
	return mask;
}

static int qtcc_setimom(struct tcc_quasar *tcc, struct q_tcc_set *qset, int im)
{
	volatile u8 __iomem *reg;

	if (im)
	{
		reg = ioremap(SB_TCC_IM0, 12);
		writel(qset->data0, reg);
		writel(qset->data1, reg + 0x4);
		writel(qset->data2, reg + 0x8);
		iounmap(reg);
		reg = ioremap(TCC_IMON0_CTRL, 12);
		writel(qset->cb + (qset->data0 ? 0x10 : 0x0), reg);
		writel(qset->cb + (qset->data1 ? 0x10 : 0x0), reg + 0x4);
		writel(qset->cb + (qset->data2 ? 0x10 : 0x0), reg + 0x8);
		iounmap(reg);
	}
	else
	{
		reg = ioremap(SB_TCC_OM0, 16);
		writel(qset->data0, reg);
		writel(qset->data1, reg + 0x4);
		writel(qset->data2, reg + 0x8);
		writel(qset->data3, reg + 0xc);
		iounmap(reg);
		reg = ioremap(TCC_OMON0_CTRL, 16);
		writel(qset->cb + (qset->data0 ? 0x10 : 0x0), reg);
		writel(qset->cb + (qset->data1 ? 0x10 : 0x0), reg + 0x4);
		writel(qset->cb + (qset->data2 ? 0x10 : 0x0), reg + 0x8);
		writel(qset->cb + (qset->data3 ? 0x10 : 0x0), reg + 0xc);
		iounmap(reg);
	}
	return 0;
}

static long qtcc_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct tcc_quasar *tcc;	
	struct q_tcc_set qset;
	unsigned long flags;
	int ret = 0;
	
	tcc = (struct tcc_quasar*)filp->private_data;
	
	switch(cmd)
	{
	case QGETDONE:
		if(tcc->di_head == tcc->di_tail)
			return -EAGAIN;
		copy_to_user((void*)arg,
			(void*)&tcc->doneinfo[tcc->di_tail], sizeof(struct q_doneinfo));
		spin_lock_irqsave(&tcc->lock, flags);
		tcc->di_tail++;
		if(tcc->di_tail >= MAX_MAILBOX)
			tcc->di_tail = 0;
		spin_unlock_irqrestore(&tcc->lock, flags);
		break;
	case QWAITDONE:
		while(tcc->di_head == tcc->di_tail) {
			/* TODO - check for dma in progress? */
			if (wait_event_interruptible(tcc->mbq,
				(tcc->di_head != tcc->di_tail)))
				return -ERESTARTSYS;
		}
		if(tcc->di_head != tcc->di_tail) {
			copy_to_user((void*)arg,
				(void*)&tcc->doneinfo[tcc->di_tail], sizeof(struct q_doneinfo));
			spin_lock_irqsave(&tcc->lock, flags);
			tcc->di_tail++;
			if(tcc->di_tail >= MAX_MAILBOX)
				tcc->di_tail = 0;
			spin_unlock_irqrestore(&tcc->lock, flags);
			ret = 0;
		}
		else {
			/* TODO - check for dma in progress? */
			ret = -EAGAIN;
		}
		break;
	case QTCCSETIM:
	case QTCCSETOM:
		if(copy_from_user((void*)&qset,
			(void*)arg, sizeof(struct q_tcc_set)))
			return -EINVAL;
		ret = qtcc_setimom(tcc, &qset, cmd == QTCCSETIM);
		break;
	case QTCCGETSTATE:
		copy_to_user((void*)arg,
			(void*)&tcc->is_tcclut_rst, sizeof(int));
		// printk("qtcc - reset state: %d\n", tcc->is_tcclut_rst);
		break;
	case QTCCSETSTATE:
		if(copy_from_user((void*)&tcc->is_tcclut_rst,
			(void*)arg, sizeof(int)))
			return -EINVAL;
		// printk("qtcc - reset state: %d\n", tcc->is_tcclut_rst);
		break;
	default:
		printk("qtcc - bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static int qtcc_mmap(struct file* filp, struct vm_area_struct* vma)
{
	struct tcc_quasar *tcc;	
	int length, ret = 0;
	
	tcc = (struct tcc_quasar*)filp->private_data;	
	if(! tcc) return -ENODEV;

	/* !! mark pages as uncached for now !! */
	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	length = vma->vm_end - vma->vm_start;
	if(length > (tcc->ioreg_end - tcc->ioreg_start)) {
		printk(KERN_WARNING
				"TCC VMA length %d truncated to io length %d\n",
				length,
				(int)(tcc->ioreg_end - tcc->ioreg_start));
		length = (tcc->ioreg_end - tcc->ioreg_start); 
	}
	ret = remap_pfn_range(
				vma,
				vma->vm_start,
				tcc->ioreg_start >> PAGE_SHIFT,
				length,
				vma->vm_page_prot
			);
	return ret;
}

static irqreturn_t quasar_tcc_interrupt(int irq, void *dev_id)
{
	struct tcc_quasar *tcc = (struct tcc_quasar *)dev_id; 
	struct timeval tv;
	int ret;
	unsigned val;
	unsigned long flags;
	
// TBD: separate for 6600?  
#if defined(Q6300B0) || defined(Q6600)
    // check int status
    val = readl(tcc->regs + TCC_DBG_OFF);
    if (val & TCC_DBG__RESP_ERR__MASK)
    {
        printk("\nQTCC DMA response error: 0x%08x\n", val);
    }
#endif
	/* clear interrupt bit in TCC (or we keep getting them!)
	*/
	val = readl(tcc->regs + TCC_CST_OFF);
	writel(val | TCC_CST__CLR_INT__MASK, tcc->regs + TCC_CST_OFF);
	
	spin_lock_irqsave(&tcc->lock, flags);
	tcc->doneinfo[tcc->di_head].msg     = QMSG_DONE;
	tcc->doneinfo[tcc->di_head].detail  = 0;
	do_gettimeofday(&tv);
	tcc->doneinfo[tcc->di_head].endtime = (unsigned long long)tv.tv_sec * 1000000 + (unsigned long long)tv.tv_usec;
	tcc->doneinfo[tcc->di_head].cycles  = 0;
	tcc->di_head++;
	if(tcc->di_head >= MAX_MAILBOX)
		tcc->di_head = 0;
	spin_unlock_irqrestore(&tcc->lock, flags);

	/* wakeup tasks waiting on dma done
	*/
	wake_up_interruptible(&tcc->mbq);

	ret = IRQ_HANDLED;
	return ret;
}

static struct file_operations quasar_tcc_ops = {
	.owner		= THIS_MODULE,
	.open		= qtcc_open,
	.release	= qtcc_release,
	.read		= qtcc_read,
	.write		= qtcc_write,
	.poll		= qtcc_poll,
	.mmap		= qtcc_mmap,
	.unlocked_ioctl	= qtcc_ioctl,
	.compat_ioctl	= qtcc_ioctl 
};

static int __init quasar_tcc_probe(struct platform_device *pdev)
{
	struct tcc_quasar *tcc;
	struct resource	*regs;
	dev_t  tccn;
	int    ret;
	static int ntccs = 0;

	tcc = kzalloc(sizeof(struct tcc_quasar), GFP_KERNEL);
	if (!tcc) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	cdev_init(&tcc->cdev, &quasar_tcc_ops);
	tccn = MKDEV(QTCC_MAJOR, ntccs);
	ret = cdev_add(&tcc->cdev, tccn, 1);
	if (ret) {
		dev_dbg(&pdev->dev, "could not create char dev %d\n", ntccs);
		goto out_err;
	}
	regs = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!regs) {
		dev_dbg(&pdev->dev, "no mmio reg resource defined\n");
		ret = -ENXIO;
		goto out_rerr;
	}
	tcc->ioreg_start = regs->start;
	tcc->ioreg_end   = regs->end;

	tcc->regs = ioremap(regs->start, regs->end - regs->start + 1);
	if(!tcc->regs) {
		ret = -ENOMEM;
		dev_dbg(&pdev->dev, "could not map reg I/O memory\n");
		goto out_ioerr;
	}

	tcc->irq = platform_get_irq(pdev, 0);
	if (tcc->irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		ret = -ENXIO;
		goto out_ioerr;
	}
	ret = request_irq(tcc->irq, quasar_tcc_interrupt,
		0, "tccdma", tcc);
	if (ret) {
		dev_dbg(&pdev->dev, "could not request irq %d\n", tcc->irq);
		tcc->irq = -1;
		goto out_ioerr;
	}
	spin_lock_init(&tcc->lock);
	init_waitqueue_head(&tcc->mbq);

	ntccs++;
	platform_set_drvdata(pdev, tcc);
	printk("QTCC - mapped at %px, irq %d\n", tcc->regs, tcc->irq);
	device_init_wakeup(&pdev->dev, 1);
	return 0;
out_ioerr:
	iounmap(tcc->regs);
out_rerr:
	cdev_del(&tcc->cdev);		
out_err:
	kfree(tcc);
	return ret;
}

static int __exit quasar_tcc_remove(struct platform_device *pdev)
{
	struct tcc_quasar *tcc = platform_get_drvdata(pdev);

	if(tcc->irq > 0)
		free_irq(tcc->irq, tcc);
	iounmap(tcc->regs);
	cdev_del(&tcc->cdev);		
	kfree(tcc);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static int quasar_tcc_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct tcc_quasar *tcc = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM1CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (tcc->minor)
		{
		default:
			break;
		case 0:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM1_TCC0__MASK, reg);
			writel(readl(ctl) | IPM1CLKDISCTRL__IPM1_TCC0__MASK, ctl);
			while ((readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_TCC0__MASK) == 0x0) {;}
			break;
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
	}
#endif
	return 0;
}

static int quasar_tcc_resume(struct platform_device *pdev)
{
	struct tcc_quasar *tcc = platform_get_drvdata(pdev);

	tcc->is_tcclut_rst = 1;

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM1CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (tcc->minor)
		{
		default:
			break;
		case 0:
			writel(readl(ctl) & IPM1CLKDISCTRL__IPM1_TCC0__INV_MASK, ctl);
			while (readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_TCC0__MASK) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_TCC0__INV_MASK, reg);
			break;
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
	}
#endif
	return 0;
}

MODULE_ALIAS("platform:quasar-tcc");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-tcc" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver quasar_tcc_driver_ops = {
	.probe		= quasar_tcc_probe,
	.remove		= quasar_tcc_remove,
	.suspend	= quasar_tcc_suspend,
	.resume		= quasar_tcc_resume,
	.driver		= {
		.name	= "quasar-tcc",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init quasar_tcc_init(void)
{
	int ret;
	
	ret = platform_driver_register(&quasar_tcc_driver_ops);
	return ret;
}
module_init(quasar_tcc_init);

static void __exit quasar_tcc_exit(void)
{
	platform_driver_unregister(&quasar_tcc_driver_ops);
}
module_exit(quasar_tcc_exit);

MODULE_DESCRIPTION("Quasar TCC driver");
MODULE_LICENSE("Dual BSD/GPL");

