/*
 * qsem.c --  A driver for the QBit hardware semaphores
 *
 * Copyright (c) 2014, 2015, The Linux Foundation. All rights reserved.
 *
 * Copyright (c) 2010-2011 Abc Incorporated
 *
 * Copyright (c) 2016, QBit Semiconductor LTD.
 * Quasar semaphore kernel driver
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
// =========================================================
//
//  $DateTime: 2022/01/19 08:48:00 $
//  $Change: 57110 $
//
// =========================================================

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/icu.h>
#include <quasar/qioctl.h>

#define SEM_DEVNAME	        "qsem"

#define MAX_IOREGS  		1

struct quasar_io_region {
	u32						start;
	u32						end;
	volatile u8	__iomem		*base;
};

struct quasar_sem {
	struct cdev				cdev;
	struct device			*frdev;

	struct quasar_io_region	ioregs[MAX_IOREGS];
	int						nioregs;

	int						refs;				/* reference count */
	int						minor;				/* device minor number */
};

static volatile u8 __iomem* sem_addr(struct quasar_sem* qsem, u32 reg)
{
	volatile u8 __iomem* addr = 0;
 	int      i;

	/* check for existing mapping */
	for (i = 0; i < qsem->nioregs; i++) {
		if(reg >= qsem->ioregs[i].start && reg < qsem->ioregs[i].end) {
			return(qsem->ioregs[i].base + (reg - qsem->ioregs[i].start));
		}
	}
	printk(KERN_WARNING SEM_DEVNAME "sem_addr - could not map reg %08x\n", reg);
	return addr;
}

#if 0
static int sem_readl(struct quasar_sem* qsem, u32 reg, unsigned* val)
{
	volatile u32 rv = 0;
	int ret = 0;
	volatile u8 __iomem* addr = sem_addr(qsem, reg);
	
	if(addr == NULL)
		return -1;
	rv = readl(addr);
	/*printk("readl %08X [%08X] -> %08X\n", reg, addr, rv);*/
	*val = (unsigned long)rv;
	return ret;
}
#endif

static int sem_writel(struct quasar_sem* qsem, struct q_sem_io *semio, u32 val)
{
	volatile u8 __iomem* addr = sem_addr(qsem, semio->sem);
	
	semio->val = -1;
	if(addr == NULL)
		return -1;
	/* printk("writel %08X [%08X] <= %08x\n", reg, addr, val);*/
	writel(val, addr);
	semio->val = 0;
	return 0;
}

static int sem_get(struct quasar_sem* qsem, struct q_sem_io *semio)
{
    unsigned long timeout, msec;
	volatile u8 __iomem* addr = sem_addr(qsem, semio->sem);
	
    semio->val = -1;
	if(addr == NULL)
		return -1;
		
	/* poll until semaphore acquired or timeout */
	msec = semio->secs * 1000 + semio->usecs / 1000;
	timeout = jiffies + msecs_to_jiffies(msec);

	while (!time_after(jiffies, timeout)) 
	{
		if (0 == readl(addr))
		{
			// semaphore get
			semio->val = 0;
			return 0;
		}

		schedule_timeout(1);
	}
	
	// timeout
	semio->val = 1;   
	return 0;
}

static int qsem_open(struct inode* inode, struct file* filp)
{
	struct quasar_sem *qsem;
	int minor;
	qsem = container_of(inode->i_cdev, struct quasar_sem, cdev);
	minor = iminor(inode);
	qsem->refs++;
	filp->private_data = qsem;
	return 0;
}

static int qsem_release(struct inode* inode, struct file* filp)
{
	struct quasar_sem* qsem;
	
	qsem = (struct quasar_sem*)filp->private_data;
	if(qsem->refs <= 0)
		return -EBADF;
	qsem->refs--;
	filp->private_data = NULL;
	return 0;
}

static long qsem_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct quasar_sem* qsem;
	struct q_sem_io semio;
	int ret = 0;
	
	qsem = (struct quasar_sem*)filp->private_data;
	if(! qsem)
		return -ENODEV;

	switch(cmd)
	{
	case QHWSEM_RELEASE:
		ret = copy_from_user(&semio, (void*)arg, sizeof(struct q_sem_io));      
		ret = sem_writel(qsem, &semio, 0);
		break;
	case QHWSEM_GET:
		ret = copy_from_user(&semio, (void*)arg, sizeof(struct q_sem_io)); 
		ret = sem_get(qsem, &semio);
		ret = copy_to_user((void*)arg, &semio, sizeof(struct q_sem_io));
		break;
	default:
		printk(KERN_WARNING SEM_DEVNAME "Bad ioctl %d\n", cmd);
		ret = -EINVAL;
	}

	return ret;
}

static struct file_operations qsem_ops = {
	.owner		= THIS_MODULE,
	.open		= qsem_open, 
	.release	= qsem_release,
	.unlocked_ioctl	= qsem_ioctl,
	.compat_ioctl	= qsem_ioctl    
};

static int __init qsem_probe(struct platform_device *pdev)
{
	struct resource	*ioreg;
	struct quasar_sem *qsem;
    static int ngpios=0;
	dev_t  qsemn;
	int ret = 0;

	qsem = kzalloc(sizeof(struct quasar_sem), GFP_KERNEL);
	if(! qsem) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	qsem->frdev = &pdev->dev;
	qsem->nioregs = 0;
	
	do {
		ioreg = platform_get_resource(pdev, IORESOURCE_MEM, qsem->nioregs);
		if (ioreg) {
			qsem->ioregs[qsem->nioregs].start = ioreg->start;
			qsem->ioregs[qsem->nioregs].end = ioreg->end;
			qsem->ioregs[qsem->nioregs].base =
				ioremap(ioreg->start, ioreg->end - ioreg->start + 4);
            if (!qsem->ioregs[qsem->nioregs].base) {
				ret = -ENOMEM;
				dev_dbg(&pdev->dev, "could not map I/O memory %0d\n", qsem->nioregs);
				goto out_abort;
			}
			/*
			printk("qsem - mapped qsem %px %x to %x at %x\n", 
					qsem, ioreg->start, ioreg->end, qsem->ioregs[qsem->nioregs].base);
			*/
			qsem->nioregs++;
		}
	} while (ioreg && (qsem->nioregs < MAX_IOREGS));
		
	cdev_init(&qsem->cdev, &qsem_ops);
	qsemn = MKDEV(QSEM_MAJOR, ngpios);
	ret = cdev_add(&qsem->cdev, qsemn, 1);
	if (ret) {
		printk(KERN_WARNING "qsem - could not create char dev %d\n", ngpios);
		ret = -ENODEV;
		goto out_abort;
	}
	printk("qsem - adding char dev %d:%d\n", MAJOR(qsemn), MINOR(qsemn));
	qsem->minor = MINOR(qsemn);
	ngpios++;
	ret = 0;

	platform_set_drvdata(pdev, qsem);
	device_init_wakeup(&pdev->dev, 1);
	dev_info(&pdev->dev, "Quasar qbsem\n");
	return 0;

out_abort:
	while (qsem->nioregs > 0) {
		qsem->nioregs--;
		iounmap(qsem->ioregs[qsem->nioregs].base);
	}
	kfree(qsem);
	return ret;
}

static int __exit qsem_remove(struct platform_device *pdev)
{
	struct quasar_sem *qsem = platform_get_drvdata(pdev);

	cdev_del(&qsem->cdev);
	while (qsem->nioregs > 0) {
		qsem->nioregs--;
		iounmap(qsem->ioregs[qsem->nioregs].base);
	}
	kfree(qsem);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

MODULE_ALIAS("platform:qsem");

static const struct of_device_id  qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-sem" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver qsem_driver_ops = {
	.probe		= qsem_probe,
	.remove		= qsem_remove,
	.driver		= {
		.name	= "quasar-sem",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init qsem_init(void)
{
	int ret;
	ret = platform_driver_register(&qsem_driver_ops);
	return ret;
}
module_init(qsem_init);

static void __exit qsem_exit(void)
{
	platform_driver_unregister(&qsem_driver_ops);
}
module_exit(qsem_exit);

MODULE_DESCRIPTION("Quasar sem Driver");
MODULE_LICENSE("Dual BSD/GPL");

