/* Kernel driver for Scrubber DMA on Quatro processors
 *
 * Quasar QSDMA kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/01/19 08:48:00 $
//  $Change: 57110 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/version.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

/* this is the max ahead on ints rcvd. keep it kinda big
*/
#define MAX_MAILBOX 256

struct sdma_quasar {
	struct cdev		cdev;
	volatile u8		__iomem	*regs;
	unsigned		ioreg_start;
	unsigned		ioreg_end;
	int			irq;
	struct device*		dev;
	wait_queue_head_t	mbq;	/* wait-queue for dma-doneinterrupt events */
	struct q_doneinfo	doneinfo[MAX_MAILBOX];
	int			di_head, di_tail;
	spinlock_t		lock;
};

static inline int sdmaRead32(struct sdma_quasar* dma, u32 reg, volatile unsigned* val)
{
	volatile u32 rv = 0;
	int ret = 0;

	reg += dma->ioreg_start;
	if(reg >= dma->ioreg_start && reg <= dma->ioreg_end) {
		rv = readl(dma->regs + (reg - dma->ioreg_start));
	}
	else {
		printk(KERN_WARNING "qgdma ioctl getreg addr range error\n");
		ret = -EINVAL;
	}
	*val = (unsigned long)rv;
	return ret;
}

static int sdmaWrite32(struct sdma_quasar* dma, u32 reg, u32 val)
{
	reg += dma->ioreg_start;
	if(reg >= dma->ioreg_start && reg <= dma->ioreg_end)
		writel(val, dma->regs + (reg - dma->ioreg_start));
	else {
		printk(KERN_WARNING
	   		"qsdma ioctl setreg addr range error reg %08X not in %08X to %08X\n",
			reg, dma->ioreg_start, dma->ioreg_end);
		return -EINVAL;
	}
	return 0;
}

static int sdma_open(struct inode* inode, struct file* filp)
{
	struct sdma_quasar *dma;
	
	dma = container_of(inode->i_cdev, struct sdma_quasar, cdev);
	filp->private_data = dma;
	dma->di_head = dma->di_tail = 0;
	return 0;
}

static int sdma_release(struct inode* inode, struct file* filp)
{
	struct sdma_quasar *dma;
	
	dma = container_of(inode->i_cdev, struct sdma_quasar, cdev);
	filp->private_data = NULL;
	return 0;
}

static ssize_t sdma_read(struct file* filp, char __user *buffer, size_t length, loff_t* offset)
{
	struct sdma_quasar *dma;	
	
	dma = (struct sdma_quasar*)filp->private_data;
	return -EINVAL;
}

static ssize_t sdma_write(struct file* filp, const char __user *buffer, size_t length, loff_t* offset)
{
	struct sdma_quasar *dma;	
	
	dma = (struct sdma_quasar*)filp->private_data;
	return -EINVAL;
}

static unsigned int sdma_poll(struct file *filp, poll_table *wait)
{
	struct sdma_quasar* dma;	
	unsigned int mask = 0;
	unsigned long flags;
	
	dma = (struct sdma_quasar*)filp->private_data;

	poll_wait(filp, &dma->mbq, wait);
	spin_lock_irqsave(&dma->lock, flags);
	if(dma->di_head != dma->di_tail)
		mask |= POLLIN | POLLRDNORM; /* readable */
	spin_unlock_irqrestore(&dma->lock, flags);
	return mask;
}

static int sdma_xfer(struct sdma_quasar *dma, struct q_sdma_xfer *xfer, int tophys)
{
	struct page *page, **pages;
	int pagecnt, firstpage, lastpage;
	int pageno;
	int voff;
	int size;
	int nleft;
	int timeout;
	int result;
	dma_addr_t dma_d;
	u8 *physoff;
	u8 *vs;
	u32 i;    

	/* validate parameters */
	if(! xfer || ((! xfer->virt_addr) && (! xfer->phys_addr))) 
		return -EINVAL;
	if(xfer->bytes & QBSCRUBDMA_SZE0__SZE__INV_MASK) {
		printk("sdma xfer count 0x%08x must be cache-line aligned\n", xfer->bytes);
		return -EINVAL;
	}
	if((unsigned long)xfer->virt_addr & 0x3f) {
		printk("sdma xfer addr must be cache-line aligned\n");
		return -EINVAL;
	}
	if((unsigned long)xfer->phys_addr & 0x3f) {
		printk("sdma xfer addr must be cache-line aligned\n");
		return -EINVAL;
	}

	/* alloc a page array */
	firstpage = (unsigned long)xfer->virt_addr >> PAGE_SHIFT;
	lastpage  = ((unsigned long)xfer->virt_addr + xfer->bytes) >> PAGE_SHIFT;
	pagecnt = lastpage - firstpage + 1;

	pages = (struct page**)kmalloc(pagecnt * sizeof(struct page*), GFP_KERNEL);
	if(! pages)
		return -ENOMEM;

	/* lock mm for read */
	down_read(&current->mm->mmap_sem);

	/* Map the user I/O buffer. */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,19,0)
	result = get_user_pages(
				(unsigned long) xfer->virt_addr & PAGE_MASK,
				pagecnt,
				tophys ? 0 : 1,
				pages,
				NULL
				);
#else
	result = get_user_pages(
				current,
				current->mm,
				(unsigned long) xfer->virt_addr & PAGE_MASK,
				pagecnt,
				tophys ? 0 : 1,
				0,
				pages,
				NULL
				);
#endif

	up_read(&current->mm->mmap_sem);

	if(result < 0 || result < pagecnt) {
		printk(KERN_WARNING "get pages failed %d\n", result);
		kfree(pages);
		return result;
 	}
	physoff = (u8*)xfer->phys_addr;
	nleft = xfer->bytes;

	/* for each page in virt, xfer to the phys location */
	for(pageno = 0; pageno < pagecnt; pageno++) {
		page = pages[pageno];		
		size = PAGE_SIZE;

		/* if first page, reduce count by offset into page of va */
		if(pageno == 0) {
			voff = (unsigned long)xfer->virt_addr & ~PAGE_MASK;
			size -= voff;
		} else
			voff = 0;

		/* if last page, reduce count by bytes to right of end of xfer */
		if(pageno == (pagecnt - 1))
			size -=  PAGE_SIZE - (((unsigned long)xfer->virt_addr + xfer->bytes) & ~PAGE_MASK);

		if(size > nleft)
			size = nleft;

		vs = kmap(page);

		dma_d = dma_map_single(
					dma->dev,
					tophys ? __va((unsigned long)physoff) : (void*)vs,
					PAGE_SIZE, DMA_TO_DEVICE
				);

		if(size > 0) {
			int starthead;

			/* setup scrubbering */
			i = (QBSCRUBDMA_MOD0__CE__HW_DEFAULT << QBSCRUBDMA_MOD0__CE__SHIFT) +
				(QBSCRUBDMA_MOD0__PF__HW_DEFAULT << QBSCRUBDMA_MOD0__PF__SHIFT) +
				(QBSCRUBDMA_MOD0__BRST__HW_DEFAULT << QBSCRUBDMA_MOD0__BRST__SHIFT);
			sdmaWrite32(dma, QBSCRUBDMA_MOD0_OFF + (xfer->ch * 0x10), i);
			/* shift addr [35:6] to reg [31:0] */
			sdmaWrite32(dma, QBSCRUBDMA_BUF0_OFF + (xfer->ch * 0x10),
				((unsigned)dma_d + (tophys ? 0 : voff)) >> QBSCRUBDMA_SZE0__SZE__SHIFT);
			/* started by programming size, only [26:6] are valid */
			size &= QBSCRUBDMA_SZE0__SZE__MASK;
			sdmaWrite32(dma, QBSCRUBDMA_SZE0_OFF + (xfer->ch * 0x10), size);

			/* remember where the interrupt record is, we'll pull it ourselves
			*/
			starthead = dma->di_head;

			/* wait for done, should be like a microsecond or less */
			timeout = 0;
			while(dma->di_head == dma->di_tail) {
				if (wait_event_interruptible(dma->mbq, (dma->di_head != dma->di_tail)))
					return -ERESTARTSYS;
				if (timeout++ > 1000000)
					break;
			}
			if (dma->di_head == dma->di_tail) {
				printk("sdma didn't complete?\n");
			}
			/* pull the interrupt record */
			dma->di_head = starthead;
		}

		dma_unmap_single(dma->dev, dma_d, PAGE_SIZE, DMA_FROM_DEVICE);

		kunmap(page);

		physoff += size;
		nleft -= size;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,19,0)
		put_page(page);
#else  	        
		page_cache_release(page);
#endif
	}
	/* Clean up and return. */
	kfree(pages);
	return xfer->bytes - nleft;
}

static long sdma_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct sdma_quasar *dma;	
	struct q_sdma_xfer xfer;
	unsigned long flags;
	int ret = 0;
	
	dma = (struct sdma_quasar*)filp->private_data;
	
	switch(cmd)
	{
	case QGETDONE:
		if(dma->di_head == dma->di_tail)
			return -EAGAIN;
		ret = copy_to_user((void*)arg,
			(void*)&dma->doneinfo[dma->di_tail], sizeof(struct q_doneinfo));
		spin_lock_irqsave(&dma->lock, flags);
		dma->di_tail++;
		if(dma->di_tail >= MAX_MAILBOX)
			dma->di_tail = 0;
		spin_unlock_irqrestore(&dma->lock, flags);
		break;
	case QWAITDONE:
		while(dma->di_head == dma->di_tail) {
			if (wait_event_interruptible(dma->mbq,
				(dma->di_head != dma->di_tail)))
				return -ERESTARTSYS;
		}
		if(dma->di_head != dma->di_tail) {
			ret = copy_to_user((void*)arg, (void*)&dma->doneinfo[dma->di_tail], sizeof(struct q_doneinfo));
			spin_lock_irqsave(&dma->lock, flags);
			dma->di_tail++;
			if(dma->di_tail >= MAX_MAILBOX)
				dma->di_tail = 0;
			spin_unlock_irqrestore(&dma->lock, flags);
			ret = 0;
		}
		else
			ret = -EAGAIN;
		break;
	case SCRUB_PHYS:
	case SCRUB_VIRT:
		/* like copy_from_user or copy_to_user but uses sdma to 
		 * be sdram bus friendly
		 */
		if(copy_from_user((void*)&xfer,
			(void*)arg, sizeof(struct q_sdma_xfer)))
			return -EINVAL;
		if (cmd == SCRUB_PHYS)
			xfer.virt_addr = NULL;
		if (cmd == SCRUB_VIRT)
			xfer.phys_addr = NULL;
		ret = sdma_xfer(dma, &xfer, cmd == SCRUB_PHYS /*QVIRT2PHYS*/);
		break;
	default:
		printk(KERN_WARNING "sdma - bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static int sdma_mmap(struct file* filp, struct vm_area_struct* vma)
{
	struct sdma_quasar *dma;	
	int length, ret = 0;

	dma = (struct sdma_quasar*)filp->private_data;	
	if(! dma) return -ENODEV;

	/* !! mark pages as uncached for now !! */
	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	length = vma->vm_end - vma->vm_start;
	if(length > (dma->ioreg_end - dma->ioreg_start + 1)) {
		printk(KERN_WARNING "SDMA VMA length(%08x) truncated to io region(%08x to %08x)\n", length, dma->ioreg_start, dma->ioreg_end);
		length = (dma->ioreg_end - dma->ioreg_start); 
	}

	if((dma->ioreg_start >> PAGE_SHIFT) == vma->vm_pgoff) {
		ret = remap_pfn_range(
			vma,
			vma->vm_start,
			dma->ioreg_start >> PAGE_SHIFT,
			length,
			vma->vm_page_prot
			);
	}
	return ret;
}

static irqreturn_t quasar_sdma_interrupt(int irq, void *dev_id)
{
	struct sdma_quasar *dma = (struct sdma_quasar *)dev_id; 
	struct timeval tv;
	int ret;
    u32 i;
	unsigned long flags;

	dma->doneinfo[dma->di_head].msg     = QMSG_DONE;
	dma->doneinfo[dma->di_head].detail  = 0;
	do_gettimeofday(&tv);
	dma->doneinfo[dma->di_head].endtime = (unsigned long long)tv.tv_sec * 1000000 +
			(unsigned long long)tv.tv_usec;
	dma->doneinfo[dma->di_head].cycles  = 0;
	spin_lock_irqsave(&dma->lock, flags);
	dma->di_head++;
	if(dma->di_head >= MAX_MAILBOX)
		dma->di_head = 0;
	spin_unlock_irqrestore(&dma->lock, flags);
	ret = sdmaRead32(dma, QBSCRUBDMA_CHST_OFF, &i);
	sdmaWrite32(dma, QBSCRUBDMA_CHST_OFF, i);   /* clear DONE bits */
	/* wakeup tasks waiting on dma done
	*/
	wake_up_interruptible(&dma->mbq);

	ret = IRQ_HANDLED;
	return ret;
}

static struct file_operations quasar_sdma_ops = {
	.owner		= THIS_MODULE,
	.open		= sdma_open,
	.release	= sdma_release,
	.read		= sdma_read,
	.write		= sdma_write,
	.poll		= sdma_poll,
	.mmap		= sdma_mmap,
	.unlocked_ioctl	= sdma_ioctl,
	.compat_ioctl	= sdma_ioctl 
};

static int __init quasar_sdma_probe(struct platform_device *pdev)
{
	struct sdma_quasar *dma;
	struct resource	*regs;
	dev_t  dman;
	int    ret;
    u32    i;

	dma = kzalloc(sizeof(struct sdma_quasar), GFP_KERNEL);
	if (!dma) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	cdev_init(&dma->cdev, &quasar_sdma_ops);
	dman = MKDEV(QSDMA_MAJOR, 0 /*ndmas*/);
	ret = cdev_add(&dma->cdev, dman, 1);
	if (ret) {
		dev_dbg(&pdev->dev, "could not create sdma char dev %d\n", ret);
		goto out_err;
	}
	regs = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!regs) {
		dev_dbg(&pdev->dev, "no mmio reg resource defined\n");
		ret = -ENXIO;
		goto out_rerr;
	}
	dma->ioreg_start = regs->start;
	dma->ioreg_end   = regs->end;

	dma->regs = ioremap(dma->ioreg_start, dma->ioreg_end - dma->ioreg_start + 1);
	if(!dma->regs) {
		ret = -ENOMEM;
		dev_dbg(&pdev->dev, "could not map reg I/O memory\n");
		goto out_ioerr;
	}

	dma->irq = platform_get_irq(pdev, 0);
	if (dma->irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		ret = -ENXIO;
		goto out_ioerr;
	}
	spin_lock_init(&dma->lock);
	init_waitqueue_head(&dma->mbq);

	sdmaWrite32(dma, QBSCRUBDMA_CHST_OFF, QBSCRUBDMA_CHST__DNE__MASK);   /* clear done bits */
	ret = request_irq(dma->irq, quasar_sdma_interrupt,
		0, "sdma", dma);
	if (ret) {
		dev_dbg(&pdev->dev, "could not request irq %d\n", dma->irq);
		dma->irq = -1;
		goto out_ioerr;
	}
	ret = sdmaRead32(dma, QBSCRUBDMA_CST_OFF, &i);
	sdmaWrite32(dma, QBSCRUBDMA_CST_OFF, i);
	platform_set_drvdata(pdev, dma);
	printk("QSDMA - mapped at %px, irq %d\n", dma->regs, dma->irq);
	device_init_wakeup(&pdev->dev, 1);	
	dma->dev = &pdev->dev;
	return 0;
out_ioerr:
	iounmap(dma->regs);
out_rerr:
	cdev_del(&dma->cdev);		
out_err:
	kfree(dma);
	return ret;
}

static int __exit quasar_sdma_remove(struct platform_device *pdev)
{
	struct sdma_quasar *dma = platform_get_drvdata(pdev);

	if(dma->irq > 0)
		free_irq(dma->irq, dma);
	iounmap(dma->regs);
	cdev_del(&dma->cdev);		
	kfree(dma);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

MODULE_ALIAS("platform:quasar-sdma");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-sdma" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver quasar_sdma_driver_ops = {
	.probe		= quasar_sdma_probe,
	.remove		= quasar_sdma_remove,
	.driver		= {
		.name	= "quasar-sdma",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init quasar_sdma_init(void)
{
	int ret;

	ret = platform_driver_register(&quasar_sdma_driver_ops);
	return ret;
}
module_init(quasar_sdma_init);

static void __exit quasar_sdma_exit(void)
{
	platform_driver_unregister(&quasar_sdma_driver_ops);
}
module_exit(quasar_sdma_exit);

MODULE_DESCRIPTION(" Quasar Scrubber DMA driver");
MODULE_LICENSE("Dual BSD/GPL");

