/*
 * A driver for SCRN h/w block found on Quatro processors
 *
 * Quasar SCRN kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/01/19 08:48:00 $
//  $Change: 57110 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#define QSCRN_DEVNAME	"qscrn"

#define MAX_MAILBOX 64

/* The fireregs headers split the screen region into two separate areas
 * with "SCRN_IP_SPARE" starting the SCRN_IP region.  For simplicity
 * the kernel driver just maps the scrn region as one, so just be sure
 * to not use any _OFF in the IP region directly
 */
#if defined Q6600 || defined Q6300
	#define QSCRN_SCRN_IP_INTERRUPT_ENABLE_OFF (SCRN0SCRN_IP_INTERRUPT_ENABLE - SCRN0_BASE)
	#define QSCRN_SCRN_LL_MISC_STAT_OFF (SCRN0SCRN_LL_MISC_STAT - SCRN0_BASE)
	#define SCRN_LL_MISC_STAT__IEXEC_ILLEGAL_INST__MASK SCRN0SCRN_LL_MISC_STAT__IEXEC_ILLEGAL_INST__MASK
#endif
	
struct scrn_quasar {
	int			ref;
	int			minor;
	struct cdev		cdev;
	volatile u8 __iomem	*regs[1];
	unsigned long		ioreg_start[1];
	unsigned long		ioreg_end[1];
	int			nioregs;
	int			irq;

	wait_queue_head_t	mbq;	/* wait-queue for dma-doneinterrupt events */
	struct q_doneinfo	doneinfo[MAX_MAILBOX];
	int			di_head, di_tail;
	spinlock_t		lock;
};

static int qscrn_open(struct inode* inode, struct file* filp)
{
	struct scrn_quasar *scrn;
    volatile u8 __iomem *reg;
    volatile u8 __iomem *sts;
	
	scrn = container_of(inode->i_cdev, struct scrn_quasar, cdev);
	if(scrn->ref > 0)
		return -EBUSY;
	scrn->ref++;
	scrn->minor = iminor(inode);
	filp->private_data = scrn;
	scrn->di_head = scrn->di_tail = 0;
// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
    reg = ioremap(IPM2CLKDISCTRL, 4);
    sts = ioremap(IPM2CLKDISSTAT, 4);
    switch (scrn->minor)
    {
    case 0:
        writel(readl(reg) & IPM2CLKDISCTRL__IPM2_SCRN0__INV_MASK, reg);
        while (readl(sts) & IPM2CLKDISSTAT__IPM2_SCRN0__MASK) {;}
        break;
    case 1:
        writel(readl(reg) & IPM2CLKDISCTRL__IPM2_SCRN1__INV_MASK, reg);
        while (readl(sts) & IPM2CLKDISSTAT__IPM2_SCRN1__MASK) {;}
        break;
    default:
        break;
    }
	iounmap(reg);
	iounmap(sts);
	reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
    switch (scrn->minor)
    {
    case 0:
        writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_SCRN0__INV_MASK, reg);
		printk(KERN_INFO "QSCRN %d Reset\n", scrn->minor);
        break;
    case 1:
        writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_SCRN1__INV_MASK, reg);
		printk(KERN_INFO "QSCRN %d Reset\n", scrn->minor);
        break;
    default:
        printk(KERN_INFO "QSCRN - unknown minor: %d\n", scrn->minor);
        break;
    }
    iounmap(reg);
#endif
	return 0;
}

static int qscrn_release(struct inode* inode, struct file* filp)
{
	struct scrn_quasar *scrn;
	
	scrn = container_of(inode->i_cdev, struct scrn_quasar, cdev);
	if(scrn->ref <= 0)
		return -EFAULT;
	scrn->ref--;
	filp->private_data = NULL;
	return 0;
}

static ssize_t qscrn_read(struct file* filp, char __user *buffer, size_t length, loff_t* offset)
{
	struct scrn_quasar *scrn;	
	
	scrn = (struct scrn_quasar*)filp->private_data;
	return -EINVAL;
}

static ssize_t qscrn_write(struct file* filp, const char __user *buffer, size_t length, loff_t* offset)
{
	struct scrn_quasar *scrn;	
	
	scrn = (struct scrn_quasar*)filp->private_data;
	return -EINVAL;
}

static unsigned int qscrn_poll(struct file *filp, poll_table *wait)
{
	struct scrn_quasar* scrn;	
	unsigned int mask = 0;
	unsigned long flags;
	
	scrn = (struct scrn_quasar*)filp->private_data;

	poll_wait(filp, &scrn->mbq, wait);
	spin_lock_irqsave(&scrn->lock, flags);
	if(scrn->di_head != scrn->di_tail) {
		mask |= POLLIN | POLLRDNORM; /* readable */
	}
	spin_unlock_irqrestore(&scrn->lock, flags);
	return mask;
}

static int qscrn_setimom(struct scrn_quasar *scrn, struct q_scrn_set *qset, int im)
{
	volatile u8 __iomem *reg;
	unsigned long offset;

	if (im)
	{
		offset = scrn->minor ? SB_SCRN1_IM0 : SB_SCRN0_IM0;
		reg = ioremap(offset, 16);
		writel(qset->data0, reg);
		writel(qset->data1, reg + 0x4);
		writel(qset->data2, reg + 0x8);
		writel(qset->data3, reg + 0xc);
		iounmap(reg);
		offset = scrn->minor ? SCRN1_IMON0_CTRL : SCRN0_IMON0_CTRL;
		reg = ioremap(offset, 16);
		writel(qset->cb + (qset->data0 ? 0x10 : 0x0), reg);
		writel(qset->cb + (qset->data1 ? 0x10 : 0x0), reg + 0x4);
		writel(qset->cb + (qset->data2 ? 0x10 : 0x0), reg + 0x8);
		writel(qset->cb + (qset->data3 ? 0x10 : 0x0), reg + 0xc);
		iounmap(reg);
	}
	else
	{
		offset = scrn->minor ? SB_SCRN1_OM0 : SB_SCRN0_OM0;
		reg = ioremap(offset, 16);
		writel(qset->data0, reg);
		writel(qset->data1, reg + 0x4);
		writel(qset->data2, reg + 0x8);
		writel(qset->data3, reg + 0xc);
		iounmap(reg);
		offset = scrn->minor ? SCRN1_OMON0_CTRL : SCRN0_OMON0_CTRL;
		reg = ioremap(offset, 16);
		writel(qset->cb + (qset->data0 ? 0x10 : 0x0), reg);
		writel(qset->cb + (qset->data1 ? 0x10 : 0x0), reg + 0x4);
		writel(qset->cb + (qset->data2 ? 0x10 : 0x0), reg + 0x8);
		writel(qset->cb + (qset->data3 ? 0x10 : 0x0), reg + 0xc);
		iounmap(reg);
	}
	return 0;
}

static long qscrn_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct scrn_quasar *scrn;	
	struct q_scrn_set qset;
	unsigned long flags;
	int ret = 0;
	
	scrn = (struct scrn_quasar*)filp->private_data;
	
	switch(cmd)
	{
	case QGETDONE:
		if(scrn->di_head == scrn->di_tail)
			return -EAGAIN;
		copy_to_user((void*)arg,
			(void*)&scrn->doneinfo[scrn->di_tail], sizeof(struct q_doneinfo));
		spin_lock_irqsave(&scrn->lock, flags);
		scrn->di_tail++;
		if(scrn->di_tail >= MAX_MAILBOX)
			scrn->di_tail = 0;
		spin_unlock_irqrestore(&scrn->lock, flags);
		break;
	case QWAITDONE:
		while(scrn->di_head == scrn->di_tail) {
			/* TODO - check for dma in progress? */
			if (wait_event_interruptible(scrn->mbq,
				(scrn->di_head != scrn->di_tail)))
				return -ERESTARTSYS;
		}
		if(scrn->di_head != scrn->di_tail) {
			copy_to_user((void*)arg,
				(void*)&scrn->doneinfo[scrn->di_tail], sizeof(struct q_doneinfo));
			spin_lock_irqsave(&scrn->lock, flags);
			scrn->di_tail++;
			if(scrn->di_tail >= MAX_MAILBOX)
				scrn->di_tail = 0;
			spin_unlock_irqrestore(&scrn->lock, flags);
			ret = 0;
		}
		else {
			/* TODO - check for dma in progress? */
			ret = -EAGAIN;
		}
		break;
	case QSCRNSETIM:
	case QSCRNSETOM:
		if(copy_from_user((void*)&qset,
			(void*)arg, sizeof(struct q_scrn_set)))
			return -EINVAL;
		ret = qscrn_setimom(scrn, &qset, cmd == QSCRNSETIM);
		break;
	default:
		printk(QSCRN_DEVNAME " bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static int qscrn_mmap(struct file* filp, struct vm_area_struct* vma)
{
	struct scrn_quasar *scrn;	
	int ret = 0;
	int i;

	scrn = (struct scrn_quasar*)filp->private_data;	
	if(! scrn) return -ENODEV;

	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	
	// find region in iorange
	//
	for(i = 0; i < scrn->nioregs; i++) {
		if((scrn->ioreg_start[i] >> PAGE_SHIFT) == vma->vm_pgoff) {
			ret = remap_pfn_range(
						vma,
						vma->vm_start,
						scrn->ioreg_start[i] >> PAGE_SHIFT,
						vma->vm_end - vma->vm_start,
						vma->vm_page_prot
					);
			return ret;
		}
	}
	printk(KERN_WARNING 
		"SCRN - Could not remap region pgoff: %0lx (addr=%08lx)\n",
		vma->vm_pgoff,
		vma->vm_pgoff << PAGE_SHIFT);
	return -ENOMEM;
}

static irqreturn_t quasar_scrn_interrupt(int irq, void *dev_id)
{
	struct scrn_quasar *scrn = (struct scrn_quasar *)dev_id; 
	struct timeval tv;
	int ret;
	unsigned long flags;
	unsigned long err;
	
	/* disable sw interrupt from SCRN core */
	writel(0, scrn->regs[0] + QSCRN_SCRN_IP_INTERRUPT_ENABLE_OFF);
	
	/* handle errors */
	err = readl(scrn->regs[0] + QSCRN_SCRN_LL_MISC_STAT_OFF);
	if(err & SCRN_LL_MISC_STAT__IEXEC_ILLEGAL_INST__MASK)
		printk("SCRN LLDMA aborted due to an illegal instruction!\n");

	do_gettimeofday(&tv);

	spin_lock_irqsave(&scrn->lock, flags);
	scrn->doneinfo[scrn->di_head].msg     = QMSG_DONE;
	scrn->doneinfo[scrn->di_head].detail  = err;
	scrn->doneinfo[scrn->di_head].endtime = (unsigned long long)tv.tv_sec * 1000000 + (unsigned long long)tv.tv_usec;
	scrn->doneinfo[scrn->di_head].cycles  = 0;
	scrn->di_head++;
	if(scrn->di_head >= MAX_MAILBOX)
		scrn->di_head = 0;
	spin_unlock_irqrestore(&scrn->lock, flags);

	/*printk("scrn int %d\n", scrn->ndmas);*/

	/* wakeup tasks waiting on dma done
	*/
	wake_up_interruptible(&scrn->mbq);

	ret = IRQ_HANDLED;
	return ret;
}

static struct file_operations quasar_scrn_ops = {
	.owner		= THIS_MODULE,
	.open		= qscrn_open,
	.release	= qscrn_release,
	.read		= qscrn_read,
	.write		= qscrn_write,
	.poll		= qscrn_poll,
	.mmap		= qscrn_mmap,
	.unlocked_ioctl	= qscrn_ioctl,
	.compat_ioctl	= qscrn_ioctl 
};

static int __init quasar_scrn_probe(struct platform_device *pdev)
{
	struct scrn_quasar *scrn;
	struct resource	*regs;
	dev_t  scrnn;
	int    ret, i;
	static int nscrns = 0;

	scrn = kzalloc(sizeof(struct scrn_quasar), GFP_KERNEL);
	if (!scrn) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	cdev_init(&scrn->cdev, &quasar_scrn_ops);
	scrnn = MKDEV(QSCRN_MAJOR, nscrns);
	ret = cdev_add(&scrn->cdev, scrnn, 1);
	if (ret) {
		dev_dbg(&pdev->dev, "could not create char dev %d\n", nscrns);
		goto out_err;
	}
	for (scrn->nioregs = 0; scrn->nioregs < 1; scrn->nioregs++) {
		regs = platform_get_resource(pdev, IORESOURCE_MEM, scrn->nioregs);
		if (!regs) {
			dev_dbg(&pdev->dev, "no mmio reg resource defined\n");
			ret = -ENXIO;
			goto out_rerr;
		}
		scrn->ioreg_start[scrn->nioregs] = regs->start;
		scrn->ioreg_end[scrn->nioregs]   = regs->end;

		scrn->regs[scrn->nioregs] = ioremap(regs->start, regs->end - regs->start + 1);
		if(!scrn->regs[scrn->nioregs]) {
			ret = -ENOMEM;
			dev_dbg(&pdev->dev, "could not map reg I/O memory %d\n", scrn->nioregs);
			goto out_ioerr;
		}
	}
	scrn->irq = platform_get_irq(pdev, 0);
	if (scrn->irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		ret = -ENXIO;
		goto out_ioerr;
	}
	ret = request_irq(scrn->irq, quasar_scrn_interrupt,
		0, "scrndma", scrn);
	if (ret) {
		dev_dbg(&pdev->dev, "could not request irq %d\n", scrn->irq);
		scrn->irq = -1;
		goto out_ioerr;
	}
	spin_lock_init(&scrn->lock);
	init_waitqueue_head(&scrn->mbq);

	nscrns++;
	platform_set_drvdata(pdev, scrn);
	printk("QSCRN - mapped at %px, irq %d\n", scrn->regs, scrn->irq);
	device_init_wakeup(&pdev->dev, 1);
	return 0;
out_ioerr:
	for (i = 0; i < scrn->nioregs; i++)
		iounmap(scrn->regs[i]);
out_rerr:
	cdev_del(&scrn->cdev);		
out_err:
	kfree(scrn);
	return ret;
}

static int __exit quasar_scrn_remove(struct platform_device *pdev)
{
	struct scrn_quasar *scrn = platform_get_drvdata(pdev);
	int i;

	if(scrn->irq > 0)
		free_irq(scrn->irq, scrn);
	for (i = 0; i < scrn->nioregs; i++)
		iounmap(scrn->regs[i]);
	cdev_del(&scrn->cdev);		
	kfree(scrn);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static int quasar_scrn_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct scrn_quasar *scrn = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM2CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (scrn->minor)
		{
		default:
			break;
		case 0:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM2_SCRN0__MASK, reg);
			writel(readl(ctl) | IPM2CLKDISCTRL__IPM2_SCRN0__MASK, ctl);
			while ((readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SCRN0__MASK) == 0x0) {;}
			break;
		case 1:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM2_SCRN1__MASK, reg);
			writel(readl(ctl) | IPM2CLKDISCTRL__IPM2_SCRN1__MASK, ctl);
			while ((readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SCRN1__MASK) == 0x0) {;}
			break;
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
	}
#endif
	return 0;
}

static int quasar_scrn_resume(struct platform_device *pdev)
{
	struct scrn_quasar *scrn = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM2CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (scrn->minor)
		{
		default:
			break;
		case 0:
			writel(readl(ctl) & IPM2CLKDISCTRL__IPM2_SCRN0__INV_MASK, ctl);
			while (readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SCRN0__MASK) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_SCRN0__INV_MASK, reg);
			break;
		case 1:
			writel(readl(ctl) & IPM2CLKDISCTRL__IPM2_SCRN1__INV_MASK, ctl);
			while (readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SCRN1__MASK) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_SCRN1__INV_MASK, reg);
			break;
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
	}
#endif
	return 0;
}

MODULE_ALIAS("platform:quasar-scrn");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-scrn" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver quasar_scrn_driver_ops = {
	.probe		= quasar_scrn_probe,
	.remove		= quasar_scrn_remove,
	.suspend	= quasar_scrn_suspend,
	.resume		= quasar_scrn_resume,
	.driver		= {
		.name	= "quasar-scrn",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init quasar_scrn_init(void)
{
	int ret;
	
	ret = platform_driver_register(&quasar_scrn_driver_ops);
	return ret;
}
module_init(quasar_scrn_init);

static void __exit quasar_scrn_exit(void)
{
	platform_driver_unregister(&quasar_scrn_driver_ops);
}
module_exit(quasar_scrn_exit);

MODULE_DESCRIPTION("Quasar WSCRN driver");
MODULE_LICENSE("Dual BSD/GPL");

