/*
 * A kernel driver for SCAL h/w block found on Quatro processors
 *
 * Quasar SCAL kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/01/19 08:48:00 $
//  $Change: 57110 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#define QSCAL_DEVNAME	"qscal"

#define MAX_MAILBOX 64

/* The fireregs headers splits the scale h/e region into two separate areas
 * with "SCAL_IP_SPARE" starting the SCAL_IP region.  For simplicity
 * the kernel driver just maps the scal region as one, so just be sure
 * to not use any _OFF in the IP region directly
 */
#if defined Q6600 || defined Q6300
	#define QSCAL_SCAL_IP_INTERRUPT_ENABLE_OFF (SCL0SCAL_IP_INTERRUPT_ENABLE - SCL0_BASE)
	#define QSCAL_SCAL_LL_MISC_STAT_OFF (SCL0SCAL_LL_MISC_STAT - SCL0_BASE)
	#define SCAL_LL_MISC_STAT__IEXEC_ILLEGAL_INST__MASK SCL0SCAL_LL_MISC_STAT__IEXEC_ILLEGAL_INST__MASK
#endif
	
struct scal_quasar {
	int			ref;
	int			minor;
	struct cdev		cdev;
	volatile u8	__iomem	*regs[1];
	unsigned long		ioreg_start[1];
	unsigned long		ioreg_end[1];
	int			nioregs;
	int			irq;

	wait_queue_head_t	mbq;	/* wait-queue for dma-doneinterrupt events */
	struct q_doneinfo	doneinfo[MAX_MAILBOX];
	int			di_head, di_tail;
	spinlock_t		lock;
};

static int qscal_open(struct inode* inode, struct file* filp)
{
	struct scal_quasar *scal;
    volatile u8 __iomem *reg;
#if defined(Q6600)
    volatile u8 __iomem *sts;
#endif
	
	scal = container_of(inode->i_cdev, struct scal_quasar, cdev);
	if(scal->ref > 0)
		return -EBUSY;
	scal->ref++;
	scal->minor = iminor(inode);
	filp->private_data = scal;
	scal->di_head = scal->di_tail = 0;
#ifdef Q6300
	{
		reg = ioremap(RSTGEN_SWRSTPULSE3, 4);
		writel(RSTGEN_SWRSTPULSE3__IPM2_SCL0__MASK, reg);
		printk(KERN_INFO "QSCL%d Reset\n", scal->minor);
		iounmap(reg);
	}
#endif
#if defined(Q6600)
    reg = ioremap(IPM2CLKDISCTRL, 4);
    sts = ioremap(IPM2CLKDISSTAT, 4);
    switch (scal->minor)
    {
    case 0:
        writel(readl(reg) & IPM2CLKDISCTRL__IPM2_SCL0__INV_MASK, reg);
        while (readl(sts) & IPM2CLKDISSTAT__IPM2_SCL0__MASK) {;}
        break;
    case 1:
        writel(readl(reg) & IPM2CLKDISCTRL__IPM2_SCL1__INV_MASK, reg);
        while (readl(sts) & IPM2CLKDISSTAT__IPM2_SCL1__MASK) {;}
        break;
    default:
        break;
    }
	iounmap(reg);
	iounmap(sts);
	reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
    switch (scal->minor)
    {
    case 0:
        writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_SCL0__INV_MASK, reg);
		printk(KERN_INFO "QSCL%d Reset\n", scal->minor);
        break;
    case 1:
        writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_SCL1__INV_MASK, reg);
		printk(KERN_INFO "QSCL%d Reset\n", scal->minor);
        break;
    default:
        printk(KERN_INFO "QSCL - unknown minor: %d\n", scal->minor);
        break;
    }
    iounmap(reg);
#endif
	return 0;
}

static int qscal_release(struct inode* inode, struct file* filp)
{
	struct scal_quasar *scal;
	
	scal = container_of(inode->i_cdev, struct scal_quasar, cdev);
	if(scal->ref <= 0)
		return -EFAULT;
	scal->ref--;
	filp->private_data = NULL;
	return 0;
}

static ssize_t qscal_read(struct file* filp, char __user *buffer, size_t length, loff_t* offset)
{
	struct scal_quasar *scal;	
	
	scal = (struct scal_quasar*)filp->private_data;
	return -EINVAL;
}

static ssize_t qscal_write(struct file* filp, const char __user *buffer, size_t length, loff_t* offset)
{
	struct scal_quasar *scal;	
	
	scal = (struct scal_quasar*)filp->private_data;
	return -EINVAL;
}

static unsigned int qscal_poll(struct file *filp, poll_table *wait)
{
	struct scal_quasar* scal;	
	unsigned int mask = 0;
	unsigned long flags;
	
	scal = (struct scal_quasar*)filp->private_data;

	poll_wait(filp, &scal->mbq, wait);
	spin_lock_irqsave(&scal->lock, flags);
	if(scal->di_head != scal->di_tail) {
		mask |= POLLIN | POLLRDNORM; /* readable */
	}
	spin_unlock_irqrestore(&scal->lock, flags);
	return mask;
}

static long qscal_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct scal_quasar *scal;	
	unsigned long flags;
	int ret = 0;
	
	scal = (struct scal_quasar*)filp->private_data;
	
	switch(cmd)
	{
	case QGETDONE:
		if(scal->di_head == scal->di_tail)
			return -EAGAIN;
		copy_to_user((void*)arg,
			(void*)&scal->doneinfo[scal->di_tail], sizeof(struct q_doneinfo));
		spin_lock_irqsave(&scal->lock, flags);
		scal->di_tail++;
		if(scal->di_tail >= MAX_MAILBOX)
			scal->di_tail = 0;
		spin_unlock_irqrestore(&scal->lock, flags);
		break;
	case QWAITDONE:
		while(scal->di_head == scal->di_tail) {
			/* TODO - check for dma in progress? */
			if (wait_event_interruptible(scal->mbq,
				(scal->di_head != scal->di_tail)))
				return -ERESTARTSYS;
		}
		if(scal->di_head != scal->di_tail) {
			copy_to_user((void*)arg,
				(void*)&scal->doneinfo[scal->di_tail], sizeof(struct q_doneinfo));
			spin_lock_irqsave(&scal->lock, flags);
			scal->di_tail++;
			if(scal->di_tail >= MAX_MAILBOX)
				scal->di_tail = 0;
			spin_unlock_irqrestore(&scal->lock, flags);
			ret = 0;
		}
		else {
			/* TODO - check for dma in progress? */
			ret = -EAGAIN;
		}
		break;
	default:
		printk(KERN_WARNING QSCAL_DEVNAME " bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static int qscal_mmap(struct file* filp, struct vm_area_struct* vma)
{
	struct scal_quasar *scal;	
	int ret = 0;
	int i;

	scal = (struct scal_quasar*)filp->private_data;	
	if(! scal) return -ENODEV;

	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	
	// find region in iorange
	//
	for(i = 0; i < scal->nioregs; i++) {
		if((scal->ioreg_start[i] >> PAGE_SHIFT) == vma->vm_pgoff) {
			ret = remap_pfn_range(
					vma,
					vma->vm_start,
					scal->ioreg_start[i] >> PAGE_SHIFT,
					vma->vm_end - vma->vm_start,
					vma->vm_page_prot
				);
			return ret;
		}
	}
	printk(KERN_WARNING
		"SCAL - Could not remap region pgoff: %0lx (addr=%08lx)\n",
		vma->vm_pgoff,
		vma->vm_pgoff << PAGE_SHIFT);
	return -ENOMEM;
}

static irqreturn_t quasar_scal_interrupt(int irq, void *dev_id)
{
	struct scal_quasar *scal = (struct scal_quasar *)dev_id; 
	struct timeval tv;
	int ret;
	unsigned long flags;
	unsigned long err;

	/* disable sw interrupt from SCAL core */
	writel(0, scal->regs[0] + QSCAL_SCAL_IP_INTERRUPT_ENABLE_OFF);

	/* handle errors */
	err = readl(scal->regs[0] + QSCAL_SCAL_LL_MISC_STAT_OFF);
	if(err & SCAL_LL_MISC_STAT__IEXEC_ILLEGAL_INST__MASK)
		printk("SCAL LLDMA aborted due to an illegal instruction!\n");

	do_gettimeofday(&tv);

	spin_lock_irqsave(&scal->lock, flags);
	scal->doneinfo[scal->di_head].msg     = QMSG_DONE;
	scal->doneinfo[scal->di_head].detail  = err;
	scal->doneinfo[scal->di_head].endtime = (unsigned long long)tv.tv_sec * 1000000 + (unsigned long long)tv.tv_usec;
	scal->doneinfo[scal->di_head].cycles  = 0;
	scal->di_head++;
	if(scal->di_head >= MAX_MAILBOX)
		scal->di_head = 0;
	spin_unlock_irqrestore(&scal->lock, flags);

	/* wakeup tasks waiting on dma done
	*/
	wake_up_interruptible(&scal->mbq);

	ret = IRQ_HANDLED;
	return ret;
}

static struct file_operations quasar_scal_ops = {
	.owner		= THIS_MODULE,
	.open		= qscal_open,
	.release	= qscal_release,
	.read		= qscal_read,
	.write		= qscal_write,
	.poll		= qscal_poll,
	.mmap		= qscal_mmap,
	.unlocked_ioctl	= qscal_ioctl,
	.compat_ioctl	= qscal_ioctl 
};

static int __init quasar_scal_probe(struct platform_device *pdev)
{
	struct scal_quasar *scal;
	struct resource	*regs;
	dev_t  scaln;
	int    ret, i;
	static int nscals = 0;

	scal = kzalloc(sizeof(struct scal_quasar), GFP_KERNEL);
	if (!scal) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	cdev_init(&scal->cdev, &quasar_scal_ops);
	scaln = MKDEV(QSCAL_MAJOR, nscals);
	ret = cdev_add(&scal->cdev, scaln, 1);
	if (ret) {
		dev_dbg(&pdev->dev, "could not create char dev %d\n", nscals);
		goto out_err;
	}
	for (scal->nioregs = 0; scal->nioregs < 1; scal->nioregs++) {
		regs = platform_get_resource(pdev, IORESOURCE_MEM, scal->nioregs);
		if (!regs) {
			dev_dbg(&pdev->dev, "no mmio reg resource defined\n");
			ret = -ENXIO;
			goto out_rerr;
		}
		scal->ioreg_start[scal->nioregs] = regs->start;
		scal->ioreg_end[scal->nioregs]   = regs->end;

		scal->regs[scal->nioregs] = ioremap(regs->start, regs->end - regs->start + 1);
		if(!scal->regs[scal->nioregs]) {
			ret = -ENOMEM;
			dev_dbg(&pdev->dev,
				"could not map reg I/O memory %d\n", scal->nioregs);
			goto out_ioerr;
		}
	}
	scal->irq = platform_get_irq(pdev, 0);
	if (scal->irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		ret = -ENXIO;
		goto out_ioerr;
	}
	ret = request_irq(scal->irq, quasar_scal_interrupt,
		0, "scaldma", scal);
	if (ret) {
		dev_dbg(&pdev->dev, "could not request irq %d\n", scal->irq);
		scal->irq = -1;
		goto out_ioerr;
	}
	spin_lock_init(&scal->lock);
	init_waitqueue_head(&scal->mbq);

	nscals++;
	platform_set_drvdata(pdev, scal);
	printk("QSCAL - mapped at %px, irq %d\n", scal->regs, scal->irq);
	device_init_wakeup(&pdev->dev, 1);
	return 0;
out_ioerr:
	for (i = 0; i < scal->nioregs; i++)
		iounmap(scal->regs[i]);
out_rerr:
	cdev_del(&scal->cdev);		
out_err:
	kfree(scal);
	return ret;
}

static int __exit quasar_scal_remove(struct platform_device *pdev)
{
	struct scal_quasar *scal = platform_get_drvdata(pdev);
	int i;

	if(scal->irq > 0)
		free_irq(scal->irq, scal);
	for (i = 0; i < scal->nioregs; i++)
		iounmap(scal->regs[i]);
	cdev_del(&scal->cdev);		
	kfree(scal);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static int quasar_scal_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct scal_quasar *scal = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM2CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (scal->minor)
		{
		default:
			break;
		case 0:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM2_SCL0__MASK, reg);
			writel(readl(ctl) | IPM2CLKDISCTRL__IPM2_SCL0__MASK, ctl);
			while ((readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SCL0__MASK) == 0x0) {;}
			break;
#if defined(Q6600)
		case 1:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM2_SCL1__MASK, reg);
			writel(readl(ctl) | IPM2CLKDISCTRL__IPM2_SCL1__MASK, ctl);
			while ((readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SCL1__MASK) == 0x0) {;}
			break;
#endif
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
	}
#endif
	return 0;
}

static int quasar_scal_resume(struct platform_device *pdev)
{
	struct scal_quasar *scal = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM2CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (scal->minor)
		{
		default:
			break;
		case 0:
			writel(readl(ctl) & IPM2CLKDISCTRL__IPM2_SCL0__INV_MASK, ctl);
			while (readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SCL0__MASK) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_SCL0__INV_MASK, reg);
			break;
#if defined(Q6600)
		case 1:
			writel(readl(ctl) & IPM2CLKDISCTRL__IPM2_SCL1__INV_MASK, ctl);
			while (readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SCL1__MASK) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_SCL1__INV_MASK, reg);
			break;
#endif
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
	}
#endif
	return 0;
}

MODULE_ALIAS("platform:quasar-scal");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-scal" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver quasar_scal_driver_ops = {
	.probe		= quasar_scal_probe,
	.remove		= quasar_scal_remove,
	.suspend	= quasar_scal_suspend,
	.resume		= quasar_scal_resume,
	.driver		= {
		.name	= "quasar-scal",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init quasar_scal_init(void)
{
	int ret;
	
	ret = platform_driver_register(&quasar_scal_driver_ops);
	return ret;
}
module_init(quasar_scal_init);

static void __exit quasar_scal_exit(void)
{
	platform_driver_unregister(&quasar_scal_driver_ops);
}
module_exit(quasar_scal_exit);

MODULE_DESCRIPTION("Quasar SCAL driver");
MODULE_LICENSE("Dual BSD/GPL");

