/*
 * A kernel driver for SBE found on Quatro processors
 *
 *  Quasar SBE kernel driver
 *
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/01/27 08:19:40 $
//  $Change: 57409 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#define QSBE_DEVNAME	"qsbe"

#define MAX_SBEIOREGS		2

#define MAX_MAILBOX 64

#if defined Q6600 || defined Q6300
#define SBE_CST_OFF                    SBE0SBE_CST_OFF
#define SBE_CST__CLR_INT__MASK         SBE0SBE_CST__CLR_INT__MASK
#define SBE_STAT_RGB_OFF               SBE0SBE_STAT_RGB_OFF
#define SBE_STAT_RGB__RESP_ERR__MASK   SBE0SBE_STAT_RGB__RESP_ERR__MASK
#endif

struct quasar_io_region {
	u32						start;
	u32						end;
	volatile u8	__iomem				*base;
};

struct sbe_quasar {
	int			ref;
	int			minor;
	struct cdev		cdev;
	struct quasar_io_region	ioregs[MAX_SBEIOREGS];
	int			nioregs;

	int			irq;

	wait_queue_head_t	mbq;	/* wait-queue for dma-doneinterrupt events */
	struct q_doneinfo	doneinfo[MAX_MAILBOX];
	int			di_head, di_tail;
	spinlock_t		lock;
};

static int qsbe_open(struct inode* inode, struct file* filp)
{
	struct sbe_quasar *sbe;
	volatile u8 __iomem *reg;
	volatile u8 __iomem *sts;
	
	sbe = container_of(inode->i_cdev, struct sbe_quasar, cdev);
	if(sbe->ref > 0)
		return -EBUSY;
	sbe->ref++;
	sbe->minor = iminor(inode);
	filp->private_data = sbe;
	sbe->di_head = sbe->di_tail = 0;
// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	reg = ioremap(IPM2CLKDISCTRL, 4);
	sts = ioremap(IPM2CLKDISSTAT, 4);
	switch (sbe->minor)
	{
	case 0:
		writel(readl(reg) & (IPM2CLKDISCTRL__IPM2_SBE0__INV_MASK & IPM2CLKDISCTRL__IPM2_SEG0__INV_MASK), reg);
		while (readl(sts) & (IPM2CLKDISSTAT__IPM2_SBE0__MASK | IPM2CLKDISSTAT__IPM2_SEG0__MASK)) {;}
		break;
	case 1:
		writel(readl(reg) & (IPM2CLKDISCTRL__IPM2_SBE1__INV_MASK & IPM2CLKDISCTRL__IPM2_SEG1__INV_MASK), reg);
		while (readl(sts) & (IPM2CLKDISSTAT__IPM2_SBE1__MASK | IPM2CLKDISSTAT__IPM2_SEG1__MASK)) {;}
		break;
	default:
		break;
	}
	iounmap(reg);
	iounmap(sts);
	reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
	switch (sbe->minor)
	{
	case 0:
		writel(readl(reg) & (RSTGEN_SWRSTSTATIC3__IPM2_SBE0__INV_MASK & RSTGEN_SWRSTSTATIC3__IPM2_SEG0__INV_MASK), reg);
		printk(KERN_INFO "QSBE %d Reset\n", sbe->minor);
		break;
	case 1:
		writel(readl(reg) & (RSTGEN_SWRSTSTATIC3__IPM2_SBE1__INV_MASK & RSTGEN_SWRSTSTATIC3__IPM2_SEG1__INV_MASK), reg);
		printk(KERN_INFO "QSBE %d Reset\n", sbe->minor);
		break;
	default:
		printk(KERN_INFO "QSBE - unknown minor: %d\n", sbe->minor);
		break;
	}
	iounmap(reg);
#endif
	return 0;
}

static int qsbe_release(struct inode* inode, struct file* filp)
{
	struct sbe_quasar *sbe;
	
	sbe = container_of(inode->i_cdev, struct sbe_quasar, cdev);
	if(sbe->ref <= 0)
		return -EFAULT;
	sbe->ref--;
	filp->private_data = NULL;
	return 0;
}

static ssize_t qsbe_read(struct file* filp, char __user *buffer, size_t length, loff_t* offset)
{
	struct sbe_quasar *sbe;	
	
	sbe = (struct sbe_quasar*)filp->private_data;
	return -EINVAL;
}

static ssize_t qsbe_write(struct file* filp, const char __user *buffer, size_t length, loff_t* offset)
{
	struct sbe_quasar *sbe;	
	
	sbe = (struct sbe_quasar*)filp->private_data;
	return -EINVAL;
}

static unsigned int qsbe_poll(struct file *filp, poll_table *wait)
{
	struct sbe_quasar* sbe;	
	unsigned int mask = 0;
	unsigned long flags;
	
	sbe = (struct sbe_quasar*)filp->private_data;

	poll_wait(filp, &sbe->mbq, wait);
	spin_lock_irqsave(&sbe->lock, flags);
	if(sbe->di_head != sbe->di_tail) {
		mask |= POLLIN | POLLRDNORM; /* readable */
	}
	spin_unlock_irqrestore(&sbe->lock, flags);
	return mask;
}

static int qsbe_reset(struct sbe_quasar *sbe)
{
	volatile u8 __iomem *reg;

	reg = ioremap(RSTGEN_SWRSTPULSE3, 4);
	if (sbe->minor == 0)
	{
		writel(RSTGEN_SWRSTPULSE3__IPM2_SBE0__MASK, reg);
	}
	else if (sbe->minor == 1)
	{
		writel(RSTGEN_SWRSTPULSE3__IPM2_SBE1__MASK, reg);
	}
	iounmap(reg);
    return 0;
}

static long qsbe_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct sbe_quasar *sbe;	
	unsigned long flags;
	int ret = 0;
	
	sbe = (struct sbe_quasar*)filp->private_data;
	
	switch(cmd)
	{
	case QRESET:
		qsbe_reset(sbe);
		break;
	case QGETDONE:
		if(sbe->di_head == sbe->di_tail)
			return -EAGAIN;
		copy_to_user((void*)arg, (void*)&sbe->doneinfo[sbe->di_tail], sizeof(struct q_doneinfo));
		spin_lock_irqsave(&sbe->lock, flags);
		sbe->di_tail++;
		if(sbe->di_tail >= MAX_MAILBOX)
			sbe->di_tail = 0;
		spin_unlock_irqrestore(&sbe->lock, flags);
		break;
	case QWAITDONE:
		while(sbe->di_head == sbe->di_tail) {
			/* TODO - check for dma in progress? */
			if (wait_event_interruptible(sbe->mbq,
				(sbe->di_head != sbe->di_tail)))
				return -ERESTARTSYS;
		}
		if(sbe->di_head != sbe->di_tail) {
			copy_to_user((void*)arg,
				(void*)&sbe->doneinfo[sbe->di_tail], sizeof(struct q_doneinfo));
			spin_lock_irqsave(&sbe->lock, flags);
			sbe->di_tail++;
			if(sbe->di_tail >= MAX_MAILBOX)
				sbe->di_tail = 0;
			spin_unlock_irqrestore(&sbe->lock, flags);
			ret = 0;
		}
		else {
			/* TODO - check for dma in progress? */
			ret = -EAGAIN;
		}
		break;
	default:
		printk("qsbe - bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static int qsbe_mmap(struct file* filp, struct vm_area_struct* vma)
{
	struct sbe_quasar *sbe;	
	long length;
	int ret = 0;
	int i;

	sbe = (struct sbe_quasar*)filp->private_data;
	if(! sbe) return -ENODEV;

	/* !! mark pages as uncached for now !! */
	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);

	printk("vma: 0x%px, vma->vm_start: 0x%lx, vma->vm_end: 0x%lx\n", 
		vma, (unsigned long)vma->vm_start, (unsigned long)vma->vm_end);

	/* !! mark pages as uncached for now !! */
	for(i = 0; i < sbe->nioregs; i++) {
		length = vma->vm_end - vma->vm_start;
		printk("length: 0x%lx\n", length);
		if(length > (sbe->ioregs[i].end - sbe->ioregs[i].start)) {
			printk("SBE VMA length truncated to io region\n");
			length = (sbe->ioregs[i].end - sbe->ioregs[i].start);
			printk("new length: 0x%lx\n", length);
		}
		if((sbe->ioregs[i].start >> PAGE_SHIFT) == vma->vm_pgoff) {
			printk("in For, i: %d, vma: 0x%px, vma->vm_start: 0x%lx, vma->vm_end: 0x%lx, len: 0x%lx\n", 
									(unsigned int)i, 
									vma, 
									(unsigned long)vma->vm_start, 
									(unsigned long)vma->vm_end, 
									(long)length);
			ret = remap_pfn_range(
						vma,
						vma->vm_start,
						sbe->ioregs[i].start >> PAGE_SHIFT,
						length,
						vma->vm_page_prot
						);
			return ret;
		} 
	}
	printk("Could not remap region pgoff: %0ld\n", 
		vma->vm_pgoff);
	return -ENOMEM;
}

static irqreturn_t quasar_sbe_interrupt(int irq, void *dev_id)
{
	struct sbe_quasar *sbe = (struct sbe_quasar *)dev_id; 
	struct timeval tv;
	int ret;
	unsigned val;
	unsigned long flags;
	
// TBD: separate for 6600? 
#if defined(Q6300B0) || defined(Q6600)
	// check int status
	val = readl(sbe->ioregs[0].base + SBE_STAT_RGB_OFF);
	if (val & SBE_STAT_RGB__RESP_ERR__MASK)
	{
		printk("\nQSBE DMA response error: 0x%08x\n", val);
	}
#endif
	/* clear interrupt bit in sbe (or we keep getting them!)
	*/
	val = readl(sbe->ioregs[0].base + SBE_CST_OFF);
	writel(val | SBE_CST__CLR_INT__MASK, sbe->ioregs[0].base + SBE_CST_OFF);
	
	spin_lock_irqsave(&sbe->lock, flags);
	sbe->doneinfo[sbe->di_head].msg     = QMSG_DONE;
	sbe->doneinfo[sbe->di_head].detail  = 0;
	do_gettimeofday(&tv);
	sbe->doneinfo[sbe->di_head].endtime = (unsigned long long)tv.tv_sec * 1000000 + (unsigned long long)tv.tv_usec;
	sbe->doneinfo[sbe->di_head].cycles  = 0;
	sbe->di_head++;
	if(sbe->di_head >= MAX_MAILBOX)
		sbe->di_head = 0;
	spin_unlock_irqrestore(&sbe->lock, flags);

	/* wakeup tasks waiting on dma done
	*/
	wake_up_interruptible(&sbe->mbq);

	ret = IRQ_HANDLED;
	return ret;
}

static struct file_operations quasar_sbe_ops = {
	.owner		= THIS_MODULE,
	.open		= qsbe_open,
	.release	= qsbe_release,
	.read		= qsbe_read,
	.write		= qsbe_write,
	.poll		= qsbe_poll,
	.mmap		= qsbe_mmap,
	.unlocked_ioctl	= qsbe_ioctl,
	.compat_ioctl	= qsbe_ioctl 
};

static int __init quasar_sbe_probe(struct platform_device *pdev)
{
	struct sbe_quasar *sbe;
	struct resource	*ioreg;
	dev_t  sben;
	int    ret;
	static int nsbes = 0;

	sbe = kzalloc(sizeof(struct sbe_quasar), GFP_KERNEL);
	if (!sbe) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}

	sbe->nioregs = 0;
	
	printk("init sbe_probe ...\n"); 
	do {
		ioreg = platform_get_resource(pdev, IORESOURCE_MEM, sbe->nioregs);
		if (ioreg) {
			sbe->ioregs[sbe->nioregs].start = ioreg->start;
			sbe->ioregs[sbe->nioregs].end   = ioreg->end;
			sbe->ioregs[sbe->nioregs].base  = 
							ioremap(ioreg->start, ioreg->end - ioreg->start + 4);
			printk("SBE [%d]-- start=0x%08X, end=0x%08X, base=0x%px\n", sbe->nioregs, 
										(unsigned int)sbe->ioregs[sbe->nioregs].start,
										(unsigned int)sbe->ioregs[sbe->nioregs].end,
										sbe->ioregs[sbe->nioregs].base);
			if (!sbe->ioregs[sbe->nioregs].base) {
				ret = -ENOMEM;
				dev_dbg(&pdev->dev, "could not map I/O memory %0d\n", sbe->nioregs);
				goto out_ioerr;
			}
			sbe->nioregs++;
		}
	} while (ioreg && (sbe->nioregs < MAX_SBEIOREGS));

	cdev_init(&sbe->cdev, &quasar_sbe_ops);
	sben = MKDEV(QSBE_MAJOR, nsbes);
	ret = cdev_add(&sbe->cdev, sben, 1);
	if (ret) {
		dev_dbg(&pdev->dev, "could not create char dev %d\n", nsbes);
		goto out_ioerr;
	}

	sbe->irq = platform_get_irq(pdev, 0);
	if (sbe->irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		ret = -ENXIO;
		goto out_ioerr;
	}
	ret = request_irq(sbe->irq, quasar_sbe_interrupt,
		0, "sbedma", sbe);
	if (ret) {
		dev_dbg(&pdev->dev, "could not request irq %d\n", sbe->irq);
		sbe->irq = -1;
		goto out_rerr;
	}
	spin_lock_init(&sbe->lock);
	init_waitqueue_head(&sbe->mbq);

	nsbes++;
	platform_set_drvdata(pdev, sbe);
	printk("QSBE - mapped at base 0: 0x%px, base 1: 0x%px, irq %d\n", 
		sbe->ioregs[0].base, sbe->ioregs[1].base, sbe->irq);
	device_init_wakeup(&pdev->dev, 1);
	dev_info(&pdev->dev, "Quasar SBE\n");
	return 0;

out_rerr:
	cdev_del(&sbe->cdev);		
out_ioerr:
	iounmap(sbe->ioregs[0].base);
	while (sbe->nioregs > 0) {
		sbe->nioregs--;
		iounmap(sbe->ioregs[sbe->nioregs].base);
	}
	kfree(sbe);
	return ret;
}

static int __exit quasar_sbe_remove(struct platform_device *pdev)
{
	struct sbe_quasar *sbe = platform_get_drvdata(pdev);

	if(sbe->irq > 0)
		free_irq(sbe->irq, sbe);
	iounmap(sbe->ioregs[0].base);
	iounmap(sbe->ioregs[1].base);
	cdev_del(&sbe->cdev);
	kfree(sbe);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static int quasar_sbe_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct sbe_quasar *sbe = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM2CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (sbe->minor)
		{
		default:
			break;
		case 0:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM2_SBE0__MASK | RSTGEN_SWRSTSTATIC3__IPM2_SEG0__MASK, reg);
			writel(readl(ctl) | IPM2CLKDISCTRL__IPM2_SBE0__MASK | IPM2CLKDISCTRL__IPM2_SEG0__MASK, ctl);
			while ((readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SBE0__MASK) == 0x0) {;}
			while ((readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SEG0__MASK) == 0x0) {;}
			//printk(KERN_INFO "QSBE%d Suspend\n", sbe->minor);
			break;
		case 1:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM2_SBE1__MASK | RSTGEN_SWRSTSTATIC3__IPM2_SEG1__MASK, reg);
			writel(readl(ctl) | IPM2CLKDISCTRL__IPM2_SBE1__MASK | IPM2CLKDISCTRL__IPM2_SEG1__MASK, ctl);
			while ((readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SBE1__MASK) == 0x0) {;}
			while ((readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_SEG1__MASK) == 0x0) {;}
			//printk(KERN_INFO "QSBE%d Suspend\n", sbe->minor);
			break;
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
	}
#endif
	return 0;
}

static int quasar_sbe_resume(struct platform_device *pdev)
{
	struct sbe_quasar *sbe = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM2CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (sbe->minor)
		{
		default:
			break;
		case 0:
			writel(readl(ctl) & IPM2CLKDISCTRL__IPM2_SBE0__INV_MASK & IPM2CLKDISCTRL__IPM2_SEG0__INV_MASK, ctl);
			while (readl(ctl + 4) & (IPM2CLKDISSTAT__IPM2_SBE0__MASK | IPM2CLKDISSTAT__IPM2_SEG0__MASK)) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_SBE0__INV_MASK & RSTGEN_SWRSTSTATIC3__IPM2_SEG0__INV_MASK, reg);
			//printk(KERN_INFO "QSBE%d Resume\n", sbe->minor);
			break;
		case 1:
			writel(readl(ctl) & IPM2CLKDISCTRL__IPM2_SBE1__INV_MASK & IPM2CLKDISCTRL__IPM2_SEG1__INV_MASK, ctl);
			while (readl(ctl + 4) & (IPM2CLKDISSTAT__IPM2_SBE1__MASK | IPM2CLKDISSTAT__IPM2_SEG1__MASK)) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_SBE1__INV_MASK & RSTGEN_SWRSTSTATIC3__IPM2_SEG1__INV_MASK, reg);
			//printk(KERN_INFO "QSBE%d Resume\n", sbe->minor);
			break;
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
	}
#endif
	return 0;
}

MODULE_ALIAS("platform:quasar-sbe");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-sbe" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver quasar_sbe_driver_ops = {
	.probe		= quasar_sbe_probe,
	.remove		= quasar_sbe_remove,
	.suspend	= quasar_sbe_suspend,
	.resume		= quasar_sbe_resume,
	.driver		= {
		.name	= "quasar-sbe",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),
	},
};

static int __init quasar_sbe_init(void)
{
	int ret;
	
	ret = platform_driver_register(&quasar_sbe_driver_ops);
	return ret;
}
module_init(quasar_sbe_init);

static void __exit quasar_sbe_exit(void)
{
	platform_driver_unregister(&quasar_sbe_driver_ops);
}
module_exit(quasar_sbe_exit);

MODULE_DESCRIPTION(" Quasar SBE driver");
MODULE_LICENSE("Dual BSD/GPL");

