/* Linux kernel driver for resistive touch panel on some Quatro reference boards
 *
 * Quasar resistive touch panel kernel driver
 * 
 * Copyright (c) 2014, 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/01/19 08:48:00 $
//  $Change: 57110 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/input.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/io.h>
#include <linux/vmalloc.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#define LCD_PANEL_WIDTH		800
#define LCD_PANEL_HEIGHT	600

static int lcd_width = LCD_PANEL_WIDTH;
module_param(lcd_width, int, 0644);
MODULE_PARM_DESC(lcd_width, "LCD Panel width");

static int lcd_height = LCD_PANEL_HEIGHT;
module_param(lcd_height, int, 0644);
MODULE_PARM_DESC(lcd_height, "LCD Panel height");

#ifdef Q5500
	#define IRQ_ADCR IRQ_ADC_CRIT
	#define ADC_X   14
	#define ADC_Y   13
#else
	#define IRQ_ADCR IRQ_ADC_CRIT
	#define ADC_X   14
	#define ADC_Y   13
#endif

/* Range of valid Y from ADC
*/
#define Y_MIN   186
#define Y_MAX   1023

/* how often to poll X then Y when touched down (milliseconds)
*/
#define TOUCH_POLL  12

/* how many samples of each to take, to form average.
// for 2 samples at 6ms per, it takes 24ms per actual sample
// which is a decent response.  probably have to poll
// faster to add samples
*/
#define TOUCH_SAMPLES 3

/* how many actual samples to represent under each press position, to 
// make a grid out of the screen, so the area of samples is always
// given the same coordinate for the touch.
*/
#define TOUCH_GRID_X	12
#define TOUCH_GRID_Y	8

static u32 s_valInts = 0, s_timInts = 0;

struct rtp_quasar {
	struct			input_dev *input;
	char			physname[64];
	int			w, h;
	int			ref;
	int			minor;
	volatile u8 __iomem	*regs[4];
	unsigned		ioreg_start[4];
	unsigned		ioreg_end[4];
	spinlock_t		lock;
	int			irq;
	int			poll_delay;
	struct timer_list	timer;
	struct device		*dev;
};

static inline u32 adccoreRead(struct rtp_quasar* qrtp, u32 reg)
{
	volatile u32 rv = 0;

	if (reg >= 0 && reg < (qrtp->ioreg_end[0] - qrtp->ioreg_start[0]))
		rv = readl(qrtp->regs[0] + reg);
	else
		printk(KERN_WARNING "qrtp getreg addr range error off: %08X\n", reg);
	return rv;
}

static int adccoreWrite(struct rtp_quasar* qrtp, u32 reg, u32 val)
{
	if (reg >= 0 && reg < (qrtp->ioreg_end[0] - qrtp->ioreg_start[0]))
		writel(val, qrtp->regs[0] + reg);
	else {
		printk(KERN_WARNING
	   		"qrtp setreg addr range error reg %08X not in %08X to %08X\n",
			reg, qrtp->ioreg_start[0], qrtp->ioreg_end[0]);
		return -EINVAL;
	}
	return 0;
}

static inline u32 adcctrlRead(struct rtp_quasar* qrtp, u32 reg)
{
	volatile u32 rv = 0;

	if (reg >= 0 && reg < (qrtp->ioreg_end[1] - qrtp->ioreg_start[1]))
		rv = readl(qrtp->regs[1] + reg);
	else
		printk(KERN_WARNING "qrtp getreg addr range error off: %08X\n", reg);
	return rv;
}

static int adcctrlWrite(struct rtp_quasar* qrtp, u32 reg, u32 val)
{
	if (reg >= 0 && reg < (qrtp->ioreg_end[1] - qrtp->ioreg_start[1]))
		writel(val, qrtp->regs[1] + reg);
	else {
		printk(KERN_WARNING
	   		"qrtp setreg addr range error reg %08X not in %08X to %08X\n",
			reg, qrtp->ioreg_start[1], qrtp->ioreg_end[1]);
		return -EINVAL;
	}
	return 0;
}

static inline u32 tgenbaseRead(struct rtp_quasar* qrtp, u32 reg)
{
	volatile u32 rv = 0;

	if (reg >= 0 && reg < (qrtp->ioreg_end[2] - qrtp->ioreg_start[2]))
		rv = readl(qrtp->regs[2] + reg);
	else
		printk(KERN_WARNING "qrtp getreg addr range error off: %08X\n", reg);
	return rv;
}

static int tgenbaseWrite(struct rtp_quasar* qrtp, u32 reg, u32 val)
{
	if (reg >= 0 && reg < (qrtp->ioreg_end[2] - qrtp->ioreg_start[2]))
		writel(val, qrtp->regs[2] + reg);
	else {
		printk(KERN_WARNING
	   		"qrtp setreg addr range error reg %08X not in %08X to %08X\n",
			reg, qrtp->ioreg_start[2], qrtp->ioreg_end[2]);
		return -EINVAL;
	}
	return 0;
}

static u32 s_xv[32], s_yv[32], s_nxv, s_nyv;

static int setupPin(struct rtp_quasar* qrtp, int pin, int isoutput, int initval)
{
	u32 pinv, vnip, mask;
	u32 srcreg, dirreg, valreg;

	/* Set TGEN bank in access mode, so we r/w it directly
	* (No need, default is ACC mode)
	*/
	
	/* tgen registers are masked with a 0 in the upper 16 bits for
	// the pin in question, so form that make (i.e. 0xFFFE0000 means
	// this is for pin0)
	*/
	pinv = (1 << ((u32)pin & 0xF));
	vnip = ~pinv & 0xFFFF; 	/* inverted pin bit, in lower 16 bits only */
	mask = ~pinv << 16;

	switch (pin / 16) {
	case 0:
		srcreg = TG_BANK0_OFF + TG_BANK0_IOSRC_OFF;
		dirreg = TG_BANK0_OFF + TG_BANK0_IODIR_OFF;
		valreg = TG_BANK0_OFF + TG_BANK0_IOOUT_OFF;
		break;
	case 1:
		srcreg = TG_BANK1_OFF + TG_BANK1_IOSRC_OFF;
		dirreg = TG_BANK1_OFF + TG_BANK1_IODIR_OFF;
		valreg = TG_BANK1_OFF + TG_BANK1_IOOUT_OFF;
		break;
	case 2:
		srcreg = TG_BANK2_OFF + TG_BANK2_IOSRC_OFF;
		dirreg = TG_BANK2_OFF + TG_BANK2_IODIR_OFF;
		valreg = TG_BANK2_OFF + TG_BANK2_IOOUT_OFF;
		break;
	case 3:
		srcreg = TG_BANK3_OFF + TG_BANK3_IOSRC_OFF;
		dirreg = TG_BANK3_OFF + TG_BANK3_IODIR_OFF;
		valreg = TG_BANK3_OFF + TG_BANK3_IOOUT_OFF;
		break;
	case 4:
		srcreg = TG_BANK4_OFF + TG_BANK4_IOSRC_OFF;
		dirreg = TG_BANK4_OFF + TG_BANK4_IODIR_OFF;
		valreg = TG_BANK4_OFF + TG_BANK4_IOOUT_OFF;
		break;
	case 5:
		srcreg = TG_BANK5_OFF + TG_BANK5_IOSRC_OFF;
		dirreg = TG_BANK5_OFF + TG_BANK5_IODIR_OFF;
		valreg = TG_BANK5_OFF + TG_BANK5_IOOUT_OFF;
		break;
	default:
		return -1;
	}
	/* Setup IOSRC to use iodir/ioout as meaning
	*/
	tgenbaseWrite(qrtp, srcreg, mask | ~pinv);

	/* Setup IOOUT as initial value
	*/
	if (isoutput)
		tgenbaseWrite(qrtp, valreg, mask | (initval ? pinv : vnip));

	/* Setup IODIR as direction of pin
	*/
	tgenbaseWrite(qrtp, dirreg, mask | (isoutput ? vnip : pinv));
	return 0;
}

static void touchPower(struct rtp_quasar* qrtp, int xm, int xp, int ym, int yp)
{
	setupPin(qrtp, 90, 1, xm);
	setupPin(qrtp, 91, 1, xp);
	setupPin(qrtp, 92, 1, ym);
	setupPin(qrtp, 93, 1, yp);
}

static u32 readADC(struct rtp_quasar* qrtp, int channel)
{
	u32 reg, val;
	
	switch (channel) {
	case 0: case 1:
		reg = ADCD0_OFF;
		break;
	case 2: case 3:
		reg = ADCD1_OFF;
		break;
	case 4: case 5:
		reg = ADCD2_OFF;
		break;
	case 6: case 7:
		reg = ADCD3_OFF;
		break;
	case 8: case 9:
		reg = ADCD4_OFF;
		break;
	case 10: case 11:
		reg = ADCD5_OFF;
		break;
	case 12: case 13:
		reg = ADCD6_OFF;
		break;
	case 14: case 15:
		reg = ADCD7_OFF;
		break;
	default:
		return 0;
	}
	val = adccoreRead(qrtp, reg);
	if (channel & 1)
		val >>= 16;
	else
		val &= 0xFFFF;
	return val;
}

static u32 s_lastX = 0, s_lastY = 0, s_isDown = 0;

static void onTPchange(struct rtp_quasar* qrtp, int isdown, int xv, int yv)
{
	int x, y;
	
	/* tp is 800x600 pixels, so convert ranges
	*/
	if (isdown) {
		/* scale x to range 0-800 from input range of 910-190
		*/
		x = (int)yv;
		if (x < 190) x = 190;
		if (x > 910) x = 910;
		x -= 190;
		x = 720 - x;
		x *= lcd_width;
		x /= 720;
		
		y = (int)xv;
		if (y < 200) y = 200;
		if (y > 900) y = 900;
		y -= 200;
		y *= lcd_height;
		y /= 700;
		
		/* snap to grid
		*/
		x /= TOUCH_GRID_X;
		x *= TOUCH_GRID_X;

		y /= TOUCH_GRID_Y;
		y *= TOUCH_GRID_Y;
		
		s_isDown = 1;
		if (x == s_lastX && y == s_lastY)
			return;

		s_lastX = x;
		s_lastY = y;

		input_report_key(qrtp->input, BTN_TOUCH, 1);
		input_report_abs(qrtp->input, ABS_X, x);
		input_report_abs(qrtp->input, ABS_Y, y);
		input_sync(qrtp->input);
	}
	else {
		x = 0;
		y = 0;
		
		input_report_key(qrtp->input, BTN_TOUCH, 0);
		input_sync(qrtp->input);

		s_lastX = -1;
		s_lastY = -1;
		
		s_isDown = 0;
	}
	//printk("x=%3d y=%3d  raw=%d,%d\n", x, y, yv, xv);
}

static void timerFired(unsigned long handle)
{
	struct rtp_quasar* qrtp = (struct rtp_quasar*)handle;
	u32 xv, yv;

	s_timInts++;
	if (s_timInts & 1) {
		/* read X value and setup to read Y value on next timer
		*/
		s_xv[s_nxv++] = readADC(qrtp, ADC_X);

		/* setup to read Y on next timer tick
		*/
		touchPower(qrtp, 1, 0, 0, 1); 

		mod_timer(&qrtp->timer, jiffies + qrtp->poll_delay);		
	}
	else {
		int i;

		/* read Y and form a range to interrupt on
		*/
		yv = readADC(qrtp, ADC_Y);
		
		if (yv > Y_MIN)	{
			s_yv[s_nyv++] = yv;

			if (s_nyv >= TOUCH_SAMPLES) {
				/* got a bunch of readings, so average them
				*/
				for (i = 0, yv = xv = 0; i < TOUCH_SAMPLES; i++)
				{
					yv += s_yv[i];
					xv += s_xv[i];
				}
				yv /= TOUCH_SAMPLES;
				xv /= TOUCH_SAMPLES;
				
				/* reset poll index
				*/
				s_nyv = 0;
				s_nxv = 0;

				onTPchange(qrtp, 1, xv, yv);
			}
			/* touched down, keep timer active to keep reading X/Y spot
			*/
			mod_timer(&qrtp->timer, jiffies + qrtp->poll_delay);

			/* setup to read X
			*/
			touchPower(qrtp, 0, 1, 1, 0); 
		}
		else {
			/* no touch, use invalid range as crit range to restart timer
			*/
			del_timer(&qrtp->timer);
			onTPchange(qrtp, 0, 0, 0);
			adccoreWrite(qrtp, ADCCRT_OFF, (Y_MIN << 16) | 0);
		}
	}
}

static irqreturn_t quasar_rtp_interrupt(int irq, void *handle)
{
	struct rtp_quasar *qrtp = (struct rtp_quasar *)handle;
	u32 rval;

	s_valInts++;

	/* Y > Y_MIN, so in valid range, setup for reading X
	* and setup to interrupt on timer to read X value
	*/
	
	/* put a wide range for critical range to avoid
	// other interrupts here while processing presses
	*/
	adccoreWrite(qrtp, ADCCRT_OFF, (0x3FF << 16) | 0); 

	/* clear int flag in adc
	*/
	rval = adccoreRead(qrtp, ADCINT_OFF);
	rval |= ADCINT__CI__MASK;
	adccoreWrite(qrtp, ADCINT_OFF, rval);
	
	/* setup to read X and insure timeInt state is 0
	*/
	touchPower(qrtp, 0, 1, 1, 0);
	s_timInts = 0;
	s_nxv = s_nyv = 0;

	/* start a timer to poll the adc
	*/
	qrtp->timer.data = (unsigned long)qrtp;
	qrtp->timer.function = &timerFired;
	qrtp->timer.expires = jiffies + qrtp->poll_delay;
	init_timer(&qrtp->timer);
	mod_timer(&qrtp->timer, qrtp->timer.expires);

	return IRQ_HANDLED;
}

static int rtp_init_hw(struct rtp_quasar *qrtp)
{
	u32 rval, adcval;
	int timer;
	u8 __iomem *ioreg;

	/* Enable clock for ADC
	*/
#ifdef Q5300
	ioreg = ioremap(CLKGEN_BASE, 1024);
	rval = readl(ioreg + CLKGENCLKSELCTRL1_OFF);
	rval &= (~0x00C00000);
	rval |= (1 << 22);
	writel(rval, ioreg + CLKGENCLKSELCTRL1_OFF);
	writel(0x00000080E, ioreg + CLKGENCLKDIVCTRL_ADCDAC_OFF);
	iounmap(ioreg);
#else
	ioreg = ioremap(RSTGEN_BASE, 1024);
	rval = readl(ioreg + RSTGENSWRSTSTATIC6_OFF);
	writel(rval & ~RSTGENSWRSTSTATIC6__MFP_ADCDAC__MASK, ioreg + RSTGENSWRSTSTATIC6_OFF);
	rval = readl(ioreg + RSTGENSWRSTSTATIC8_OFF);
	writel(rval & ~RSTGENSWRSTSTATIC8__MFP_TGEN__MASK, ioreg + RSTGENSWRSTSTATIC8_OFF);
	iounmap(ioreg);
	ioreg = ioremap(SYSCG_BASE, 2048);
	writel(0x00000080E, ioreg + SYSCG_CLKDIVCTRL_ADCDAC_OFF);
	iounmap(ioreg);
#endif
	qrtp->regs[3] = ioremap(qrtp->ioreg_start[3],
			qrtp->ioreg_end[3] - qrtp->ioreg_start[3] + 1);
	if(!qrtp->regs[3]) {
		dev_dbg(qrtp->dev, "could not map ctrl reg I/O memory\n");
		return -ENXIO;
	}
	/* configure digital i/o for ACH9-ACH12 = X-,X+,Y-,Y+ power and ACH8 (led1)
	*/
	rval = readl(qrtp->regs[3] + IODACFG_ACH_OFF);
	rval |= (1 << 12) | (1 << 11) | (1 << 10) | (1 << 9) | (1 << 8);
	writel(rval, qrtp->regs[3] + IODACFG_ACH_OFF);

	/* configure analog i/o for ACH13-ACH14 = X,Y read value
	*/
	rval = readl(qrtp->regs[3] + IODACFG_ACH_OFF);
	rval &= ~((1 << 13) | (1 << 14));
	writel(rval, qrtp->regs[3] + IODACFG_ACH_OFF);

	rval = readl(qrtp->regs[3] + IOPDCFG_ACH_OFF);
	rval |= ((1 << 13) | (1 << 14));
	writel(rval, qrtp->regs[3] + IOPDCFG_ACH_OFF);

	iounmap(qrtp->regs[3]);
	qrtp->regs[3] = NULL;
#ifdef Q5300	
	ioreg = ioremap(MOVL_BASE, 1024);
	
	// configure overlays to route ACH9:12 to TGENPO90:93 and ACH8 to TGENPO89
	//
	rval = readl(ioreg + MFP_OVLCTL12_OFF);
	rval &=
	 	~(0xF << MFP_OVLCTL12__ACH9__SHIFT)
	    &   ~(0xF << MFP_OVLCTL12__ACH10__SHIFT)
	    &   ~(0xF << MFP_OVLCTL12__ACH11__SHIFT)
	    &   ~(0xF << MFP_OVLCTL12__ACH12__SHIFT);
	rval |=
	 	(OVLCTL12_ACH9_TGENPI90_TGENPO90 << MFP_OVLCTL12__ACH9__SHIFT)
	    |   (OVLCTL12_ACH10_TGENPI91_TGENPO91 << MFP_OVLCTL12__ACH10__SHIFT)
	    |   (OVLCTL12_ACH11_TGENPI92_TGENPO92 << MFP_OVLCTL12__ACH11__SHIFT)
	    |   (OVLCTL12_ACH12_TGENPI93_TGENPO93 << MFP_OVLCTL12__ACH12__SHIFT);
	writel(rval, ioreg + MFP_OVLCTL12_OFF);
	
	// route pio27 for LED1
	rval = readl(ioreg + MFP_OVLCTL4_OFF);
	rval &=
	 	~(0xF << MFP_OVLCTL4__PRIO27__SHIFT);
	rval |=
	 	(OVLCTL4_PRIO27_TGENPI27_TGENPO27 << MFP_OVLCTL4__PRIO27__SHIFT);
	writel(rval, ioreg + MFP_OVLCTL4_OFF);
	iounmap(ioreg);
#elif defined(Q5500)
	/* configure overlays to route ACH9:12 to TGENPO90:93 and ACH8 to TGENPO89
	*/
	ioreg = ioremap(MOVL_BASE, 1024);
	rval = readl(ioreg + MFP_OVLCTL17_OFF);
	rval &=
	 	~(0xF << MFP_OVLCTL17__ACH8__SHIFT)
	    &   ~(0xF << MFP_OVLCTL17__ACH9__SHIFT)
	    &   ~(0xF << MFP_OVLCTL17__ACH10__SHIFT);
	rval |=
	 	(OVLCTL17_ACH8_TGENPI89_TGENPO89 << MFP_OVLCTL17__ACH8__SHIFT)
	    |   (OVLCTL17_ACH9_TGENPI90_TGENPO90 << MFP_OVLCTL17__ACH9__SHIFT)
	    |   (OVLCTL17_ACH10_TGENPI91_TGENPO91 << MFP_OVLCTL17__ACH10__SHIFT);
	writel(rval, ioreg + MFP_OVLCTL17_OFF);

	rval = readl(ioreg + MFP_OVLCTL18_OFF);
	rval &=
	 	~(0xF << MFP_OVLCTL18__ACH11__SHIFT)
	    &   ~(0xF << MFP_OVLCTL18__ACH12__SHIFT);
	rval |=
	 	(OVLCTL18_ACH11_TGENPI92_TGENPO92 << MFP_OVLCTL18__ACH11__SHIFT)
	    |   (OVLCTL18_ACH12_TGENPI93_TGENPO93 << MFP_OVLCTL18__ACH12__SHIFT);
	writel(rval, ioreg + MFP_OVLCTL18_OFF);

	/* configure overlay to route PRIO35 via PO35 for other LED
	*/
	rval = readl(ioreg + MFP_OVLCTL5_OFF);
	rval &= ~(0xF << MFP_OVLCTL5__PRIO35__SHIFT);
	rval |= (OVLCTL5_PRIO35_TGENPI35_TGENPO35 << MFP_OVLCTL5__PRIO35__SHIFT);
	writel(rval, ioreg + MFP_OVLCTL5_OFF);
	iounmap(ioreg);
#endif
	/* ------ the following code assumes complete ownership of the ADC 
	// ------ and should be replaced with an ADC subsystem that 
	// ------ is more programmable/friendly
	*/
	rval = adcctrlRead(qrtp, ADC_INPUT_OFF);
	rval &= ~(
				ADC_INPUT__ADC_OFFSET_ENABLE__MASK
	 	   |	ADC_INPUT__IREF_ENABLE__MASK
	 	   |	ADC_INPUT__ADC_ENABLE__MASK
		   );
	adcctrlWrite(qrtp, ADC_INPUT_OFF, rval);

	/* clear out ADC interrupt enables */
	adccoreWrite(qrtp, ADCINT_OFF, 0);
	adccoreWrite(qrtp, ADCINT_OFF, ADCINT__CI__MASK);
	
	/* reset ADC */
	adccoreWrite(qrtp, ADCSTAT_OFF, 0);
	
	/* set ADC for prescaler of 1 and fullspeed */
	rval = adccoreRead(qrtp, ADCSTAT_OFF);
	adccoreWrite(qrtp, ADCSTAT_OFF, (0x1 | rval));
	
	/* program the ADC queue to sample the X,Y channels
	*/
	adcval = (ADC_X << 24) | (ADC_X << 16) | (ADC_X << 8) | (ADC_X);
	adccoreWrite(qrtp, ADCQ0_OFF, adcval);
	adccoreWrite(qrtp, ADCQ1_OFF, adcval);
	adccoreWrite(qrtp, ADCQ2_OFF, adcval);
	adccoreWrite(qrtp, ADCQ3_OFF, adcval);
	adcval = (ADC_Y << 24) | (ADC_Y << 16) | (ADC_Y << 8) | (ADC_Y);
	adccoreWrite(qrtp, ADCQ4_OFF, adcval);
	adccoreWrite(qrtp, ADCQ5_OFF, adcval);
	adccoreWrite(qrtp, ADCQ6_OFF, adcval);
	adccoreWrite(qrtp, ADCQ7_OFF, adcval);

	/* Setup the critical value interrupt handler
	*/
	adccoreWrite(qrtp, ADCCRT_OFF, (0x3FF << 16) | 0);  // 0 - 1023 = max range (never an int)
	adccoreWrite(qrtp, ADCINT_OFF, (ADC_Y << 16));		// critical chanel is (Y)

	/* setup input control to power up s/h and core
	*/
	rval = adcctrlRead(qrtp, ADC_INPUT_OFF);
	rval |=
			ADC_INPUT__ADC_OFFSET_ENABLE__MASK
	 	   |	ADC_INPUT__IREF_ENABLE__MASK
	 	   |	ADC_INPUT__ADC_ENABLE__MASK;
	adcctrlWrite(qrtp, ADC_INPUT_OFF, rval);
	
	/* wait for ADC to calibrate (afaik it never does...)
	*/
	timer = 500;
	do {
	    adcval = adccoreRead(qrtp, ADCSTAT_OFF);
	}
	while (timer-- > 0 && (adcval & (1 << 31)));

	/* enable, cyclically
	*/
	adcval  |=
	 	 	ADCSTAT__CD__MASK	 	// ADC command enabled
	 	   |	ADCSTAT__CY__MASK	 	// cyclic
	 	   |	(31 << ADCSTAT__QEND__SHIFT);   // Q end index (wrap sampling queue at 15)
	adcval &= ~ADCSTAT__CS__MASK;	
	adccoreWrite(qrtp, ADCSTAT_OFF, adcval);

	adcval = adccoreRead(qrtp, ADCSTAT_OFF);

	/* init x/y value state and setup to read X
	*/
	s_nxv = 0;
	s_nyv = 0;

	touchPower(qrtp, 0, 1, 1, 0);
	return 0;
}

static int __init quasar_rtp_probe(struct platform_device *pdev)
{
	struct rtp_quasar *qrtp;
	struct resource	*regs;
	int	ret;
	static int nqrtps = 0;

	qrtp = kzalloc(sizeof(struct rtp_quasar), GFP_KERNEL);
	if (!qrtp)
		return -ENOMEM;
	qrtp->w = lcd_width;
	qrtp->h = lcd_height;

	qrtp->input = input_allocate_device();
	if (! qrtp->input) {
		kfree(qrtp);
		return -ENOMEM;
	}
	/* ADC Core register area */
	regs = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!regs) {
		dev_dbg(&pdev->dev, "no adc-core reg resource defined\n");
		ret = -ENXIO;
		goto out_rerr;
	}
	qrtp->ioreg_start[0] = regs->start;
	qrtp->ioreg_end[0]   = regs->end;
	qrtp->regs[0] = ioremap(regs->start, regs->end - regs->start + 1);
	if(!qrtp->regs[0]) {
		dev_dbg(&pdev->dev, "could not map reg I/O memory\n");
		ret = -ENXIO;
		goto out_rerr;
	}

	/* ADC Ctrl register area */
	regs = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	if (!regs) {
		dev_dbg(&pdev->dev, "no adc-ctrl reg resource defined\n");
		ret = -ENXIO;
		goto out_ioerr;
	}
	qrtp->ioreg_start[1] = regs->start;
	qrtp->ioreg_end[1]   = regs->end;
	qrtp->regs[1] = ioremap(regs->start, regs->end - regs->start + 1);
	if(!qrtp->regs[1]) {
		dev_dbg(&pdev->dev, "could not map ctrl reg I/O memory\n");
		ret = -ENXIO;
		goto out_ioerr;
	}

	/* TGEN register area */
	regs = platform_get_resource(pdev, IORESOURCE_MEM, 2);
	if (!regs) {
		dev_dbg(&pdev->dev, "no tgen-base reg resource defined\n");
		ret = -ENXIO;
		goto out_ioerr1;
	}
	qrtp->ioreg_start[2] = regs->start;
	qrtp->ioreg_end[2]   = regs->end;
	qrtp->regs[2] = ioremap(regs->start, regs->end - regs->start + 1);
	if(!qrtp->regs[2]) {
		dev_dbg(&pdev->dev, "could not map tgen reg I/O memory\n");
		ret = -ENXIO;
		goto out_ioerr1;
	}

	/* GPF-MFP core register area */
	regs = platform_get_resource(pdev, IORESOURCE_MEM, 3);
	if (!regs) {
		dev_dbg(&pdev->dev, "no mfpgpf reg resource defined\n");
		ret = -ENXIO;
		goto out_ioerr1;
	}
	qrtp->ioreg_start[3] = regs->start;
	qrtp->ioreg_end[3]   = regs->end;
	/* -- mfp-gpf is only mapped when used, as its a large
	      space and is used only briefly at init time
	*/
	qrtp->regs[3] = NULL;

	/* ADC Critical IRQ */
	qrtp->irq = platform_get_irq(pdev, 0);
	if (qrtp->irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		ret = -ENXIO;
		goto out_ioerr2;
	}
	spin_lock_init(&qrtp->lock);

	ret = request_irq(qrtp->irq, quasar_rtp_interrupt,
		0, "qrtp", qrtp);
	if (ret) {
		dev_dbg(&pdev->dev, "could not request irq %d\n", qrtp->irq);
		qrtp->irq = -1;
		goto out_ioerr2;
	}
	platform_set_drvdata(pdev, qrtp);
	printk("QRTP %dx%d mapped at %px, irq %d\n",
		qrtp->w, qrtp->h, qrtp->regs, qrtp->irq);
	nqrtps++;
	device_init_wakeup(&pdev->dev, 1);
	qrtp->dev = &pdev->dev;

	qrtp->poll_delay = msecs_to_jiffies(TOUCH_POLL);
	if (qrtp->poll_delay <= 0)
		qrtp->poll_delay = 1;
	ret = rtp_init_hw(qrtp);
	if (ret)
		goto failed_init;

	snprintf(qrtp->physname, sizeof(qrtp->physname), "%s/input0", dev_name(&pdev->dev));

	qrtp->input->name = "Quasar Touchscreen";
	qrtp->input->phys = qrtp->physname;
	qrtp->input->dev.parent = &pdev->dev;

	__set_bit(EV_KEY, qrtp->input->evbit);
	__set_bit(BTN_TOUCH, qrtp->input->keybit);
	__set_bit(EV_ABS, qrtp->input->evbit);
	__set_bit(ABS_X, qrtp->input->absbit);
	__set_bit(ABS_Y, qrtp->input->absbit);

	input_set_abs_params(qrtp->input, ABS_X, 0, qrtp->w, 0, 0);
	input_set_abs_params(qrtp->input, ABS_Y, 0, qrtp->h, 0, 0);

	ret = input_register_device(qrtp->input);
	if (ret)
	    goto failed_init;

	// setup a critical interrupt to get ready for press
	adccoreWrite(qrtp, ADCCRT_OFF, (Y_MIN << 16) | 0); // new critical range	
	
	return 0;
failed_init:
	free_irq(qrtp->irq, qrtp);
out_ioerr2:
	iounmap(qrtp->regs[2]);
out_ioerr1:
	iounmap(qrtp->regs[1]);
out_ioerr:
	iounmap(qrtp->regs[0]);
out_rerr:
	input_free_device(qrtp->input);
	kfree(qrtp);
	platform_set_drvdata(pdev, NULL);
	return ret;
}

static int __exit quasar_rtp_remove(struct platform_device *pdev)
{
	struct rtp_quasar *qrtp = platform_get_drvdata(pdev);

	input_unregister_device(qrtp->input);
	if(qrtp->irq > 0)
		free_irq(qrtp->irq, qrtp);
	iounmap(qrtp->regs[0]);
	iounmap(qrtp->regs[1]);
	iounmap(qrtp->regs[2]);
	platform_set_drvdata(pdev, NULL);
	del_timer_sync(&qrtp->timer);
	kfree(qrtp);
	return 0;
}

MODULE_ALIAS("platform:quasar-touchpanel");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-resistive-touchpanel" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver quasar_rtp_driver_ops = {
	.probe		= quasar_rtp_probe,
	.remove		= quasar_rtp_remove,
	.driver		= {
		.name	= "quasar-qrtouch",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init quasar_rtp_init(void)
{
	int ret;

	ret = platform_driver_register(&quasar_rtp_driver_ops);
	return ret;
}
module_init(quasar_rtp_init);

static void __exit quasar_rtp_exit(void)
{
	platform_driver_unregister(&quasar_rtp_driver_ops);
}
module_exit(quasar_rtp_exit);

MODULE_DESCRIPTION("Quasar touch panel driver");
MODULE_LICENSE("Dual BSD/GPL");

