/*
 * A driver for the power control for QBit SOCs
 *
 * Quasar OTP I/F kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/01/19 08:48:00 $
//  $Change: 57110 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#define PWR_DEVNAME	"qotp"

#define MAX_IOREGS		    1

struct quasar_io_region {
	u32						start;
	u32						end;
	volatile u8	__iomem		*base;
};

struct quasar_otp {
	struct cdev				cdev;
	struct device			*frdev;

	struct quasar_io_region	ioregs[MAX_IOREGS];
	int						nioregs;
	int						refs;				/* reference count */
	int						minor;				/* device minor number */
};

static volatile u8 __iomem* otp_addr(struct quasar_otp* qotp, u32 reg)
{
	volatile u8 __iomem* addr = 0;
 	int      i;

	//printk(KERN_WARNING PWR_DEVNAME "otp_addr - addr map, reg %08x, ord=%d, <%08X-%08X>--<%08X-%08X>--<%08X-%08X>--<%08X-%08X>\n", 
	//       reg, qotp->minor, qotp->ioregs[0].start, qotp->ioregs[0].end, qotp->ioregs[1].start, qotp->ioregs[1].end, qotp->ioregs[2].start, qotp->ioregs[2].end, qotp->ioregs[3].start, qotp->ioregs[3].end);
	/* check for existing mapping */
	for (i = 0; i < qotp->nioregs; i++) {
		if(reg >= qotp->ioregs[i].start && reg < qotp->ioregs[i].end) {
			return(qotp->ioregs[i].base + (reg - qotp->ioregs[i].start));
		}
	}
	printk(KERN_WARNING PWR_DEVNAME "qotp_addr - could not map reg %08x\n", reg);
	return addr;
}

static int otp_readl(struct quasar_otp* qotp, u32 reg, unsigned* val)
{
	volatile u32 rv = 0;
	int ret = 0;
	volatile u8 __iomem* addr = otp_addr(qotp, reg);
	
	if(addr == NULL)
		return -1;
	rv = readl(addr);
	/*printk("readl %08X [%08X] -> %08X\n", reg, addr, rv);*/
	*val = (unsigned long)rv;
	return ret;
}

static int otp_writel(struct quasar_otp* qotp, u32 reg, u32 val)
{
	volatile u8 __iomem* addr = otp_addr(qotp, reg);
	if(addr == NULL)
		return -1;
	/* printk("writel %08X [%08X] <= %08x\n", reg, addr, val);*/
	writel(val, addr);
	return 0;
}

///////////////////////////////////////////////////////////////////////////
static u32 s_otp_read_shift;
static u32 s_otp_read_mask;
static int  s_otp_cached;

#define NANODELAY(cycles) /* EMPTY */
#define OTP_SIZE                                128

#define OTPCMD__NO_OPERATION                    0
#define OTPCMD__REDUNDACY_READ                  1
#define OTPCMD__UPDATE_BOOTSTRAPS_AND_RP_MAP    2
#define OTPCMD__POWER_DOWN                      3
#define OTPCMD__OTP_DATA_READ                   4
#define OTPCMD__SECURE_KEY_READ                 5
#define OTPCMD__OTP_WRITE                       6
#define OTPCMD__REDUNDACY_WRITE                 7
#define OTPCMD__BLANK_CHECK_TEST                8
#define OTPCMD__CRC_TEST                        9

#define OTP_MINIMUM_PULSE_WIDTH_NS              12
#define OTP_MAXIMUM_FREQUENCY_MHZ               (1000/OTP_MINIMUM_PULSE_WIDTH_NS)

#define CEILING_DIVIDE(n,d) (((n) + (d) - 1) / (d))
static void OTPsetTiming(struct quasar_otp* qotp)
{
    u32 write_strobe;
    u32 system_clock_mhz = 250;//BIOSgetSYSfrequency() / (1000*1000);
    u32 prescaler =
        CEILING_DIVIDE(system_clock_mhz, OTP_MAXIMUM_FREQUENCY_MHZ);

    if (prescaler > OTPTIM__PRSC__MASK)
    {
        prescaler = OTPTIM__PRSC__MASK;
    }

    otp_writel(qotp, OTPTIM, prescaler);

    // require (prescale*write_strobe)/system_clock_mhz = 12us
    write_strobe = (system_clock_mhz / prescaler) * 12;
#ifdef Q6300A0    
    if (write_strobe > OTP_T_WRITE_STROBE__PARAM__MASK)
    {
        write_strobe = OTP_T_WRITE_STROBE__PARAM__MASK;
    }
    otp_writel(qotp, OTP_T_WRITE_STROBE, write_strobe);
#endif    
}

static void OTPinit(struct quasar_otp* qotp)
{
    u32 status;
    int ret;

    OTPsetTiming(qotp);

    ret = otp_readl(qotp, OTPSTAT, &status);

    // Coming out of RESET, the RSTGEN state machine reads the first word
    // of OTP to get the OTP bootstraps. Make sure the read completed before
    // proceeding.
    if ((status & OTPSTAT__BSY__MASK) != 0)
    {
        // Wait until the OTP read is no longer busy
        do
        {
            ret = otp_readl(qotp, OTPSTAT, &status);
        } 
        while ((status & OTPSTAT__BSY__MASK) != 0);

        // Wait until OTP says the read is done
        while ((status & OTPSTAT__DONE__MASK) == 0)
        {
            ret = otp_readl(qotp, OTPSTAT, &status);
        }
    }

    // If RSTGEN did not clear the DONE bit (the normal chip behavior), clear
    // it now.
    // If RSTGEN is modified to clear the DONE bit, we don't have to clear it.
    if ((status & OTPSTAT__DONE__MASK) != 0)
    {
        otp_writel(qotp, OTPCMD, OTPCMD__DONECLR__MASK);

        // Wait until OTP acknowledges that DONE is cleared
        do
        {
            ret = otp_readl(qotp, OTPSTAT, &status);
        } 
        while ((status & OTPSTAT__DONE__MASK) != 0);
    }
}

static int OTPstartRead(struct quasar_otp* qotp, u16 byte, u8 count)
{
/// Always use 4-byte reads to match hardware implementation
#define LENGTH_4_BYTES 2
    static u16 s_last_byte;

    s_otp_read_shift = (byte & 0x3) << 3;
    byte &= ~0x3;

    switch (count)
    {
    case 1:
        s_otp_read_mask = 0xFF;
        break;
    case 2:
        s_otp_read_mask = 0xFFFF;
        break;
    case 4:
        s_otp_read_mask = 0xFFFFFFFF;
        break;
    default:
        printk("OTP count out of range {1,2,4}: %d\n", count);
        return -1;
    }
    
    if (byte >= OTP_SIZE)
    {
        printk("OTP byte number out of range [0..%d]: %d\n", OTP_SIZE, byte);
        return -3;
    }

    if (byte == s_last_byte && s_otp_cached)
    {
        // Skip OTP read
    }
    else
    {
        u32 add;

        s_last_byte  = byte;
        s_otp_cached = 0;
        add = byte;

        // LONGWORD unit of ADD for read command
        otp_writel(qotp, OTPCMD,
                     (OTPCMD__OTP_DATA_READ << OTPCMD__CMD__SHIFT) |
                     (LENGTH_4_BYTES << OTPCMD__LEN__SHIFT) |
                     ((add >> 1) << OTPCMD__ADD__SHIFT));
    }

    return 0;
}

static int OTPwaitForDone(struct quasar_otp* qotp)
{
    u32 status;
    int ret;

    if (s_otp_cached)
    {
        return 0;
    }

    do
    {
        ret = otp_readl(qotp, OTPSTAT, &status);
    } 
    while ((status & OTPSTAT__BSY__MASK) != 0);

    if ((status & OTPSTAT__DONE__MASK) != 0)
    {
        otp_writel(qotp, OTPCMD, OTPCMD__DONECLR__MASK);
        do
        {
            ret = otp_readl(qotp, OTPSTAT, &status);
        } 
        while ((status & OTPSTAT__DONE__MASK) != 0);
    }

    if ((status & OTPSTAT__ERR__MASK) != 0)
    {
        otp_writel(qotp, OTPCMD, OTPCMD__ERRCLR__MASK);
        do
        {
            ret = otp_readl(qotp, OTPSTAT, &status);
        } 
        while ((status & OTPSTAT__ERR__MASK) != 0);

        return 1;
    }

    if ((status & OTPSTAT__CMD__MASK) ==
        (OTPCMD__OTP_DATA_READ << OTPSTAT__CMD__SHIFT))
    {
        s_otp_cached = 1;
    }

    return 0;
}

static u32 OTPreadResults(struct quasar_otp* qotp)
{
    u32 result;
    int ret;
    
    ret = otp_readl(qotp, OTPRRES, &result);

    result >>= s_otp_read_shift;
    result &=  s_otp_read_mask;

    return result;
}

static int OTPread(struct quasar_otp* qotp,
            u16 byte,
            u8  count,
            u32 *p_value)
{
    int status = OTPstartRead(qotp, byte, count);

    if (status < 0)
    {
        return status;
    }

    if (OTPwaitForDone(qotp) != 0)
    {
        printk("OTPread: OTPwaitForDone reported error\n");
        return -6;
    }
    else
    {
        *p_value = OTPreadResults(qotp);
        return 0;
    }
}

static int OTPwrite(struct quasar_otp* qotp,
            u16 byte,
             u8  bit)
{
    int i;
    u32 add;

    if (byte >= OTP_SIZE)
    {
        printk("OTPwrite: byte number out of range [0..%d]: %d\n", OTP_SIZE, byte);
        return -3;
    }

    if (bit >= 8)
    {
        printk("OTPwrite: bit number out of range [0..7]: %d\n", bit);
        return -5;
    }

    s_otp_cached = 0; // Conservatively invalidate on write

    // For double bits write, add[11:7] is the bit address (32 bit), 
    // add[6:0] is the word address (128 words)
    add = byte / 2;
    add |= (bit << 8);
    add |= ((byte % 2) << 11);
    for (i=0; i < 32; i++)
    {
        int status;
        u32 value;

        // bit unit of ADD for write command
        otp_writel(qotp, OTPCMD,
                     (OTPCMD__OTP_WRITE << OTPCMD__CMD__SHIFT) |
                     (add << OTPCMD__ADD__SHIFT));

        if (OTPwaitForDone(qotp) != 0)
        {
            printk("OTPwrite: OTPwaitForDone reported error\n");
            return -6;
        }

        NANODELAY(16);
        status = OTPread(qotp, byte, 1, &value);
        if (status != 0)
        {
            printk("OTPwrite: readback failed, retry required\n");
        }
        else if ((value & (1 << bit)) == 0)
        {
            printk("OTPwrite: bit set failed, retry required\n");
            status = -1;
        }
        else
        {
            return 0;
        }

        NANODELAY(8);
        s_otp_cached = 0; // Conservatively invalidate on write
    }

    printk("OTPwrite: programming failed\n");

    return -1;
}
////////////////////////////////////////////////////////////////////////////

static int otp_writebit(struct quasar_otp* qotp, struct q_otp_io *otpio)
{
	return OTPwrite(qotp, otpio->byteOffset, otpio->bitindex);
}

static int otp_read(struct quasar_otp* qotp, struct q_otp_io *otpio)
{
	return OTPread(qotp, otpio->byteOffset, otpio->count, &otpio->val);
}
        
static int qotp_open(struct inode* inode, struct file* filp)
{
	struct quasar_otp *qotp;
	int minor;
	qotp = container_of(inode->i_cdev, struct quasar_otp, cdev);
	minor = iminor(inode);
	qotp->refs++;
	filp->private_data = qotp;
	return 0;
}

static int qotp_release(struct inode* inode, struct file* filp)
{
	struct quasar_otp* qotp;
	qotp = (struct quasar_otp*)filp->private_data;
	if(qotp->refs <= 0)
		return -EBADF;
	qotp->refs--;
	filp->private_data = NULL;
	return 0;
}

static long qotp_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct quasar_otp* qotp;
	struct q_otp_io ioval;
	int ret = 0;
	qotp = (struct quasar_otp*)filp->private_data;

	if(! qotp)
		return -ENODEV;

	switch(cmd)
	{
	case QOTP_WRITE_BIT:
		ret = copy_from_user(&ioval, (void*)arg, sizeof(struct q_otp_io));
		ret = otp_writebit(qotp, &ioval);
		break;
	case QOTP_READ:
		ret = copy_from_user(&ioval, (void*)arg, sizeof(struct q_otp_io));
		ret = otp_read(qotp, &ioval);
		ret = copy_to_user((void*)arg, &ioval, sizeof(struct q_otp_io));
		break;

	default:
		printk(KERN_WARNING PWR_DEVNAME "Bad ioctl %d\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static struct file_operations qotp_ops = {
	.owner		= THIS_MODULE,
	.open		= qotp_open,
	.release	= qotp_release,
	.unlocked_ioctl	= qotp_ioctl,
	.compat_ioctl	= qotp_ioctl 
};

static int __init qotp_probe(struct platform_device *pdev)
{
	struct resource	*ioreg;
    struct quasar_otp *qotp;
    static int ngpios=0;
	dev_t  qotpn;
	int ret = 0;
printk("=========== OTP probe ===============\n");
	qotp = kzalloc(sizeof(struct quasar_otp), GFP_KERNEL);
	if(! qotp) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	qotp->frdev = &pdev->dev;
	qotp->nioregs = 0;
	
    printk(KERN_WARNING PWR_DEVNAME "init qotp_probe ...\n"); 
	do {
		ioreg = platform_get_resource(pdev, IORESOURCE_MEM, qotp->nioregs);
		if (ioreg) {
			qotp->ioregs[qotp->nioregs].start = ioreg->start;
			qotp->ioregs[qotp->nioregs].end = ioreg->end;
			qotp->ioregs[qotp->nioregs].base =
				ioremap(ioreg->start, ioreg->end - ioreg->start + 4);
            //printk(KERN_WARNING PWR_DEVNAME "[%d]-- start=0x%08X, end=0x%08X, base=0x%08X\n", qotp->nioregs, ioreg->start, ioreg->end, qotp->ioregs[qotp->nioregs].base); 
			if (!qotp->ioregs[qotp->nioregs].base) {
				ret = -ENOMEM;
				dev_dbg(&pdev->dev, "could not map I/O memory %0d\n", qotp->nioregs);
				goto out_abort;
			}
			/*
			printk("qotp - mapped %px %x to %x at %x\n", 
					qotp, ioreg->start, ioreg->end, qotp->ioregs[qotp->nioregs].base);
			*/
			qotp->nioregs++;
		}
	} while (ioreg && (qotp->nioregs < MAX_IOREGS));

	cdev_init(&qotp->cdev, &qotp_ops);
	qotpn = MKDEV(QOTP_MAJOR, ngpios);
	ret = cdev_add(&qotp->cdev, qotpn, 1);
	if (ret) {
		printk(KERN_WARNING "qotp - could not create char dev %d\n", ngpios);
		ret = -ENODEV;
		goto out_abort;
	}
	ngpios++;
	ret = 0;
    OTPinit(qotp);
    
	platform_set_drvdata(pdev, qotp);
	device_init_wakeup(&pdev->dev, 1);
    
	dev_info(&pdev->dev, "Quasar QOTP\n");
	return 0;

out_abort:
	while (qotp->nioregs > 0) {
		qotp->nioregs--;
		iounmap(qotp->ioregs[qotp->nioregs].base);
	}

	kfree(qotp);
	return ret;
}

static int __exit qotp_remove(struct platform_device *pdev)
{
	struct quasar_otp *qotp = platform_get_drvdata(pdev);
printk("===== qotp remove - init\n");
	cdev_del(&qotp->cdev);
	while (qotp->nioregs > 0) {
		qotp->nioregs--;
		iounmap(qotp->ioregs[qotp->nioregs].base);
	}

	kfree(qotp);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

MODULE_ALIAS("platform:qotp");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-otp" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver qotp_driver_ops = {
	.probe		= qotp_probe,
	.remove		= qotp_remove,
	.driver		= {
		.name	= "quasar-otp",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init qotp_init(void)
{
	int ret;
printk("===== qotp - init\n");
		ret = platform_driver_register(&qotp_driver_ops);
	return ret;
}
module_init(qotp_init);

static void __exit qotp_exit(void)
{
printk("===== qotp - exit\n");    
	platform_driver_unregister(&qotp_driver_ops);
}
module_exit(qotp_exit);

MODULE_DESCRIPTION("Quasar OTP Driver");
MODULE_LICENSE("Dual BSD/GPL");

