/*
 * A driver for allocation of large blocks of contiguous physical memory for
 * Quatro processors
 *
 * Quasar large blocks memory kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2021/07/30 10:39:17 $
//  $Change: 51780 $
//
// =========================================================
//mknod /dev/qmem c 212 0

#define USING_VOS_DPL 2 //0-not use VOS POOL, 1-use vos_mtmPool(multi memory pool), 2-use vos memory pool

#if (USING_VOS_DPL == 0)
	#include "qmem.c.old"
#else

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>
#include "../vos/vos_if.h"

#ifndef SCRATCHPAD
	#define SCRATCHPAD      MAIN_SP
#endif

#define SCRATCHPAD_PA SCRATCHPAD
static int debug_mmap = 0;
module_param(debug_mmap, int, 0644);
MODULE_PARM_DESC(debug_mmap, "allow mmap of entire register space");

static int no_freelist = 0;
module_param(no_freelist, int, 0644);
MODULE_PARM_DESC(no_freelist, "free blocks to system, not to ourselves as a cache");

static int extra_reserve = 0;
module_param(extra_reserve, int, 0644);
MODULE_PARM_DESC(extra_reserve, "amount extra bytes to add to system reserved (for IPS for example)");


#define PMX_QMEM_POOL_SIZE (88*1024*1024) //added by Bob 80M
#define PMX_QMEM_POOL_START_PHY_ADDR 0x9A700000
struct mem_quasar *g_qmem = NULL;


/* Systems build with PAE configured will have a 64 bit physical address type
 * The only use of addresses with more than 32 bits is the mrHIGH alloc type
 * which keeps the upper bits in the "logiAddr" field and the physAddr field
 * will have the lower 32 bits.  We don't ever expect to map high mem allocs
 * via this driver so they will never be in cache
 */

#define QMEM_DEVNAME	"qmem"

struct mem_rec {
	enum { mrKERN, mrDMA, mrSPAD, 
#if defined(Q6600)
	mrJBSP,
#endif
	mrHIGH } type;
	struct file*		filp;
	struct list_head	alloclist;
	size_t		logiAddr;	/* for mrHIGH, upper 32 of phys addr */
	size_t		virtAddr;	/* for mrHIGH, always 0 */
	size_t		physAddr;	/* lower 32 bits of phys addr */
	struct page		*pages;
	int			size;
	int			actsize;
	int			cached;
};

struct mem_quasar {
	int			ref;
	int			minor;
	struct cdev		cdev;
	int			maxblock;	// largest contiguous block
	u32			curalloc;	// how much is currently alloced 
	u32			maxalloc;	// largest total alloced
	struct list_head	allocs;		// list of current allocations
	struct list_head	frees;		// list of freed max-size blocks
	spinlock_t		lock;
	HVOSMTP hVosMtp;
	
	unsigned long		spadalloc;
	unsigned long		spadsize;
	unsigned long		spadused;
	unsigned long		spadbase_pa;
	u8 __iomem*		spadbase_va;
};

static int qmem_open(struct inode* inode, struct file* filp)
{
	struct mem_quasar *mem;
	
	mem = container_of(inode->i_cdev, struct mem_quasar, cdev);
	mem->ref++;
	mem->minor = iminor(inode);
	filp->private_data = mem;
	return 0;
}

static int qmem_release(struct inode* inode, struct file* filp)
{
	struct mem_quasar *mem;
	struct mem_rec    *rec = NULL;
	struct list_head  *recl, *q;
	unsigned long  flags;
	unsigned long  va;
	int            pages;

	// vosPrintf("qmem_release............\r\n");
	mem = container_of(inode->i_cdev, struct mem_quasar, cdev);
	if(mem->ref <= 0)
		return -EFAULT;
	
	list_for_each_safe(recl, q, &mem->allocs) {
		rec = list_entry(recl, struct mem_rec, alloclist);
		if(rec->filp == filp || mem->ref == 1) {
			spin_lock_irqsave(&mem->lock, flags);
			list_del(recl);
			if(rec->type == mrSPAD)
			{ /* spad */
					if(rec->logiAddr == mem->spadalloc + mem->spadused - rec->actsize) {
						/* the end of the blocks, easy to free */
						mem->spadused -= rec->actsize;
					}
					else {
						printk("qmem ERROR - spad allocs must be alloced/freed in LIFO order\n");
					}
				}
				spin_unlock_irqrestore(&mem->lock, flags);
				kfree(rec);
		}
		if(rec == NULL)
		{
			
		}
	}
	mem->ref--;
	if (mem->ref == 0)
	{
	#if (USING_VOS_DPL == 1)	
		vos_mtpFreeAll(mem->hVosMtp);
	#else
		vos_mplFreeAll(mem->hVosMtp);
	#endif	
		
	}
	if (mem->ref == 0) {
		/* no more references to this, free up any free blocks */
		list_for_each_safe(recl, q, &mem->frees) {
			rec = list_entry(recl, struct mem_rec, alloclist);
			spin_lock_irqsave(&mem->lock, flags);
			list_del(recl);
			va = rec->logiAddr;
			for(pages = rec->actsize / PAGE_SIZE; pages > 0; pages--) {
				ClearPageReserved(virt_to_page(va));
				va += PAGE_SIZE;
			}
			kfree((void*)rec->logiAddr);
			kfree(rec);
			spin_unlock_irqrestore(&mem->lock, flags);
		}
		mem->spadused = 0;
	}
	filp->private_data = NULL;
	return 0;
}


static ssize_t qmem_read(struct file* filp, char __user *buffer, size_t length, loff_t* offset)
{
	struct mem_quasar *mem;	
	
	mem = (struct mem_quasar*)filp->private_data;
	return -EINVAL;
}

static ssize_t qmem_write(struct file* filp, const char __user *buffer, size_t length, loff_t* offset)
{
	struct mem_quasar *mem;	
	
	mem = (struct mem_quasar*)filp->private_data;
	return -EINVAL;
}

static u32 qmem_getUsedSize(struct mem_quasar *mem)
{
	u32 usedSize=0;
	
	usedSize = vos_mtpGetUsedSize(mem->hVosMtp);
	return usedSize;
}

static long qmem_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct mem_quasar *mem;	
	struct q_mem_alloc qma;
	void *pkVirAddr = NULL;
	int ret = 0, flag=0;
	/* int localloced, localsmalls; */
	unsigned long va, pa;
	int spadsize = 0;
	unsigned long flags;
	struct mem_rec    *rec, *item;
	struct list_head  *recl, *q;
	
	mem = (struct mem_quasar*)filp->private_data;
	
	switch(cmd)
	{		
		case 0x9cf1: //show qmem info
		ret = copy_from_user((void*)&flag, (void*)arg, 4);
		#if (USING_VOS_DPL == 1)		
			mem->curalloc = vos_mtpGetUsedSize(mem->hVosMtp);
			vosPrintf("hVosMtp(0x%x), ref=%d, minor=%d, curalloc=0x%x\r\n", mem->hVosMtp, mem->ref, mem->minor, mem->curalloc);
			vos_mtpShow(mem->hVosMtp, flag);
		#else
			vos_mplInfo(mem->hVosMtp, NULL, flag);
		#endif
			break;
	
		case QMEMGETMAXALLOCSIZE:
			put_user(mem->maxblock, (unsigned long*)arg);
			break;
	
		case QMEMGETCURALLOCS:
		#if (USING_VOS_DPL == 1)
			mem->curalloc = vos_mtpGetUsedSize(mem->hVosMtp);
		#else
			vos_mplGetSize(mem->hVosMtp, NULL, &mem->curalloc, NULL);
		#endif
			put_user(mem->curalloc, (unsigned long*)arg);
			break;
		
		case QMEMGETMAXALLOCS:
			put_user(mem->maxalloc, (unsigned long*)arg);
			break;
	
		case QMEMGETAVAILABLE:		/* get just what's already alloced in free list */
		case QMEMGETTOTALAVAILABLE:	/* get total avail ram (free list + system low + system high) */
		case QMEMGETLOWAVAILABLE:	/* get low avail (free list + system low) */
		case QMEMGETHIGHAVAILABLE:	/* get avail system high memory */
		case QMEMGETSPADAVAILABLE:
			
		#if (USING_VOS_DPL == 1)
			mem->curalloc = vos_mtpGetUsedSize(mem->hVosMtp);
		#else
			vos_mplGetSize(mem->hVosMtp, NULL, &mem->curalloc, NULL);
		#endif
			put_user((mem->maxalloc - mem->curalloc), (unsigned long*)arg);
			break;
		
		case QMEMALLOC:
		case QMEMALLOCSPAD:
		case QMEMALLOCHIGH:
			/* validate parms */
			ret = copy_from_user((void*)&qma, (void*)arg, sizeof(struct q_mem_alloc));
			if(qma.req_size > mem->maxblock)
				return -EINVAL;
			if(cmd != QMEMALLOCSPAD)
			{
				qma.ret_size = qma.req_size;
			#if (USING_VOS_DPL == 1)	
				pkVirAddr = vos_mtpMalloc(mem->hVosMtp, qma.req_size, 32, qma.strFunc, qma.numLine);
			#else
				pkVirAddr = vos_mplAlloc(mem->hVosMtp, qma.req_size, 32, qma.strFunc, qma.numLine);
			#endif
				/* return the physaddr up to user */
				qma.ret_addr = vos_mplGetPhyAddr(1, pkVirAddr);
				//vosPrintf("[qmem]QMEMALLOC: q_size=%d, pkVirAddr=0x%x, phy_addr=0x%x\r\n", qma.req_size, pkVirAddr, qma.ret_addr);
				if (copy_to_user((void*)arg, (void*)&qma, sizeof(struct q_mem_alloc)))
					return -EFAULT;
			}
			if(cmd == QMEMALLOCSPAD)
			{	
				
				/* create a record for this alloc */
				rec = kmalloc(sizeof(struct mem_rec), GFP_KERNEL);
				if(! rec)
					return -ENOMEM;
				/* stuff filep into rec as key to auto-free on close */
				rec->filp = filp; 
				
				INIT_LIST_HEAD(&rec->alloclist);
				rec->size	 = qma.req_size;
				
				/* enforce page size alloc size chunking */
				qma.ret_size = qma.req_size;
				qma.ret_size += PAGE_SIZE-1;
				qma.ret_size &= PAGE_MASK;
				rec->actsize = qma.ret_size;
				rec->cached  = qma.cached;
				spadsize = mem->spadsize;
				if((rec->actsize + mem->spadused) > spadsize) 
				{
					printk("qmem: %d bytes not available in spad\n", rec->actsize);
					kfree(rec);
					return -ENOMEM;
				}
				spin_lock_irqsave(&mem->lock, flags);
				rec->type     = mrSPAD;
				rec->virtAddr = 0;
				rec->physAddr = SCRATCHPAD_PA + mem->spadused;
				rec->logiAddr = (unsigned long)(mem->spadbase_va + mem->spadused);
				mem->spadused += rec->actsize;
				spin_unlock_irqrestore(&mem->lock, flags);
		
				/* enlist this record */
				spin_lock_irqsave(&mem->lock, flags);		
				list_add_tail(&rec->alloclist, &mem->allocs);
				spin_unlock_irqrestore(&mem->lock, flags);
				/* return the physaddr up to user */
				qma.ret_addr = (void*)rec->physAddr;
				if (copy_to_user((void*)arg, (void*)&qma, sizeof(struct q_mem_alloc)))
					return -EFAULT;
			}
			break;
		
	case QMEMFREE:
		get_user(pa, (unsigned long*)arg);

		spin_lock_irqsave(&mem->lock, flags);
		/* find location of alloc in list */
		rec = NULL;
		list_for_each_safe(recl, q, &mem->allocs) 
		{
			item = list_entry(recl, struct mem_rec, alloclist);
			if(item->physAddr == pa) {
				list_del(recl);
				rec = item;
				break;
			}
		}
		if(rec == NULL)
		{
		#if (USING_VOS_DPL == 1)	
			pkVirAddr = vos_mtpGetKvirAddr(mem->hVosMtp, (void *)pa);
			if(NULL == pkVirAddr)
			{
				printk("ERROR: QMEMFREE phyAddr=0x%x\n", pa);
				break;
			}
			vos_mtpFree(mem->hVosMtp, pkVirAddr);
			//vosPrintf("[qmem]QMEMFREE: pkVirAddr=0x%x\r\n", pkVirAddr);
		#else
			pkVirAddr = vos_mplGetKvirAddr(mem->hVosMtp, (void *)pa);
			if(NULL == pkVirAddr)
			{
				printk("ERROR: QMEMFREE phyAddr=0x%x\n", pa);
				break;
			}
			vos_mplFree(mem->hVosMtp, pkVirAddr);
		#endif
		}
		if(rec)
		{
			if(rec->type == mrSPAD) 
			{
				if(rec->logiAddr == mem->spadalloc + mem->spadused - rec->actsize) {
					/* the end of the blocks, easy to free */
					mem->spadused -= rec->actsize;
					kfree(rec);
				}
				else {
					printk("qmem ERROR - spad allocs must be alloced/freed in LIFO order\n");
				}
			}
			ret = 0;
		}		
		spin_unlock_irqrestore(&mem->lock, flags);
		break;

	case QMEMFLUSH:
	case QMEMFLUSHINVALIDATE:
		ret = copy_from_user((void*)&qma, (void*)arg, sizeof(struct q_mem_alloc));
		if (qma.cached != 0)
			dma_sync_single_for_device(0, (dma_addr_t)qma.ret_addr, qma.ret_size,
				cmd == QMEMFLUSH ? DMA_TO_DEVICE : DMA_BIDIRECTIONAL);
		break;
	case QMEMINVALIDATE:
		ret = copy_from_user((void*)&qma, (void*)arg, sizeof(struct q_mem_alloc));
		if (qma.cached != 0)
			dma_sync_single_for_cpu(0, (dma_addr_t)qma.ret_addr, qma.ret_size, DMA_FROM_DEVICE);
		break;
	default:
		printk("qmem - bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}

	return ret;
}

static int qmem_mmap(struct file* filp, struct vm_area_struct* vma)
{
	struct mem_quasar *mem;	
	struct mem_rec    *rec, *item;
	struct list_head  *recl;
	unsigned long size, start;
	unsigned long flags;
	int ret = 0;
	mem = (struct mem_quasar*)filp->private_data;
	
	if(! mem) return -ENODEV;
	
	start = vma->vm_pgoff * PAGE_SIZE;
	size  = vma->vm_end - vma->vm_start;

	spin_lock_irqsave(&mem->lock, flags);
	rec = NULL;
	list_for_each(recl, &mem->allocs) {
		item = list_entry(recl, struct mem_rec, alloclist);
		if(item->physAddr == start) {
			/*printk("qmem - mmap phys %08lx\n", rec->physAddr);*/
			rec = item;
			break;
		}
	}
	spin_unlock_irqrestore(&mem->lock, flags);
	if (! rec) 
	{
		/* Map part of Quasar register space if requested for mapping things
		   outside of any particular drivers address space
		*/
		if (debug_mmap) {
			/*printk("qmem - mmap PHYS %08lx\n", start);*/
			vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
			ret = remap_pfn_range(
					vma,
					vma->vm_start,
					vma->vm_pgoff,
					vma->vm_end - vma->vm_start,
					vma->vm_page_prot
			);
			return ret;
		}
		//printk("qmem - mmap: no such alloc %08lx\n", start);
		return -ENODEV;
	}

	if(item->type != mrSPAD)
	{
		printk("qmem - alloc mmap size only support SPAD\n");
		return -EINVAL;
	}
	/* make sure user knows what they are dealing with, allow only mmapping of
	 * entire block
	 */
	if (size != (unsigned long)rec->actsize) {
		printk("qmem - alloc mmap size %lu != allocated size:%d\n", size, rec->actsize);
		return -EINVAL;
	}
	/* if this block has already been mmapped, don't allow another mmapping
	*/
	if (rec->virtAddr != 0) {
		printk("qmem - block at %08lx already mapped\n", (unsigned long)rec->physAddr);
		return -EINVAL;
	}
	/* remember mapped address until its unmappped */
	rec->virtAddr = vma->vm_start;

	if (! rec->cached) {
		/* mark uncached pages as uncached */
		vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	}
	// workaround for IPM/DSP valid access of cacheable memory 
	else {
		//vma->vm_page_prot = pgprot_dmacoherent(vma->vm_page_prot);
	}

	ret = remap_pfn_range(
				vma,
				vma->vm_start,
				rec->physAddr >> PAGE_SHIFT,
				size,
				vma->vm_page_prot
			);
	return ret;
}

static struct file_operations quasar_mem_ops = {
	.owner		= THIS_MODULE,
	.open		= qmem_open,
	.release	= qmem_release,
	.read		= qmem_read,
	.write		= qmem_write,
	.mmap		= qmem_mmap,
	.unlocked_ioctl	= qmem_ioctl
};

static int __init quasar_mem_probe(struct platform_device *pdev)
{
	struct mem_quasar *mem;
	dev_t  memn;
	int    ret;
	static int nmems = 0;
	struct sysinfo info;

	mem = kzalloc(sizeof(struct mem_quasar), GFP_KERNEL);
	if (!mem) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	cdev_init(&mem->cdev, &quasar_mem_ops);
	spin_lock_init(&mem->lock);
	INIT_LIST_HEAD(&mem->allocs);
	INIT_LIST_HEAD(&mem->frees);
	memn = MKDEV(QMEM_MAJOR, nmems);
	ret = cdev_add(&mem->cdev, memn, 1);
	if (ret) {
		dev_dbg(&pdev->dev, "could not create char dev %d\n", nmems);
		kfree(mem);
		return ret;
	}
	nmems++;
	platform_set_drvdata(pdev, mem);

	mem->ref = 0;
	mem->maxblock = 0x400000;
	mem->maxalloc = PMX_QMEM_POOL_SIZE;
	mem->curalloc = 0;
	g_qmem = mem;

#if (USING_VOS_DPL == 1)
	//USING_EXTERNAL_NODE and using vos_kmalloc/vos_kmalloc_coherent
	mem->hVosMtp = vos_mtpCreatePool("qmem", 2, 1, mem->maxalloc/mem->maxblock, mem->maxblock);
#else
{
	#if 0
	void *dplKvAddr;
	unsigned int nPhyStartAddr = 0X9f000000, nPoolSize = (16*1024*1024);
	
	dplKvAddr = (unsigned long)ioremap(nPhyStartAddr, nPoolSize);
	if(NULL == dplKvAddr)
	{
		dev_dbg(&pdev->dev, "ioremap fail\n");
		return -ENOMEM;;
	}
	
	mem->hVosMtp = vos_mplCreatePoolPhy("qmem_pool", 2, (void *)dplKvAddr, nPoolSize, nPhyStartAddr, 1);
	#else
	mem->hVosMtp = vos_mplCreatePoolPhy("qmem", 2, 0, PMX_QMEM_POOL_SIZE, PMX_QMEM_POOL_START_PHY_ADDR, 1);
	#endif
}
#endif
	mem->spadused  = 0;
	mem->spadsize  = MAIN_SP_END - MAIN_SP + 1;
	mem->spadbase_pa = SCRATCHPAD_PA;
	mem->spadbase_va = ioremap(mem->spadbase_pa, mem->spadsize);
	mem->spadalloc = (unsigned long)mem->spadbase_va;

	return 0;
}

void qmem_show(int flag)
{
	struct mem_quasar *mem = g_qmem;
	if(mem == NULL)return;
	
#if (USING_VOS_DPL == 1)		
	mem->curalloc = vos_mtpGetUsedSize(mem->hVosMtp);
	vosPrintf("hVosMtp(0x%x), ref=%d, minor=%d, curalloc=0x%x, maxalloc=0x%x\r\n", mem->hVosMtp, mem->ref, mem->minor, mem->curalloc, mem->maxalloc);
	vos_mtpShow(mem->hVosMtp, flag);
#else
	vos_mplGetSize(mem->hVosMtp, NULL, &mem->curalloc, NULL);
	vosPrintf("hVosMtp(0x%x), ref=%d, minor=%d, curalloc=0x%x, maxalloc=0x%x\r\n", mem->hVosMtp, mem->ref, mem->minor, mem->curalloc, mem->maxalloc);
	vos_mplInfo(mem->hVosMtp, NULL, 2);
#endif
}

static int __exit quasar_mem_remove(struct platform_device *pdev)
{
	struct mem_quasar *mem = platform_get_drvdata(pdev);
	struct mem_rec    *rec;
	unsigned long  va;
	int            pages;

	vosPrintf("quasar_mem_remove............\r\n");
	
	cdev_del(&mem->cdev);	

#if (USING_VOS_DPL == 1)	
	vos_mtpDestroyPool(mem->hVosMtp);
#else
{
	void *pKvAddrStart;
	
	pKvAddrStart = (void *)vos_mplGetPoolAddrStart(mem->hVosMtp);
	vos_mplDestroy(mem->hVosMtp);
	if(pKvAddrStart)iounmap(pKvAddrStart);
}
#endif

	while (!list_empty(&mem->allocs)) 
	{
		rec = list_entry(mem->allocs.next, struct mem_rec, alloclist);
		list_del_init(&rec->alloclist);
		/*
		printk("qmem: freeing alloced block rec %lx pa=%lx (log=%lx) of %d bytes type:%d\n",
				rec, rec->physAddr, rec->logiAddr, rec->actsize, rec->type);
		*/
		kfree(rec);
	}
	while (!list_empty(&mem->frees)) 
	{
		rec = list_entry(mem->frees.next, struct mem_rec, alloclist);
		list_del_init(&rec->alloclist);
		/*
		printk("qmem: freeing alloced block rec %lx pa=%lx (log=%lx) of %d bytes type:%d\n",
				rec, rec->physAddr, rec->logiAddr, rec->actsize, rec->type);
		*/
		
		kfree(rec);
	}
	
	kfree(mem);

	platform_set_drvdata(pdev, NULL);

	return 0;
}

MODULE_ALIAS("platform:quasar-mem");

static struct resource quasar_qmem_resources[] = {
};

static void qmem_dev_release(struct device *dev)
{
	/* no-op */
}

static u64 quasar_qmem_dmamask = 0xffffffffUL;

static struct platform_device quasar_qmem_device_ops = {
	.name		= "quasar-mem",
	.id			= 0,
	.dev		= {
		.release	= qmem_dev_release,
		.dma_mask	= &quasar_qmem_dmamask,
		.coherent_dma_mask = 0xffffffff,
	},
	.num_resources	= ARRAY_SIZE(quasar_qmem_resources),
	.resource	= quasar_qmem_resources,
};

static struct platform_driver quasar_mem_driver_ops = {
	.probe		= quasar_mem_probe,
	.remove		= quasar_mem_remove,
	.driver		= {
		.name	= "quasar-mem",
		.owner	= THIS_MODULE,
	},
};

static int __init quasar_mem_init(void)
{
	int ret;
	
	ret = platform_device_register(&quasar_qmem_device_ops);
	if(!ret)
		ret = platform_driver_register(&quasar_mem_driver_ops);
	return ret;
}
module_init(quasar_mem_init);

static void __exit quasar_mem_exit(void)
{
	platform_driver_unregister(&quasar_mem_driver_ops);
	platform_device_unregister(&quasar_qmem_device_ops);
}
module_exit(quasar_mem_exit);

MODULE_DESCRIPTION(" Quasar QMEM driver");
MODULE_LICENSE("Dual BSD/GPL");

#endif

