/*
 * A driver for the Cortex M3 processors in some Quasar SOCs
 *
 * Quasar CM3 I/F kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/03/31 10:38:25 $
//  $Change: 59512 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

static int irq_mask = 0;
module_param(irq_mask, int, 0644);
MODULE_PARM_DESC(irq_mask, "disable mbox interrupts (bits 2:0 = Mbox 2, 1, 0)");

static int dbg_ints = 0;
module_param(dbg_ints, int, 0644);
MODULE_PARM_DESC(dbg_ints, "print mbox interrupt info");

#define M3_DEVNAME	"qm3"

#define MAX_M3S  			2 /* get this from a header! */
#define MAX_M3IRQS    		3
#define MAX_M3IOREGS		4
#define MAX_M3ARGS			32
#define QM3_MAX_MSG_ENTRIES 16
#define MAX_M3_QUEUE		64

struct quasar_io_region {
	u64						start;
	u64						end;
	volatile u8	__iomem		*base;
};

struct quasar_m3 {
	struct cdev				cdev;
	struct device			*frdev;

	struct quasar_io_region	ioregs[MAX_M3IOREGS];
	int						nioregs;
	
	unsigned long			irqs[MAX_M3IRQS];
	int						nirqs;
	
	int						refs;				/* reference count */
	int						minor;				/* device minor number */
	
	wait_queue_head_t		m3_rdq, m3_ntq;		/* wait-queue for m3 I/O */

	struct q_m3_io			responses[MAX_M3_QUEUE];
	int						m3_rhead, m3_rtail;

	struct q_m3_io			notifications[MAX_M3_QUEUE];
	int						m3_nhead, m3_ntail;	
	
	spinlock_t				lock;	/* lock for m3 mail ring buffer access */
	
	int						ordinal;
};

static int g_nm3s = 0;

static volatile u8 __iomem* m3_addr(struct quasar_m3* qm3, u32 reg)
{
	volatile u8 __iomem* addr = 0;
 	int      i;

	//printk(KERN_WARNING M3_DEVNAME "m3_addr - m3addr map, reg %08x, ord=%d, <%08X-%08X>--<%08X-%08X>--<%08X-%08X>--<%08X-%08X>\n", 
	//       reg, qm3->minor, qm3->ioregs[0].start, qm3->ioregs[0].end, qm3->ioregs[1].start, qm3->ioregs[1].end, qm3->ioregs[2].start, qm3->ioregs[2].end, qm3->ioregs[3].start, qm3->ioregs[3].end);
	/* check for existing mapping */
	for (i = 0; i < qm3->nioregs; i++) {
		if(reg >= qm3->ioregs[i].start && reg < qm3->ioregs[i].end) {
			return(qm3->ioregs[i].base + (reg - qm3->ioregs[i].start));
		}
	}
	printk(KERN_WARNING M3_DEVNAME "m3_addr - could not map reg %08x\n", reg);
	return addr;
}

static int m3_readl(struct quasar_m3* qm3, u32 reg, unsigned* val)
{
	volatile u32 rv = 0;
	int ret = 0;
	volatile u8 __iomem* addr = m3_addr(qm3, reg);
	
	if(addr == NULL)
		return -1;
	rv = readl(addr);
	/*printk("readl %08X [%08X] -> %08X\n", reg, addr, rv);*/
	*val = (unsigned long)rv;
	return ret;
}

static int m3_writel(struct quasar_m3* qm3, u32 reg, u32 val)
{
	volatile u8 __iomem* addr = m3_addr(qm3, reg);
	if(addr == NULL)
		return -1;
	/* printk("writel %08X [%08X] <= %08x\n", reg, addr, val);*/
	writel(val, addr);
	return 0;
}

static int qm3_rsp_wait(struct quasar_m3* qm3, int block)
{
	while (qm3->m3_rhead == qm3->m3_rtail)
	{
		if(! block)
			return -EAGAIN;

		if (wait_event_interruptible(qm3->m3_rdq, (qm3->m3_rhead != qm3->m3_rtail)))
			return -ERESTARTSYS;
	}
	return 0;
}

static int qm3_rsp_read(struct quasar_m3* qm3, struct q_m3_io* reply)
{
	unsigned long flags;

	if(qm3->m3_rhead == qm3->m3_rtail) {
		return -EAGAIN;
	}
	memcpy((char*)reply, &qm3->responses[qm3->m3_rtail], sizeof(struct q_m3_io));
	
	spin_lock_irqsave(&qm3->lock, flags);
	qm3->m3_rtail++;
	if(qm3->m3_rtail >= MAX_M3_QUEUE)
		qm3->m3_rtail = 0;
	spin_unlock_irqrestore(&qm3->lock, flags);
	return 0;	
}	

static int qm3_ntf_wait(struct quasar_m3* qm3, int block)
{
	while (qm3->m3_nhead == qm3->m3_ntail)
	{
		if(! block)
			return -EAGAIN;

		if (wait_event_interruptible(qm3->m3_ntq, (qm3->m3_nhead != qm3->m3_ntail)))
			return -ERESTARTSYS;
	}
	return 0;
}

/*  return -EAGAIN (-11): nothing in the queue
                 0      : success
*/
static int qm3_ntf_read(struct quasar_m3* qm3, struct q_m3_io* reply)
{
	unsigned long flags;

	if(qm3->m3_nhead == qm3->m3_ntail)
		return -EAGAIN;

	memcpy((char*)reply, &qm3->notifications[qm3->m3_ntail], sizeof(struct q_m3_io));
	/*printk("readnote=%02X\n", reply->val[0]);*/
	spin_lock_irqsave(&qm3->lock, flags);
	qm3->m3_ntail++;
	if(qm3->m3_ntail >= MAX_M3_QUEUE)
		qm3->m3_ntail = 0;
	spin_unlock_irqrestore(&qm3->lock, flags);
	return 0;
}


static void qm3_add_rsp(struct quasar_m3 *qm3,
		unsigned char msg, unsigned int *v, int nv)
{
	unsigned long flags;
    int i;

	qm3->responses[qm3->m3_rhead].msg  = msg;
	qm3->responses[qm3->m3_rhead].args = nv;
	if(dbg_ints)
		printk("add resp 0x%02X a0=%02X 1=%02X 2=%02X 3=%02X\n", msg,
				v[0], v[1], v[2], v[3]);
    for (i = 0; i < nv && i < QM3_MAX_ARGS; i++)
        qm3->responses[qm3->m3_rhead].val[i] = v[i];
	spin_lock_irqsave(&qm3->lock, flags);
	qm3->m3_rhead++;
	if(qm3->m3_rhead >= MAX_M3_QUEUE)
		qm3->m3_rhead = 0;
	if(qm3->m3_rhead == qm3->m3_rtail) {
		/* overrun of rsp buffer!! */
		printk(KERN_WARNING M3_DEVNAME "add-response overrun\n");
		qm3->m3_rtail++;
		if(qm3->m3_rtail >= MAX_M3_QUEUE)
			qm3->m3_rtail = 0;
	}
	spin_unlock_irqrestore(&qm3->lock, flags);
	wake_up_interruptible(&qm3->m3_rdq);
}

static void qm3_add_ntf(struct quasar_m3* qm3,
		unsigned char msg, unsigned int *v, int nv)
{
	unsigned long flags;
    int i;

	qm3->notifications[qm3->m3_nhead].msg  = msg;
	qm3->notifications[qm3->m3_nhead].args = nv;
	if(dbg_ints)
		printk("add nty 0x%02X 0=%02X 1=%02X 2=%02X 3=%02X\n", msg,
				v[0], v[1], v[2], v[3]);
    for (i = 0; i < nv && i < QM3_MAX_ARGS; i++)
        qm3->notifications[qm3->m3_nhead].val[i] = v[i];
	spin_lock_irqsave(&qm3->lock, flags);
	qm3->m3_nhead++;
	if(qm3->m3_nhead >= MAX_M3_QUEUE)
		qm3->m3_nhead = 0;
	if(qm3->m3_nhead == qm3->m3_ntail) {
		/* overrun of ntf buffer!! */
		printk(KERN_WARNING M3_DEVNAME "add-notification overrun\n");
		qm3->m3_ntail++;
		if(qm3->m3_ntail >= MAX_M3_QUEUE)
			qm3->m3_ntail = 0;
	}
	spin_unlock_irqrestore(&qm3->lock, flags);
	wake_up_interruptible(&qm3->m3_ntq);
}

static irqreturn_t qm3_interrupt(int irq, void *dev_id)
{
	struct quasar_m3 *qm3 = (struct quasar_m3 *)dev_id;
	
	u32 mboxstat, mboxmail, mboxmcnt;
	u32 entries, msgcnt;
	u32 vals[QM3_MAX_ARGS];
	u32 header, msgtype, msgid, msgcount;
	int ret, i;

	switch(qm3->ordinal)	{
    case 0:
#if defined(Q6300) || defined(Q6600)
        mboxstat = QBM3_0_MBS1;
        mboxmail = QBM3_0_MB1;
        mboxmcnt = QBM3_0_MBMC1;
#endif		
		break;
	case 1:
#if defined(Q6300) || defined(Q6600)
        mboxstat = QBM3_1_MBS1;
        mboxmail = QBM3_1_MB1;
        mboxmcnt = QBM3_1_MBMC1;
#endif 	
		break;
	default:
		printk(M3_DEVNAME "Spurious irq %d\n", irq);
		return IRQ_HANDLED;
	}

	do {
		m3_readl(qm3, mboxstat, &entries);
		msgcnt = (entries >> 16) & 0x1F;		
		entries &= 0x1F;
	
		if (! msgcnt || ! entries) {
			break;
		}
		/* Header format
	 	*	 Message Type |  Message Id  | Sequence No. | Message Size |
	 	*		0xAA : for commands and responses
	 	*		0x55 : for notification
		 */
		m3_readl(qm3, mboxmail, &header);
		entries--;
		msgtype  = header >> 24;
		msgid	= (header >> 16) & 0xFF;
		msgcount = (header & 0xFF); /* number of dwords, excluding header */

		if (entries < msgcount) {
			printk(M3_DEVNAME "protocol error: only %d of expected %d\n", entries, msgcount);
		}
		/* read msg arguments into buffer
		*/
		for (i = 0; i < (msgcount - 1) && (entries > 0) && (i < QM3_MAX_ARGS); i++) {
	 	   m3_readl(qm3, mboxmail, &vals[i]);
			entries--;
		}
		/* read last msg word and dec msg count
		*/
		m3_readl(qm3, mboxmcnt, &vals[i]);
		entries--;

		if (msgtype == 0xAA /* response */)
			qm3_add_rsp(qm3, msgid, vals, msgcount);
		else
			qm3_add_ntf(qm3, msgid, vals, msgcount);
	}
	while (entries > 0);
	ret = IRQ_HANDLED;
	return ret;
}

static unsigned int qm3_poll(struct file *filp, poll_table *wait)
{
	struct quasar_m3* qm3;	
	unsigned int mask = 0;
	unsigned long flags;
	
	/* polling only cares if there are NOTIFICATIONs available
	 */
	qm3 = (struct quasar_m3*)filp->private_data;

	poll_wait(filp, &qm3->m3_ntq, wait);
	spin_lock_irqsave(&qm3->lock, flags);
	if(qm3->m3_nhead != qm3->m3_ntail) {
		mask |= POLLIN | POLLRDNORM; /* readable */
	}
	spin_unlock_irqrestore(&qm3->lock, flags);
	return mask;
}

static int qm3_open(struct inode* inode, struct file* filp)
{
	struct quasar_m3 *qm3;
	int minor;
	
	qm3 = container_of(inode->i_cdev, struct quasar_m3, cdev);
	minor = iminor(inode);
	if(minor < 0 || minor >= g_nm3s)
		return -ENODEV;
	if(qm3->refs > 0)
		return -EBUSY;
	qm3->refs++;

	filp->private_data = qm3;
	return 0;
}

static int qm3_release(struct inode* inode, struct file* filp)
{
	struct quasar_m3* qm3;
	
	qm3 = (struct quasar_m3*)filp->private_data;
	if(qm3->refs <= 0)
		return -EBADF;
	qm3->refs--;
	filp->private_data = NULL;
	return 0;
}

static long qm3_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct quasar_m3* qm3;
	struct q_regio ioval;
	struct q_m3_io m3io;
	int iparm = 1;
	int ret = 0;
	int argdex;
	unsigned mask, val;
	volatile u32 __iomem	*rstgenreg;
	
	qm3 = (struct quasar_m3*)filp->private_data;

	if(! qm3)
		return -ENODEV;

	switch(cmd)
	{
	case QSETREG:
		ret = copy_from_user(&ioval, (void*)arg, sizeof(struct q_regio));
		ret = m3_writel(qm3, ioval.reg, (unsigned long)ioval.val);
		break;
	case QGETREG:
		ret = copy_from_user(&ioval, (void*)arg, sizeof(struct q_regio));
		ret = m3_readl(qm3, ioval.reg, (unsigned*)&ioval.val);
		ret = copy_to_user((void*)arg, &ioval, sizeof(struct q_regio));
		break;
	case QM3_SEND_COMMAND:
		ret = copy_from_user(&m3io, (void*)arg, sizeof(struct q_m3_io));
		if (m3io.args > 0) {
#if defined(Q6300) || defined(Q6600)
		    m3_writel(qm3, (qm3->minor == 0) ? QBM3_0_MB0 :  QBM3_1_MB0, m3io.msg);
	        for (argdex = 0; argdex < (m3io.args - 1); argdex++)
	            m3_writel(qm3, (qm3->minor == 0) ? QBM3_0_MB0 : QBM3_1_MB0, m3io.val[argdex]);
	        m3_writel(qm3, (qm3->minor == 0) ? QBM3_0_MBMC0 : QBM3_1_MBMC0, m3io.val[argdex]);
#endif	        
		}
		else
#if defined(Q6300) || defined(Q6600)
            m3_writel(qm3, (qm3->minor == 0) ? QBM3_0_MBMC0 : QBM3_1_MBMC0, m3io.msg);
#endif	        
		ret = 0;		
		break;
	case QM3_GET_RESPONSE:
		ret = qm3_rsp_read(qm3, &m3io);
		if(! ret)
			ret = copy_to_user((struct q_m3_io*)arg, &m3io, sizeof(m3io));
		break;
	case QM3_GET_NOTIFICATION:
		ret = qm3_ntf_read(qm3, &m3io);
		if (! ret)
			ret = copy_to_user((struct q_m3_io*)arg, &m3io, sizeof(m3io));
		break;
	case QM3_WAIT_RESPONSE:
		iparm = (int)arg;
		ret = qm3_rsp_wait(qm3, iparm);
		break;
	case QM3_WAIT_NOTIFICATION:
		iparm = (int)arg;
		ret = qm3_ntf_wait(qm3, iparm);
		break;
	case QM3_EXECUTE:
		iparm = (int)arg;
		ret = 0;
		switch(qm3->minor)
		{
		case 0:
#if defined(Q6300) || defined(Q6600)
            mask = RSTGEN_SWRSTSTATIC5__SCA_M3_0_H__MASK;
            rstgenreg = ioremap(RSTGEN_SWRSTSTATIC5, 32);
#endif		
			break;
		case 1:
#if defined(Q6300) || defined(Q6600)
            mask = RSTGEN_SWRSTSTATIC6__PRT_M3_1_H__MASK;
            rstgenreg = ioremap(RSTGEN_SWRSTSTATIC6, 32);
#endif		
			break;
		default:
			ret = -EINVAL;
			return ret;
		}
		val = readl(rstgenreg);
		// reset 
		if (iparm == 2)
		{
			if (qm3->minor == 1)
			{
				// force POR for M31
				unsigned pormask=RSTGEN_SWRSTSTATIC6__PRT_M3_1_POR__MASK;
				writel(val | mask | pormask, rstgenreg);
//printk("=== qm3: execute write POR 0x%08X = 0x%0X ==\n", rstgenreg, readl(rstgenreg));
			}
			
			// stop M3
			iparm = 0;
		}
		if (iparm)
			val &= ~mask;
		else
			val |= mask;
        //printk(M3_DEVNAME "qm3: START M3 %d, val=%08X\n", qm3->minor, val);
		writel(val, rstgenreg);
		iounmap(rstgenreg);
		break;
	default:
		printk(KERN_WARNING M3_DEVNAME "Bad ioctl %d\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static int qm3_mmap(struct file* filp, struct vm_area_struct* vma)
{
	int i, ret;
	struct quasar_m3* qm3;
	
	qm3 = (struct quasar_m3*)filp->private_data;
	if(! qm3)
		return -ENODEV;
	
	/* !! reg access should not be cached !! */
	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	
	for(i = 0; i < qm3->nioregs; i++) {
		if((qm3->ioregs[i].start >> PAGE_SHIFT) == vma->vm_pgoff) {
			ret = remap_pfn_range(
								vma,
								vma->vm_start,
								qm3->ioregs[i].start >> PAGE_SHIFT,
								vma->vm_end - vma->vm_start,
								vma->vm_page_prot
								);
			return ret;
		} 
	}
	printk(KERN_WARNING M3_DEVNAME "Could not remap region pgoff: %0ld\n", 
	       vma->vm_pgoff);
	return -ENOMEM;
}

static struct file_operations qm3_ops = {
	.owner		= THIS_MODULE,
	.open		= qm3_open,
	.poll		= qm3_poll,	
	.release	= qm3_release,
	.mmap           = qm3_mmap,
	.unlocked_ioctl	= qm3_ioctl,
	.compat_ioctl	= qm3_ioctl 
};

static int __init qm3_probe(struct platform_device *pdev)
{
	struct resource	*ioreg, *irqs;
	struct quasar_m3 *qm3;
	dev_t  qm3n;
	int irq;
	int ret = 0;
	int irqrecs = 0;
	int disable_rsp = (irq_mask >> (pdev->id * 2)) & 1;
	int disable_ntf = (irq_mask >> (pdev->id * 2 + 1)) & 1;
	
	if(g_nm3s >= MAX_M3S)
		return -ENXIO;

	qm3 = kzalloc(sizeof(struct quasar_m3), GFP_KERNEL);
	if(! qm3) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	qm3->frdev = &pdev->dev;
	qm3->nioregs = 0;
	qm3->nirqs = 0;
	
    printk(KERN_WARNING M3_DEVNAME "init qm3_probe ...\n"); 
	do {
		ioreg = platform_get_resource(pdev, IORESOURCE_MEM, qm3->nioregs);
		if (ioreg) {
			qm3->ioregs[qm3->nioregs].start = ioreg->start;
			qm3->ioregs[qm3->nioregs].end = ioreg->end;
			qm3->ioregs[qm3->nioregs].base =
				ioremap(ioreg->start, ioreg->end - ioreg->start + 4);
            //printk(KERN_WARNING M3_DEVNAME "[%d]-- start=0x%llX, end=0x%llX, base=0x%llX\n", qm3->nioregs, ioreg->start, ioreg->end, qm3->ioregs[qm3->nioregs].base); 
			if (!qm3->ioregs[qm3->nioregs].base) {
				ret = -ENOMEM;
				dev_dbg(&pdev->dev, "could not map I/O memory %0d\n", qm3->nioregs);
				goto out_abort;
			}
			/*
			printk("qm3 - mapped qm3 %px %x to %x at %x\n", 
					qm3, ioreg->start, ioreg->end, qm3->ioregs[qm3->nioregs].base);
			*/
			qm3->nioregs++;
		}
	} while (ioreg && (qm3->nioregs < MAX_M3IOREGS));

	do {
		irqs = platform_get_resource(pdev, IORESOURCE_IRQ, irqrecs++);
		if (irqs) {
			for(irq = irqs->start; irq <= irqs->end; irq++) {
				if (((irq == irqs->start) && (disable_rsp == 0))
					 || ((irq == irqs->end) && (disable_ntf == 0))) {
					if (qm3->nirqs >= MAX_M3IRQS) {
						dev_dbg(&pdev->dev, "too many irqs\n");
						goto out_abort;
					}
					qm3->irqs[qm3->nirqs++] = irq;
					ret = request_irq(irq, qm3_interrupt, 0, "qm3", qm3);
					if (ret) {
						dev_dbg(&pdev->dev,
								  "could not request irq %d\n", irq);
						goto out_abort;
					}
				}
			}
		}
	} while (irqs && (qm3->nirqs < MAX_M3IRQS));
		
	spin_lock_init(&qm3->lock);

	cdev_init(&qm3->cdev, &qm3_ops);
	qm3n = MKDEV(QM3_MAJOR, g_nm3s);
	ret = cdev_add(&qm3->cdev, qm3n, 1);
	if (ret) {
		printk(KERN_WARNING "qm3 - could not create char dev %d\n", g_nm3s);
		ret = -ENODEV;
		goto out_abort;
	}
	/*printk("qm3 - adding char dev %d:%d\n", MAJOR(qm3n), MINOR(qm3n));*/
	qm3->minor = MINOR(qm3n);
	qm3->ordinal = g_nm3s;
	g_nm3s++;
	ret = 0;

	/* wait queues for m3 response and notifications
	*/
	init_waitqueue_head(&qm3->m3_rdq);
	init_waitqueue_head(&qm3->m3_ntq);

	platform_set_drvdata(pdev, qm3);
	device_init_wakeup(&pdev->dev, 1);
	dev_info(&pdev->dev, "Quasar M3\n");
	return 0;

out_abort:
	while (qm3->nioregs > 0) {
		qm3->nioregs--;
		iounmap(qm3->ioregs[qm3->nioregs].base);
	}
	for(irq = 0; irq < qm3->nirqs; irq++)
		free_irq(qm3->irqs[irq], qm3);
	kfree(qm3);
	return ret;
}

static int __exit qm3_remove(struct platform_device *pdev)
{
	struct quasar_m3 *qm3 = platform_get_drvdata(pdev);
	int irq;

	cdev_del(&qm3->cdev);
	while (qm3->nioregs > 0) {
		qm3->nioregs--;
		iounmap(qm3->ioregs[qm3->nioregs].base);
	}
	for(irq = 0; irq < qm3->nirqs; irq++)
		free_irq(qm3->irqs[irq], qm3);
	kfree(qm3);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static int qm3_suspend(struct platform_device *pdev, pm_message_t state)
{
    struct quasar_m3 *qm3 = platform_get_drvdata(pdev);
    
    if (qm3->ordinal == 0)
    {
        // m30 must be powered down and hold in reset in the Quasar sleep process, so keep it alive ths moment
        //unsigned mask, val;
        //volatile u32 __iomem	*rstgenreg;

        // hold reset of m30
        //mask = RSTGEN_SWRSTSTATIC5__SCA_M3_0_H__MASK;
        //rstgenreg = ioremap(RSTGEN_SWRSTSTATIC5, 32);
        //val = readl(rstgenreg);
        //val |= mask;
        //writel(val, rstgenreg);
        //iounmap(rstgenreg);
    }
    else
    {
        // m31 is the master of low power processor, keep it alive
    }
            
    //printk("    ==== qm3_suspend for cm3%d, state=%d ====\n", qm3->ordinal, state);
	return 0;
}

static int qm3_resume(struct platform_device *pdev)
{
	struct quasar_m3 *qm3 = platform_get_drvdata(pdev);
    unsigned mask, val;
	volatile u32 __iomem	*rstgenreg;

    if (qm3->ordinal == 0)
    {
        // Deassert reset of M30
        mask = RSTGEN_SWRSTSTATIC5__SCA_M3_0_H__MASK;
        rstgenreg = ioremap(RSTGEN_SWRSTSTATIC5, 32);
    }
    else
    {
        // Deassert reset of M31
        mask = RSTGEN_SWRSTSTATIC6__PRT_M3_1_H__MASK;
        rstgenreg = ioremap(RSTGEN_SWRSTSTATIC6, 32);
    }
    
    val = readl(rstgenreg);
    val &= ~mask;
    writel(val, rstgenreg);
    iounmap(rstgenreg);
    
    //printk("    ==== qm3_resume for cm3%d ====\n", qm3->ordinal);
	return 0;
}

MODULE_ALIAS("platform:qm3");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-cm3" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver qm3_driver_ops = {
	.probe		= qm3_probe,
	.remove		= qm3_remove,
	.suspend	= qm3_suspend,
	.resume		= qm3_resume,
	.driver		= {
		.name	= "quasar-cm3",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init qm3_init(void)
{
	int ret;

		ret = platform_driver_register(&qm3_driver_ops);
	return ret;
}
module_init(qm3_init);

static void __exit qm3_exit(void)
{
	platform_driver_unregister(&qm3_driver_ops);
}
module_exit(qm3_exit);

MODULE_DESCRIPTION("Quasar Cortex M3 Driver");
MODULE_LICENSE("Dual BSD/GPL");

