/*
 * A driver for JPEG h/w block found on some Quatro processors
 *
 *  Quasar JPEG kernel driver
 *
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/01/19 08:48:00 $
//  $Change: 57110 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#define QJPEG_DEVNAME	"qjpeg"

#define MAX_MAILBOX 64

struct jpeg_quasar {
	int			ref;
	int			minor;
	struct cdev		cdev;
	volatile u8		__iomem	*regs;
	unsigned long		ioreg_start;
	unsigned long		ioreg_end;
	int			irq;

	wait_queue_head_t	mbq;	/* wait-queue for dma-doneinterrupt events */
	struct q_doneinfo	doneinfo[MAX_MAILBOX];
	int			di_head, di_tail;
	spinlock_t		lock;
};

static int qjpeg_open(struct inode* inode, struct file* filp)
{
	struct jpeg_quasar *jpeg;
	volatile u8 __iomem *reg;
	volatile u8 __iomem *sts;

	jpeg = container_of(inode->i_cdev, struct jpeg_quasar, cdev);
	if(jpeg->ref > 0) {
		printk(KERN_WARNING " qjpeg busy!!!!!!!!\n");
		return -EBUSY;
	}
	jpeg->ref++;
	jpeg->minor = iminor(inode);
	filp->private_data = jpeg;
	jpeg->di_head = jpeg->di_tail = 0;

	/* reset */
	reg = ioremap(IPM1CLKDISCTRL, 4);
	sts = ioremap(IPM1CLKDISSTAT, 4);
	switch (jpeg->minor)
	{
	case 0:
		writel(readl(reg) & IPM1CLKDISCTRL__IPM1_JPG0__INV_MASK, reg);
		while (readl(sts) & IPM1CLKDISSTAT__IPM1_JPG0__MASK) {;}
		break;
	case 1:
		writel(readl(reg) & IPM1CLKDISCTRL__IPM1_JPG1__INV_MASK, reg);
		while (readl(sts) & IPM1CLKDISSTAT__IPM1_JPG1__MASK) {;}
		break;
	default:
		break;
	}
	iounmap(reg);
	iounmap(sts);
	reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
	switch (jpeg->minor)
	{
	case 0:
		writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_QBJPG0__INV_MASK, reg);
		printk(KERN_INFO "QJPEG %d Reset\n", jpeg->minor);
		break;
	case 1:
		writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_QBJPG1__INV_MASK, reg);
		printk(KERN_INFO "QJPEG %d Reset\n", jpeg->minor);
		break;
	default:
		printk(KERN_INFO "QJPEG - unknown minor: %d\n", jpeg->minor);
		break;
	}
	iounmap(reg);

	return 0;
}

static int qjpeg_release(struct inode* inode, struct file* filp)
{
	struct jpeg_quasar *jpeg;

	jpeg = container_of(inode->i_cdev, struct jpeg_quasar, cdev);
	if(jpeg->ref <= 0)
		return -EFAULT;
	jpeg->ref--;
	filp->private_data = NULL;
	return 0;
}

static ssize_t qjpeg_read(struct file* filp, char __user *buffer, size_t length, loff_t* offset)
{
	struct jpeg_quasar *jpeg;

	jpeg = (struct jpeg_quasar*)filp->private_data;
	return -EINVAL;
}

static ssize_t qjpeg_write(struct file* filp, const char __user *buffer, size_t length, loff_t* offset)
{
	struct jpeg_quasar *jpeg;

	jpeg = (struct jpeg_quasar*)filp->private_data;
	return -EINVAL;
}

static unsigned int qjpeg_poll(struct file *filp, poll_table *wait)
{
	struct jpeg_quasar* jpeg;
	unsigned int mask = 0;
	unsigned long flags;

	jpeg = (struct jpeg_quasar*)filp->private_data;

	poll_wait(filp, &jpeg->mbq, wait);
	spin_lock_irqsave(&jpeg->lock, flags);
	if(jpeg->di_head != jpeg->di_tail) {
		mask |= POLLIN | POLLRDNORM; /* readable */
	}
	spin_unlock_irqrestore(&jpeg->lock, flags);
	return mask;
}

static long qjpeg_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct jpeg_quasar *jpeg;
	unsigned long flags;
	int ret = 0;

	jpeg = (struct jpeg_quasar*)filp->private_data;

	switch(cmd)
	{
	case QGETDONE:
		if(jpeg->di_head == jpeg->di_tail)
			return -EAGAIN;
		copy_to_user((void*)arg, (void*)&jpeg->doneinfo[jpeg->di_tail], sizeof(struct q_doneinfo));
		spin_lock_irqsave(&jpeg->lock, flags);
		jpeg->di_tail++;
		if(jpeg->di_tail >= MAX_MAILBOX)
			jpeg->di_tail = 0;
		spin_unlock_irqrestore(&jpeg->lock, flags);
		break;
	case QWAITDONE:
		while(jpeg->di_head == jpeg->di_tail) {
			/* TODO - check for dma in progress? */
			if (wait_event_interruptible(jpeg->mbq,
				(jpeg->di_head != jpeg->di_tail)))
				return -ERESTARTSYS;
		}
		if(jpeg->di_head != jpeg->di_tail) {
			copy_to_user((void*)arg,
				(void*)&jpeg->doneinfo[jpeg->di_tail], sizeof(struct q_doneinfo));
			spin_lock_irqsave(&jpeg->lock, flags);
			jpeg->di_tail++;
			if(jpeg->di_tail >= MAX_MAILBOX)
				jpeg->di_tail = 0;
			spin_unlock_irqrestore(&jpeg->lock, flags);
			ret = 0;
		}
		else {
			/* TODO - check for dma in progress? */
			ret = -EAGAIN;
		}
		break;
	case QRESET:
		{
			volatile u8 __iomem *reg;

			reg = ioremap(RSTGEN_SWRSTPULSE3, 4);
			switch(jpeg->minor)
			{
			case 0:
				writel(RSTGEN_SWRSTPULSE3__IPM1_QBJPG0__MASK, reg);
				break;

			case 1:
				writel(RSTGEN_SWRSTPULSE3__IPM1_QBJPG1__MASK, reg);
				break;

			default:
				printk(KERN_INFO "QJPEG - unknown minor: %d\n", jpeg->minor);
				break;
			}
			iounmap(reg);
		}
		break;
	default:
		printk(KERN_WARNING "qjpeg - bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static int qjpeg_mmap(struct file* filp, struct vm_area_struct* vma)
{
	struct jpeg_quasar *jpeg;
	int length, ret = 0;

	jpeg = (struct jpeg_quasar*)filp->private_data;
	if(! jpeg) { printk(KERN_ERR "JPEG NOT OPEN\n"); return -ENODEV; }

	/* !! mark pages as uncached for now !! */

	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	length = vma->vm_end - vma->vm_start;
	if(length > (jpeg->ioreg_end - jpeg->ioreg_start)) {
		printk(KERN_WARNING "JPEG VMA length truncated to io region\n");
		length = (jpeg->ioreg_end - jpeg->ioreg_start); 
	}
	ret = remap_pfn_range(
				vma,
				vma->vm_start,
				jpeg->ioreg_start >> PAGE_SHIFT,
				length,
				vma->vm_page_prot
			);
	return ret;
}

static irqreturn_t quasar_jpeg_interrupt(int irq, void *dev_id)
{
	struct jpeg_quasar *jpeg = (struct jpeg_quasar *)dev_id;
	struct timeval tv;
	int ret;
	unsigned long flags;
	
	unsigned long int_status;
	unsigned long err_status = 0;

	/// @todo if encoding, read input dma bit
	/// @todo if decoding, read output dma bit
	// printk("qjpeg: interrupt\n");

	/* clear interrupts on JPEG core */
	// disable interrupt
	writel(0,  jpeg->regs + (JPG0_IP_INTERRUPT_ENABLE - JPEG0_BASE));

	/* read dma */
	int_status = readl(jpeg->regs + (JPG0_IP_DMA_DONE_FLAG - JPEG0_BASE));

	spin_lock_irqsave(&jpeg->lock, flags);
	jpeg->doneinfo[jpeg->di_head].msg     = int_status;
	jpeg->doneinfo[jpeg->di_head].detail  = err_status;
	do_gettimeofday(&tv);
	jpeg->doneinfo[jpeg->di_head].endtime = (unsigned long long)tv.tv_sec * 1000000 + (unsigned long long)tv.tv_usec;
	jpeg->doneinfo[jpeg->di_head].cycles  = 0;
	jpeg->di_head++;
	if(jpeg->di_head >= MAX_MAILBOX)
		jpeg->di_head = 0;
	spin_unlock_irqrestore(&jpeg->lock, flags);

	/* wakeup tasks waiting on dma done
	*/
	wake_up_interruptible(&jpeg->mbq);

	ret = IRQ_HANDLED;
	return ret;
}

static struct file_operations quasar_jpeg_ops = {
	.owner		= THIS_MODULE,
	.open		= qjpeg_open,
	.release	= qjpeg_release,
	.read		= qjpeg_read,
	.write		= qjpeg_write,
	.poll		= qjpeg_poll,
	.mmap		= qjpeg_mmap,
	.unlocked_ioctl	= qjpeg_ioctl,
	.compat_ioctl	= qjpeg_ioctl 
};

static int __init quasar_jpeg_probe(struct platform_device *pdev)
{
	struct jpeg_quasar *jpeg;
	struct resource	*regs;
	dev_t  jpegn;
	int    ret;
	static int njpegs = 0;

	jpeg = kzalloc(sizeof(struct jpeg_quasar), GFP_KERNEL);
	if (!jpeg) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	cdev_init(&jpeg->cdev, &quasar_jpeg_ops);
	jpegn = MKDEV(QJPEG_MAJOR, njpegs);
	ret = cdev_add(&jpeg->cdev, jpegn, 1);
	if (ret) {
		dev_dbg(&pdev->dev, "could not create char dev %d\n", njpegs);
		goto out_err;
	}
	regs = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!regs) {
		dev_dbg(&pdev->dev, "no mmio reg resource defined\n");
		ret = -ENXIO;
		goto out_rerr;
	}
	jpeg->ioreg_start = regs->start;
	jpeg->ioreg_end   = regs->end;

	jpeg->regs = ioremap(regs->start, regs->end - regs->start + 1);
	if(!jpeg->regs) {
		ret = -ENOMEM;
		dev_dbg(&pdev->dev, "could not map reg I/O memory\n");
		goto out_ioerr;
	}

	jpeg->irq = platform_get_irq(pdev, 0);
	if (jpeg->irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		ret = -ENXIO;
		goto out_ioerr;
	}
	ret = request_irq(jpeg->irq, quasar_jpeg_interrupt,
		0, "jpegdma", jpeg);
	if (ret) {
		dev_dbg(&pdev->dev, "could not request irq %d\n", jpeg->irq);
		jpeg->irq = -1;
		goto out_ioerr;
	}
	spin_lock_init(&jpeg->lock);
	init_waitqueue_head(&jpeg->mbq);

	njpegs++;
	platform_set_drvdata(pdev, jpeg);
	printk(KERN_INFO "QJPEG - mapped at %px, irq %d\n", jpeg->regs, jpeg->irq);
	device_init_wakeup(&pdev->dev, 1);      
	dev_info(&pdev->dev, "Quasar JPEG\n");
	return 0;
out_ioerr:
	iounmap(jpeg->regs);
out_rerr:
	cdev_del(&jpeg->cdev);
out_err:
	kfree(jpeg);
	return ret;
}

static int __exit quasar_jpeg_remove(struct platform_device *pdev)
{
	struct jpeg_quasar *jpeg = platform_get_drvdata(pdev);

	if(jpeg->irq > 0)
		free_irq(jpeg->irq, jpeg);
	iounmap(jpeg->regs);
	cdev_del(&jpeg->cdev);
	kfree(jpeg);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static int quasar_jpeg_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct jpeg_quasar *jpeg = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM1CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (jpeg->minor)
		{
		default:
			break;
		case 0:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM1_QBJPG0__MASK, reg);
			writel(readl(ctl) | IPM1CLKDISCTRL__IPM1_JPG0__MASK, ctl);
			while ((readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JPG0__MASK) == 0x0) {;}
			break;
		case 1:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM1_QBJPG1__MASK, reg);
			writel(readl(ctl) | IPM1CLKDISCTRL__IPM1_JPG1__MASK, ctl);
			while ((readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JPG1__MASK) == 0x0) {;}
			break;
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
	}
#endif
	return 0;
}

static int quasar_jpeg_resume(struct platform_device *pdev)
{
	struct jpeg_quasar *jpeg = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM1CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (jpeg->minor)
		{
		default:
			break;
		case 0:
			writel(readl(ctl) & IPM1CLKDISCTRL__IPM1_JPG0__INV_MASK, ctl);
			while (readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JPG0__MASK) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_QBJPG0__INV_MASK, reg);
			break;
		case 1:
			writel(readl(ctl) & IPM1CLKDISCTRL__IPM1_JPG1__INV_MASK, ctl);
			while (readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JPG1__MASK) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_QBJPG1__INV_MASK, reg);
			break;
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
	}
#endif
	return 0;
}

MODULE_ALIAS("platform:quasar-jpeg");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-jpeg" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver quasar_jpeg_driver_ops = {
	.probe		= quasar_jpeg_probe,
	.remove		= quasar_jpeg_remove,
	.suspend	= quasar_jpeg_suspend,
	.resume		= quasar_jpeg_resume,
	.driver		= {
		.name	= "quasar-jpeg",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),
	},
};

static int __init quasar_jpeg_init(void)
{
	int ret;

	ret = platform_driver_register(&quasar_jpeg_driver_ops);
	return ret;
}
module_init(quasar_jpeg_init);

static void __exit quasar_jpeg_exit(void)
{
	platform_driver_unregister(&quasar_jpeg_driver_ops);
}
module_exit(quasar_jpeg_exit);

MODULE_DESCRIPTION("Quasar JPEG driver");
MODULE_LICENSE("Dual BSD/GPL");

