/*
 * jbig driver header
 *
 *
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#define JBIG_DEBUG 0

#ifdef JBIG1_BASE
	#define JBIG_NUM_CONTEXTS	2
#else
	#define JBIG_NUM_CONTEXTS	1
#endif

/**@enum JbigBihOrder
 * Values for the order field in JBIG_BIH.
 *
 * @note Since the JBIG hardware does not support progressive coding,
 *       none of these values are supported.
 */
typedef enum
{
    JBIG_HITOLO = 0x08,
    /**< When this bit is set the BIE will contain the higher resolution
     *   layers before the lower layers.
     */
    JBIG_SEQ    = 0x04,
    /**< Usually, at first all stripes of one resolution layer are
     *   written to the BIE and then all stripes of the next layer, and so
     *   on. When the SEQ bit is set however, then all layers of the first
     *   stripe will be written, followed by all layers of the second
     *   stripe, etc.
     */
    JBIG_ILEAVE = 0x02,
    /**< If this bit is set, then at first all layers of one plane are
     *   written before the encoder starts with the next plane.
     */
    JBIG_SMID   = 0x01,
    /**< In case there exist several bit planes, then the order of the
     *   stripes is determined by 3 loops over all stripes, all planes and
     *   all layers. When SMID is set, the loop over all stripes is the
     *   middle loop.
     */

    JBIG_ORDER_DEFAULT=0 ///< No order bits set
}
JbigBihOrder;


/**@enum JbigBihOptions
 * Values for the options field in JBIG_BIH
 */
typedef enum
{
    JBIG_LRLTWO = 0x40,
    /**< Normally, in the lowest resolution layer, pixels from three
     *   lines around the next pixel are used in order to determine the
     *   context in which the next pixel is encoded. If JBG_LRLTWO is set,
     *   the slightly faster but 5% less well compressing two line
     *   alternative is selected.
     */

    JBIG_VLENGTH = 0x20,
    /**< If the image length is unknown at the time encoding begins, the
     *   encoder may set VLENGTH, and then later in the JBIG datastream
     *   emit a NEWLEN marker segment providing the actual image length.
     */

    JBIG_TPDON = 0x10, /**< UNSUPPORTED */

    JBIG_TPBON = 0x08,
    /**< TPBON activates the "typical prediction" algorithm in the lowest
     *   resolution layer to make coding of large areas of equal color
     *   unnecessary.
     */

    JBIG_DPON   = 0x04, /**< UNSUPPORTED */
    JBIG_DPPRIV = 0x02, /**< UNSUPPORTED */
    JBIG_DPLAST = 0x01, /**< UNSUPPORTED */

    JBIG_OPTIONS_NONE    = 0,                ///< No JBIG options selected
    JBIG_OPTIONS_DEFAULT = JBIG_OPTIONS_NONE ///< By default, no options selected
}
JbigBihOptions;


/**
 * @enum JbigEscapeCodes
 *
 * JBIG escape codes defined by ITU-T Recommendation T.82
 */
typedef enum
{
    JBIG_STUFF_MARKER   = 0,
    JBIG_RESERVE_MARKER = 1, ///< Must never appear in a public JBIG datastream
    JBIG_SDNORM_MARKER  = 2,
    JBIG_SDRESET_MARKER = 3,
    JBIG_ABORT_MARKER   = 4,
    JBIG_NEWLEN_MARKER  = 5,
    JBIG_ATMOVE_MARKER  = 6,
    JBIG_COMMENT_MARKER = 7,
    JBIG_ESCAPE         = 0xFF
}
JbigEscapeCodes;

/** The JBIG BIH. The values xd, yd, and l0 integers are coded most
 *  significant byte first.  This struct is used internaly by
 *  jbigenc/dec.c to keep track of the BIH.  Use pack function to
 *  read/write the 20 byte BIH from a stream.
 */
typedef struct tag_jbig_bih
{
    u8  dl; ///< The initial resolution layer to be specified in this BIE
    u8  d;  ///< The final resolution layer to be specified in this BIE
    u8  p;  ///< The number of bit-planes. For a bi-level image p == 1
    u8  reserved; ///< Pad. Should always be 0
    u32 xd; ///< The horizontal dimension at highest resolution
    u32 yd; ///< The vertical dimension at highest resolution
    u32 l0; ///< The number of lines per stripe at lowest resolution
    u8  mx; ///< The maximum horizontal offset allowed for the AT pixel
    u8  my; ///< The maximum vertical offset allowed for the AT pixel
    JbigBihOrder order;     ///< Zero, since multiple planes are not supported
    u8  options; ///< JbigBihOptions: LRLTWO, VLENGTH, and TPBON are supported
}
JBIG_BIH;

// note that all JBIG unit access is done using _OFF semantics so units can be treated equally
// so use JBIG_ not JBIGN_ to access units
//
#define jbigWrite32(x, r, v)		\
	writel((v), ((s_jbig_context[x])->regs + r))

#define jbigRead32(x, r)			\
	(volatile u32)readl((s_jbig_context[x])->regs + r)
	
#ifndef JBIG0_BASE
#ifdef  JBIGCB0_BASE
// Q6300
#define JBIG0_BASE JBIGCB0_BASE
#define JBJMR      JBCB0JMR_OFF
/* JBIG Reset Register */
#define JBJRST     JBCB0JRST_OFF
/* JBIG Status Register */
#define JBJSR      JBCB0JSR_OFF
/* JBIG A-ALU Register */
#define JBAREG     JBCB0AREG_OFF
/* JBIG C-ALU Register */
#define JBCREG     JBCB0CREG_OFF
/* JBIG Image Dimensions Register */
#define JBJIDR     JBCB0JIDR_OFF
/* JBIG Template Definition Register */
#define JBTDEF     JBCB0TDEF_OFF
/* JBIG Line Count */
#define JBJLC      JBCB0JLC_OFF
/* JBIG DMA Actual Source Count */
#define JBJASC     JBCB0JASC_OFF
/* JBIG DMA Actual Count */
#define JBJDAC     JBCB0JDAC_OFF
/* JBIG DMA Source Address */
#define JBJSRC     JBCB0JSRC_OFF
/* JBIG DMA Destination Address */
#define JBJDES     JBCB0JDES_OFF
/* JBIG DMA Clip Address */
#define JBJDCL     JBCB0JDCL_OFF
/* JBIG Adapter Memory */
#define JBJMEM     JBCB0JMEM_OFF
#else
#define JBIG0_BASE JBIG_BASE
#define JB0JMEM_END JMEM_END
#define IRQ_JBIG0 IRQ_JBIG
// older cpu, only one jbig
#define JBJMR	JMR_OFF
/* JBIG Reset Register */
#define JBJRST	JRST_OFF
/* JBIG Status Register */
#define JBJSR	JSR_OFF
/* JBIG A-ALU Register */
#define JBAREG	AREG_OFF
/* JBIG C-ALU Register */
#define JBCREG	CREG_OFF
/* JBIG Image Dimensions Register */
#define JBJIDR	JIDR_OFF
/* JBIG Template Definition Register */
#define JBTDEF	TDEF_OFF
/* JBIG Line Count */
#define JBJLC	JLC_OFF
/* JBIG DMA Actual Source Count */
#define JBJASC	JASC_OFF
/* JBIG DMA Actual Count */
#define JBJDAC	JDAC_OFF
/* JBIG DMA Source Address */
#define JBJSRC	JSRC_OFF
/* JBIG DMA Destination Address */
#define JBJDES	JDES_OFF
/* JBIG DMA Clip Address */
#define JBJDCL	JDCL_OFF
/* JBIG Adapter Memory */
#define JBJMEM	JMEM_OFF
#endif
#else
#define JBJMR	JB0JMR_OFF
/* JBIG Reset Register */
#define JBJRST	JB0JRST_OFF
/* JBIG Status Register */
#define JBJSR	JB0JSR_OFF
/* JBIG A-ALU Register */
#define JBAREG	JB0AREG_OFF
/* JBIG C-ALU Register */
#define JBCREG	JB0CREG_OFF
/* JBIG Image Dimensions Register */
#define JBJIDR	JB0JIDR_OFF
/* JBIG Template Definition Register */
#define JBTDEF	JB0TDEF_OFF
/* JBIG Line Count */
#define JBJLC	JB0JLC_OFF
/* JBIG DMA Actual Source Count */
#define JBJASC	JB0JASC_OFF
/* JBIG DMA Actual Count */
#define JBJDAC	JB0JDAC_OFF
/* JBIG DMA Source Address */
#define JBJSRC	JB0JSRC_OFF
/* JBIG DMA Destination Address */
#define JBJDES	JB0JDES_OFF
/* JBIG DMA Clip Address */
#define JBJDCL	JB0JDCL_OFF
/* JBIG Adapter Memory */
#define JBJMEM	JB0JMEM_OFF
#endif

#define JBIG_READ32(r, u)     jbigRead32(u, r)
#define JBIG_WRITE32(r, u, d) jbigWrite32(u, r, d)

/* JBIG module registers*/

/**@name JBIG Mode Register Defines */
//@{
#define JMR_DST_SHIFT 19   ///< Destination Start Offset Shift Amount
#define JMR_SST_SHIFT 16   ///< Source Start Offset Shift Amount
#define JMR_FS     (1<<15) ///< First Stripe
#define JMR_PC     (1<<14) ///< Preserve Coder
#define JMR_ENCODE (0<<13) ///< Encode mode or adapter memory load/store/clear operation
#define JMR_DECODE (1<<13) ///< Decode mode
#define JMR_SBO    (1<<12) ///< Stop on Buffer Overflow
#define JMR_TH     (1<<9)  ///< Throttle
#define JMR_CE     (1<<8)  ///< Conditional Exchange Enable (JBIG Mode)
#define JMR_TP     (1<<6)  ///< Typical Prediction
#define JMR_SE     (1<<5)  ///< Keep Statistics at End of Stripe (SDNORM behavior)
#define JMR_CLP    (1<<1)  ///< Destination Buffer Clipping
#define JMR_EN     (1<<0)  ///< JBIG Macro Enable

/// JBJMR setting to start a JBIG Memory Erase operation
#define JBIGHW_START_ERASE (JMR_FS | JBIG_INIT_JMEM  | JMR_EN)

/// JBJMR setting to start a JBIG Memory Store operation
#define JBIGHW_START_SAVE  (JMR_FS | JBIG_STORE_JMEM | JMR_EN)

/// JBJMR setting to start a JBIG Memory Load operation
#define JBIGHW_START_LOAD  (JMR_FS | JBIG_LOAD_JMEM  | JMR_EN)

#define JBIG_HALF_SPEED_ENCODE 1
#define JBIG_HALF_SPEED_DECODE 1

#if JBIG_HALF_SPEED_ENCODE
/// Use Half-Speed (throttled) JBIG Encode
#  define ENCODE_SPEED JMR_TH
#else
/// Use Full-Speed (unthrottled) JBIG Encode
#  define ENCODE_SPEED 0
#endif

#if JBIG_HALF_SPEED_DECODE
/// Use Half-Speed (throttled) JBIG Decode
#  define DECODE_SPEED JMR_TH
#else
/// Use Full-Speed (unthrottled) JBIG Decode
#  define DECODE_SPEED 0
#endif

/// JMR setting to start a JBIG Encode operation
#define JBIGHW_START_ENCODE(etc) \
    (etc | ENCODE_SPEED | JMR_CE | JMR_ENCODE | JMR_EN)

/// JMR setting to start a JBIG Decode operation
#define JBIGHW_START_DECODE(etc) \
    (etc | DECODE_SPEED | JMR_CE | JMR_DECODE | JMR_EN)

//@}


/**@name JBIG Status Register Defines */
//@{
#define JSR_BO 0x4 /**< Buffer Overflow */
#define JSR_SS 0x2 /**< Stripe Corrupted (unknown escape sequence found) */
#define JSR_EF 0x1 /**< Escape Found */
//@}

#if JBIG_DEBUG
/// Print JBIG debug information.
#  define JBGPRINTF(...) printk(__VA_ARGS__)
#else
/// Placeholder for printing JBIG debug information.
#  define JBGPRINTF(...)
#endif

#define JbigGetBytesWritten(unit)		JBIG_READ32(JBJDAC, unit)
#define JbigUseTwoLineTemplate(unit)	JBIG_WRITE32(JBTDEF, unit, JBIG_2LINE_TMPL)
#define JbigUseThreeLineTemplate(unit)	JBIG_WRITE32(JBTDEF, unit, JBIG_3LINE_TMPL)

/** TDEF.ATD is specified as (ATMOVE.tx - 2) for non-zero tx */
#define ATMOVE_TX_TO_TDEF_ATD(tx) (tx ? (tx) - 2 : 0)

/** Choose TDEF.TSEL based on the selected JBIG template */
#define TDEF_SELECT_TEMPLATE(is_two_line) \
        ((is_two_line) ? JBIG_2LINE_TMPL : JBIG_3LINE_TMPL)


#define JbigGetLineCount(unit)		JBIG_READ32(JBJLC, unit)
#define JbigReset(unit)			    JBIG_WRITE32(JBJRST, unit, 0);

#define JbigSetSrcAddress(unit, src)	\
    JBIG_WRITE32(JBJSRC, unit, ((unsigned long)(src) & 0xFFFFFFFF) & ~7UL)

#define JbigSetDstAddress(unit, dst)	\
    JBIG_WRITE32(JBJDES, unit, ((unsigned long)(dst) & 0xFFFFFFFF) & ~7UL)

#define JbigGetActualSrcCount(unit) JBIG_READ32(JBJASC, unit)
#define JbigGetStatusRegister(unit)	JBIG_READ32(JBJSR, unit)

/// Typical Prediction State mask for TDEF
#define TPST 0x10000

#define JbigGetTPState(iunit)	\
	(JBIG_READ32(JBTDEF, unit) & TPST)

#define JbigStartJmemErase(unit)	\
    JBIG_WRITE32(JBJMR, unit, JBIGHW_START_ERASE);

#define JbigGetEscapeCode(unit)	\
	((JbigGetStatusRegister(unit) & JSR_EF) ? ((JbigGetStatusRegister(unit) >> 8) & 0xFF) : 0)

#define JbigGetBytesRead(unit) JbigGetActualSrcCount(unit)
#define JbigStripeCorrupted(unit) (JbigGetStatusRegister(unit) & JSR_SS)

/// The size of a JBIG Bitonal Image Header (BIH) in bytes.
#define JBIG_BIH_SIZE 20

/** Size of JBIG Adapter Memory in bytes */
#define JMEM_SIZE_BYTES 1024

/** Size of JBIG Adapter Memory in bits */
#define JMEM_SIZE (JMEM_SIZE_BYTES * 8)


