/*
 * jbig decoder using Quatro 4xxx h/w for jbig-on-the-fly decompression
 *
 * Quasar jbig decoder
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2022/01/06 10:11:28 $
//  $Change: 56812 $
//
// =========================================================
#define JBIG_DECODER_DEBUG 0

#define JBIG_DEC_UNIT (JBIG_NUM_CONTEXTS-1)
//#define JBIG_DEC_UNIT 0
	
#if JBIG_DECODER_DEBUG
/// Print JBIG Decoder debug information.
#  define JDPRINTF(...) printk(__VA_ARGS__)
#else
/// Placeholder for printing JBIG Decoder debug information.
#  define JDPRINTF(...)
#endif

void JbigSelectTemplate(int    unit,
						int    use_two_line_template)
{
	int tp_state = 0;
	int hw_atmove = 0;

	if(use_two_line_template) {
		JBGPRINTF("%s(%d,2L,AT%2d,TP:%c) TDEF=0x%08X\n",
				  __func__, unit, atmove_tx, tp_state ? '1' : '0',
				  JBIG_2LINE_TMPL | hw_atmove | tp_state);

		JBIG_WRITE32(JBTDEF, unit, JBIG_2LINE_TMPL | hw_atmove | tp_state);
	}
	else {
		JBGPRINTF("%s(%d,3L,AT%2d,TP:%c) TDEF=0x%08X\n",
				  __func__, unit, atmove_tx, tp_state ? '1' : '0',
				  JBIG_3LINE_TMPL | hw_atmove | tp_state);

		JBIG_WRITE32(JBTDEF, unit, JBIG_3LINE_TMPL | hw_atmove | tp_state);
	}
}

/** Set the height and width of the image's stripes.
 *
 *  Height goes into the top 16 bits of JIDR and width into the lower 16.
 *
 *  @param[in] unit JBIG hardware unit identifier
 *  @param[in] height number of lines in image
 *  @param[in] width width of image in pixels 
 */
void JbigSetDimensions(int unit, u16 height, u16 width)
{
	u32 dim = height;
	dim <<= 16;
	dim |= width;

	JBGPRINTF("%s(%d,%d,%d) JIDR=0x%08X\n",
			  __func__, unit, height, width, dim);

	JBIG_WRITE32(JBJIDR, unit, dim);
}

#if 0
/** Stringify name of JBIG Escape code
 ** @param[in] code JBIG marker byte
 ** @return string
 */
static const char *EscName(u8 code)
{
	switch (code)
	{
	case JBIG_RESERVE_MARKER: return "RESERVE";
	case JBIG_SDNORM_MARKER:  return "SDNORM";
	case JBIG_SDRESET_MARKER: return "SDRESET";
	case JBIG_ABORT_MARKER:   return "ABORT";
	case JBIG_NEWLEN_MARKER:  return "NEWLEN";
	case JBIG_ATMOVE_MARKER:  return "ATMOVE";
	case JBIG_COMMENT_MARKER: return "COMMENT";

	default:
		{
		static char hex[3];
		const char digits[17] = "0123456789ABCDEF";

		hex[0] = digits[(code >> 4) & 0xF];
		hex[1] = digits[code & 0xF];
		hex[2] = '\0';

		return hex;
		}
	}
}
#endif

#if 0
int JbigWaitInterrupt(PJBIGCTX pjbig)
{
	fd_set  fds;
	struct  timeval timeout;
	struct  q_doneinfo doneinfo;
	int sv;
	
	doneinfo.msg = 0;
	
	// timeout waiting for a JBIG dma to finish.  First select with timeout
	// on all active dma operations, then select on each one to see if its the one done
	//
	do
	{
		// 250 ms
		//
		timeout.tv_sec  = 0;
		timeout.tv_usec = 250000;
		
		FD_ZERO (&fds);
		FD_SET  (pjbig->jbigDriver, &fds);
		
		sv = select(pjbig->jbigDriver + 1, &fds, NULL, NULL, &timeout);
				
		if(sv == 0)
		{
			if(! (jbigRead32(pjbig->ordinal, JBJMR) & JMR_EN))
			{
				Log(1, "Apparently Missed an interrupt!\n");
				sv = 1;
				return 1;
			}
		}
		else if(sv < 0)
		{
			Error("JBIG driver broken\n");
			return -1;
		}
		else /* (sv > 0) *//* have int-complete pending! */
		{
			int rc;
			
			rc = ioctl(pjbig->jbigDriver, QGETDONE, &doneinfo);
			if(rc < 0)
			{
				Log(1, "JBIGwaitInterrupt - no int record!!\n");
			}
			return 0;
		}
	}
	while(1);
	
	return 0;
}
#endif

static void JbigWaitForDone(int unit)
{
//	JbigWaitInterrupt(&g_jbig_context[unit]);
	volatile u32 s;

	do
	{	
		s = JBIG_READ32(JBJMR, unit);
	}
	while(s	& JMR_EN);
}

static void JbigJmemErase(int unit)
{
	JbigReset(unit);
	JbigSetDimensions(unit, 1, JMEM_SIZE);
	JbigStartJmemErase(unit);
	JbigWaitForDone(unit);
}

/** Start a decode on the JBIG engine
 *
 *  @param[in] unit JBIG hardware unit identifier
 *  @param[in] tp TRUE if image was encoded using Typical Prediction.
 *  @param[in] dstoffset Low-order three bits of destination address.
 *  @param[in] srcoffset Low-order three bits of source address.
 *
 *  @note It would be better to pass in source and destination buffer addresses
 *  	  rather than offsets, giving JbigStartEncode()/JbigStartDecode()
 *  	  the responsibility to control JbigSetSrcAddress()/JbigSetDstAddress()
 *  	  and handle the JBJMR offsets, rather than pushing that burden out to
 *  	  callers.
 */
static void JbigStartDecode(int	 unit,
					 int tp,
					 u32  dstoffset,
					 u32  srcoffset)
{
	u32 jmr = (srcoffset << JMR_SST_SHIFT) | (dstoffset << JMR_DST_SHIFT);

	if (tp)
	{
		jmr |= JMR_TP;
	}

	jmr |= JMR_FS;

	JBIG_WRITE32(JBJMR, unit, JBIGHW_START_DECODE(jmr));
}

/** Read an unsigned long in MSB format.
*/
static u32 JbigBihUnpackLong(const u8 *src)
{
	u32 result = src[0];

	result = (result << 8) | src[1];
	result = (result << 8) | src[2];
	result = (result << 8) | src[3];
	return result;
}

#if 0
/**
   Read in a BIH.  BIH is a 20 byte structure at the start of JBIG
   stream. JbigDecReadBih gives the decoder the image parameters.

   @param[in,out] d   The decoder
   @param[in]     bih Pointer to a twenty byte JBIG BIH.

   @return int;
*/
static int JbigDecReadBih(struct jbig_quasar *d, const u8 *bih)
{
	/* read the BIH from src */
	d->bih.dl = bih[0];
	d->bih.d  = bih[1];
	d->bih.p  = bih[2];

	d->bih.xd = JbigBihUnpackLong(&bih[4]);
	d->bih.yd = JbigBihUnpackLong(&bih[8]);
	d->bih.l0 = JbigBihUnpackLong(&bih[12]);

#if JBIG_DECODER_DEBUG
	JDPRINTF("JBIG BIH:\n"
		 "\tDL=%d D=%d P=%d rsvd=%x XD=%d YD=%d L0=%d MX=%d MY=%d\n"
		 "\tOrder=%02x Options=%02x:",
		 bih[0], bih[1], bih[2], bih[3],
		 d->bih.xd, d->bih.yd, d->bih.l0,
		 bih[16], bih[17], bih[18], bih[19]);

	if(bih[19] == 0)
	{
		JDPRINTF(" none\n");
	}
	else
	{
		if(bih[19] & JBIG_LRLTWO)  JDPRINTF(" LRLTWO");
		if(bih[19] & JBIG_VLENGTH) JDPRINTF(" VLENGTH");
		if(bih[19] & JBIG_TPDON)   JDPRINTF(" TPDON");
		if(bih[19] & JBIG_TPBON)   JDPRINTF(" TPBON");
		if(bih[19] & JBIG_DPON)	JDPRINTF(" DPON");
		if(bih[19] & JBIG_DPPRIV)  JDPRINTF(" DPPRIV");
		if(bih[19] & JBIG_DPLAST)  JDPRINTF(" DPLAST");
		JDPRINTF("\n");
	}
#endif

	/* if this isn't zero then this isn't a JBIG header */
	d->bih.reserved = bih[3];
	if(d->bih.reserved != 0)
	{
		return -1;
	}
	d->bih.mx = bih[16];
	d->bih.my = bih[17];
	d->bih.order = (JbigBihOrder)bih[18];
	if(bih[18] != 0)
	{
		JDPRINTF("Unexpected order:%s%s%s%s\n",
			 bih[18] & (1 << 3) ? " HITOLO" : "",
			 bih[18] & (1 << 2) ? " SEQ" : "",
			 bih[18] & (1 << 1) ? " ILEAVE" : "",
			 bih[18] & (1 << 0) ? " SMID" : "");
	}
	d->bih.options = (JbigBihOptions)bih[19];
	if((bih[19] & 0x80) != 0)
	{
		return -1;
	}
	if(
		   d->bih.p  != 1
		|| d->bih.dl != 0
		|| d->bih.d  != 0
		|| d->bih.my != 0
		|| (d->bih.options
			& (JBIG_TPDON | JBIG_DPON | JBIG_DPPRIV | JBIG_DPLAST)) != 0
	) {
		printk("JBIG data stream incompatible with JBIG unit\n");
		return -1;
	}
	/* One reference line for two-line template
	*/
	d->referenceLineBytes = (d->bih.xd + 7) / 8;
	if((d->bih.options & JBIG_LRLTWO) == 0) {
		/* ...two reference lines for three-line template 
		*/
		d->referenceLineBytes *= 2;
	}
	return 0;
}
#endif

static int JbigDecInit(
		struct jbig_quasar* d,
		u32 w,
		u32 h,
		u32 l0)
{
	d->bih.dl = 0;
	d->bih.d  = 0;
	d->bih.p  = 1;
	d->bih.reserved = 0;
	d->bih.xd = w;
	d->bih.yd = h;
	d->bih.l0 = l0;
	d->bih.mx = 0;
	d->bih.my = 0;
	d->bih.order   = JBIG_ORDER_DEFAULT;
	d->bih.options = JBIG_OPTIONS_DEFAULT;

	d->referenceLineBytes = ((w+7) / 8 * 2);

	return 0;
}

static int JbigDecInitForOakRaster(struct jbig_quasar *d, u32 w, u32 h)
{
	int status;

	status = JbigDecInit(d, w, h, h);
	d->bih.options = JBIG_TPBON | JBIG_LRLTWO;
	return status;
}

/** Parse JBIG Floating Marker Segments before next stripe.
 *
 *  Note that this routine requires \em complete floating marker segments.
 *  Behavior in the presence of partial floating marker segments is undefined.
 *
 * @param[in,out] decoder An initialized JBIG decoder.
 * @param[in]     src     A buffer that holds JBIG encoded data.
 * @param[in]     srcsize The size in bytes of JBIG encoded data buffer.
 * @param[out]    srcused Returns the number of bytes of JBIG floating marker
 *  					  segment data read from src.
 * @return int
 */
static int JbigDecParseFloatingMarkerSegments(struct jbig_quasar     *d,
													 const u8 *src,
													 u32	   srcsize,
													 u32	  *srcused)
{
	int done=0;
	int retval = 0;

	*srcused = 0;

	JDPRINTF("(src => %px[%d])\n", src, srcsize);

	while(! done && (src[*srcused] == JBIG_ESCAPE)) {
		int marker_length = 0;

		switch (src[*srcused + 1])
		{
		case JBIG_ABORT_MARKER:
			printk("ABORT marker segment found; NOW WHAT?\n");
			retval = -1;
			done = 1;
			break;
			
		case JBIG_SDNORM_MARKER:
		case JBIG_SDRESET_MARKER:
			/*
			** -------------------------------------------------------------
			** Found a bare stripe end marker.
			** There are no more floating marker segments to be parsed here
			** -------------------------------------------------------------
			*/
			retval = 0;
			done = 1;
			break;
			
		case JBIG_STUFF_MARKER:
			/* Next stripe starts with a stuff, get back to decoding */
			retval = 0;
			done = 1;
			break;
			
		case JBIG_NEWLEN_MARKER:
			{
			u32 new_length = JbigBihUnpackLong(&src[*srcused + 2]);
				
			marker_length = 6 /*NEWLEN_SIZE*/;
			printk("NEWLEN marker segment found: %u\n", new_length);
			d->bih.yd = new_length;
			d->bih.options &= ~JBIG_VLENGTH;
			}
			break;
			
		case JBIG_ATMOVE_MARKER:
			{
			int yat = JbigBihUnpackLong(&src[*srcused + 2]);
			
			u8 tx = src[*srcused + 6];
			u8 ty = src[*srcused + 7];
			
			marker_length = 8;
			
			printk("ATMOVE marker segment found:\n"
				"\tYat=%d, tx=%d, ty=%d\n", yat, tx, ty);
			return -1;
			}
			break;
		
		case JBIG_COMMENT_MARKER:
			{
			int i;
			int length = JbigBihUnpackLong(&src[*srcused + 2]);
			
			marker_length = 6 + length;
			
			printk("\nCOMMENT marker seg found: (%d bytes)\n", length);
			
			for(i=0; i < length; i++) {
				char ch=src[*srcused + 6+i];
				
				if((ch >= '\t' && ch <= '\r')
					|| (ch >= ' '  && ch <= '~')) {
					JDPRINTF("%c", ch);
				}
				else {
					JDPRINTF("<0x%02x>", ch);
				}
			}
			JDPRINTF("\n");
			}
			break;

		default:
			printk("Unknown JBIG marker segment found: 0x%02x\n",
				src[*srcused + 1]);
			retval = -1;
			done = 1;
			break;
		}
		/** Flag an error if a partial floating marker segment is found.
		*/
		if(*srcused + marker_length > srcsize) {
			int i;

			printk("Partial floating marker segment found:");
			for(i=0; i < srcsize; i++) {
				printk(" %02x", src[*srcused+i]);
			}
			printk("\n");

			retval = -1;
		}
		*srcused += marker_length;
	}
	JDPRINTF("(*srcused => %d/%d) => %s\n", *srcused, srcsize,
		retval == 0 ? "Success" : "Failure");
	return retval;
}

/// Check whether 8-bit value matches 0x02 or 0x03
#define IS_STRIPE_END_CODE(c) (((c) & 0xFE) == 0x02)

static int JbigDecDecodeStripe(
	struct jbig_quasar *d,
	u8  	 *dstpa,
	u32 	  dstsize,
	u8  	 *srcpa,
	u8  	 *srcva,
	u32 	  srcsize,
	u32 	 *dstused,
	u32 	 *srcused)
{
	int status;
	int typical_prediction;
	u32 srcoffset, dstoffset;
	u32 nlines;
	int n_substituted_bytes = 0;


	JDPRINTF("%s(src => %px[%d], dst => %px[%d])\n",
			 __func__, srcva, srcsize, dstpa, dstsize);

	*srcused = 0;

	typical_prediction = (d->bih.options & JBIG_TPBON) != 0;

	/*
	** Check for JBIG floating marker segments before the PSCD.
	*/
	status = JbigDecParseFloatingMarkerSegments(d, srcva, srcsize, srcused);

	if(status != 0) {
		return status;
	}
	else if(0) {
		JDPRINTF("%s: Nothing left to decode\n", __func__);
		*srcused = srcsize;
		*dstused = 0;
		return 0;
	}
	else if(srcsize == 0) {
		printk("Insert a pad stripe to avoid decoding garbage\n");

		d->padded_short_stripe[0] = 0x00;
		d->padded_short_stripe[1] = 0x00;
		d->padded_short_stripe[2] = 0xFF;
		d->padded_short_stripe[3] = 0x03;

		srcva = d->padded_short_stripe;
		srcpa = d->padded_short_stripe;
	}
	else {
		srcpa += *srcused;
		srcva += *srcused;
	}
	/*
	** Check for short stripes:
	**
	**    a. 0 bytes of PSCD, just a stripe end marker
	** or
	**    b. 1 byte of PSCD followed by a stripe end marker
	**
	** The Quasar needs at least 2 bytes of PSCD preceding a stripe end
	** marker.  If a short stripe is found, pad the PSCD with zeroes before
	** giving it to the JBIG unit for processing.
	*/
	if(srcva != NULL && srcva[0] == 0xFF && IS_STRIPE_END_CODE(srcva[1])) {
		d->padded_short_stripe[0] = 0x00;
		d->padded_short_stripe[1] = 0x00;
		d->padded_short_stripe[2] = srcva[0];
		d->padded_short_stripe[3] = srcva[1];

		srcva = d->padded_short_stripe;
		srcpa = d->padded_short_stripe;

		n_substituted_bytes = 2;
	}
	else if(srcva != NULL && srcva[1] == 0xFF && IS_STRIPE_END_CODE(srcva[2])) {
		d->padded_short_stripe[0] = srcva[0];
		d->padded_short_stripe[1] = 0x00;
		d->padded_short_stripe[2] = srcva[1];
		d->padded_short_stripe[3] = srcva[2];

		srcva = d->padded_short_stripe;
		srcpa = d->padded_short_stripe;

		n_substituted_bytes = 1;
	}
	if(JBIG_READ32(JBJMR, d->unit) & JMR_EN)
	{
		printk("b");
//		JbigWaitForDone(d->unit);
	}

	JbigJmemErase(d->unit);
	nlines = dstsize * 8 / d->bih.xd;

	/*
	** Set up the JBIG hardware registers and do the decode.
	*/
	JbigSelectTemplate(d->unit, d->bih.options & JBIG_LRLTWO);
	JbigSetDimensions(d->unit, (u16)nlines, (u16)d->bih.xd);
	JbigSetSrcAddress(d->unit, srcpa);
	JbigSetDstAddress(d->unit, dstpa);

	srcoffset = (unsigned long)srcpa & 0x7;
	dstoffset = (unsigned long)dstpa & 0x7;

	JbigStartDecode(d->unit,
			typical_prediction,
			dstoffset, srcoffset);

	return 0;
	
	JbigWaitForDone(d->unit);

	// Calculate how much of the input and output buffer space was used.
	*srcused += JbigGetBytesRead(d->unit) - srcoffset - n_substituted_bytes;
	*dstused = JbigGetBytesWritten(d->unit) - dstoffset;

	if(JbigStripeCorrupted(d->unit)) {
		printk("JBIG decode error: stripe corrupted\n");
		JbigReset(d->unit);
		return -1;
	}
	if(*srcused > srcsize) {
		JDPRINTF("(*srcused => %d/%d, *dstused => %d/%d) => UNDERFLOW!\n",
				*srcused, srcsize, *dstused, dstsize);
		return -1;
	}
	JDPRINTF("(*srcused => %d/%d, *dstused => %d/%d) => Success\n",
			*srcused, srcsize, *dstused, dstsize);
	return 0;
}

/****
*/
int QJBIGdecodeBand(int w, int h, int d, u8* dstpa, u8* srcpa, u8* srcva, int srcsize)
{
	struct jbig_quasar *jd;
	int status;
	int srcused = 0, dstused = 0;
	int s;

	jd = s_jbig_context[JBIG_DEC_UNIT];
	if(! jd) {
		printk("JBIG - no context for band decode\n");
		return -1;
	}
	status = JbigDecInitForOakRaster(jd, w * d, h);
	if(status != 0)
		return status;
	
	s = (w * d + 7) / 8;

	// start off jbig decode
	//
	status = JbigDecDecodeStripe(
							jd,
							dstpa,
							h * s,
							srcpa,
							srcva,
							srcsize,
							&dstused,
							&srcused
						);
	return status;
}

EXPORT_SYMBOL(QJBIGdecodeBand);


