/*
 * A Kernel driver for JBIG unit found on Quatro processors
 *
 * Quasar JBIG kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2022/01/19 08:48:00 $
//  $Change: 57110 $
//
// ========================================================= 
#include "qjbigx.h"

#define QJBIG_DEVNAME	"qjbig"

#define MAX_MAILBOX 64

struct jbig_quasar {
	int			ref;	///< reference count
	int			minor;	///< device minor number
	struct cdev		cdev;
	volatile u8	__iomem	*regs;
	unsigned long		ioreg_start;
	unsigned long		ioreg_end;
	int			irq;

	wait_queue_head_t	mbq;	///< wait-queue for dma-done events
	struct q_doneinfo	doneinfo[MAX_MAILBOX];
	int			di_head;
	int			di_tail;
	int			ints;	///< int count (debug only)
	spinlock_t		lock;	///< lock for irq/ioctl access


	spinlock_t		oplock;	///< lock for operations
	int			unit;	///< my ordinal
	int			op;	///< 0=none, 1=decode, 2=encode
	u8			padded_short_stripe[4];
	u32			referenceLineBytes;
	JBIG_BIH		bih;
};

struct jbig_quasar *s_jbig_context[JBIG_NUM_CONTEXTS];

#include "qjbigdec.c"

static int qjbig_open(struct inode* inode, struct file* filp)
{
	struct jbig_quasar *jbig;
    volatile u8 __iomem *reg;
    volatile u8 __iomem *sts;

	jbig = container_of(inode->i_cdev, struct jbig_quasar, cdev);
	if (jbig->ref > 0) {
		printk("qjbig busy!!!!!!!!\n");
		return -EBUSY;
	}
	jbig->ref++;
	jbig->minor = iminor(inode);
	filp->private_data = jbig;
	jbig->di_head = jbig->di_tail = 0;
// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
    reg = ioremap(IPM1CLKDISCTRL, 4);
    sts = ioremap(IPM1CLKDISSTAT, 4);
    switch (jbig->minor)
    {
    case 0:
        writel(readl(reg) & IPM1CLKDISCTRL__IPM1_JBIG0__INV_MASK, reg);
        while (readl(sts) & IPM1CLKDISSTAT__IPM1_JBIG0__MASK) {;}
        break;
    case 1:
        writel(readl(reg) & IPM1CLKDISCTRL__IPM1_JBIG1__INV_MASK, reg);
        while (readl(sts) & IPM1CLKDISSTAT__IPM1_JBIG1__MASK) {;}
        break;
#if defined(Q6600)
    case 2:
        writel(readl(reg) & IPM1CLKDISCTRL__IPM1_JBIG2__INV_MASK, reg);
        while (readl(sts) & IPM1CLKDISSTAT__IPM1_JBIG2__MASK) {;}
        break;
    case 3:
        writel(readl(reg) & IPM1CLKDISCTRL__IPM1_JBIG3__INV_MASK, reg);
        while (readl(sts) & IPM1CLKDISSTAT__IPM1_JBIG3__MASK) {;}
        break;
#endif
    default:
        break;
    }
	iounmap(reg);
	iounmap(sts);
	reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
    switch (jbig->minor)
    {
    case 0:
        writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_JBIG0__INV_MASK, reg);
		printk(KERN_INFO "QJBIG %d Reset\n", jbig->minor);
        break;
    case 1:
        writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_JBIG1__INV_MASK, reg);
		printk(KERN_INFO "QJBIG %d Reset\n", jbig->minor);
        break;
#if defined(Q6600)
    case 2:
        writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_JBIG2__INV_MASK, reg);
		printk(KERN_INFO "QJBIG %d Reset\n", jbig->minor);
        break;
    case 3:
        writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_JBIG3__INV_MASK, reg);
		printk(KERN_INFO "QJBIG %d Reset\n", jbig->minor);
        break;
#endif
    default:
        printk(KERN_INFO "QJBIG - unknown minor: %d\n", jbig->minor);
        break;
    }
    iounmap(reg);
#endif
	return 0;
}

static int qjbig_release(struct inode* inode, struct file* filp)
{
	struct jbig_quasar *jbig;

	jbig = container_of(inode->i_cdev, struct jbig_quasar, cdev);
	if (jbig->ref <= 0)
		return -EFAULT;
	jbig->ref--;
	filp->private_data = NULL;
	return 0;
}

static ssize_t qjbig_read(
	struct file* filp, char __user *buffer, size_t length, loff_t* offset)
{
	struct jbig_quasar *jbig;

	jbig = (struct jbig_quasar*)filp->private_data;
	return -EINVAL;
}

static ssize_t qjbig_write(
	struct file* filp, const char __user *buffer, size_t length,
	loff_t* offset)
{
	struct jbig_quasar *jbig;

	jbig = (struct jbig_quasar*)filp->private_data;
	return -EINVAL;
}

static unsigned int qjbig_poll(struct file *filp, poll_table *wait)
{
	struct jbig_quasar* jbig;
	unsigned int mask = 0;
	unsigned long flags;

	jbig = (struct jbig_quasar*)filp->private_data;

	poll_wait(filp, &jbig->mbq, wait);
	spin_lock_irqsave(&jbig->lock, flags);
	if (jbig->di_head != jbig->di_tail)
		mask |= POLLIN | POLLRDNORM; /* readable */
	spin_unlock_irqrestore(&jbig->lock, flags);
	return mask;
}

static long qjbig_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct jbig_quasar *jbig;
	unsigned long flags;
	int ret = 0;

	jbig = (struct jbig_quasar*)filp->private_data;

	switch (cmd)
	{
	case QGETDONE:
		if (jbig->di_head == jbig->di_tail)
			return -EAGAIN;
		ret = copy_to_user(
			(void*)arg,
			(void*)&jbig->doneinfo[jbig->di_tail],
			sizeof(jbig->doneinfo[0]));
		spin_lock_irqsave(&jbig->lock, flags);
		jbig->di_tail++;
		if (jbig->di_tail >= MAX_MAILBOX)
			jbig->di_tail = 0;
		spin_unlock_irqrestore(&jbig->lock, flags);
		break;
	case QWAITDONE:
		while (jbig->di_head == jbig->di_tail)
		{
			/* TODO - check for dma in progress? */
			if (wait_event_interruptible(
				jbig->mbq, (jbig->di_head != jbig->di_tail)))
				return -ERESTARTSYS;
		}
		if (jbig->di_head != jbig->di_tail) {
			ret = copy_to_user(
				(void*)arg,
				(void*)&jbig->doneinfo[jbig->di_tail],
				sizeof(jbig->doneinfo[0]));
			spin_lock_irqsave(&jbig->lock, flags);
			jbig->di_tail++;
			if (jbig->di_tail >= MAX_MAILBOX)
				jbig->di_tail = 0;
			spin_unlock_irqrestore(&jbig->lock, flags);
			ret = 0;
		} else {
			/* TODO - check for dma in progress? */
			ret = -EAGAIN;
		}
		break;
	default:
		printk(KERN_WARNING "qjbig - bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static int qjbig_mmap(struct file* filp, struct vm_area_struct* vma)
{
	struct jbig_quasar *jbig;
	int length, ret = 0;

	jbig = (struct jbig_quasar*)filp->private_data;
	if (! jbig)
		return -ENODEV;

	/* !! mark pages as uncached for now !! */
	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	length = vma->vm_end - vma->vm_start;
	if(length > (jbig->ioreg_end - jbig->ioreg_start)) {
		printk(KERN_WARNING "JBIG VMA length truncated to io region\n");
		length = (jbig->ioreg_end - jbig->ioreg_start); 
	}

	ret = remap_pfn_range(
        		vma,
        		vma->vm_start,
        		jbig->ioreg_start >> PAGE_SHIFT,
        		length,
        		vma->vm_page_prot
        	);
	return ret;
}

static irqreturn_t quasar_jbig_interrupt(int irq, void *dev_id)
{
	struct jbig_quasar *jbig = (struct jbig_quasar *)dev_id;
	struct timeval tv;
	int ret;
	unsigned long flags;

    readl(jbig->regs + JBJSR);

	spin_lock_irqsave(&jbig->lock, flags);
	jbig->doneinfo[jbig->di_head].msg     = QMSG_DONE;
	jbig->doneinfo[jbig->di_head].detail  = 0;
	do_gettimeofday(&tv);
	jbig->doneinfo[jbig->di_head].endtime =
		(unsigned long long)tv.tv_sec * 1000000 +
		(unsigned long long)tv.tv_usec;
	jbig->doneinfo[jbig->di_head].cycles  = 0;
	jbig->di_head++;
	if (jbig->di_head >= MAX_MAILBOX)
		jbig->di_head = 0;
	spin_unlock_irqrestore(&jbig->lock, flags);

	/*printk("jbig int %d\n", jbig->ints++);*/

	/* wakeup tasks waiting on dma done */
	wake_up_interruptible(&jbig->mbq);

	ret = IRQ_HANDLED;
	return ret;
}

static struct file_operations quasar_jbig_ops = {
	.owner		= THIS_MODULE,
	.open		= qjbig_open,
	.release	= qjbig_release,
	.read		= qjbig_read,
	.write		= qjbig_write,
	.poll		= qjbig_poll,
	.mmap		= qjbig_mmap,
	.unlocked_ioctl	= qjbig_ioctl,
	.compat_ioctl	= qjbig_ioctl 
};

static int __init quasar_jbig_probe(struct platform_device *pdev)
{
	struct jbig_quasar *jbig;
	struct resource *regs;
	dev_t  jbign;
	int    ret;
	static int njbigs = 0;

	jbig = kzalloc(sizeof(*jbig), GFP_KERNEL);
	if (!jbig) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	cdev_init(&jbig->cdev, &quasar_jbig_ops);
	jbign = MKDEV(QJBIG_MAJOR, njbigs);
	ret = cdev_add(&jbig->cdev, jbign, 1);
	if (ret) {
		dev_dbg(&pdev->dev, "could not create char dev %d\n", njbigs);
		goto out_err;
	}
	regs = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!regs) {
		dev_dbg(&pdev->dev, "no mmio reg resource defined\n");
		ret = -ENXIO;
		goto out_rerr;
	}
	jbig->ioreg_start = regs->start;
	jbig->ioreg_end   = regs->end;

	jbig->regs = ioremap(regs->start, regs->end - regs->start + 1);
	if (!jbig->regs) {
		ret = -ENOMEM;
		dev_dbg(&pdev->dev, "could not map reg I/O memory\n");
		goto out_ioerr;
	}
	jbig->irq = platform_get_irq(pdev, 0);
	if (jbig->irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		ret = -ENXIO;
		goto out_ioerr;
	}
	ret = request_irq(
		jbig->irq, quasar_jbig_interrupt, 0, "jbigdma",
		jbig);
	if (ret) {
		dev_dbg(&pdev->dev, "could not request irq %d\n", jbig->irq);
		jbig->irq = -1;
		goto out_ioerr;
	}
	spin_lock_init(&jbig->lock);
	init_waitqueue_head(&jbig->mbq);

	platform_set_drvdata(pdev, jbig);
	printk("QJBIG %d - mapped at %px, irq %d\n",
		njbigs, jbig->regs, jbig->irq);
	s_jbig_context[njbigs] = jbig;
	jbig->unit = njbigs++;
	device_init_wakeup(&pdev->dev, 1);
	return 0;
out_ioerr:
	iounmap(jbig->regs);
out_rerr:
	cdev_del(&jbig->cdev);
out_err:
	kfree(jbig);
	return ret;
}

static int __exit quasar_jbig_remove(struct platform_device *pdev)
{
	struct jbig_quasar *jbig = platform_get_drvdata(pdev);

	if (jbig->irq > 0)
		free_irq(jbig->irq, jbig);
	iounmap(jbig->regs);
	cdev_del(&jbig->cdev);
	kfree(jbig);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static int quasar_jbig_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct jbig_quasar *jbig = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM1CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (jbig->minor)
		{
		default:
			break;
		case 0:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM1_JBIG0__MASK, reg);
			writel(readl(ctl) | IPM1CLKDISCTRL__IPM1_JBIG0__MASK, ctl);
			while ((readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JBIG0__MASK) == 0x0) {;}
			break;
		case 1:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM1_JBIG1__MASK, reg);
			writel(readl(ctl) | IPM1CLKDISCTRL__IPM1_JBIG1__MASK, ctl);
			while ((readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JBIG1__MASK) == 0x0) {;}
			break;
#if defined(Q6600)
		case 2:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM1_JBIG2__MASK, reg);
			writel(readl(ctl) | IPM1CLKDISCTRL__IPM1_JBIG2__MASK, ctl);
			while ((readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JBIG2__MASK) == 0x0) {;}
			break;
		case 3:
			writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM1_JBIG3__MASK, reg);
			writel(readl(ctl) | IPM1CLKDISCTRL__IPM1_JBIG3__MASK, ctl);
			while ((readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JBIG3__MASK) == 0x0) {;}
			break;
#endif
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
    }
#endif
            
    //printk("    ==== quasar_jbig_suspend for qjbig%d, state=%d ====\n", jbig->minor, state);
	return 0;
}

static int quasar_jbig_resume(struct platform_device *pdev)
{
	struct jbig_quasar *jbig = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u8 __iomem *ctl;
		u32 temp;

		ctl = ioremap(IPM1CLKDISCTRL, 8);
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		switch (jbig->minor)
		{
		default:
			break;
		case 0:
			writel(readl(ctl) & IPM1CLKDISCTRL__IPM1_JBIG0__INV_MASK, ctl);
			while (readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JBIG0__MASK) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_JBIG0__INV_MASK, reg);
			break;
		case 1:
			writel(readl(ctl) & IPM1CLKDISCTRL__IPM1_JBIG1__INV_MASK, ctl);
			while (readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JBIG1__MASK) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_JBIG1__INV_MASK, reg);
			break;
#if defined(Q6600)
		case 2:
			writel(readl(ctl) & IPM1CLKDISCTRL__IPM1_JBIG2__INV_MASK, ctl);
			while (readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JBIG2__MASK) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_JBIG2__INV_MASK, reg);
			break;
		case 3:
			writel(readl(ctl) & IPM1CLKDISCTRL__IPM1_JBIG3__INV_MASK, ctl);
			while (readl(ctl + 4) & IPM1CLKDISSTAT__IPM1_JBIG3__MASK) {;}
			writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM1_JBIG3__INV_MASK, reg);
			break;
#endif
		}
		temp = readl(reg);
		iounmap(reg);
		iounmap(ctl);
	}
#endif
	
	//printk("    ==== quasar_jbig_resume for qjbig%d ====\n", jbig->minor);
	return 0;
}

MODULE_ALIAS("platform:quasar-jbig");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-jbig" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver quasar_jbig_driver_ops = {
	.probe	    = quasar_jbig_probe,
	.remove	    = quasar_jbig_remove,
	.suspend	= quasar_jbig_suspend,
	.resume		= quasar_jbig_resume,
	.driver	= {
		.name	= "quasar-jbig",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init quasar_jbig_init(void)
{
	int ret;

	ret = platform_driver_register(&quasar_jbig_driver_ops);
	return ret;
}
module_init(quasar_jbig_init);

static void __exit quasar_jbig_exit(void)
{
	platform_driver_unregister(&quasar_jbig_driver_ops);
}
module_exit(quasar_jbig_exit);

MODULE_DESCRIPTION("Quasar JBIG driver");
MODULE_LICENSE("Dual BSD/GPL");

