/*
 * A driver for the power control for QBit SOCs
 *
 * Quasar GPIO I/F kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/01/19 08:48:00 $
//  $Change: 57110 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#define PWR_DEVNAME	"qgpio"

#define MAX_IOREGS		    4

struct quasar_io_region {
	u32						start;
	u32						end;
	volatile u8	__iomem		*base;
};

struct quasar_gpio {
	struct cdev				cdev;
	struct device			*frdev;

	struct quasar_io_region	ioregs[MAX_IOREGS];
	int						nioregs;
	int						refs;				/* reference count */
	int						minor;				/* device minor number */
};

static volatile u8 __iomem* gpio_addr(struct quasar_gpio* qgpio, u32 reg)
{
	volatile u8 __iomem* addr = 0;
 	int      i;

	//printk(KERN_WARNING PWR_DEVNAME "gpio_addr - addr map, reg %08x, ord=%d, <%08X-%08X>--<%08X-%08X>--<%08X-%08X>--<%08X-%08X>\n", 
	//       reg, qgpio->minor, qgpio->ioregs[0].start, qgpio->ioregs[0].end, qgpio->ioregs[1].start, qgpio->ioregs[1].end, qgpio->ioregs[2].start, qgpio->ioregs[2].end, qgpio->ioregs[3].start, qgpio->ioregs[3].end);
	/* check for existing mapping */
	for (i = 0; i < qgpio->nioregs; i++) {
		if(reg >= qgpio->ioregs[i].start && reg < qgpio->ioregs[i].end) {
			return(qgpio->ioregs[i].base + (reg - qgpio->ioregs[i].start));
		}
	}
	printk(KERN_WARNING PWR_DEVNAME "qgpio_addr - could not map reg %08x\n", reg);
	return addr;
}

static int gpio_readl(struct quasar_gpio* qgpio, u32 reg, unsigned* val)
{
	volatile u32 rv = 0;
	int ret = 0;
	volatile u8 __iomem* addr = gpio_addr(qgpio, reg);
	
	if(addr == NULL)
		return -1;
	rv = readl(addr);
	/*printk("readl %08X [%08X] -> %08X\n", reg, addr, rv);*/
	*val = (unsigned long)rv;
	return ret;
}

static int gpio_writel(struct quasar_gpio* qgpio, u32 reg, u32 val)
{
	volatile u8 __iomem* addr = gpio_addr(qgpio, reg);
	if(addr == NULL)
		return -1;
	/* printk("writel %08X [%08X] <= %08x\n", reg, addr, val);*/
	writel(val, addr);
	return 0;
}

static int qgpio_open(struct inode* inode, struct file* filp)
{
	struct quasar_gpio *qgpio;
	int minor;
	
	qgpio = container_of(inode->i_cdev, struct quasar_gpio, cdev);
	minor = iminor(inode);
	qgpio->refs++;
	filp->private_data = qgpio;
	return 0;
}

static int qgpio_release(struct inode* inode, struct file* filp)
{
	struct quasar_gpio* qgpio;
	
	qgpio = (struct quasar_gpio*)filp->private_data;
	if(qgpio->refs <= 0)
		return -EBADF;
	qgpio->refs--;
	filp->private_data = NULL;
	return 0;
}

static long qgpio_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct quasar_gpio* qgpio;
	struct q_regio ioval;
	int ret = 0;
	
	qgpio = (struct quasar_gpio*)filp->private_data;

	if(! qgpio)
		return -ENODEV;

	switch(cmd)
	{
	case QSETREG:
		ret = copy_from_user(&ioval, (void*)arg, sizeof(struct q_regio));
		ret = gpio_writel(qgpio, ioval.reg, (unsigned long)ioval.val);
		break;
	case QGETREG:
		ret = copy_from_user(&ioval, (void*)arg, sizeof(struct q_regio));
		ret = gpio_readl(qgpio, ioval.reg, (unsigned*)&ioval.val);
		ret = copy_to_user((void*)arg, &ioval, sizeof(struct q_regio));
		break;

	default:
		printk(KERN_WARNING PWR_DEVNAME "Bad ioctl %d\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static struct file_operations qgpio_ops = {
	.owner		= THIS_MODULE,
	.open		= qgpio_open,
	.release	= qgpio_release,
	.unlocked_ioctl	= qgpio_ioctl,
	.compat_ioctl	= qgpio_ioctl 
};

static int __init qgpio_probe(struct platform_device *pdev)
{
	struct resource	*ioreg;
	struct quasar_gpio *qgpio;
    static int ngpios=0;
	dev_t  qgpion;
	int ret = 0;

	qgpio = kzalloc(sizeof(struct quasar_gpio), GFP_KERNEL);
	if(! qgpio) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	qgpio->frdev = &pdev->dev;
	qgpio->nioregs = 0;
	
    printk(KERN_WARNING PWR_DEVNAME "init qgpio_probe ...\n"); 
	do {
		ioreg = platform_get_resource(pdev, IORESOURCE_MEM, qgpio->nioregs);
		if (ioreg) {
			qgpio->ioregs[qgpio->nioregs].start = ioreg->start;
			qgpio->ioregs[qgpio->nioregs].end = ioreg->end;
			qgpio->ioregs[qgpio->nioregs].base =
				ioremap(ioreg->start, ioreg->end - ioreg->start + 4);
            //printk(KERN_WARNING PWR_DEVNAME "[%d]-- start=0x%08X, end=0x%08X, base=0x%08X\n", qgpio->nioregs, ioreg->start, ioreg->end, qgpio->ioregs[qgpio->nioregs].base); 
			if (!qgpio->ioregs[qgpio->nioregs].base) {
				ret = -ENOMEM;
				dev_dbg(&pdev->dev, "could not map I/O memory %0d\n", qgpio->nioregs);
				goto out_abort;
			}
			/*
			printk("qgpio - mapped %px %x to %x at %x\n", 
					qgpio, ioreg->start, ioreg->end, qgpio->ioregs[qgpio->nioregs].base);
			*/
			qgpio->nioregs++;
		}
	} while (ioreg && (qgpio->nioregs < MAX_IOREGS));

	cdev_init(&qgpio->cdev, &qgpio_ops);
	qgpion = MKDEV(QGPIO_MAJOR, ngpios);
	ret = cdev_add(&qgpio->cdev, qgpion, 1);
	if (ret) {
		printk(KERN_WARNING "qgpio - could not create char dev %d\n", ngpios);
		ret = -ENODEV;
		goto out_abort;
	}
	ngpios++;
	ret = 0;

	platform_set_drvdata(pdev, qgpio);
	device_init_wakeup(&pdev->dev, 1);
	dev_info(&pdev->dev, "Quasar QGPIO\n");
	return 0;

out_abort:
	while (qgpio->nioregs > 0) {
		qgpio->nioregs--;
		iounmap(qgpio->ioregs[qgpio->nioregs].base);
	}

	kfree(qgpio);
	return ret;
}

static int __exit qgpio_remove(struct platform_device *pdev)
{
	struct quasar_gpio *qgpio = platform_get_drvdata(pdev);

	cdev_del(&qgpio->cdev);
	while (qgpio->nioregs > 0) {
		qgpio->nioregs--;
		iounmap(qgpio->ioregs[qgpio->nioregs].base);
	}

	kfree(qgpio);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

MODULE_ALIAS("platform:qgpio");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-gpio" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver qgpio_driver_ops = {
	.probe		= qgpio_probe,
	.remove		= qgpio_remove,
	.driver		= {
		.name	= "quasar-gpio",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init qgpio_init(void)
{
	int ret;

		ret = platform_driver_register(&qgpio_driver_ops);
	return ret;
}
module_init(qgpio_init);

static void __exit qgpio_exit(void)
{
	platform_driver_unregister(&qgpio_driver_ops);
}
module_exit(qgpio_exit);

MODULE_DESCRIPTION("Quasar GPIO Driver");
MODULE_LICENSE("Dual BSD/GPL");

