/*
 * A driver for FIR h/w block found on Quatro processors
 *
 *  Quasar FIR kernel driver
 *
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/01/19 08:48:00 $
//  $Change: 57110 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#define QFIR_DEVNAME	"qfir"

#define MAX_MAILBOX 64

/* The fireregs headers split the fir region into two separate areas
 * with "FIR_IP_SPARE" starting the FIR_IP region.  For simplicity
 * the kernel driver just maps the fir region as one, so just be sure
 * to not use any _OFF in the IP region directly
 */
// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	#define QFIR_FIR_IP_INTERRUPT_ENABLE_OFF (FIR0FIR_IP_INTERRUPT_ENABLE - FIR0_BASE) 
	#define QFIR_FIR_LL_MISC_STAT_OFF (FIR0FIR_LL_MISC_STAT - FIR0_BASE)
	#define FIR_LL_MISC_STAT__IEXEC_ILLEGAL_INST__MASK FIR0FIR_LL_MISC_STAT__IEXEC_ILLEGAL_INST__MASK
#endif
	
struct fir_quasar {
	int			ref;
	int			minor;
	struct cdev		cdev;
	volatile u8		__iomem	*regs[1];
	unsigned long		ioreg_start[1];
	unsigned long		ioreg_end[1];
	int			nioregs;
	int			irq;

	wait_queue_head_t	mbq;	/* wait-queue for dma-doneinterrupt events */
	struct q_doneinfo	doneinfo[MAX_MAILBOX];
	int			di_head, di_tail;
	spinlock_t		lock;
};

static int qfir_open(struct inode* inode, struct file* filp)
{
	struct fir_quasar *fir;
    volatile u8 __iomem *reg;
    volatile u8 __iomem *sts;
	
	fir = container_of(inode->i_cdev, struct fir_quasar, cdev);
	if(fir->ref > 0) {
		return -EBUSY;
	}
	fir->ref++;
	fir->minor = iminor(inode);
	filp->private_data = fir;
	fir->di_head = fir->di_tail = 0;
// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
    reg = ioremap(IPM2CLKDISCTRL, 4);
    sts = ioremap(IPM2CLKDISSTAT, 4);
    switch (fir->minor)
    {
    case 0:
        writel(readl(reg) & IPM2CLKDISCTRL__IPM2_FIR0__INV_MASK, reg);
        while (readl(sts) & IPM2CLKDISSTAT__IPM2_FIR0__MASK) {;}
        break;
    case 1:
        writel(readl(reg) & IPM2CLKDISCTRL__IPM2_FIR1__INV_MASK, reg);
        while (readl(sts) & IPM2CLKDISSTAT__IPM2_FIR1__MASK) {;}
        break;
#if defined(Q6600)
    case 2:
        writel(readl(reg) & IPM2CLKDISCTRL__IPM2_FIR2__INV_MASK, reg);
        while (readl(sts) & IPM2CLKDISSTAT__IPM2_FIR2__MASK) {;}
        break;
#endif
    default:
        break;
    }
	iounmap(reg);
	iounmap(sts);
	reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
    switch (fir->minor)
    {
    case 0:
        writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_FIR0__INV_MASK, reg);
        printk(KERN_INFO "QFIR %d Reset\n", fir->minor);
        break;
    case 1:
        writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_FIR1__INV_MASK, reg);
        printk(KERN_INFO "QFIR %d Reset\n", fir->minor);
        break;
#if defined(Q6600)
    case 2:
        writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_FIR2__INV_MASK, reg);
        printk(KERN_INFO "QFIR %d Reset\n", fir->minor);
        break;
#endif
    default:
        printk(KERN_INFO "QFIR - unknown minor: %d\n", fir->minor);
        break;
    }
    iounmap(reg);
#endif
	return 0;
}

static int qfir_release(struct inode* inode, struct file* filp)
{
	struct fir_quasar *fir;
	
	fir = container_of(inode->i_cdev, struct fir_quasar, cdev);
	if(fir->ref <= 0)
		return -EFAULT;
	fir->ref--;
	filp->private_data = NULL;
	return 0;
}

static ssize_t qfir_read(struct file* filp, char __user *buffer, size_t length, loff_t* offset)
{
	struct fir_quasar *fir;	
	
	fir = (struct fir_quasar*)filp->private_data;
	return -EINVAL;
}

static ssize_t qfir_write(struct file* filp, const char __user *buffer, size_t length, loff_t* offset)
{
	struct fir_quasar *fir;	
	
	fir = (struct fir_quasar*)filp->private_data;
	return -EINVAL;
}

static unsigned int qfir_poll(struct file *filp, poll_table *wait)
{
	struct fir_quasar* fir;	
	unsigned int mask = 0;
	unsigned long flags;
	
	fir = (struct fir_quasar*)filp->private_data;

	poll_wait(filp, &fir->mbq, wait);
	spin_lock_irqsave(&fir->lock, flags);
	if(fir->di_head != fir->di_tail) {
		mask |= POLLIN | POLLRDNORM; /* readable */
	}
	spin_unlock_irqrestore(&fir->lock, flags);
	return mask;
}

static int qfir_setimom(struct fir_quasar *fir, struct q_fir_set *qset, int im)
{
	volatile u8 __iomem *reg;
	unsigned long offset;

	if (im)
	{
		offset = fir->minor ? SB_FIR1_IM0 : SB_FIR0_IM0;
#if defined(Q6600)
		if (fir->minor == 2)
		{
			offset = SB_FIR2_IM0;
		}
#endif
		reg = ioremap(offset, 16);
		writel(qset->data0, reg);
		writel(qset->data1, reg + 0x4);
		writel(qset->data2, reg + 0x8);
		writel(qset->data3, reg + 0xc);
		iounmap(reg);
		offset = fir->minor ? FIR1_IMON0_CTRL : FIR0_IMON0_CTRL;
#if defined(Q6600)
		if (fir->minor == 2)
		{
			offset = FIR2_IMON0_CTRL;
		}
#endif
		reg = ioremap(offset, 16);
		writel(qset->cb + (qset->data0 ? 0x10 : 0x0), reg);
		writel(qset->cb + (qset->data1 ? 0x10 : 0x0), reg + 0x4);
		writel(qset->cb + (qset->data2 ? 0x10 : 0x0), reg + 0x8);
		writel(qset->cb + (qset->data3 ? 0x10 : 0x0), reg + 0xc);
		iounmap(reg);
	}
	else
	{
		offset = fir->minor ? SB_FIR1_OM0 : SB_FIR0_OM0;
#if defined(Q6600)
		if (fir->minor == 2)
		{
			offset = SB_FIR2_OM0;
		}
#endif
		reg = ioremap(offset, 16);
		writel(qset->data0, reg);
		writel(qset->data1, reg + 0x4);
		writel(qset->data2, reg + 0x8);
		writel(qset->data3, reg + 0xc);
		iounmap(reg);
		offset = fir->minor ? FIR1_OMON0_CTRL : FIR0_OMON0_CTRL;
#if defined(Q6600)
		if (fir->minor == 2)
		{
			offset = FIR2_OMON0_CTRL;
		}
#endif
		reg = ioremap(offset, 16);
		writel(qset->cb + (qset->data0 ? 0x10 : 0x0), reg);
		writel(qset->cb + (qset->data1 ? 0x10 : 0x0), reg + 0x4);
		writel(qset->cb + (qset->data2 ? 0x10 : 0x0), reg + 0x8);
		writel(qset->cb + (qset->data3 ? 0x10 : 0x0), reg + 0xc);
		iounmap(reg);
	}
	return 0;
}

static long qfir_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct fir_quasar *fir;	
	struct q_fir_set qset;
	unsigned long flags;
	int ret = 0;
	
	fir = (struct fir_quasar*)filp->private_data;
	
	switch(cmd)
	{
	case QGETDONE:
		if(fir->di_head == fir->di_tail)
			return -EAGAIN;
		copy_to_user((void*)arg, (void*)&fir->doneinfo[fir->di_tail], sizeof(struct q_doneinfo));
		spin_lock_irqsave(&fir->lock, flags);
		fir->di_tail++;
		if(fir->di_tail >= MAX_MAILBOX)
			fir->di_tail = 0;
		spin_unlock_irqrestore(&fir->lock, flags);
		break;
	case QWAITDONE:
		while(fir->di_head == fir->di_tail)
		{
			/* TODO - check for dma in progress? */
			if (wait_event_interruptible(fir->mbq,
				(fir->di_head != fir->di_tail)))
				return -ERESTARTSYS;
		}
		if(fir->di_head != fir->di_tail) {
			copy_to_user((void*)arg,
				(void*)&fir->doneinfo[fir->di_tail], sizeof(struct q_doneinfo));
			spin_lock_irqsave(&fir->lock, flags);
			fir->di_tail++;
			if(fir->di_tail >= MAX_MAILBOX)
				fir->di_tail = 0;
			spin_unlock_irqrestore(&fir->lock, flags);
			ret = 0;
		}
		else {
			/* TODO - check for dma in progress? */
			ret = -EAGAIN;
		}
		break;
	case QFIRSETIM:
	case QFIRSETOM:
		if(copy_from_user((void*)&qset,
			(void*)arg, sizeof(struct q_fir_set)))
			return -EINVAL;
		ret = qfir_setimom(fir, &qset, cmd == QFIRSETIM);
		break;
	default:
		printk(KERN_WARNING "qfir - bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static int qfir_mmap(struct file* filp, struct vm_area_struct* vma)
{
	struct fir_quasar *fir;	
	int ret = 0;
	int i;

	fir = (struct fir_quasar*)filp->private_data;	
	if(! fir) { printk("FIR NOT OPEN\n"); return -ENODEV; }

	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	
	// find region in iorange
	//
	for(i = 0; i < fir->nioregs; i++) {
		if((fir->ioreg_start[i] >> PAGE_SHIFT) == vma->vm_pgoff) {
			ret = remap_pfn_range(
						vma,
						vma->vm_start,
						fir->ioreg_start[i] >> PAGE_SHIFT,
						vma->vm_end - vma->vm_start,
						vma->vm_page_prot
					);
			return ret;
		}
	}
	printk(KERN_WARNING
			"FIR - Could not remap region pgoff: %0lx (addr=%08lx)\n",
			vma->vm_pgoff,
			vma->vm_pgoff << PAGE_SHIFT);
	return -ENOMEM;
}

static irqreturn_t quasar_fir_interrupt(int irq, void *dev_id)
{
	struct fir_quasar *fir = (struct fir_quasar *)dev_id; 
	struct timeval tv;
	int ret;
	unsigned long flags;
	unsigned long err;

	/* disable sw interrupt from FIR core */
	writel(0, fir->regs[0] + QFIR_FIR_IP_INTERRUPT_ENABLE_OFF);
	
	/* handle errors */
	err = readl(fir->regs[0] + QFIR_FIR_LL_MISC_STAT_OFF);
	if(err & FIR_LL_MISC_STAT__IEXEC_ILLEGAL_INST__MASK)
		printk(KERN_WARNING "FIR LLDMA aborted due to an illegal instruction!\n");

	do_gettimeofday(&tv);

	spin_lock_irqsave(&fir->lock, flags);
	fir->doneinfo[fir->di_head].msg     = QMSG_DONE;
	fir->doneinfo[fir->di_head].detail  = err;
	fir->doneinfo[fir->di_head].endtime = (unsigned long long)tv.tv_sec * 1000000 + (unsigned long long)tv.tv_usec;
	fir->doneinfo[fir->di_head].cycles  = 0;
	fir->di_head++;
	if(fir->di_head >= MAX_MAILBOX)
		fir->di_head = 0;
	spin_unlock_irqrestore(&fir->lock, flags);

	/* wakeup tasks waiting on dma done
	*/
	wake_up_interruptible(&fir->mbq);

	ret = IRQ_HANDLED;
	return ret;
}

static struct file_operations quasar_fir_ops = {
	.owner		= THIS_MODULE,
	.open		= qfir_open,
	.release	= qfir_release,
	.read		= qfir_read,
	.write		= qfir_write,
	.poll		= qfir_poll,
	.mmap		= qfir_mmap,
	.unlocked_ioctl	= qfir_ioctl,
	.compat_ioctl	= qfir_ioctl 
};

static int __init quasar_fir_probe(struct platform_device *pdev)
{
	struct fir_quasar *fir;
	struct resource	*regs;
	dev_t  firn;
	int    ret, i;
	static int nfirs = 0;

	fir = kzalloc(sizeof(struct fir_quasar), GFP_KERNEL);
	if (!fir) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	cdev_init(&fir->cdev, &quasar_fir_ops);
	firn = MKDEV(QFIR_MAJOR, nfirs);
	ret = cdev_add(&fir->cdev, firn, 1);
	if (ret) {
		dev_dbg(&pdev->dev, "could not create char dev %d\n", nfirs);
		goto out_err;
	}
	for (fir->nioregs = 0; fir->nioregs < 1; fir->nioregs++) {
		regs = platform_get_resource(pdev, IORESOURCE_MEM, fir->nioregs);
		if (!regs) {
			dev_dbg(&pdev->dev, "no mmio reg resource defined\n");
			ret = -ENXIO;
			goto out_rerr;
		}
		fir->ioreg_start[fir->nioregs] = regs->start;
		fir->ioreg_end[fir->nioregs]   = regs->end;

		fir->regs[fir->nioregs] = ioremap(regs->start,
			regs->end - regs->start + 1);
		if(!fir->regs[fir->nioregs]) {
			ret = -ENOMEM;
			dev_dbg(&pdev->dev, "could not map reg I/O memory %d\n", fir->nioregs);
			goto out_ioerr;
		}
	}
	fir->irq = platform_get_irq(pdev, 0);
	if (fir->irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		ret = -ENXIO;
		goto out_ioerr;
	}
	ret = request_irq(fir->irq, quasar_fir_interrupt,
		0, "firdma", fir);
	if (ret) {
		dev_dbg(&pdev->dev, "could not request irq %d\n", fir->irq);
		fir->irq = -1;
		goto out_ioerr;
	}
	spin_lock_init(&fir->lock);
	init_waitqueue_head(&fir->mbq);

	nfirs++;
	platform_set_drvdata(pdev, fir);
	printk("QFIR - mapped at %px, irq %d\n", fir->regs, fir->irq);
	device_init_wakeup(&pdev->dev, 1);
	return 0;
out_ioerr:
	for (i = 0; i < fir->nioregs; i++)
		iounmap(fir->regs[i]);
out_rerr:
	cdev_del(&fir->cdev);		
out_err:
	kfree(fir);
	return ret;
}

static int __exit quasar_fir_remove(struct platform_device *pdev)
{
	struct fir_quasar *fir = platform_get_drvdata(pdev);
	int i;

	if(fir->irq > 0)
		free_irq(fir->irq, fir);
	for (i = 0; i < fir->nioregs; i++)
		iounmap(fir->regs[i]);
	cdev_del(&fir->cdev);		
	kfree(fir);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static int quasar_fir_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct fir_quasar *fir = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
        u8 __iomem *reg;
        u8 __iomem *ctl;
        u32 temp;

        ctl = ioremap(IPM2CLKDISCTRL, 8);
        reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
        switch (fir->minor)
        {
        default:
            break;
        case 0:
            writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM2_FIR0__MASK, reg);
            writel(readl(ctl) | IPM2CLKDISCTRL__IPM2_FIR0__MASK, ctl);
            while ((readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_FIR0__MASK) == 0x0) {;}
            break;
        case 1:
            writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM2_FIR1__MASK, reg);
            writel(readl(ctl) | IPM2CLKDISCTRL__IPM2_FIR1__MASK, ctl);
            while ((readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_FIR1__MASK) == 0x0) {;}
            break;
#if defined(Q6600)
        case 2:
            writel(readl(reg) | RSTGEN_SWRSTSTATIC3__IPM2_FIR2__MASK, reg);
            writel(readl(ctl) | IPM2CLKDISCTRL__IPM2_FIR2__MASK, ctl);
            while ((readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_FIR2__MASK) == 0x0) {;}
            break;
#endif
        }
        temp = readl(reg);
        iounmap(reg);
        iounmap(ctl);
    }
#endif
	return 0;
}

static int quasar_fir_resume(struct platform_device *pdev)
{
	struct fir_quasar *fir = platform_get_drvdata(pdev);

// TBD: separate for 6600?
#if defined(Q6300) || defined(Q6600)
	{
        u8 __iomem *reg;
        u8 __iomem *ctl;
        u32 temp;

        ctl = ioremap(IPM2CLKDISCTRL, 8);
        reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
        switch (fir->minor)
        {
        default:
            break;
        case 0:
            writel(readl(ctl) & IPM2CLKDISCTRL__IPM2_FIR0__INV_MASK, ctl);
            while (readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_FIR0__MASK) {;}
            writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_FIR0__INV_MASK, reg);
            break;
        case 1:
            writel(readl(ctl) & IPM2CLKDISCTRL__IPM2_FIR1__INV_MASK, ctl);
            while (readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_FIR1__MASK) {;}
            writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_FIR1__INV_MASK, reg);
            break;
#if defined(Q6600)
        case 2:
            writel(readl(ctl) & IPM2CLKDISCTRL__IPM2_FIR2__INV_MASK, ctl);
            while (readl(ctl + 4) & IPM2CLKDISSTAT__IPM2_FIR2__MASK) {;}
            writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_FIR2__INV_MASK, reg);
            break;
#endif
        }
        temp = readl(reg);
        iounmap(reg);
        iounmap(ctl);
	}
#endif
	return 0;
}

MODULE_ALIAS("platform:quasar-fir");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-fir" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver quasar_fir_driver_ops = {
	.probe		= quasar_fir_probe,
	.remove		= quasar_fir_remove,
	.suspend	= quasar_fir_suspend,
	.resume		= quasar_fir_resume,
	.driver		= {
		.name	= "quasar-fir",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),			
	},
};

static int __init quasar_fir_init(void)
{
	int ret;
	
	ret = platform_driver_register(&quasar_fir_driver_ops);
	return ret;
}
module_init(quasar_fir_init);

static void __exit quasar_fir_exit(void)
{
	platform_driver_unregister(&quasar_fir_driver_ops);
}
module_exit(quasar_fir_exit);

MODULE_DESCRIPTION("Quasar FIR driver");
MODULE_LICENSE("Dual BSD/GPL");

