/*
 * Quasar LCD controller kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2022/01/06 10:11:28 $
//  $Change: 56812 $
//
// =========================================================
#define DEBUG

#include <linux/dma-mapping.h>
#include <linux/device.h>
#include "uapi/qfb.h"

size_t qfb_get_im_size(size_t width, size_t height, int bpp, int packed)
{
	size_t im_size;

	im_size = width * height;
	if (bpp <= 8) {
		im_size /= (8/bpp);
	} else if (bpp <= 16) {
		im_size *= ((bpp+7)/8);
	} else {
		im_size *= (packed) ? 3 : 4;
	}

	return im_size;
}


static void qfb_dump_bi_prop(struct device *dev, struct qfb_bi_prop* prop)
{
	dev_dbg(dev, "== qfb_bi_prop: width(%d), height(%d), packed(%d)\n", 
			prop->width,
			prop->height,
			prop->packed);
}

static void qfb_dump_bi(
		struct device *dev, 
		struct qfb_bi_prop *prop, 
		struct qfb_bi_node* bi,
		int bpp)
{
	int i;
	size_t im_size;

	im_size = qfb_get_im_size(prop->width, prop->height,
			bpp,
			prop->packed);

	dev_dbg(dev, "image size = %zu\n", im_size);

	dev_dbg(dev, "== (bi) (%px) bin[0:2]=%02x %02x %02x\n", 
			bi->bin, 
			bi->bin[0], bi->bin[1], bi->bin[2]);

	dev_dbg(dev, "== (bi) bin[x-3:x-1]=%02x %02x %02x\n",
			bi->bin[im_size-3], bi->bin[im_size-2], bi->bin[im_size-1]);

	dev_dbg(dev, "== (bi) en_ow(%d), ow_prio(%d)\n", bi->en_ow, bi->ow_prio);

	for (i=0; i< MAX_OW; i++) {
		if (bi->ow[i].bin) {
			dev_dbg(dev, "== (bi:ow[%d]) %px\n", 
					i, bi->ow[i].bin); 
			dev_dbg(dev, "== (bi:ow) bin[0:2] = %02x %02x %02x\n", 
					bi->ow[i].bin[0],
					bi->ow[i].bin[1],
					bi->ow[i].bin[2]);
			dev_dbg(dev, "== (bi:ow) x(%d), y(%d), width(%d), height(%d), alpha_src(%d), const_alpha(%d), packed(%d)\n",
					bi->ow[i].x,
					bi->ow[i].y,
					bi->ow[i].width,
					bi->ow[i].height,
					bi->ow[i].alpha_src,
					bi->ow[i].const_alpha,
					bi->ow[i].packed);
		}
	}
}

void qfb_dump_devinfo(struct device *dev, struct qfb_devinfo *conf)
{
	int i;

	dev_dbg(dev, "DUMP :: qfb_devinfo :: %px\n", conf);
	qfb_dump_bi_prop(dev, &conf->bi_prop);

	for (i=0; i<MAX_BI; i++) {
		if (conf->bi[i].bin) {
			dev_dbg(dev, "== bi[%d]\n", i); 
			qfb_dump_bi(dev, &conf->bi_prop, &conf->bi[i], conf->bpp);
		}
	}
	dev_dbg(dev, "== fs: node_cnt(%d), end_act(%d)\n", conf->fs.node_cnt, conf->fs.end_act);
	for (i=0; i<conf->fs.node_cnt; i++) {
		dev_dbg(dev, "== (fs) node[%d]\n", i);
		dev_dbg(dev, "== (fs) en_snm1(%d), en_snm2(%d)\n", 
				conf->fs.nodes[i].en_snm1, conf->fs.nodes[i].en_snm2);
		qfb_dump_bi_prop(dev, &conf->fs.nodes[i].bi_prop);
		qfb_dump_bi(dev, 
				&conf->fs.nodes[i].bi_prop, 
				&conf->fs.nodes[i].bi, 
				conf->bpp);
	}
}
