/*
 * Quasar LCD controller kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2022/04/22 09:45:19 $
//  $Change: 60182 $
//
// =========================================================
#ifndef __DRIVERS_QFB_UAPI_QFB_H
#define __DRIVERS_QFB_UAPI_QFB_H

#define RGB_FMT_555             1
#define RGB_FMT_565             2
#define RGB_FMT_666             3
#define RGB_FMT_888             4

#define ALPHA_SRC_NONE          0
#define ALPHA_SRC_CONSTANT      1
#define ALPHA_SRC_PER_PIXEL     2

#define MAX_BI                  2
#define MAX_OW                  2
#define MAX_FSN                 10

#define FS_END_ACT_STOP         0
#define FS_END_ACT_LOOP         1

#define PCLK_HIGH               1
#define PCLK_NORMAL             2
#define PCLK_LOW                3
#define PCLK_DIV_3              4
#define PCLK_DIV_4              5
#define PCLK_DIV_4_5            6
#define PCLK_DIV_5              7
#define PCLK_DIV_6              8
#define PCLK_DIV_7_5            9
#define PCLK_DIV_10             10 
#define PCLK_DIV_12             11
#define PCLK_DIV_14             12


#define QFB_EN_OW(q_dev, id, en) \
	(q_dev)->en_ow = ((q_dev)->en_ow & ~(1 << (id))) | ((en) << (id));

/*
 * Horizontal line alignment w/ VSYNC
 * 
 * QFB_HV_ALIGN_HFRIST: HV alignment, where H line edge 1 PCLK before VSYNC edge
 * QFB_HV_ALIGN_COINCIDE: HV alignment, where H line edge concides with HSYNC edge
 * QFB_HV_ALIGN_VFIRST: HV alignment, where H line edge 1 PCLK after VSYNC edge
 */ 
enum qfb_hv_align {
	QFB_HV_ALIGN_HFRIST = 0,
	QFB_HV_ALIGN_COINCIDE,
	QFB_HV_ALIGN_VFIRST,
};

typedef uint8_t* qfb_ptr_t;

struct qfb_plt_node
{
	qfb_ptr_t bin;
	int       bpp;            // 15 stands for 5:5:5, 16 for 5:6:5
	int       size;           // size in bytes
};

/**
 * Define common properties of BI
 */
struct qfb_bi_prop
{
	int         width;
	int         height;
	int         packed;        // 1 for packed; 0 for unpacked
};

struct qfb_ow_node
{
	qfb_ptr_t bin;                // binary content

	int       x;
	int       y;
	int       width;
	int       height;

	int       alpha_src;          // source of alpha (see ALPHA_SRC_xxx)
	uint8_t   const_alpha;        // constant alpha value

	int       packed;             // 1 for packed; 0 for unpacked
};

struct qfb_bi_node
{
	struct qfb_ow_node ow[MAX_OW];     // Overlay windows
	qfb_ptr_t bin;            // binary content
	int       en_ow;          // enable overlay (bit 0 for window 0, ... etc)
	int       ow_prio;        // overlay priority, 0 for ow1, 1 for ow2
};

/**
 * Internal memory layout for lcdctrl's tag
 * The order of each member in the struct is important
 */
struct __attribute__((__packed__)) qfb_fs_ow
{
	uint32_t ow_addr;          // OW address
	uint32_t ow_size;          // total width and height of OW
	uint32_t ow_pos;           // OW position
};

struct __attribute__((__packed__)) qfb_fs_tag
{
	uint32_t          next_tag_addr;    // next tag address
	uint32_t          ctrl_attr;        // includes STNA_Valid, snm1, snm2, overlay ctrl reg
	uint32_t          bi_addr;          // BI address
	uint32_t          bi_attrs;         // BI attributes, including size, bpp
	struct qfb_fs_ow  ows[MAX_OW];      // ow attributes
	uint32_t          ow_vhsize;        // OW visible horizontal width
};

struct qfb_fs_node
{
	struct qfb_bi_node bi;
	struct qfb_bi_prop bi_prop;

	int en_snm1;        // enable snm1 interrupt
	int en_snm2;        // enable snm2 interrupt

	struct qfb_fs_tag __tag  __attribute__((aligned (4)));  // internal use tag for FS engine
};

struct qfb_fs_seq
{
	struct qfb_fs_node  nodes[MAX_FSN];
	int node_cnt;
	int end_act;        // action to take when reach last node
};

/**
 * struct qfb_timing - panel timing
 * 
 * @hfp_width: Horizontal front porch width
 * @hbp_width: Horizontal back porch width
 * @hsy_width: Horizontal sync width
 * @vfp_width: Vertical front porch width
 * @vbp_width: Vertical back porch width
 * @vsy_width: Vertical sync width
 * 
 */
struct qfb_timing
{
	uint8_t             hfp_width;
	uint8_t             hbp_width;
	uint8_t             hsy_width;

	uint8_t             vfp_width;
	uint8_t             vbp_width;
	uint8_t             vsy_width;
};

/*
 * FIXME: The following structure and QFB_IOCTL_* should be moved
 * to `quasar/qioctl.h`
 */
struct qfb_devinfo
{
	struct qfb_bi_prop  bi_prop;            // Properties of BIs
	struct qfb_bi_node  bi[MAX_BI];         // input BI
	struct qfb_fs_seq   fs;                 // frame sequencer only
	struct qfb_plt_node plt;                // palette

	int                 bpp;                // bit per pixel
	int                 out_port;           // left or right justified (ref spec)
	int                 en_fs;              // enable frame sequencer
	int                 en_db;              // enable double buffer
	int                 en_pd;              // enable pixel duplication mode

	int                 en_cc;              // enable cache coherence

	int                 pclk_speed;         // (FIXME: remove me, this is for emulation) 
											// speed of the pixel clock, could be one of 
											// PCLK_HIGH/PCLK_NORMAL/PCLK_SLOW
};

/**
 * struct qfb_intr - bookkeeping interrupts from hardware
 * 
 * @st_vert_fp: the start of vertical front porch
 * @st_vert_bp: the start of vertical back porch
 * @fs_mark1: a frame sequencer node with this marker is executed
 * @fs_mark2: a frame sequencer node with this marker is executed
 * @db_switched: in double buffer mode, another buffer is switched
 * @underrun: whether an buffer underrun occurs. The occurance might
 *            cause flicker.
 */
struct qfb_intr 
{
	int     st_vert_fp;
	int     st_vert_bp;
	int     st_vsync;

	int     fs_mark1;
	int     fs_mark2;
	int     db_switched;
	int     underrun;
};

/**
 * IOCTL for QFB
 */
#define QFB_IOCTL_SET_DEVINFO    _IOW('F', 0x77, struct qfb_devinfo)
#define QFB_IOCTL_RESET_DEFAULT  _IO( 'F', 0x78)
#define QFB_IOCTL_SHUTDOWN       _IO( 'F', 0x79)
#define QFB_IOCTL_POLL_INTR      _IOR('F', 0x80, struct qfb_intr)
#define QFB_IOCTL_CUR_DBID       _IOR('F', 0x81, uint32_t)
#define QFB_IOCTL_SET_DBID       _IOW('F', 0x82, uint32_t)
#define QFB_IOCTL_DUMP_REGS      _IO( 'F', 0x83)

/*
 * Update the timing for the current panel settings.
 * Note that if you don't want to modify specific timing, fill 0
 */
#define QFB_IOCTL_UPDATE_TIMING  _IOW('F', 0x83, struct qfb_timing)

#endif
