/*
 * Quasar LCD controller kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2022/03/14 10:30:50 $
//  $Change: 58883 $
//
// =========================================================
#define DEBUG

#include "core.h"

static void qfb_wait_pll(struct qfb_info *qfb_info, uint32_t wait_cnt)
{
	/*
	 * xin0_clk is running at 25MHz, the total waiting time would be:
	 * 
	 *      1/25MHz * `wait_cnt` * 10^6 (us)
	 */
	uint32_t val;

	// Disable
	val  = qfb_rreg32(qfb_info->base_gpf, LCD_LVDSVX1_PLLLOCK1_OFF);
	val &= LCD_LVDSVX1_PLLLOCK1__LOCK_TIME_EN__INV_MASK; // Lock timer disabled
	qfb_wreg32(qfb_info->base_gpf, LCD_LVDSVX1_PLLLOCK1_OFF, val);

	// LOCK_TIME
	val = qfb_rreg32(qfb_info->base_gpf, LCD_LVDSVX1_PLLLOCK1_OFF);
	val = (val & LCD_LVDSVX1_PLLLOCK1__LOCK_TIME__INV_MASK) |
		  (wait_cnt << LCD_LVDSVX1_PLLLOCK1__LOCK_TIME__SHIFT);
	qfb_wreg32(qfb_info->base_gpf, LCD_LVDSVX1_PLLLOCK1_OFF, val);

	// Enable
	val  = qfb_rreg32(qfb_info->base_gpf, LCD_LVDSVX1_PLLLOCK1_OFF);
	val |= (0x1 << LCD_LVDSVX1_PLLLOCK1__LOCK_TIME_EN__SHIFT); // Lock timer enabled
	qfb_wreg32(qfb_info->base_gpf, LCD_LVDSVX1_PLLLOCK1_OFF, val);
 
	// Wait for PLL locked
	do {
		val = qfb_rreg32(qfb_info->base_gpf, LCD_LVDSVX1_PLLLOCK2_OFF);
	} while (!(val & LCD_LVDSVX1_PLLLOCK2__LOCK_TIMER_DONE__MASK));
}

void qfb_setup_pll(struct device *dev, struct qfb_info *qfb_info)
{
	uint32_t val;
	uint32_t upcnt  = qfb_info->panel.pll.upcnt;
	uint32_t dkin   = qfb_info->panel.pll.dkin;
	uint32_t min    = qfb_info->panel.pll.min;
	uint32_t kin    = qfb_info->panel.pll.kin;
	uint32_t reg1   = qfb_info->panel.pll.reg1;
	uint32_t reg2   = qfb_info->panel.pll.reg2;
	uint32_t reg3   = qfb_info->panel.pll.reg3;
	uint32_t reg4   = qfb_info->panel.pll.reg4;
	uint32_t fidiv  = qfb_info->panel.pll.fidiv;
	uint32_t pdiv   = qfb_info->panel.pll.pdiv;
	uint32_t qdiv   = qfb_info->panel.pll.qdiv;
	uint32_t divl   = qfb_info->panel.pll.divl;
	uint32_t vcolmt = qfb_info->panel.pll.vcolmt;
	uint32_t cp2en  = qfb_info->panel.pll.cp2en;
	uint32_t cp2en2 = qfb_info->panel.pll.cp2en2;

	dev_dbg(dev, 
			"%s: upcnt(%x) dkin(%x) min(%x) kin(%x) reg1(%x) reg2(%x) " 
			"reg3(%x) reg4(%x) fidiv(%x) pdiv(%x) qdiv(%x) divl(%x) "
			"vcolmt(%x) cp2en(%x) cp2en2(%x)\n",
			__func__, upcnt, dkin, min, kin, reg1, reg2, 
			reg3, reg4, fidiv, pdiv, qdiv, divl, 
			vcolmt, cp2en, cp2en2);

	// DRVAMP (Amplitude): SRS_SIG0 = LVDS, 350 mV (0x6)
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_DRVAMP_OFF, 0x6 << LCDVX1TX_DRVAMP__SRS_SIG0__SHIFT);

	//	EMVL (Emphasis Strength): EMLVL0 = 0% (0x0) [IGNORE]
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_EMLVL_OFF, 0x0 << LCDVX1TX_EMLVL__EMLVL0__SHIFT);

	//	DRVCNT
	val = (0x0  << LCDVX1TX_DRVCNT__DRVDAT__SHIFT)     |
		  (0x0  << LCDVX1TX_DRVCNT__DRVMEM__SHIFT)     | // Off
		  (0x3f << LCDVX1TX_DRVCNT__NLVLS__SHIFT)      | // 0.56V/1.26V (0x3F), Level Shift Bias Control for DRV
		  (0x0  << LCDVX1TX_DRVCNT__TMCOROFF__SHIFT)   | // Off, Termination Resistance Correction
		  (0x0  << LCDVX1TX_DRVCNT__EME__SHIFT)        | // Off, Emphasis Enable
		  (0x0  << LCDVX1TX_DRVCNT__NVCOMO0125__SHIFT) | // Mode Selection
		  (0x1  << LCDVX1TX_DRVCNT__NPORTEN__SHIFT)    | // On, Built-in Terminating Resistor
		  (0x0  << LCDVX1TX_DRVCNT__LVDSEN__SHIFT)     | // Off, LVDS Mode Enable
		  (0x0  << LCDVX1TX_DRVCNT__EMT__SHIFT)        | // 338ps, Emphasis Period
		  (0x0  << LCDVX1TX_DRVCNT__PNDIFF_ADJ__SHIFT);  // -5.5ps, IntraPair Skew Control
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_DRVCNT_OFF, val);

	//	DRVCNT.DRVMEM = On (0x1)
	val  = qfb_rreg32(qfb_info->base_intf, LCDVX1TX_DRVCNT_OFF);
	val |= (0x1 << LCDVX1TX_DRVCNT__DRVMEM__SHIFT);
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_DRVCNT_OFF, val);

	/* 
	 * PLL setup
	 */
	val = (divl << LCDVX1TX_PLLANACNT__DIVL__SHIFT)   |
		  (cp2en2 << LCDVX1TX_PLLANACNT__CP2EN2__SHIFT) |
		  (cp2en << LCDVX1TX_PLLANACNT__CP2EN__SHIFT)  |
		  (vcolmt << LCDVX1TX_PLLANACNT__VCOLMT__SHIFT) |
		  (reg4 << LCDVX1TX_PLLANACNT__REG4__SHIFT)   |
		  (reg3 << LCDVX1TX_PLLANACNT__REG3__SHIFT)   |
		  (reg2 << LCDVX1TX_PLLANACNT__REG2__SHIFT)   |
		  (reg1 << LCDVX1TX_PLLANACNT__REG1__SHIFT);    
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_PLLANACNT_OFF, val);

	val = (0x0 << LCDVX1TX_PLLDIGCNT__PLLBYPS__SHIFT)    |
		  (0x0 << LCDVX1TX_PLLDIGCNT__DIVOUTON__SHIFT)   |
		  (0x0 << LCDVX1TX_PLLDIGCNT__VCOSEL__SHIFT)     |
		  (0x0 << LCDVX1TX_PLLDIGCNT__TESTCNT__SHIFT)    |
		  (0x1 << LCDVX1TX_PLLDIGCNT__VCOOUT25EN__SHIFT) |
		  (0x0 << LCDVX1TX_PLLDIGCNT__OFFSETON__SHIFT)   |
		  (qdiv << LCDVX1TX_PLLDIGCNT__QDIV__SHIFT)       | 
		  (pdiv << LCDVX1TX_PLLDIGCNT__PDIV__SHIFT)       |
		  (fidiv << LCDVX1TX_PLLDIGCNT__FIDIV__SHIFT);
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_PLLDIGCNT_OFF, val);

	//	SSCNT1
	val = (min << LCDVX1TX_SSCCNT1__MIN__SHIFT)     |
		  (kin  << LCDVX1TX_SSCCNT1__KIN__SHIFT);
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_SSCCNT1_OFF, val);

	//	SSCNT2
	val = (0x0  << LCDVX1TX_SSCCNT2__NRST__SHIFT)     |
		  (0x0  << LCDVX1TX_SSCCNT2__NRSTDS__SHIFT)   |
		  (0x0  << LCDVX1TX_SSCCNT2__SSCEN__SHIFT)    |
		  (0x0  << LCDVX1TX_SSCCNT2__DSIGMODE__SHIFT) |
		  (upcnt << LCDVX1TX_SSCCNT2__UPCNTIN__SHIFT) |
		  (dkin << LCDVX1TX_SSCCNT2__DKIN__SHIFT);
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_SSCCNT2_OFF, val);

	// VCO Low Frequency Oscillation Control: Turn On (OFFSETON)
	val  = qfb_rreg32(qfb_info->base_intf, LCDVX1TX_PLLDIGCNT_OFF);
	val |= (0x1 << LCDVX1TX_PLLDIGCNT__OFFSETON__SHIFT);
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_PLLDIGCNT_OFF, val);
	
	// PHY Power Up (NPDWN)
	val  = qfb_rreg32(qfb_info->base_intf, LCDVX1TX_ENTCNT1_OFF);
	val |= (0x1 << LCDVX1TX_ENTCNT1__NPDOWN__SHIFT); // Normal operation
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_ENTCNT1_OFF, val);

	// PHY Reset > Cancel (NRESET)
	val  = qfb_rreg32(qfb_info->base_intf, LCDVX1TX_ENTCNT1_OFF);
	val |= (0x1 << LCDVX1TX_ENTCNT1__NRESET__SHIFT); // Normal operation
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_ENTCNT1_OFF, val);

	// SSC Reset > Cancel (NRST)
	val  = qfb_rreg32(qfb_info->base_intf, LCDVX1TX_SSCCNT2_OFF);
	val |= (0x1 << LCDVX1TX_SSCCNT2__NRST__SHIFT); // Set
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_SSCCNT2_OFF, val);

	// Driver Power > Up (ENT)
	val  = qfb_rreg32(qfb_info->base_intf, LCDVX1TX_ENTCNT2_OFF);
	val |= (0x1 << LCDVX1TX_ENTCNT2__ENT__SHIFT); // DRV current ON
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_ENTCNT2_OFF, val);

	// Wait 10us
	qfb_wait_pll(qfb_info, 0xfa);

	// Driver Test > Turn Off (DRVMEN)
	//	LCDVX1TX_DRVCNT.DRVMEM = Off (0x0)
	val  = qfb_rreg32(qfb_info->base_intf, LCDVX1TX_DRVCNT_OFF);
	val &= LCDVX1TX_DRVCNT__DRVMEM__INV_MASK; // Off
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_DRVCNT_OFF, val);

	// Wait 40us
	qfb_wait_pll(qfb_info, 0x3e8);

	// Wait 950us
	qfb_wait_pll(qfb_info, 0x5cc6);

	// VCO Low Frequency Oscillation Control > Turn Off (OFFSETON)
	val  = qfb_rreg32(qfb_info->base_intf, LCDVX1TX_PLLDIGCNT_OFF);
	val &= LCDVX1TX_PLLDIGCNT__OFFSETON__INV_MASK; // Off
	qfb_wreg32(qfb_info->base_intf, LCDVX1TX_PLLDIGCNT_OFF, val);
	
	// Wait 4ms
	qfb_wait_pll(qfb_info, 0x186A0);
	dev_dbg(dev, "%s: finish\n", __func__);
}
