/*
 * Quasar LCD controller kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2021/10/28 10:29:17 $
//  $Change: 54504 $
//
// =========================================================
#define DEBUG

#include <linux/list.h>
#include <linux/device.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>

/**
 * qfb_mem_kind - uniquely reference a memory region by some attrs
 * 
 * @ndid: (fs) node id, specify negative to ignore this field
 * @biid: background image
 * @owid: overlay image id, specify negative to ignore this field
 * @pal: this field can only be 0 or negative. We only have one palette.
 * 
 * This may not be a good design, since it has some implications.
 * For example, if the attributes are: niid(-1), biid(0), owid(-1), pal(-1),
 * it means we target to the memory of background image id 0. However,
 * if the attrs are: niid(-1), biid(0), owid(0), pal(-1), it means the target is
 * the first overlay image for the first background buffer.
 * 
 * Anyway, it is the current design, and luckily, `qfb_mem_kind` is only
 * used in this file, so it could be easily refactored without affecting
 * the other modules.
 */
struct qfb_mem_kind {
	int8_t ndid;
	int8_t biid;
	int8_t owid;
	int8_t pal;
};

#define QFB_SAME_KIND(a, b) ( \
		(a.ndid==b.ndid) &&   \
		(a.biid==b.biid) &&   \
		(a.owid==b.owid) &&   \
		(a.pal==b.pal))

struct qfb_mem_region {
	struct list_head mem_list;
	struct qfb_mem_kind kind;

	void        *virt;
	dma_addr_t  dma;
	size_t      sz;
};

static DEFINE_SPINLOCK(qfb_mem_alloc_lock);
static LIST_HEAD(qfb_mem_pool);


static struct qfb_mem_region *__qfb_mem_alloc(
		struct device *dev, 
		struct qfb_mem_kind kind,
		size_t sz)
{
	struct qfb_mem_region *mem = NULL;

	mem = kzalloc(sizeof(*mem), GFP_KERNEL);
	if (!mem) {
		dev_err(dev, "%s: cannot alloc mem region\n", __func__);
		goto safe_release;
	}

	mem->sz   = sz;
	mem->kind = kind;
	mem->virt = kzalloc(sz, GFP_KERNEL);

	if (!mem->virt) {
		dev_err(dev, "%s: cannot alloc mem for image buffer\n", __func__);
		goto safe_release;
	}

	mem->dma = dma_map_single(dev->parent, mem->virt, sz, DMA_TO_DEVICE);

	if (dma_mapping_error(dev->parent, mem->dma)) {
		dev_err(dev, "%s: cannot do dma mapping\n", __func__);
		goto safe_release;
	}

	return mem;

safe_release:
	if (mem) {
		if (mem->virt) {
			kfree(mem->virt);
		}
		kfree(mem);
	}
	return NULL;
}

static void __qfb_mem_free(
		struct device *dev,
		struct qfb_mem_region *mem)
{
	dma_unmap_single(dev->parent, mem->dma, mem->sz, DMA_TO_DEVICE);
	kfree(mem->virt);
	kfree(mem);
}

static struct qfb_mem_region *qfb_find_mem(
		struct device *dev,
		struct qfb_mem_kind kind)
{
	struct qfb_mem_region *tmp = NULL;
	struct qfb_mem_region *mem = NULL;

	list_for_each_entry(tmp, &qfb_mem_pool, mem_list) {
		if (QFB_SAME_KIND(tmp->kind, kind)) {
			mem = tmp;
			break;
		}
	}

	return mem;
}

static void __qfb_free_and_list_del(
		struct device *dev,
		struct qfb_mem_region *mem)
{
	list_del(&mem->mem_list);
	__qfb_mem_free(dev, mem);
}

static struct qfb_mem_region *qfb_find_or_alloc(
		struct device *dev, 
		struct qfb_mem_kind kind, 
		size_t sz)
{
	struct qfb_mem_region *mem = NULL;
	struct qfb_mem_region *tmp = NULL;
	unsigned long flags;

	// We align size to PAGE_SIZE here
	sz = roundup(sz, PAGE_SIZE);

	spin_lock_irqsave(&qfb_mem_alloc_lock, flags);

	/*
	 * According to the `kind` to find existed memory region.
	 * Also check `sz` to decide if we need to recreate it.
	 */
	tmp = qfb_find_mem(dev, kind);
	if (tmp) {
		if (tmp->sz == sz) {
			mem = tmp;
		} else {
			__qfb_free_and_list_del(dev, tmp);
			dev_dbg(dev, "%s: diff size detected, free old one (kind: %d,%d,%d,%d)\n",
					__func__, kind.ndid, kind.biid, kind.owid, kind.pal);
		}
	}

	/* 
	 * Found nothing, need to create new one
	 */
	if (!mem) {
		mem = __qfb_mem_alloc(dev, kind, sz);
		list_add(&mem->mem_list, &qfb_mem_pool);
		dev_dbg(dev, "%s: create new mem region (kind: %d,%d,%d,%d)\n",
				__func__, kind.ndid, kind.biid, kind.owid, kind.pal);
	}

	spin_unlock_irqrestore(&qfb_mem_alloc_lock, flags);

	return mem;
}

void *qfb_bi_req_mem(struct device *dev, int biid, size_t sz)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = biid,
		.owid = -1,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_or_alloc(dev, kind, sz);
	return mem->virt;
}

void *qfb_ow_req_mem(struct device *dev, int biid, int owid, size_t sz)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = biid,
		.owid = owid,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_or_alloc(dev, kind, sz);
	return mem->virt;
}

void *qfb_pal_req_mem(struct device *dev, size_t sz)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = -1,
		.owid = -1,
		.pal  = 0,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_or_alloc(dev, kind, sz);
	return mem->virt;
}

void *qfb_fs_bi_req_mem(struct device *dev, int ndid, size_t sz)
{
	struct qfb_mem_kind kind = {
		.ndid = ndid,
		.biid = 0,
		.owid = -1,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_or_alloc(dev, kind, sz);
	return mem->virt;
}

void *qfb_fs_ow_req_mem(struct device *dev, int ndid, int owid, size_t sz)
{
	struct qfb_mem_kind kind = {
		.ndid = ndid,
		.biid = 0,
		.owid = owid,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_or_alloc(dev, kind, sz);
	return mem->virt;
}

size_t qfb_bi_get_sz(struct device *dev, int biid)
{	
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = biid,
		.owid = -1,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	return mem ? mem->sz : 0;
}

size_t qfb_ow_get_sz(struct device *dev, int biid, int owid)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = biid,
		.owid = owid,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	return mem ? mem->sz : 0;
}

size_t qfb_pal_get_sz(struct device *dev)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = -1,
		.owid = -1,
		.pal  = 0,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	return mem ? mem->sz : 0;
}

size_t qfb_fs_bi_get_sz(struct device *dev, int ndid)
{
	struct qfb_mem_kind kind = {
		.ndid = ndid,
		.biid = 0,
		.owid = -1,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	return mem ? mem->sz : 0;
}

size_t qfb_fs_ow_get_sz(struct device *dev, int ndid, int owid)
{
	struct qfb_mem_kind kind = {
		.ndid = ndid,
		.biid = 0,
		.owid = owid,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	return mem ? mem->sz : 0;
}

int qfb_bi_free_mem(struct device *dev, int biid)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = biid,
		.owid = -1,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	if (!mem) {
		dev_err(dev, "%s: free non-existed bi(%d) mem region\n", __func__, biid);
		return -1;
	}

	__qfb_free_and_list_del(dev, mem);
	return 0;
}

int qfb_ow_free_mem(struct device *dev, int biid, int owid)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = biid,
		.owid = owid,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	if (!mem) {
		dev_err(dev, 
				"%s: free non-existed ow(%d) for bi(%d) mem region\n", 
				__func__, owid, biid);
		return -1;
	}

	__qfb_free_and_list_del(dev, mem);
	return 0;
}

int qfb_pal_free_mem(struct device *dev)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = -1,
		.owid = -1,
		.pal  = 0,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	if (!mem) {
		dev_err(dev, "%s: free non-existed palmem region\n", __func__);
		return -1;
	}

	__qfb_free_and_list_del(dev, mem);
	return 0;
}

int qfb_fs_bi_free_mem(struct device *dev, int ndid)
{
	struct qfb_mem_kind kind = {
		.ndid = ndid,
		.biid = 0,
		.owid = -1,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	if (!mem) {
		dev_err(dev, 
				"%s: free non-existed bi for node(%d) mem region\n", 
				__func__, ndid);
		return -1;
	}

	__qfb_free_and_list_del(dev, mem);
	return 0;
}

int qfb_fs_ow_free_mem(struct device *dev, int ndid, int owid)
{
	struct qfb_mem_kind kind = {
		.ndid = ndid,
		.biid = 0,
		.owid = owid,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	if (!mem) {
		dev_err(dev, 
				"%s: free non-existed ow(%d) for bi in node(%d) mem region\n", 
				__func__, owid, ndid);
		return -1;
	}

	__qfb_free_and_list_del(dev, mem);
	return 0;
}

dma_addr_t qfb_bi_get_dma(struct device *dev, int biid)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = biid,
		.owid = -1,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	return mem ? mem->dma : (dma_addr_t)0;
}

dma_addr_t qfb_ow_get_dma(struct device *dev, int biid, int owid)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = biid,
		.owid = owid,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	return mem ? mem->dma : (dma_addr_t)0;
}

dma_addr_t qfb_pal_get_dma(struct device *dev)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = -1,
		.owid = -1,
		.pal  = 0,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	return mem ? mem->dma : (dma_addr_t)0;
}

dma_addr_t qfb_fs_bi_get_dma(struct device *dev, int ndid)
{
	struct qfb_mem_kind kind = {
		.ndid = ndid,
		.biid = 0,
		.owid = -1,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	return mem ? mem->dma : (dma_addr_t)0;
}

dma_addr_t qfb_fs_ow_get_dma(struct device *dev, int ndid, int owid)
{
	struct qfb_mem_kind kind = {
		.ndid = ndid,
		.biid = 0,
		.owid = owid,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	return mem ? mem->dma : (dma_addr_t)0;
}

int qfb_bi_flush_mem(struct device *dev, int biid)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = biid,
		.owid = -1,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	if (!mem) {
		dev_err(dev, "%s: non-exist bi(%d) for flush\n", __func__, biid);
		return -1;
	}

	dma_sync_single_for_device(dev->parent, mem->dma, mem->sz, DMA_TO_DEVICE);
	return 0;
}

int qfb_ow_flush_mem(struct device *dev, int biid, int owid)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = biid,
		.owid = owid,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	if (!mem) {
		dev_err(dev, 
				"%s: non-exist ow(%d) of bi(%d) for flush\n", 
				__func__, owid, biid);
		return -1;
	}

	dma_sync_single_for_device(dev->parent, mem->dma, mem->sz, DMA_TO_DEVICE);
	return 0;

}

int qfb_pal_flush_mem(struct device *dev)
{
	struct qfb_mem_kind kind = {
		.ndid = -1,
		.biid = -1,
		.owid = -1,
		.pal  = 0,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	if (!mem) {
		dev_err(dev, "%s: non-exist pal flush\n", __func__);
		return -1;
	}

	dma_sync_single_for_device(dev->parent, mem->dma, mem->sz, DMA_TO_DEVICE);
	return 0;
}

int qfb_fs_bi_flush_mem(struct device *dev, int ndid)
{
	struct qfb_mem_kind kind = {
		.ndid = ndid,
		.biid = 0,
		.owid = -1,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	if (!mem) {
		dev_err(dev, 
				"%s: non-exist bi in node(%d) for flush\n", 
				__func__, ndid);
		return -1;
	}

	dma_sync_single_for_device(dev->parent, mem->dma, mem->sz, DMA_TO_DEVICE);
	return 0;
}

int qfb_fs_ow_flush_mem(struct device *dev, int ndid, int owid)
{
	struct qfb_mem_kind kind = {
		.ndid = ndid,
		.biid = 0,
		.owid = owid,
		.pal  = -1,
	};
	struct qfb_mem_region *mem;

	mem = qfb_find_mem(dev, kind);
	if (!mem) {
		dev_err(dev, 
				"%s: non-exist ow(%d) of bi in node(%d) for flush\n", 
				__func__, owid, ndid);
		return -1;
	}

	dma_sync_single_for_device(dev->parent, mem->dma, mem->sz, DMA_TO_DEVICE);
	return 0;
}

void qfb_release_all_mem(struct device *dev)
{
	struct qfb_mem_region *mem, *tmp;

	list_for_each_entry_safe(mem, tmp, &qfb_mem_pool, mem_list) {
		__qfb_free_and_list_del(dev, mem);
	}

	if (!list_empty(&qfb_mem_pool)) {
		dev_warn(dev, "%s: release all mem region, but not empty\n", __func__);
	}
}
