/*
 * Quasar LCD controller kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2022/04/22 09:45:19 $
//  $Change: 60182 $
//
// =========================================================
#define DEBUG

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/fb.h>
#include <linux/wait.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/fb.h>
#include <linux/irq.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <linux/console.h>
#include <linux/err.h>
#include <linux/of_graph.h>
#include <video/display_timing.h>
#include <video/of_display_timing.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <quasar/qioctl.h>
#include "core.h"
#include "mem.h"
#include "uapi/qfb.h"

/*
 * Define default values
 */
#define DEF_WIDTH           1024
#define DEF_HEIGHT          600
#define DEF_DEPTH           4
#define DEF_FRAME_SZ       (DEF_WIDTH*DEF_HEIGHT*DEF_DEPTH)

/** 
 * qfb_sync_to_fb_info: update `fb_info` from `qfb_devinfo`
 * 
 * @info: will be synced from `q_dev`
 * @q_dev: it will be used to sync to `info`
 * 
 * Since `qfb_devinfo` has more features than `fb_info`, only background image part
 * will be synced to `fb_info`
 * 
 * return 0 if success, otherwise, return negative value
 */
static int qfb_sync_to_fb_info(struct fb_info *info, struct qfb_devinfo *q_dev)
{
	int bytes_per_pixel;

	info->var.width  = q_dev->bi_prop.width;
	info->var.height = q_dev->bi_prop.height;
	info->var.xres   = info->var.width;
	info->var.yres   = info->var.height;
	info->var.xres_virtual = info->var.xres;
	info->var.yres_virtual = info->var.yres;

	// TODO: How to handle palette mode?
	// sync bpp
	if (q_dev->bpp == 15) {
		info->var.bits_per_pixel = 16;
		info->var.blue.offset    = 0;
		info->var.blue.length    = 5;
		info->var.green.offset   = 5;
		info->var.green.length   = 5;
		info->var.red.offset     = 11;
		info->var.red.length     = 5;
	} else if (q_dev->bpp == 16) {
		info->var.bits_per_pixel = 16;
		info->var.blue.offset    = 0;
		info->var.blue.length    = 5;
		info->var.green.offset   = 5;
		info->var.green.length   = 6;
		info->var.red.offset     = 11;
		info->var.red.length     = 5;
	} else if (q_dev->bpp == 18) {
		if (q_dev->bi_prop.packed) {
			info->var.bits_per_pixel = 24;
		} else {
			info->var.bits_per_pixel = 32;
		}
		info->var.blue.offset    = 0;
		info->var.blue.length    = 6;
		info->var.green.offset   = 6;
		info->var.green.length   = 6;
		info->var.red.offset     = 12;
		info->var.red.length     = 6;
	} else if (q_dev->bpp == 24) {
		if (q_dev->bi_prop.packed) {
			info->var.bits_per_pixel = 24;
		} else {
			info->var.bits_per_pixel = 32;
		}
		info->var.blue.offset    = 0;
		info->var.blue.length    = 8;
		info->var.green.offset   = 8;
		info->var.green.length   = 8;
		info->var.red.offset     = 16;
		info->var.red.length     = 8;
	} else {
		dev_warn(info->dev, "%s: cannot map `qfb_devinfo` to `fb_info`", __func__);
	}

	// line length
	bytes_per_pixel = info->var.bits_per_pixel / 8;
	info->fix.line_length = q_dev->bi_prop.width * bytes_per_pixel;

	// setup mem related, since the data could be re-created
	info->screen_base = (void __iomem *) q_dev->bi[0].bin;
	info->fix.smem_start  = (unsigned long) qfb_bi_get_dma(info->dev, 0);
	info->fix.smem_len    = qfb_bi_get_sz(info->dev, 0);

	dev_dbg(info->dev, "%s: info->base = %px\n", __func__, info->screen_base);
	dev_dbg(info->dev, "%s: info->fix.smem_start = %px\n", __func__, (void*)info->fix.smem_start);
	dev_dbg(info->dev, "%s: info->fix.smem_len = %u\n", __func__, info->fix.smem_len);
	dev_dbg(info->dev, "%s: info->fix.line_length = %u\n", __func__, info->fix.line_length);
	dev_dbg(info->dev, "%s: info->var.bits_per_pixel = %u\n", __func__, info->var.bits_per_pixel);
	dev_dbg(info->dev, "%s: info->var.blue.offset = %u\n", __func__, info->var.blue.offset);
	dev_dbg(info->dev, "%s: info->var.blue.length = %u\n", __func__, info->var.blue.length);
	dev_dbg(info->dev, "%s: info->var.green.offset = %u\n", __func__, info->var.green.offset);
	dev_dbg(info->dev, "%s: info->var.green.length = %u\n", __func__, info->var.green.length);
	dev_dbg(info->dev, "%s: info->var.red.offset = %u\n", __func__, info->var.red.offset);
	dev_dbg(info->dev, "%s: info->var.red.length = %u\n", __func__, info->var.red.length);

	return 0;
}

/**
 * qfb_sync_to_qfb_dev: update `qfb_devinfo` from `fb_info`
 * 
 * @q_dev: will be synced from `info`
 * @info: it will be used to sync to `q_dev`
 * 
 * Once the `info` syncs to `q_dev`, it will fall back to background image mode,
 * because `fb_info` has limited feature.
 * 
 * return 0 if success, otherwise, return negative value
 */
static int qfb_sync_to_qfb_dev(struct qfb_devinfo *q_dev, struct fb_info *info)
{
	int bytes_per_pixel;

	/*
	 * `fb_info` has no feature such as double buffer and frame sequencer etc.
	 * So we disable this feature in the `qfb_devinfo`.
	 */
	q_dev->en_fs = 0;
	q_dev->en_db = 0;
	q_dev->en_pd = 0;

	if (info->var.bits_per_pixel == 16) {
		switch (info->var.green.length) {
			case 5: q_dev->bpp = 15; break;
			case 6: q_dev->bpp = 16; break;
			default: dev_err(info->dev, 
					         "%s: invalid green length(%u)\n", 
					         __func__, info->var.green.length);
		}
	} else if (info->var.bits_per_pixel == 24 || info->var.bits_per_pixel == 32) {
		// Just choose one channel to check
		switch (info->var.blue.length) {
			case 6: q_dev->bpp = 18; break;
			case 8: q_dev->bpp = 24; break;
			default: dev_err(info->dev,
					         "%s: invalid blue length(%u)\n",
					         __func__, info->var.blue.length);
		}
	}

	q_dev->bi_prop.width  = info->var.xres;
	q_dev->bi_prop.height = info->var.yres;

	bytes_per_pixel = info->fix.line_length / info->var.xres;
	q_dev->bi_prop.packed = (bytes_per_pixel == 3);

	q_dev->bi[0].bin = (qfb_ptr_t) info->screen_base;

	dev_dbg(info->dev, "%s: qfb_devinfo width = %u\n", __func__, q_dev->bi_prop.width);
	dev_dbg(info->dev, "%s: qfb_devinfo height = %u\n", __func__, q_dev->bi_prop.height);
	dev_dbg(info->dev, "%s: qfb_devinfo bpp = %u\n", __func__, q_dev->bpp);
	dev_dbg(info->dev, "%s: qfb_devinfo packed = %u\n", __func__, q_dev->bi_prop.packed);
	dev_dbg(info->dev, "%s: qfb_devinfo addr = %px\n", __func__, q_dev->bi[0].bin);

	return 0;
}

static void qfb_reset_fb_mem(struct fb_info *info)
{
	void *virt;
	dma_addr_t dma;

	virt = qfb_bi_req_mem(info->dev, 0, DEF_FRAME_SZ);
	dma  = qfb_bi_get_dma(info->dev, 0);

	info->screen_base    = (void __iomem *) virt;
	info->fix.smem_start = (unsigned long) dma;
	info->fix.smem_len   = qfb_bi_get_sz(info->dev, 0);
}

static void qfb_reset_fb_info(struct fb_info *info)
{
	strlcpy(info->fix.id, "quasar-fb", sizeof(info->fix.id));
	info->fix.type              = FB_TYPE_PACKED_PIXELS;
	info->fix.type_aux          = 0;
	info->fix.xpanstep          = 0;
	info->fix.ypanstep          = 0;
	info->fix.accel             = FB_ACCEL_NONE;
	info->fix.visual            = FB_VISUAL_TRUECOLOR;
	info->fix.line_length       = DEF_DEPTH * DEF_WIDTH;

	info->var.nonstd            = 0;
	info->var.activate          = FB_ACTIVATE_NOW;
	info->var.height            = DEF_WIDTH;
	info->var.width             = DEF_HEIGHT;
	info->var.accel_flags       = 0;
	info->var.vmode             = FB_VMODE_NONINTERLACED;
	info->var.sync              = FB_SYNC_COMP_HIGH_ACT;
	info->var.xres              = DEF_WIDTH;
	info->var.xres_virtual      = DEF_WIDTH;
	info->var.yres              = DEF_HEIGHT;
	info->var.yres_virtual      = DEF_HEIGHT;

	/*
	 * XXX: I wonder how to tell the differences 18bpp/24bpp packed/unpacked by using `var`
	 * struct.
	 * 
	 * IMHO, I think `var.bits_per_pixel`and `var.{blue|green|red}.{offset|length} could be used together 
	 * to tell the differences.
	 * 
	 * For example, for 24bpp,
	 * - packed: bits_per_pixel = 24, blue.offset=0, green.offset=8, red.offset=24
	 * - unpack: bits_per_pixel = 32, blue.offset=0, green.offset=8, red.offset=24
	 *
	 * For 18bpp,
	 * - packed: bits_per_pixel = 24, blue.offset=0, green.offset=6, red.offset=12
	 * - unpack: bits_per_pixel = 32, blue.offset=0, green.offset=6, red.offset=12
	 * 
	 * The other bpp and packed/unpacked combinations could follow the same qfb_setups.
	 */
	info->var.bits_per_pixel    = DEF_DEPTH * 8;
	info->var.grayscale         = 0;
	info->var.blue.offset       = 0;
	info->var.blue.length       = 8;
	info->var.blue.msb_right    = 0;
	info->var.green.offset      = 8;
	info->var.green.length      = 8;
	info->var.green.msb_right   = 0;
	info->var.red.offset        = 24;
	info->var.red.length        = 8;
	info->var.red.msb_right     = 0;
	info->var.transp.offset     = 0;
	info->var.transp.length     = 0;
	info->var.transp.msb_right  = 0;
}

static int qfb_check_var(struct fb_var_screeninfo *var,
		struct fb_info *info)
{
	dev_dbg(info->dev, "%s: check var\n", __func__);
	dev_dbg(info->dev, "%s: - xres(%u)\n", __func__, var->xres);
	dev_dbg(info->dev, "%s: - yres(%u)\n", __func__, var->yres);
	dev_dbg(info->dev, "%s: - xres_virtual(%u)\n", __func__, var->xres_virtual);
	dev_dbg(info->dev, "%s: - yres_virtual(%u)\n", __func__, var->yres_virtual);
	dev_dbg(info->dev, "%s: - xoffset(%u)\n", __func__, var->xoffset);
	dev_dbg(info->dev, "%s: - yoffset(%u)\n", __func__, var->yoffset);
	dev_dbg(info->dev, "%s: - bits_per_pixel(%u)\n", __func__, var->bits_per_pixel);
	dev_dbg(info->dev, "%s: - grayscale(%u)\n", __func__, var->grayscale);
	dev_dbg(info->dev, "%s: - red(%u/%u/%u)\n", __func__, var->red.offset, var->red.length, var->red.msb_right);
	dev_dbg(info->dev, "%s: - green(%u/%u/%u)\n", __func__, 
			var->green.offset, var->green.length, var->green.msb_right);
	dev_dbg(info->dev, "%s: - blue(%u/%u/%u)\n", __func__, 
			var->blue.offset, var->blue.length, var->blue.msb_right);
	dev_dbg(info->dev, "%s: - transp(%u/%u/%u)\n", __func__, 
			var->transp.offset, var->transp.length, var->transp.msb_right);

	// I think some similar attributes should follow xres/yres,
	// which includes xres_virtual/yres_virtual and width/height
	var->width  = var->xres_virtual = var->xres;
	var->height = var->yres_virtual = var->yres;

	// ignore xoffset/yoffset
	var->xoffset = var->yoffset = 0;

	// Some fixed values
	var->activate    = FB_ACTIVATE_NOW;
	var->nonstd      = 0;
	var->accel_flags = 0;
	var->vmode       = FB_VMODE_NONINTERLACED;
	var->sync        = FB_SYNC_COMP_HIGH_ACT;

	if (var->xres * var->yres < 24) {
		dev_err(info->dev, "%s: resolution too small\n", __func__);
		return -EINVAL;
	}

	return 0;
}

static void do_qfb_setup(struct device *dev, struct qfb_info *qfb_info)
{
	unsigned long flags;

	dev_dbg(dev, "%s: perform qfb setup\n", __func__);
	spin_lock_irqsave(&qfb_info->intr_lck, flags);

	memset(&qfb_info->intr, 0, sizeof(qfb_info->intr));
	qfb_setup(dev, qfb_info);

	spin_unlock_irqrestore(&qfb_info->intr_lck, flags);
	dev_dbg(dev, "%s: perform qfb setup done\n", __func__);
}

static int qfb_set_par(struct fb_info *info)
{
	int ret;
	size_t sz;
	struct qfb_info *par = info->par;

	// Adjust some properties by the `bits_per_pixel`
	// TODO: palette
	switch (info->var.bits_per_pixel) {
		case 16:
		case 24:
		case 32:
			info->fix.line_length = ((info->var.bits_per_pixel + 7) / 8) * info->var.xres;
			info->fix.visual = FB_VISUAL_TRUECOLOR;
			break;
		default:
			dev_err(info->dev, "%s: not support bits_per_pixel(%u)\n", __func__, info->var.bits_per_pixel);
			return -EINVAL;
	}

	// re-assign memory, since the size might change
	sz = info->fix.line_length * info->var.yres;
	info->screen_base = (void __iomem *) qfb_bi_req_mem(info->dev, 0, sz);
	info->fix.smem_start = (unsigned long) qfb_bi_get_dma(info->dev, 0);
	info->fix.smem_len = qfb_bi_get_sz(info->dev, 0);

	if (!par->info_mm) {
		if (qfb_init_devinfo_mm(info->dev, par) != 0) {
			dev_err(info->dev, "%s: devinfo_mm init failed\n", __func__);
			return -EFAULT;
		}
	}

	/*
	 * Sync `fb_info` to `qfb_devinfo` first.
	 * This action also set to single background
	 * image mode.
	 */
	ret = qfb_sync_to_qfb_dev(par->info_mm->devinfo, info); 
	if (ret < 0) {
		dev_err(info->dev, "%s: fail to sync `fix` to qfb device info\n", __func__);
		return ret;
	}

	do_qfb_setup(info->dev, par);
	dev_dbg(info->dev, "%s: qfb_set_par done\n", __func__);
	return 0;
}

static int qfb_mmap(struct fb_info *info, struct vm_area_struct *vma)
{
	int rc;
	struct qfb_info *par;
	size_t size;

	par  = (struct qfb_info*)info->par;
	size = vma->vm_end - vma->vm_start;

	rc = dma_mmap_coherent(info->dev->parent, 
			vma, info->screen_base, info->fix.smem_start, size);
	return rc;
}

static void qfb_update_info_mm(
		struct device *dev,
		struct qfb_info *par, 
		struct qfb_devinfo_mm *info_mm)
{
	// release old ones
	if (par->info_mm) {
		dma_unmap_single(dev->parent, par->info_mm->dma_addr, sizeof(*par->info_mm->devinfo), DMA_TO_DEVICE);
		kfree(par->info_mm->devinfo);
		kfree(par->info_mm);
	}
	par->info_mm = info_mm;
}

/*
 * qfb_setup_sync - sync `fb_info` and `qfb_devinfo` and setup hardware
 * 
 * @info: `fb_info` that will be synced from `q_dev`
 * 
 * return 0 if success, otherwise, it will return negative value
 */
static int qfb_setup_sync(struct fb_info *info)
{
	int ret;
	struct qfb_info *par = info->par;
	struct qfb_devinfo *q_dev = par->info_mm->devinfo;

	/* 
	 * Make sure part of the `qfb_devinfo` attributes
	 * sync to the `fb_info`
	 */
	ret = qfb_sync_to_fb_info(info, q_dev);
	if (ret < 0) {
		dev_err(info->dev, "%s: fail to sync qfb device info to `fix`\n", __func__);
		return ret;
	}

	/*
	 * Perform the real hardware setup
	 */
	do_qfb_setup(info->dev, par);
	return 0;
}

static int qfb_ioctl_set_devinfo(struct fb_info *info, void __user *argp) 
{
	struct qfb_info *par;
	struct qfb_devinfo_mm *info_mm;

	info_mm = (struct qfb_devinfo_mm *)kzalloc(sizeof(*info_mm), GFP_KERNEL);
	if (!info_mm) {
		dev_err(info->dev, "%s: Failed to allocate memory for qfb_devinfo\n", __func__);
		return -EFAULT;
	}

	if (qfb_devinfo_copy_from_user(info->dev, info_mm, argp)) {
		return -EFAULT;
	}

	// keep it in par, and update to HW
	par = (struct qfb_info *)info->par;

	qfb_update_info_mm(info->dev, par, info_mm);
	return qfb_setup_sync(info);
}

static int qfb_ioctl_reset_default(struct fb_info *info)
{
	struct qfb_info *par;

	par = (struct qfb_info*)info->par;

	// reset fb_info to default
	qfb_reset_fb_info(info);
	qfb_reset_fb_mem(info);

	// reset hardware to default
	qfb_reset_hw(info->dev, par);
	return 0;
}

static int qfb_ioctl_shutdown(struct fb_info *info)
{
	struct qfb_info *par;
	par = (struct qfb_info*)info->par;

	// disable irq handler first or the handler would
	// consume INT vec that is required in the `qfb_shutdown`
	disable_irq(par->irq);

	qfb_shutdown(info->dev, par);

	// re-enable it
	enable_irq(par->irq);
	return 0;
}

static int qfb_ioctl_poll_intr(struct fb_info *info, struct qfb_intr __user *intr)
{
	unsigned long flags;
	struct qfb_info *par;
	par = (struct qfb_info*)info->par;

	spin_lock_irqsave(&par->intr_lck, flags);

	copy_to_user((void __user*)intr, &par->intr, sizeof(struct qfb_intr));
	memset(&par->intr, 0, sizeof(par->intr));

	spin_unlock_irqrestore(&par->intr_lck, flags);
	return 0;
}

static int qfb_ioctl_cur_dbid(struct fb_info *info, uint32_t __user *id)
{
	int ret;
	uint32_t dbid;
	struct qfb_info *par;

	par  = (struct qfb_info*)info->par;
	dbid = qfb_cur_dbid(info->dev, par);

	ret = put_user(dbid, id);

	if (ret < 0) {
		dev_err(info->dev, "%s: cannot return db id to user\n", __func__);
		return ret;
	}

	return 0;
}

static int qfb_ioctl_set_dbid(struct fb_info *info, uint32_t __user *id)
{
	int ret;
	uint32_t dbid;
	struct qfb_info *par;

	ret = get_user(dbid, id);

	if (ret < 0) {
		dev_err(info->dev, "%s: cannot get db id from user\n", __func__);
		return ret;
	}

	par = (struct qfb_info*)info->par;
	qfb_set_dbid(info->dev, par, dbid);

	return 0;
}

static int qfb_ioctl_update_timing(struct fb_info *info, void __user *argp) 
{
	struct qfb_info *par;
	struct qfb_timing timing;

	if (copy_from_user(&timing, argp, sizeof(timing))) {
		dev_err(info->dev, "%s: fail to copy timing from user\n", __func__);
		return -EFAULT;
	}

	// keep it in par, and update to HW
	par = (struct qfb_info *)info->par;

	/*
	 * We only update those attributes without 0
	 */
	if (timing.hfp_width) {
		dev_dbg(info->dev, "%s: change hfp from %u to %u\n", 
				__func__, par->panel.timing.hfp_width, timing.hfp_width);
		par->panel.timing.hfp_width = timing.hfp_width;
	}

	if (timing.hbp_width) {
		dev_dbg(info->dev, "%s: change hbp from %u to %u\n", 
				__func__, par->panel.timing.hbp_width, timing.hbp_width);
		par->panel.timing.hbp_width = timing.hbp_width;
	}

	if (timing.hsy_width) {
		dev_dbg(info->dev, "%s: change hsy from %u to %u\n", 
				__func__, par->panel.timing.hsy_width, timing.hsy_width);
		par->panel.timing.hsy_width = timing.hsy_width;
	}

	if (timing.vfp_width) {
		dev_dbg(info->dev, "%s: change vfp from %u to %u\n", 
				__func__, par->panel.timing.vfp_width, timing.vfp_width);
		par->panel.timing.vfp_width = timing.vfp_width;
	}

	if (timing.vbp_width) {
		dev_dbg(info->dev, "%s: change vbp from %u to %u\n", 
				__func__, par->panel.timing.vbp_width, timing.vbp_width);
		par->panel.timing.vbp_width = timing.vbp_width;
	}

	if (timing.vsy_width) {
		dev_dbg(info->dev, "%s: change vsy from %u to %u\n", 
				__func__, par->panel.timing.vsy_width, timing.vsy_width);
		par->panel.timing.vsy_width = timing.vsy_width;
	}

	qfb_set_timing(info->dev, par);
	dev_dbg(info->dev, "%s: set timing done\n", __func__);

	return 0;
}

static int qfb_ioctl_dump_regs(struct fb_info *info, void __user *argp)
{
#ifdef DEBUG
#define __QFB_DUMP_REG(base, reg) {                         \
	dev_dbg(info->dev, "%s: " #reg "=%08x\n", __func__,     \
			qfb_rreg32(base, reg##_OFF));                  \
}
	struct qfb_info *par = (struct qfb_info *)info->par;

	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_CSR);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_BIBADDR);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_HORT);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_VERT);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_BISIZE);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_INT_MASK);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_FSBADDR);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_FSCNA);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OWCSR);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW0BADDR);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW0SIZE);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW0COORD);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW1BADDR);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW1SIZE);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW1COORD);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OWVHSIZE);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_BI2BADDR);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW02BADDR);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW02SIZE);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW02COORD);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW12BADDR);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW12SIZE);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW12COORD);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_OW2VHSIZE);
	__QFB_DUMP_REG(par->base_ctl, LCD_LCDCTRL_DBG1);
#endif
	return 0;
}

static int qfb_ioctl(struct fb_info *info,
		unsigned int cmd, unsigned long arg)
{
	void __user *argp;
	int ret = 0;

	argp = (void __user *)arg;

	switch (cmd) {
		case QFB_IOCTL_SET_DEVINFO:
			dev_dbg(info->dev, "%s: QFB_IOCTL_SET_DEVINFO\n", __func__);
			ret = qfb_ioctl_set_devinfo(info, argp);
			break;
		case QFB_IOCTL_RESET_DEFAULT:
			dev_dbg(info->dev, "%s: QFB_IOCTL_RESET_DEFAULT\n", __func__);
			ret = qfb_ioctl_reset_default(info);
			break;
		case QFB_IOCTL_SHUTDOWN:
			dev_dbg(info->dev, "%s: QFB_IOCTL_SHUTDOWN\n", __func__);
			ret = qfb_ioctl_shutdown(info);
			break;
		case QFB_IOCTL_POLL_INTR:
//			dev_dbg(info->dev, "%s: QFB_IOCTL_POLL_INTR\n", __func__);
			ret = qfb_ioctl_poll_intr(info, argp);
			break;
		case QFB_IOCTL_CUR_DBID:
//			dev_dbg(info->dev, "%s: QFB_IOCTL_CUR_DBID\n", __func__);
			ret = qfb_ioctl_cur_dbid(info, argp);
			break;
		case QFB_IOCTL_SET_DBID:
//			dev_dbg(info->dev, "%s: QFB_IOCTL_SET_DBID\n", __func__);
			ret = qfb_ioctl_set_dbid(info, argp);
			break;
		case QFB_IOCTL_UPDATE_TIMING:
			dev_dbg(info->dev, "%s: QFB_IOCTL_UPDATE_TIMING\n", __func__);
			ret = qfb_ioctl_update_timing(info, argp);
			break;
		case QFB_IOCTL_DUMP_REGS:
			ret = qfb_ioctl_dump_regs(info, argp);
			break;
		default:
			ret = -ENOTTY;
	}

	return ret;
}

static __poll_t qfb_poll(struct file *file, 
		struct fb_info *info, struct poll_table_struct *pt)
{
	__poll_t mask = 0;
	unsigned long flags;
	struct qfb_info *par = info->par;

	poll_wait(file, &par->wait_intr, pt);

	spin_lock_irqsave(&par->intr_lck, flags);
	if (par->intr.st_vert_fp    || 
		par->intr.st_vert_bp    ||
		par->intr.st_vsync      ||
		par->intr.fs_mark1      ||
		par->intr.fs_mark2      ||
		par->intr.db_switched   ||
		par->intr.underrun) {

		mask |= POLLIN | POLLRDNORM;
	}
	spin_unlock_irqrestore(&par->intr_lck, flags);

	return mask;
}

/* ------------------------------------------------------------------------- */

/* Frame buffer operations */

static struct fb_ops qfb_ops =
{
	.owner          = THIS_MODULE,
	.fb_check_var   = qfb_check_var,
	.fb_set_par     = qfb_set_par,
	.fb_mmap        = qfb_mmap,
	.fb_ioctl       = qfb_ioctl,
	.fb_poll        = qfb_poll,
};

/* ------------------------------------------------------------------------- */

static irqreturn_t qfb_isr(int this_irq, void *cookie)
{
	uint32_t stat = 0;
	uint32_t intr = 0;
	unsigned long flags;
	struct fb_info *info = (struct fb_info *) cookie;
	struct qfb_info *qfb_info = info->par;

	spin_lock_irqsave(&qfb_info->intr_lck, flags);

	stat = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_STAT_OFF);
	intr = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_INT_VECT_OFF);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_INT_CLR_OFF, intr);

	if (intr & LCD_LCDCTRL_INT_VECT__RESPE__MASK) {
		dev_err(info->dev, "%s: DMA response error\n", __func__);
	}

	// bookkeeping interrupts
	if (intr & LCD_LCDCTRL_INT_VECT__VSVFP__MASK) {
		qfb_info->intr.st_vert_fp = 1;
	}
	if (intr & LCD_LCDCTRL_INT_VECT__VSVBP__MASK) {
		qfb_info->intr.st_vert_bp = 1;
	}
	if (intr & LCD_LCDCTRL_INT_VECT__VSVS__MASK) {
		qfb_info->intr.st_vsync = 1;
	}
	if (intr & LCD_LCDCTRL_INT_VECT__FSNM1__MASK) {
		qfb_info->intr.fs_mark1 = 1;
	}
	if (intr & LCD_LCDCTRL_INT_VECT__FSNM2__MASK) {
		qfb_info->intr.fs_mark2 = 1;
	}
	if (intr & LCD_LCDCTRL_INT_VECT__DTC__MASK) {
		qfb_info->intr.db_switched = 1;
	}

	//
	// FIXME: A0's bug. FIFO output underrun is untrusted.
	// So is FIFOU. Re-enable it in B0.
	//
//	if (intr & LCD_LCDCTRL_INT_VECT__FIFOU__MASK) {
//		qfb_info->intr.underrun = 1;
//	}

	if (intr & (LCD_LCDCTRL_INT_VECT__BIINFIFOU__MASK |
				LCD_LCDCTRL_INT_VECT__OW0INFIFOU__MASK |
				LCD_LCDCTRL_INT_VECT__OW1INFIFOU__MASK)) {
		qfb_info->intr.underrun = 1;
		dev_err(info->dev, "%s: FIFO underrun (%08x)\n", __func__, intr);
	}

	spin_unlock_irqrestore(&qfb_info->intr_lck, flags);

	// notify other process that the interrupt is available
	wake_up_interruptible(&qfb_info->wait_intr);
	return IRQ_HANDLED;
}

static void qfb_qfb_setup_irq(struct fb_info *info)
{
	struct qfb_info *par = info->par;

	dev_dbg(info->dev, "%s: irq %d\n", __func__, par->irq);
	devm_request_threaded_irq(info->dev, par->irq, 
			NULL, qfb_isr, IRQF_ONESHOT, "qfb", info);
}

static int qfb_init_res_io_mem(struct platform_device *pdev, struct qfb_info *qfb_info)
{
	struct resource *ctls;
	struct resource *gpfs;
	struct resource *intfs;
	struct resource *iopad;
	struct resource *ddrgpfs;
	struct resource *prtpios;

	ctls = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!ctls) {
		dev_err(&pdev->dev, "%s: no `reg` defined in dts\n", __func__);
		return -ENXIO;
	}

	qfb_info->base_ctl  = devm_ioremap_resource(&pdev->dev, ctls);
	if (IS_ERR(qfb_info->base_ctl)) {
		return PTR_ERR(qfb_info->base_ctl);
	}

	gpfs = platform_get_resource(pdev, IORESOURCE_MEM, 1);
	if (!gpfs) {
		dev_err(&pdev->dev, "%s: not enough `reg` defined in dts\n", __func__);
		return -ENXIO;
	}

	qfb_info->base_gpf  = devm_ioremap_resource(&pdev->dev, gpfs);
	if (IS_ERR(qfb_info->base_gpf)) {
		return PTR_ERR(qfb_info->base_gpf);
	}

	intfs = platform_get_resource(pdev, IORESOURCE_MEM, 2);
	if (!intfs) {
		dev_err(&pdev->dev, "%s: not enough `reg` defined in dts\n", __func__);
		return -ENXIO;
	}

	qfb_info->base_intf  = devm_ioremap_resource(&pdev->dev, intfs);
	if (IS_ERR(qfb_info->base_intf)) {
		return PTR_ERR(qfb_info->base_intf);
	}

	iopad = platform_get_resource(pdev, IORESOURCE_MEM, 3);
	if (!iopad) {
		dev_err(&pdev->dev, "%s: not enough `reg` defined in dts\n", __func__);
		return -ENXIO;
	}

	qfb_info->base_iopad = devm_ioremap_resource(&pdev->dev, iopad);
	if (IS_ERR(qfb_info->base_iopad)) {
		return PTR_ERR(qfb_info->base_iopad);
	}

	ddrgpfs = platform_get_resource(pdev, IORESOURCE_MEM, 4);
	if (!ddrgpfs) {
		dev_err(&pdev->dev, "%s: not enough `reg` defined in dts\n", __func__);
		return -ENXIO;
	}

	qfb_info->base_ddrgpf = devm_ioremap_resource(&pdev->dev, ddrgpfs);
	if (IS_ERR(qfb_info->base_ddrgpf)) {
		return PTR_ERR(qfb_info->base_ddrgpf);
	}

	prtpios = platform_get_resource(pdev, IORESOURCE_MEM, 5);
	if (!prtpios) {
		dev_err(&pdev->dev, "%s: not enough `reg` defined in dts\n", __func__);
		return -ENXIO;
	}

	qfb_info->base_prtpio = devm_ioremap_resource(&pdev->dev, prtpios);
	if (IS_ERR(qfb_info->base_prtpio)) {
		return PTR_ERR(qfb_info->base_prtpio);
	}

	dev_dbg(&pdev->dev, "%s: ctls %pa~%pa iomem %px\n", 
			__func__, &ctls->start, &ctls->end, qfb_info->base_ctl);
	dev_dbg(&pdev->dev, "%s: gpfs %pa~%pa iomem %px\n", 
			__func__, &gpfs->start, &gpfs->end, qfb_info->base_gpf);
	dev_dbg(&pdev->dev, "%s: intfs %pa~%pa iomem %px\n", 
			__func__, &intfs->start, &intfs->end, qfb_info->base_intf);
	dev_dbg(&pdev->dev, "%s: iopad %pa~%pa iomem %px\n", 
			__func__, &iopad->start, &iopad->end, qfb_info->base_iopad);
	dev_dbg(&pdev->dev, "%s: ddrgpfs %pa~%pa iomem %px\n", 
			__func__, &ddrgpfs->start, &ddrgpfs->end, qfb_info->base_ddrgpf);
	dev_dbg(&pdev->dev, "%s: prtpios %pa~%pa iomem %px\n", 
			__func__, &prtpios->start, &prtpios->end, qfb_info->base_prtpio);

	return 0;
}

static int qfb_init_res_pll(struct platform_device *pdev,
		struct qfb_info *qfb_info,
		struct device_node *panel)
{
#define __READ_PLL_PROP(nd, prop, store)                \
	if (of_property_read_u32(nd, prop, store) < 0) {    \
		return -ENXIO;                                  \
	}

	struct device_node *pll;

	pll = of_find_node_by_name(panel, "pll");

	if (!pll) {
		dev_dbg(&pdev->dev, "%s: no pll node defined\n", __func__);
		return -ENXIO;
	}

	__READ_PLL_PROP(pll, "upcnt", &qfb_info->panel.pll.upcnt);
 	__READ_PLL_PROP(pll, "dkin", &qfb_info->panel.pll.dkin);
	__READ_PLL_PROP(pll, "min", &qfb_info->panel.pll.min);
	__READ_PLL_PROP(pll, "kin", &qfb_info->panel.pll.kin);
	__READ_PLL_PROP(pll, "reg1", &qfb_info->panel.pll.reg1);
	__READ_PLL_PROP(pll, "reg2", &qfb_info->panel.pll.reg2);
	__READ_PLL_PROP(pll, "reg3", &qfb_info->panel.pll.reg3);
	__READ_PLL_PROP(pll, "reg4", &qfb_info->panel.pll.reg4);
	__READ_PLL_PROP(pll, "fidiv", &qfb_info->panel.pll.fidiv);
	__READ_PLL_PROP(pll, "pdiv", &qfb_info->panel.pll.pdiv);
	__READ_PLL_PROP(pll, "qdiv", &qfb_info->panel.pll.qdiv);
	__READ_PLL_PROP(pll, "divl", &qfb_info->panel.pll.divl);
	__READ_PLL_PROP(pll, "vcolmt", &qfb_info->panel.pll.vcolmt);
	__READ_PLL_PROP(pll, "cp2en", &qfb_info->panel.pll.cp2en);
	__READ_PLL_PROP(pll, "cp2en2", &qfb_info->panel.pll.cp2en2);
	return 0;
}

static int qfb_init_res_panel(struct platform_device *pdev, struct qfb_info *qfb_info)
{
	int ret;
	struct device_node *endpoint;
	struct device_node *panel;
	struct display_timing timing;
	const char *data_mapping;

	// set default hv_align, and we can overwrite this value in devicetree
	// with the property `hv-align`. See enum `qfb_hv_align` in `uapi/qfb.h` 
	qfb_info->panel.hv_align = QFB_HV_ALIGN_HFRIST;

	/**
	 * NOTE:
	 * If there are more than one available `endpoint`, which means 
	 * the `status` properties are all `okay` or `ok`, the behavior
	 * of which one is chosen is undefined.
	 */
	for_each_endpoint_of_node(pdev->dev.of_node, endpoint) {
		if (!of_device_is_available(endpoint)) {
			dev_dbg(&pdev->dev, "%s: %s not available\n", __func__, endpoint->name);
			of_node_put(endpoint);
			continue;
		}

		panel = of_graph_get_remote_port_parent(endpoint);
		if (!panel) {
			dev_dbg(&pdev->dev, "%s: no remote panel found\n", __func__);
			of_node_put(endpoint);
			return -ENXIO;
		}

		ret = of_get_display_timing(panel, "panel-timing", &timing);
		if (ret < 0) {
			dev_dbg(&pdev->dev, "%s: no panel-timing found\n", __func__);
			of_node_put(endpoint);
			of_node_put(panel);
			return -ENXIO;
		}

		ret = qfb_init_res_pll(pdev, qfb_info, panel);
		if (ret < 0) {
			dev_dbg(&pdev->dev, "%s: fail to get pll settings\n", __func__);
			of_node_put(endpoint);
			of_node_put(panel);
			return -ENXIO;
		}

		if (of_device_is_compatible(panel, "panel-lvds")) {
			dev_dbg(&pdev->dev, "%s: panel(%s) is lvds panel\n", __func__, panel->name);

			// Possible valuse are "jeida-18", "jeida-24", "vesa-24". See 
			// 'Documentation/devicetree/bindings/display/panel/panel-lvds.txt'
			of_property_read_string(panel, 
					"data-mapping", &data_mapping);

			if (strcmp(data_mapping, "jeida-24") == 0) {
				qfb_info->panel.tx_mode = QFB_TX_LVDS_24_0;
			} else if (strcmp(data_mapping, "vesa-24") == 0 || strcmp(data_mapping, "jeida-18") == 0) {
				// XXX: For `jeida-18`, if the setup has some problem, might try
				// to set LCD_LVDSVX1_CSR1#LVCT4
				qfb_info->panel.tx_mode = QFB_TX_LVDS_24_1;
			} else {
				dev_err(&pdev->dev, "%s: invalid LVDS mode(%s)\n", __func__, data_mapping);
			}

			qfb_info->panel.data_mirror = of_property_read_bool(panel, "data-mirror");
			qfb_info->panel.panel_type  = QFB_PANEL_LVDS;
		} else if (of_device_is_compatible(panel, "panel-vx1")) {
			dev_dbg(&pdev->dev, "%s: panel(%s) is vx1 panel\n", __func__, panel->name);

			// The data-mapping attributes in vx1 represents the byte mode.
			// Since there is no indication in linux, the possible values are
			// defined by QBit.
			of_property_read_string(panel,
					"data-mapping", &data_mapping);

			if (strcmp(data_mapping, "3-byte") == 0) {
				qfb_info->panel.vx1_mode = QFB_VX1_3BYTE;
			} else if (strcmp(data_mapping, "4-byte") == 0) {
				qfb_info->panel.vx1_mode = QFB_VX1_4BYTE;
			} else if (strcmp(data_mapping, "5-byte") == 0) {
				qfb_info->panel.vx1_mode = QFB_VX1_5BYTE;
			} else if (strcmp(data_mapping, "3-byte-dup") == 0) {
				qfb_info->panel.vx1_mode = QFB_VX1_3BYTE_DUP;
			} else {
				dev_err(&pdev->dev, "%s: invalid Vx1 mode(%s)\n", __func__, data_mapping);
			}
			qfb_info->panel.data_mirror   = of_property_read_bool(panel, "data-mirror");
			qfb_info->panel.vx1_adap_lvds = of_property_read_bool(panel, "vx1-adap-lvds");

			qfb_info->panel.tx_mode    = QFB_TX_VX1;
			qfb_info->panel.panel_type = QFB_PANEL_VX1;
		} else {
			// XXX: if we have more panels, the `else` clause might not be 
			// considered as srgb
			dev_dbg(&pdev->dev, "%s: panel(%s) is srgb panel\n", __func__, panel->name);

			of_property_read_string(panel, 
					"data-mapping", &data_mapping);

			// default use 888 mode
			if (strcmp(data_mapping, "565") == 0) {
				qfb_info->panel.tx_mode = QFB_TX_RGB_565;
			} else if (strcmp(data_mapping, "666") == 0) {
				qfb_info->panel.tx_mode = QFB_TX_RGB_666;
			} else {
				qfb_info->panel.tx_mode = QFB_TX_RGB_888;
			}

			qfb_info->panel.panel_type = QFB_PANEL_SRGB;
		}

		// For the sake of simplicity, we use typical properties
		qfb_info->panel.clk = timing.pixelclock.typ;
		qfb_info->panel.h_res = timing.hactive.typ;
		qfb_info->panel.timing.hfp_width = timing.hfront_porch.typ;
		qfb_info->panel.timing.hbp_width = timing.hback_porch.typ;
		qfb_info->panel.timing.hsy_width = timing.hsync_len.typ;
		qfb_info->panel.v_res = timing.vactive.typ;
		qfb_info->panel.timing.vfp_width = timing.vfront_porch.typ;
		qfb_info->panel.timing.vbp_width = timing.vback_porch.typ;
		qfb_info->panel.timing.vsy_width = timing.vsync_len.typ;
		of_property_read_u32(panel, "hv-align", &qfb_info->panel.hv_align);

		// TODO
		// Read `display_timing#display_flags`

		dev_dbg(&pdev->dev, "%s: panel - clk = %u\n", __func__, qfb_info->panel.clk);
		dev_dbg(&pdev->dev, "%s: panel - h_res = %u\n", __func__, qfb_info->panel.h_res);
		dev_dbg(&pdev->dev, "%s: panel - hfp_width = %u\n", __func__, qfb_info->panel.timing.hfp_width);
		dev_dbg(&pdev->dev, "%s: panel - hbp_width = %u\n", __func__, qfb_info->panel.timing.hbp_width);
		dev_dbg(&pdev->dev, "%s: panel - hsy_width = %u\n", __func__, qfb_info->panel.timing.hsy_width);
		dev_dbg(&pdev->dev, "%s: panel - v_res = %u\n", __func__, qfb_info->panel.v_res);
		dev_dbg(&pdev->dev, "%s: panel - vfp_width = %u\n", __func__, qfb_info->panel.timing.vfp_width);
		dev_dbg(&pdev->dev, "%s: panel - vbp_width = %u\n", __func__, qfb_info->panel.timing.vbp_width);
		dev_dbg(&pdev->dev, "%s: panel - vsy_width = %u\n", __func__, qfb_info->panel.timing.vsy_width);
		dev_dbg(&pdev->dev, "%s: panel - tx_mode = %u\n", __func__, qfb_info->panel.tx_mode);
		dev_dbg(&pdev->dev, "%s: panel - data_mirror = %u\n", __func__, qfb_info->panel.data_mirror);

		of_node_put(endpoint);
		of_node_put(panel);
	}

	return 0;
}

static int qfb_init_resource(struct platform_device *pdev, struct qfb_info *qfb_info)
{
	int ret;

	ret = qfb_init_res_io_mem(pdev, qfb_info);
	if (ret != 0) {
		return ret;
	}

	ret = qfb_init_res_panel(pdev, qfb_info);
	if (ret != 0) {
		return ret;
	}

	return 0;
}

static void qfb_par_init(struct qfb_info *par)
{
	memset(par, 0, sizeof(*par));
	spin_lock_init(&par->intr_lck);
	init_waitqueue_head(&par->wait_intr);
}

static int __init quasar_fb_probe(struct platform_device *pdev)
{
	int ret;
	struct fb_info *info;
	struct qfb_info *par;

	dev_dbg(&pdev->dev, "%s: init start\n", __func__);

	info = framebuffer_alloc(sizeof(*par), &pdev->dev);
	if (!info) {
		dev_err(&pdev->dev, "%s: cannot allocate memory\n", __func__);
		return -ENOMEM;
	}

	par = info->par;
	qfb_par_init(par);

	ret = qfb_init_resource(pdev, par);
	if (ret < 0) {
		return ret;
	}

	par->irq = platform_get_irq(pdev, 0);
	if (par->irq < 0) {
		dev_err(&pdev->dev, "%s: no irq specified\n", __func__);
		return -ENOENT;
	}

	info->pseudo_palette        = NULL;
	info->fbops                 = &qfb_ops;
	info->flags                 = FBINFO_FLAG_DEFAULT | FBINFO_READS_FAST;

	qfb_reset_fb_info(info);

	if (register_framebuffer(info) < 0) {
		return -EINVAL;
	}

	// This should be called after `register_framebuffer`, since 
	// 'qfb_reset_fb_mem` needs `info->dev` which was constructed
	// in `register_framebuffer`
	qfb_reset_fb_mem(info);

	platform_set_drvdata(pdev, info);
	device_init_wakeup(&pdev->dev, 1);      // XXX: Need this?

	// qfb_setup irq
	qfb_qfb_setup_irq(info);

	dev_dbg(&pdev->dev, "%s: init end\n", __func__);
	return 0;
}

static int __exit quasar_fb_remove(struct platform_device *pdev)
{
	struct fb_info *info;
	struct qfb_info *par;

	/* FIXME: Need to stop hardware first */

	info = platform_get_drvdata(pdev);
	if (!info) {
		return -EINVAL;
	}

	par = info->par;
	if (!par) {
		return -EINVAL;
	}

	if (par->info_mm) {
		dma_unmap_single(info->dev->parent, 
				par->info_mm->dma_addr, sizeof(*par->info_mm->devinfo), DMA_TO_DEVICE);

		kfree(par->info_mm->devinfo);
		kfree(par->info_mm);
	}
	qfb_release_all_mem(info->dev);
	return 0;
}

MODULE_ALIAS("platform:quasar-fb");

static const struct of_device_id quasar_fb_id_table[] = {
	{ .compatible = "qbit,quasar-fb" },
	{}
};
MODULE_DEVICE_TABLE(of, quasar_fb_id_table);

static struct platform_driver quasar_fb_driver_ops = {
	.probe		= quasar_fb_probe,
	.remove		= quasar_fb_remove,
	.driver		= {
		.name	= "quasar-fb",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(quasar_fb_id_table),	
	},
};

static int __init quasar_fb_init(void)
{
	int ret;
	ret = platform_driver_register(&quasar_fb_driver_ops);
	return ret;
}
module_init(quasar_fb_init);

static void __exit quasar_fb_exit(void)
{
	platform_driver_unregister(&quasar_fb_driver_ops);
}
module_exit(quasar_fb_exit);

MODULE_DESCRIPTION(" Quasar fb driver");
MODULE_LICENSE("Dual BSD/GPL");

