// =========================================================
//
//           Q QQ QQ Q
//       QQ QQ QQ QQ QQ Q
//    QQ QQ QQ QQ QQ QQ QQ QQ     BBBBBBBB      iiii
//  Q QQ QQ             QQ QQ Q   BBBBBBBBBB    iii   ttt
// QQ QQ       QQ QQ       QQ QQ  BBB    BBB          ttt
// QQ QQ       QQ QQ       QQ QQ  BBB    BBB          tttttt
// QQ QQ       QQ QQ       QQ QQ  BBBBBBBBBB    iii   ttttt
// QQ QQ       QQ QQ       QQ QQ  BBBBBBBBBBB   iii   ttt
// QQ QQ       QQ QQ       QQ QQ  BBB     BBBB  iii   ttt
//  Q QQ       QQ QQ       QQ Q   BBB     BBBB  iii   ttt
//    QQ QQ             QQ QQ     BBBBBBBBBBBB  iii   tttttt
//     Q QQ QQ QQ QQ       Q      BBBBBBBBBB    iii    tttt
//          QQ QQ QQ QQ
//           Q QQ QQ Q            
// 
// Copyright (C) 2017-2018 QBit Semiconductor LTD.
// =========================================================
//
//  $DateTime: 2022/03/14 10:30:50 $
//  $Change: 58883 $
//
// =========================================================
#ifndef __DRIVERS_QFB_CORE_H
#define __DRIVERS_QFB_CORE_H

#include <linux/device.h>
#include <linux/types.h>
#include <linux/spinlock_types.h>
#include <quasar/qbsocregs.h>
#include "uapi/qfb.h"


#define QFB_CHK_OWEN(q_dev, id) \
	((q_dev)->en_ow & (1 << id))

#define QFB_FS_STNA_V(attr, valid)                  \
	do {                                            \
		attr = (attr & 0x7fffffff) | (valid << 31); \
	} while (0)

#define QFB_FS_OW_PRIO(attr, ow_id)                 \
	do {                                            \
		attr = (attr & 0xfeffffff) | (ow_id << 24); \
	} while (0)

#define QFB_FS_EN_OW(attr, ow_id, en)   \
	do {                                \
		attr &= ~(0x1 << (ow_id * 4));  \
		attr |= (en << (ow_id * 4));    \
	} while (0)

#define QFB_FS_SET_OW(attr, ow_id, packed, aben, avsrc, en)             \
	do {                                                                \
		uint8_t val;                                                    \
		attr &= ~(0xf << ((ow_id) * 4));                                \
		val = ((avsrc) << 3) | ((aben) << 2) | ((packed) << 1) | (en);  \
		val = val << ((ow_id) * 4);                                     \
		attr |= val;                                                    \
	} while (0)

#define QFB_FS_CONS_ALAPH(attr, ow_id, alpha)                   \
	do {                                                        \
		attr &= ~(0xff << (8 + ow_id * 8));                     \
		attr |= (alpha << (8 + ow_id * 8));                     \
	} while (0)

#define QFB_FS_SNM(attr, snm_id, en)                            \
	do {                                                        \
		attr &= ~(1 << (24 + snm_id));                          \
		attr |= (en << (24 + snm_id));                          \
	} while (0)

#define QFB_FS_VHSIZE(attr, ow_id, vis_xlen)                    \
	do {                                                        \
		attr &= ~(0xfff << (ow_id*16));                         \
		attr |= ((vis_xlen-1) << (ow_id*16));                   \
	} while (0)

#define QFB_FS_BI_ATTR(attr, width, height, packed)             \
	do {                                                        \
		attr = (packed << 28) | ((height-1) << 16) | (width-1); \
	} while (0)

#define QFB_CLRSET(val, reg, attr, attr_val)                \
	do {                                                    \
		val = (val & reg ## __ ## attr ## __INV_MASK) |     \
			  ((attr_val) << reg ## __ ## attr ## __SHIFT); \
	} while (0) 

#define QFB_CLRSET_WREG32(base, reg, attr, attr_val)        \
	do {                                                    \
		uint32_t val = qfb_rreg32(base, (reg##_OFF));       \
		val = (val & reg ## __ ## attr ## __INV_MASK) |     \
			  ((attr_val) << reg ## __ ## attr ## __SHIFT); \
		qfb_wreg32(base, reg##_OFF, val);                   \
	} while (0)

/*
 * Panel type
 * 
 * QFB_PANEL_SRGB: srgb panel
 * QFB_PANEL_LVDS: lvds panel
 * QFB_PANEL_VX1: vx1 panel
 */
enum qfb_panel_type {
	QFB_PANEL_SRGB = 0,
	QFB_PANEL_LVDS,
	QFB_PANEL_VX1,
};

/*
 * Transmit modes. 
 * 
 * Note that for LVDS, the `data-mapping` attribute in devicetree specify the LVDS format.
 * 
 * QFB_TX_NONE: no setting, which means it does not sRGB panel
 * QFB_TX_RGB_888: 8-bit Serial RGB, 3-color, No Packing, 888 Mode, in device tree, the value is '888'
 * QFB_TX_RGB_565: 8-bit Serial RGB, 3-color Packing, 565 Mode, in device tree, the value is '565'
 * QFB_TX_RGB_666: 9-bit Serial RGB, 3-color Packing, 666 Mode, in device tree, the value is '666'
 * QFB_TX_LVDS_24_0: LVDS 7x, 4 channel Mode, 24-bit RGB, 24.0 Format. In device tree, value is `jeida-24`
 * QFB_TX_LVDS_24_1: LVDS 7x, 4 channel Mode, 24-bit RGB, 24.1 Format. 
 *                Also used for LVDS 7x, 3 channel mode. In device tree, possible values
 *                are `vesa-24` and `jeida-18`
 * QFB_TX_VX1: Vx1 mode
 * 
 */
enum qfb_tx_mode {
	QFB_TX_NONE      = 0,
	QFB_TX_RGB_888   = 0b101,
	QFB_TX_RGB_565   = 0b110,
	QFB_TX_RGB_666   = 0b111,
	QFB_TX_LVDS_24_0 = 0b001,
	QFB_TX_LVDS_24_1 = 0b010,
	QFB_TX_VX1       = 0b100,
};

/*
 * Vx1 modes. The `data-mapping` attribute in devicetree specify the Vx1 mode.
 * 
 * QFB_VX1_3BYTE: vx1 3-byte mode
 * QFB_VX1_4BYTE: vx1 4-byte mode
 * QFB_VX1_5BYTE: vx1 5-byte mode
 * QFB_VX1_3BYTE_DUP: vx1 3-byte duplication mode
 * 
 */
enum qfb_vx1_mode {
	QFB_VX1_3BYTE   = 0,
	QFB_VX1_4BYTE,
	QFB_VX1_5BYTE,
	QFB_VX1_3BYTE_DUP,
};

/**
 * struct qfb_pll - PLL properties
 * 
 * Reference socionext PLL document
 */
struct qfb_pll
{
	uint32_t    upcnt;
	uint32_t    dkin;
	uint32_t    min;
	uint32_t    kin;
	uint32_t    reg1;
	uint32_t    reg2;
	uint32_t    reg3;
	uint32_t    reg4;
	uint32_t    fidiv;
	uint32_t    pdiv;
	uint32_t    qdiv;
	uint32_t    divl;
	uint32_t    vcolmt;
	uint32_t    cp2en;
	uint32_t    cp2en2;
};

/**
 * struct qfb_panel - panel properties
 * 
 * @clk: panel dot clock frequency in Hz
 * @h_res: horizontal resolution in pixels
 * @v_res: vertical resolution in pixels
 * @timing: panel timings
 * @pll: PLL properties
 * @hv_align: Horizontal line alignment w/ VSYNC, some 
 *            LCD panel needs some special alignment
 * @panel_type: types of the panel
 * @data_mirror: this only make sense for LVDS/Vx1 panel. If set, reverse the bit order
 *               of data transmission. See
 *               'Documentation/devicetree/bindings/display/panel/panel-lvds.txt'
 * @tx_mode: transmit mode
 * @vx1_mode: this is only for Vx1
 * @vx1_adap_lvds: this special mode is for vx1 to lvds adapter. The value is set
 *                 in device tree attribute 'vx1-adap-lvds'.
 *                 See LCD_LVDSVX1_CSR1#VX1TOLVDS_CONVEN for detail.
 */
struct qfb_panel
{
	uint32_t            clk;

	uint32_t            h_res;
	uint32_t            v_res;

	struct qfb_timing   timing;
	struct qfb_pll      pll;

	enum qfb_hv_align   hv_align;
	enum qfb_panel_type panel_type;

	int                 data_mirror;
	enum qfb_tx_mode    tx_mode;

	enum qfb_vx1_mode   vx1_mode;
	int                 vx1_adap_lvds;
};

/**
 * struct qfb_devinfo_mm - The mapping of virtual & DMA address of qfb_devinfo
 * 
 * @devinfo: virtual address of `struct qfb_devinfo`
 * @dma_addr: DMA address of `devinfo`
 * 
 */
struct qfb_devinfo_mm {
	struct qfb_devinfo *devinfo;
	dma_addr_t dma_addr;
};

/**
 * struct qfb_info - quasar device frame buffer
 * 
 * @info_mm: fb configuration for hardware internal use
 * @panel: panel settings from devicetree
 * @intr: bookkeeping latest interrupts from hardware
 * @intr_lck: lock for acessing `intr`
 * @wait_intr: wait queue for interrupt event
 * @base_ctl: base virtual address of controller component
 * @base_gpf: base virtual address of general purpose component
 * @base_intf: base virtual address of interface component
 * @base_ddrgpf: base virtual address of ddr general purpose component
 * @base_prtpio: base virtual address of PRT block gpio component
 * @irq: interrupt number of this device
 */
struct qfb_info
{
	struct qfb_devinfo_mm *info_mm;

	struct qfb_panel panel;
	struct qfb_intr intr;
	spinlock_t intr_lck;
	wait_queue_head_t wait_intr;

	void __iomem *base_ctl;
	void __iomem *base_gpf;
	void __iomem *base_intf;
	void __iomem *base_iopad;
	void __iomem *base_ddrgpf;
	void __iomem *base_prtpio;

	int irq;
};

void qfb_wreg32(void __iomem *base, int offset, uint32_t val);
uint32_t qfb_rreg32(void __iomem *base, int offset);
void qfb_set_timing(struct device *dev, struct qfb_info *qfb_info);
void qfb_setup(struct device *dev, struct qfb_info *qfb_info);
unsigned long qfb_devinfo_copy_from_user(struct device *dev, struct qfb_devinfo_mm *out,
		void __user *src);
void qfb_reset_hw(struct device *dev, struct qfb_info *qfb_info);
void qfb_shutdown(struct device *dev, struct qfb_info *qfb_info);
uint32_t qfb_cur_dbid(struct device *dev, struct qfb_info *qfb_info);
void qfb_set_dbid(struct device *dev, struct qfb_info *qfb_info, uint32_t id);
int qfb_init_devinfo_mm(struct device *dev, struct qfb_info *qfb_info);
#endif
