/*
 * Quasar LCD controller kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2022/03/31 10:38:25 $
//  $Change: 59512 $
//
// =========================================================
#define DEBUG

#include <linux/module.h>
#include <linux/device.h>
#include <linux/kernel.h>
#include <linux/io.h>
#include <linux/types.h>
#include <linux/uaccess.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <quasar/qbsocregs.h>
#include "uapi/qfb.h"
#include "mem.h"
#include "core.h"
#include "util.h"
#include "lvds.h"
#include "vx1.h"
#include "srgb.h"

#define QFB_REG_SHOW(dev, base, off) {                                         \
	dev_dbg(dev, "%s: " #off " = 0x%08x\n", __func__, qfb_rreg32(base, off)); \
}

#define QFB_HW_DEFAULT(bit) (bit##__HW_DEFAULT << bit##__SHIFT)

#define QFB_INTR_MASKS (LCD_LCDCTRL_INT_VECT__DTC__MASK         | \
						LCD_LCDCTRL_INT_VECT__FSNM1__MASK       | \
						LCD_LCDCTRL_INT_VECT__FSNM2__MASK       | \
						LCD_LCDCTRL_INT_VECT__VSVFP__MASK       | \
						LCD_LCDCTRL_INT_VECT__VSVBP__MASK       | \
						LCD_LCDCTRL_INT_VECT__VSVS__MASK        | \
						LCD_LCDCTRL_INT_VECT__BIINFIFOU__MASK   | \
						LCD_LCDCTRL_INT_VECT__OW0INFIFOU__MASK  | \
						LCD_LCDCTRL_INT_VECT__OW1INFIFOU__MASK  | \
						LCD_LCDCTRL_INT_VECT__OUTFIFOU__MASK    | \
						LCD_LCDCTRL_INT_VECT__RESPE__MASK       | \
						LCD_LCDCTRL_INT_VECT__FIFOU__MASK       )

// When shutdowning qfb, how many loop counts to check
// before leaving. This value should be carefully chosen.
#define QFB_SHUTDOWN_RETRY_CNT  (1000)

static inline dma_addr_t qfb_virt_to_dma_addr(
		void *virt_base, 
		void *virt_target, 
		dma_addr_t dma_base)
{
	return dma_base + (virt_target - virt_base);
}

/*
 * qfb_init_devinfo_mm - initialize structure `qfb_info_mm` and setup DMA mapping
 * 
 * @dev: qfb device
 * @qfb_info: structure of `qfb_info` passed in to setup `info_mm` member
 * 
 * return 0 if success
 */
int qfb_init_devinfo_mm(struct device *dev, struct qfb_info *qfb_info)
{
	if (qfb_info->info_mm) {
		dev_err(dev, "%s: `info_mm` should be NULL for initialization\n", __func__);
		goto safe_release;
	}

	qfb_info->info_mm = (struct qfb_devinfo_mm *)kzalloc(
			sizeof(*qfb_info->info_mm), 
			GFP_KERNEL);

	if (!qfb_info->info_mm) {
		dev_err(dev, "%s: Failed to allocate memory for qfb_devinfo_mm\n", __func__);
		goto safe_release;
	}

	qfb_info->info_mm->devinfo = (struct qfb_devinfo*)kzalloc(
			sizeof(*qfb_info->info_mm->devinfo),
			GFP_KERNEL);

	if (!qfb_info->info_mm->devinfo) {
		dev_err(dev, "%s: Failed to allocate memory for devinfo\n", __func__);
		goto safe_release;
	}

	qfb_info->info_mm->dma_addr = dma_map_single(dev->parent,
			qfb_info->info_mm->devinfo,
			sizeof(*qfb_info->info_mm->devinfo),
			DMA_TO_DEVICE);

	return 0;

safe_release:
	if (qfb_info->info_mm) {
		if (qfb_info->info_mm->devinfo) {
			// Breaking the dma mapping first
			dma_unmap_single(dev->parent, 
					qfb_info->info_mm->dma_addr, 
					sizeof(*qfb_info->info_mm->devinfo),
					DMA_TO_DEVICE);

			kfree(qfb_info->info_mm->devinfo);
		}
		kfree(qfb_info->info_mm);
	}
	return -EFAULT;
}

unsigned long qfb_devinfo_copy_from_user(
		struct device *dev, 
		struct qfb_devinfo_mm *out, 
		void __user *src)
{
	struct qfb_devinfo *q_dev;
	struct qfb_ow_node *ow;
	struct qfb_fs_node *fn;
	qfb_ptr_t img_src;
	size_t    img_size;
	int       i, j;

	q_dev = (struct qfb_devinfo*)kzalloc(sizeof(*q_dev), GFP_KERNEL);

	if (copy_from_user((void*)q_dev, src, sizeof(*q_dev))) {
		dev_err(dev, "%s: Fail to copy qfb_devinfo structure from user\n", __func__);
		return -EFAULT;
	}

	// handle BI image address
	for (i=0; i<MAX_BI; i++) {
		img_size = qfb_get_im_size(q_dev->bi_prop.width,
				q_dev->bi_prop.height,
				q_dev->bpp,
				q_dev->bi_prop.packed);

		img_src = q_dev->bi[i].bin;
		if (img_src) {
			q_dev->bi[i].bin = qfb_bi_req_mem(dev, i, img_size);
			if (copy_from_user(q_dev->bi[i].bin, img_src, img_size)) {
				dev_err(dev, "%s: Fail to copy BI images from user\n", __func__);
				return -EFAULT;
			}
		}
	}

	// handle plt address
	if (q_dev->plt.bin) {
		img_src = q_dev->plt.bin;
		q_dev->plt.bin = qfb_pal_req_mem(dev, 256*2);
		if (copy_from_user(q_dev->plt.bin, img_src, 256*2)) {
			dev_err(dev, "%s: Fail to copy plt from user\n", __func__);
			return -EFAULT;
		}
	}
	 
	// handle ow address
	for (i=0; i<MAX_BI; i++) {
		for (j=0; j<MAX_OW; j++) {
			ow = &q_dev->bi[i].ow[j];

			img_size = qfb_get_im_size(ow->width,
					ow->height,
					q_dev->bpp,
					ow->packed);
			
			img_src = ow->bin;
			if (img_src) {
				ow->bin = qfb_ow_req_mem(dev, i, j, img_size);
				if (copy_from_user(ow->bin, img_src, img_size)) {
					dev_err(dev, "%s: Fail to copy ow from user\n", __func__);
					return -EFAULT;
				}
			}
		}
	}

	// handle fs address
	for (i=0; i<q_dev->fs.node_cnt; i++) {
		fn = &q_dev->fs.nodes[i];

		// handle BI image address
		img_size = qfb_get_im_size(fn->bi_prop.width,
				fn->bi_prop.height,
				q_dev->bpp,
				fn->bi_prop.packed);

		img_src = fn->bi.bin;
		if (img_src) {
			fn->bi.bin = qfb_fs_bi_req_mem(dev, i, img_size);
			if (copy_from_user(fn->bi.bin, img_src, img_size)) {
				dev_err(dev, "%s: Fail to copy fs BI from user\n", __func__);
				return -EFAULT;
			}
		}

		// handle BI's related OW's address
		for (j=0; j<MAX_OW; j++) {
			ow = &fn->bi.ow[j];

			img_size = qfb_get_im_size(ow->width,
					ow->height,
					q_dev->bpp,
					ow->packed);

			img_src = ow->bin;
			if (img_src) {
				ow->bin = qfb_fs_ow_req_mem(dev, i, j, img_size);
				if (copy_from_user(ow->bin, img_src, img_size)) {
					dev_err(dev, "%s: Fail to copy fs OW from user\n", __func__);
					return -EFAULT;
				}
			}
		}
	}

	// dump copied content
	dev_dbg(dev, "%s: dump lcd q_dev\n", __func__);

#ifdef DEBUG
	qfb_dump_devinfo(dev, q_dev);
#endif

	out->devinfo = q_dev;
	out->dma_addr  = dma_map_single(dev->parent, q_dev, sizeof(*q_dev), DMA_TO_DEVICE);

	if (dma_mapping_error(dev->parent, out->dma_addr)) {
		dev_err(dev, "%s: DMA mapping error\n", __func__);
	}

	dev_dbg(dev, "%s: qfb_devinfo copied from user completes\n", __func__);
	return 0;
}

void qfb_wreg32(void __iomem *base, int offset, uint32_t val)
{	
	iowrite32(val, base+offset);
}

uint32_t qfb_rreg32(void __iomem *base, int offset)
{
	uint32_t val;

	val = ioread32(base+offset);
	return val;
}

static int qfb_bpp_to_regval(int bpp)
{
	int result;

	switch (bpp)
	{
		case  1: result = 0b000; break;
		case  2: result = 0b001; break;
		case  4: result = 0b010; break;
		case  8: result = 0b011; break;
		case 15: result = 0b100; break;
		case 16: result = 0b100; break;
		case 18: result = 0b101; break;
		case 24: result = 0b110; break;
		default:
		         // ERR
		         result = 0b111; 
		         break;
	}

	return result;
}

static void qfb_cal_owl_vis(
		struct device *dev,
		struct qfb_devinfo *q_dev, 
		struct qfb_ow_node *ow, 
		int       *vis_xlen, 
		int       *vis_ylen, 
		qfb_ptr_t *adj_bin)
{
	int bi_w   = q_dev->bi_prop.width;
	int bi_h   = q_dev->bi_prop.height;
	int offset = 0;
	int bytes_per_pix;

	switch (q_dev->bpp)
	{
	case 15:
	case 16:
		bytes_per_pix = 2;
		break;
	case 18:
	case 24:
		bytes_per_pix = ow->packed ? 3 : 4;
		break;
	default:
		dev_err(dev, "%s: Unsupported OW bits when calculating visibility\n", __func__);
		bytes_per_pix = 0;       // just give a init value to avoid compiler warning
	}

	//
	// calculate visible x length
	// 
	if (ow->x < 0) {
		*vis_xlen = ow->width + ow->x;
		// for the case that the overlay width larger than BI's
		*vis_xlen = (*vis_xlen > bi_w) ? bi_w : *vis_xlen;
	} else if (ow->x + ow->width > bi_w) {
		*vis_xlen = bi_w - ow->x;
	} else {
		*vis_xlen = ow->width;
	}

	//
	// calculate visible y length
	//
	if (ow->y < 0) {
		*vis_ylen = ow->height + ow->y;
		// for the case that the overlay height larger than BI's
		*vis_ylen = (*vis_ylen > bi_h) ? bi_h : *vis_ylen;
	} else if (ow->y + ow->height > bi_h) {
		*vis_ylen = bi_h - ow->y;
	} else {
		*vis_ylen = ow->height;
	}

	//
	// calculate offset of binary content addr
	// 
	if (ow->y < 0) {
		offset += (ow->y * -1) * ow->width * bytes_per_pix;
	} 

	if (ow->x < 0) {
		offset += (ow->x * -1) * bytes_per_pix;
	}

	*adj_bin = ow->bin + offset;
}

static uint32_t do_qfb_cur_dbid(struct qfb_info *qfb_info)
{
	uint32_t val;

	val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
	return (val & LCD_LCDCTRL_CSR__DBSEL__MASK) >> LCD_LCDCTRL_CSR__DBSEL__SHIFT;
}

static int qfb_is_en(struct qfb_info *qfb_info)
{
	uint32_t val;

	val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
	return !!(val & LCD_LCDCTRL_CSR__ENABLE__MASK);
}

static int qfb_setup_ow_attr(
		struct device *dev,
		struct qfb_info *qfb_info, 
		uint32_t bi_id,
		uint32_t ow_id)
{
	dma_addr_t dma;
	qfb_ptr_t adj_bin;        // owl content start addr adjusted by visiblity
	qfb_ptr_t bin;
	int x;
	int y;
	int bpp;
	int packed;
	int width;
	int height;
	int vis_xlen;
	int vis_ylen;

	struct qfb_devinfo *q_dev = qfb_info->info_mm->devinfo;
	struct qfb_bi_node *bi = &q_dev->bi[bi_id];

	/**
	 * Address / Dimension / Coord
	 */
	x      = bi->ow[ow_id].x;
	y      = bi->ow[ow_id].y;
	bin    = bi->ow[ow_id].bin;
	packed = bi->ow[ow_id].packed;
	width  = bi->ow[ow_id].width;
	height = bi->ow[ow_id].height;
	bpp    = q_dev->bpp;

	qfb_cal_owl_vis(dev, q_dev, &bi->ow[ow_id], &vis_xlen, &vis_ylen, &adj_bin);
	dev_dbg(dev, "%s: Vis: ow_x=%d, ow_y=%d, ow_w=%d, ow_h=%d, ow_bpp=%d,  \
			ow_packed=%d, ow_bin=%px, vis_xlen=%d, vis_ylen=%d, adj_bin=%px\n",
			__func__, x, y, width, height, bpp, packed, (void*)bin, vis_xlen, vis_ylen, (void*)adj_bin);

	// report error if the visible length is invalid
	if (vis_xlen <= 0 || vis_ylen <= 0) {
		dev_dbg(dev, 
				"Visible x/y length is invalid (xlen=%d, ylen=%d) will temparilty disable overlay(%u)\n", 
				vis_xlen, vis_ylen, ow_id);
		return -EINVAL;
	}

	// Special handle for negative (x, y), since HW only 
	// accepts non-negative values
	x = x >= 0 ? x : 0;
	y = y >= 0 ? y : 0;

	// get dma address of specific ow
	dma = qfb_ow_get_dma(dev, bi_id, ow_id);

	if (bi_id == 0) {
		if (ow_id == 0) {                   // bi0 + ow0
			qfb_wreg32(qfb_info->base_ctl,
					LCD_LCDCTRL_OW0BADDR_OFF, dma + adj_bin - bin);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW0SIZE, HPIX, width-1);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW0SIZE, VLINES, vis_ylen-1);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW0COORD, XSTART, x);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW0COORD, YSTART, y);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OWVHSIZE, OW0_VIS_HPIX, vis_xlen-1);
		} else {                            // bi0 + ow1
			qfb_wreg32(qfb_info->base_ctl,
					LCD_LCDCTRL_OW1BADDR_OFF, dma + adj_bin - bin);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW1SIZE, HPIX, width-1);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW1SIZE, VLINES, vis_ylen-1);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW1COORD, XSTART, x);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW1COORD, YSTART, y);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OWVHSIZE, OW1_VIS_HPIX, vis_xlen-1);
		}
	} else {
		if (ow_id == 0) {                   // bi1 + ow0
			qfb_wreg32(qfb_info->base_ctl,
					LCD_LCDCTRL_OW02BADDR_OFF, dma + adj_bin - bin);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW02SIZE, HPIX, width-1);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW02SIZE, VLINES, vis_ylen-1);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW02COORD, XSTART, x);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW02COORD, YSTART, y);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW2VHSIZE, OW02_VIS_HPIX, vis_xlen-1);
		} else {                            // bi1 + ow1
			qfb_wreg32(qfb_info->base_ctl,
					LCD_LCDCTRL_OW12BADDR_OFF, dma + adj_bin - bin);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW12SIZE, HPIX, width-1);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW12SIZE, VLINES, vis_ylen-1);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW12COORD, XSTART, x);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW12COORD, YSTART, y);
			QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_OW2VHSIZE, OW12_VIS_HPIX, vis_xlen-1);
		}
	}

	qfb_ow_flush_mem(dev, bi_id, ow_id);
	return 0;
}

static void qfb_disable_ow(struct device *dev,
		struct qfb_info *qfb_info,
		int ow_id)
{
	uint32_t val;

	val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_OWCSR_OFF); 

	switch (ow_id) {
		case 0:
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW0EN, 0);
			break;
		case 1:
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW1EN, 0);
			break;
		default:
			dev_err(dev, "%s: Err: Unspported OW id(%d)\n", __func__, ow_id);
			return;
	}

	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OWCSR_OFF, val);
}

static void qfb_setup_ow_ctrl(
		struct device *dev, 
		struct qfb_info *qfb_info, 
		struct qfb_bi_node *bi, 
		int ow_id)
{
	uint32_t val;

	/**
	 * Overlay Control
	 */
	val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_OWCSR_OFF); 

	switch (ow_id)
	{
	case 0:
		QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW0PACKED, bi->ow[0].packed);

		if (bi->ow[0].alpha_src == ALPHA_SRC_CONSTANT) {
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW0AVSRC, 0);
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW0ALPHA, bi->ow[0].const_alpha);
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW0ABEN, 1);
		} else if (bi->ow[0].alpha_src == ALPHA_SRC_PER_PIXEL) {
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW0AVSRC, 1);
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW0ABEN, 1);
		} else {
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW0AVSRC, LCD_LCDCTRL_OWCSR__OW0AVSRC__HW_DEFAULT);
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW0ABEN, 0);
		}

		QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW0EN, 1);
		break;
	case 1:
		QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW1PACKED, bi->ow[1].packed);

		if (bi->ow[1].alpha_src == ALPHA_SRC_CONSTANT) {
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW1AVSRC, 0);
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW1ALPHA, bi->ow[1].const_alpha);
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW1ABEN, 1);
		} else if (bi->ow[1].alpha_src == ALPHA_SRC_PER_PIXEL) {
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW1AVSRC, 1);
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW1ABEN, 1);
		} else {
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW1AVSRC, LCD_LCDCTRL_OWCSR__OW1AVSRC__HW_DEFAULT);
			QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW1ABEN, 0);
		}

		QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OW1EN, 1);
		break;
	default:
		dev_err(dev, "%s: Err: Unspported OW id(%d)\n", __func__, ow_id);
		return;
	}

	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OWCSR_OFF, val);
}

static void qfb_setup_fs_ow(struct device *dev, 
		struct qfb_devinfo *q_dev, 
		struct qfb_fs_node *node, 
		int nd_id,
		int ow_id)
{
	dma_addr_t dma_addr;
	struct qfb_fs_tag *tag = &node->__tag;
	struct qfb_fs_ow  *tow = &tag->ows[ow_id];

	int packed = node->bi.ow[ow_id].packed;
	int c_alph = node->bi.ow[ow_id].const_alpha;

	qfb_ptr_t bin = node->bi.ow[ow_id].bin;
	qfb_ptr_t adj_bin;
	int      vis_xlen;
	int      vis_ylen;
	int      width  = node->bi.ow[ow_id].width;
	int      x = node->bi.ow[ow_id].x;
	int      y = node->bi.ow[ow_id].y;
	int      bin_off;

	x = x >= 0 ? x : 0;
	y = y >= 0 ? y : 0;

	qfb_cal_owl_vis(dev, q_dev, &node->bi.ow[ow_id], &vis_xlen, &vis_ylen, &adj_bin);

	if (vis_xlen > 0 && vis_ylen > 0) {
		switch (node->bi.ow[ow_id].alpha_src) {
			case ALPHA_SRC_CONSTANT: 
				QFB_FS_SET_OW(tag->ctrl_attr, ow_id, packed, 1 /*aben*/, 0 /*avsrc*/, 1 /*en*/);
				QFB_FS_CONS_ALAPH(tag->ctrl_attr, ow_id, c_alph);
				break;
			case ALPHA_SRC_PER_PIXEL:
				QFB_FS_SET_OW(tag->ctrl_attr, ow_id, packed, 1 /*aben*/, 1 /*avsrc*/, 1 /*en*/);
				break;
			default:
				QFB_FS_SET_OW(tag->ctrl_attr, ow_id, packed, 0 /*aben*/, 0 /*avsrc*/, 1 /*en*/);
		}

		bin_off = adj_bin - bin;

		dma_addr = qfb_fs_ow_get_dma(dev, nd_id, ow_id);
		tow->ow_addr = dma_addr + bin_off;
		qfb_fs_ow_flush_mem(dev, nd_id, ow_id);

		tow->ow_size = ((vis_ylen-1) << 16) | (width-1);
		tow->ow_pos  = (y << 16) | x;

		QFB_FS_VHSIZE(tag->ow_vhsize, ow_id, vis_xlen);
	} else {
		// invalid visible xlen or ylen, just disable overlay
		QFB_FS_EN_OW(tag->ctrl_attr, ow_id, 0);
	}
}

static void qfb_set_fs_loop(struct qfb_devinfo_mm *info_mm, struct qfb_fs_seq *fs) 
{
	struct qfb_fs_tag *lst_tag = &fs->nodes[fs->node_cnt-1].__tag;
	struct qfb_fs_tag *fst_tag = &fs->nodes[0].__tag;

	if (fs->end_act == FS_END_ACT_LOOP) {
		lst_tag->next_tag_addr = qfb_virt_to_dma_addr(info_mm->devinfo,
				fst_tag, info_mm->dma_addr);
		QFB_FS_STNA_V(lst_tag->ctrl_attr, 1);
	} else {
		lst_tag->next_tag_addr = (dma_addr_t) 0;
		QFB_FS_STNA_V(lst_tag->ctrl_attr, 0);
	}
}

static void qfb_create_fs_tag(
		struct device *dev, 
		struct qfb_devinfo_mm *info_mm, 
		int idx, 
		dma_addr_t n_tag)
{
	struct qfb_devinfo *q_dev = info_mm->devinfo;
	struct qfb_fs_seq  *fs    = &q_dev->fs;
	struct qfb_fs_node *node  = &fs->nodes[idx];
	struct qfb_fs_tag  *tag   = &node->__tag;

	int ow_id;
	int bi_width  = node->bi_prop.width;
	int bi_height = node->bi_prop.height;
	int bi_packed = node->bi_prop.packed;

	/**
	 * next tag info
	 */
	if (n_tag) {
		tag->next_tag_addr = (uint32_t)n_tag;
		QFB_FS_STNA_V(tag->ctrl_attr, 1);
	} else {
		qfb_set_fs_loop(info_mm, fs);
	}

	/**
	 * qfb_setup BI
	 */
	tag->bi_addr = qfb_fs_bi_get_dma(dev, idx);
	qfb_fs_bi_flush_mem(dev, idx);

	QFB_FS_BI_ATTR(tag->bi_attrs, bi_width, bi_height, bi_packed);

	/**
	 * qfb_setup tag's ctrl_attr & ow
	 */
	for (ow_id = 0; ow_id < MAX_OW; ow_id++) {
		if (QFB_CHK_OWEN(&node->bi, ow_id)) {
			qfb_setup_fs_ow(dev, q_dev, node, idx, ow_id);
		}

		// Setup overlay window priority
		QFB_FS_OW_PRIO(tag->ctrl_attr, node->bi.ow_prio);
	}

	if (node->en_snm1) {
		QFB_FS_SNM(tag->ctrl_attr, 1, 1);
	}

	if (node->en_snm2) {
		QFB_FS_SNM(tag->ctrl_attr, 2, 1);
	}
}

static void qfb_tag_show(struct device *dev, struct qfb_fs_tag *tag)
{
	int ow_id;

	dev_dbg(dev, "%s: FS Tag (%px):\n", __func__, (qfb_ptr_t)tag);
	dev_dbg(dev, "%s: >> next_tag_addr = %px\n", __func__, (qfb_ptr_t)(uintptr_t)tag->next_tag_addr);
	dev_dbg(dev, "%s: >> ctrl_attr = 0x%08x\n", __func__, tag->ctrl_attr);
	dev_dbg(dev, "%s: >> bi_addr = %px\n", __func__, (qfb_ptr_t)(uintptr_t)tag->bi_addr);
	dev_dbg(dev, "%s: >> bi_attrs = 0x%08x\n", __func__, tag->bi_attrs);

	for (ow_id = 0; ow_id < MAX_OW; ow_id++) {
		dev_dbg(dev, "%s: >> ow%d_addr = %px\n", __func__, ow_id, (qfb_ptr_t)(uintptr_t)tag->ows[ow_id].ow_addr);
		dev_dbg(dev, "%s: >> ow%d_size = 0x%08x\n", __func__, ow_id, tag->ows[ow_id].ow_size);
		dev_dbg(dev, "%s: >> ow%d_pos = 0x%08x\n", __func__, ow_id, tag->ows[ow_id].ow_pos);
	}
}

static void qfb_setup_fs(struct device *dev, struct qfb_info *qfb_info)
{
	int i;
	struct qfb_devinfo_mm *info_mm = qfb_info->info_mm;
	struct qfb_devinfo *q_dev = info_mm->devinfo;
	struct qfb_fs_node *fst_n = &q_dev->fs.nodes[0];

	for (i=0; i < q_dev->fs.node_cnt; i++) {
		struct qfb_fs_node *cur_n = &q_dev->fs.nodes[i];
		dma_addr_t n_tag = 0;

		if (i < q_dev->fs.node_cnt - 1) {
			n_tag = qfb_virt_to_dma_addr(q_dev, 
					&q_dev->fs.nodes[i+1].__tag, info_mm->dma_addr);
		}

		qfb_create_fs_tag(dev, info_mm, i, n_tag);
		qfb_tag_show(dev, &cur_n->__tag);
	}

	qfb_wreg32(qfb_info->base_ctl, 
			LCD_LCDCTRL_FSBADDR_OFF,
			qfb_virt_to_dma_addr(q_dev, &fst_n->__tag, info_mm->dma_addr));

	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_FSBADDR_OFF);
}

static void qfb_setup_bi_ow_internal(
		struct device *dev, 
		struct qfb_info *qfb_info, 
		uint32_t bi_id)
{
	uint32_t val;
	uint32_t ow_id;
	struct qfb_devinfo *q_dev = qfb_info->info_mm->devinfo;
	struct qfb_bi_node *bi = &q_dev->bi[bi_id];

	for (ow_id = 0; ow_id < MAX_OW; ow_id++) {
		if (QFB_CHK_OWEN(bi, ow_id)) {
			if (qfb_setup_ow_attr(dev, qfb_info, bi_id, ow_id) == 0) {
				qfb_setup_ow_ctrl(dev, qfb_info, bi, ow_id);
			} else {
				// disable the related overlay if ow attributes
				// are not appropriated
				qfb_disable_ow(dev, qfb_info, ow_id);
			}
		}
	}

	// Setup overlay window priority
	val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_OWCSR_OFF); 
	QFB_CLRSET(val, LCD_LCDCTRL_OWCSR, OWPRIO, bi->ow_prio);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OWCSR_OFF, val);
}


/**
 * Setup BI's overlay window
 * 
 * Note that, in double buffer mode, we need to know which BI's
 * OW nodes should be used to qfb_setup overlay window.
 */
static void qfb_setup_bi_ow(struct device *dev, struct qfb_info *qfb_info)
{
	uint32_t bi_id;
	struct qfb_devinfo *q_dev = qfb_info->info_mm->devinfo;

	if (q_dev->en_db) {
		bi_id = do_qfb_cur_dbid(qfb_info);
		qfb_setup_bi_ow_internal(dev, qfb_info, bi_id);
	} else {
		qfb_setup_bi_ow_internal(dev, qfb_info, 0);
	}

	// Dump ow related registers for debugging
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_OW0BADDR_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_OW0SIZE_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_OW0COORD_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_OW1BADDR_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_OW1SIZE_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_OW1COORD_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_OWVHSIZE_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_OWCSR_OFF);
}

static void qfb_data_en(
		struct device *dev, 
		struct qfb_info *qfb_info, 
		int en)
{
	uint32_t val;

	switch (qfb_info->panel.panel_type) {
		case QFB_PANEL_SRGB:
			val = qfb_rreg32(qfb_info->base_gpf, LCD_LVDSVX1_CSR1_OFF);
			val = (val & LCD_LVDSVX1_CSR1__SRGB_LCDDATA_EN__INV_MASK) | (en << LCD_LVDSVX1_CSR1__SRGB_LCDDATA_EN__SHIFT);
			qfb_wreg32(qfb_info->base_gpf, LCD_LVDSVX1_CSR1_OFF, val);
			break;
		case QFB_PANEL_LVDS:
			val = qfb_rreg32(qfb_info->base_gpf, LCD_LVDSVX1_CSR2_OFF);
			val = (val & LCD_LVDSVX1_CSR2__LVDSPHY_LCDDATA_EN__INV_MASK) | (en << LCD_LVDSVX1_CSR2__LVDSPHY_LCDDATA_EN__SHIFT);
			qfb_wreg32(qfb_info->base_gpf, LCD_LVDSVX1_CSR2_OFF, val);
			break;
		case QFB_PANEL_VX1:
			val = qfb_rreg32(qfb_info->base_gpf, LCD_LVDSVX1_CSR2_OFF);
			val = (val & LCD_LVDSVX1_CSR2__LVDSPHY_LCDDATA_EN__INV_MASK) | (en << LCD_LVDSVX1_CSR2__LVDSPHY_LCDDATA_EN__SHIFT);
			qfb_wreg32(qfb_info->base_gpf, LCD_LVDSVX1_CSR2_OFF, val);
			break;
		default:
			dev_err(dev, "%s: invalid panel type specified(%d)\n", __func__, qfb_info->panel.panel_type);
			break;
	}
}

void qfb_start(struct device *dev, struct qfb_info *qfb_info)
{
	uint32_t val;

	/**
	  * Setup converter
	  */
	switch (qfb_info->panel.panel_type) {
		case QFB_PANEL_SRGB:
			qfb_setup_srgb(dev, qfb_info);
			break;
		case QFB_PANEL_LVDS:
			qfb_setup_lvds(dev, qfb_info);
			break;
		case QFB_PANEL_VX1:
			qfb_setup_vx1(dev, qfb_info);
			break;
	}

	val = qfb_rreg32(qfb_info->base_gpf, LCD_LVDSVX1_CSR1_OFF);
	val = (val & LCD_LVDSVX1_CSR1__TX_MODE__INV_MASK) | 
		  (qfb_info->panel.tx_mode << LCD_LVDSVX1_CSR1__TX_MODE__SHIFT);
	qfb_wreg32(qfb_info->base_gpf, LCD_LVDSVX1_CSR1_OFF, val);

	/**
	 * Start the controller
	 */
	val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
	val = (val & LCD_LCDCTRL_CSR__ENABLE__INV_MASK) | (1 << LCD_LCDCTRL_CSR__ENABLE__SHIFT);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF, val);

	/**
	  * Enable Load Data
	  */
	qfb_data_en(dev, qfb_info, 1);
}

static void qfb_set_divider(struct device *dev, struct qfb_info *qfb_info)
{
	uint32_t val;
	uint32_t div = 0x0;

	switch (qfb_info->panel.panel_type)
	{
		case QFB_PANEL_SRGB: div = 0x02010203; break;
		case QFB_PANEL_LVDS: div = 0x00000001; break;
		case QFB_PANEL_VX1:
			switch (qfb_info->panel.vx1_mode) {
				case QFB_VX1_3BYTE: div = 0x02010203; break;
				case QFB_VX1_4BYTE: div = 0x00000304; break;
				case QFB_VX1_5BYTE: div = 0x03010305; break;
				case QFB_VX1_3BYTE_DUP:
				    div = 0x02010203;
					// Enable duplication mode
					val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
					val = (val & LCD_LCDCTRL_CSR__PIXDUP_EN__INV_MASK) |
						  (1 << LCD_LCDCTRL_CSR__PIXDUP_EN__SHIFT);
					qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF, val);
				    break;
				default:
					dev_err(dev, "%s: individ vx1 mode(%d)\n", 
							__func__, qfb_info->panel.vx1_mode); 
			}
			break;
		default: 
			dev_err(dev, "%s: invalid panel type\n", __func__); 
			break;
	}

	qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_OFF, div);

	// FIXME: This register only workable for enumation environment
	// Need to decide if we need to remove it or not
	qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x00000200);
	switch(qfb_info->info_mm->devinfo->pclk_speed) {
		case PCLK_LOW: 
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x0000050c);
			break;
		case PCLK_HIGH: 
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x00000001);
			break;
		case PCLK_DIV_3:
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x02010203);
			break;
		case PCLK_DIV_4:
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x00000304);
			break;
		case PCLK_DIV_4_5:
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x07050309);
			break;
		case PCLK_DIV_5:
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x03010305);
			break;
		case PCLK_DIV_6:
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x00000406);
			break;
		case PCLK_DIV_7_5:
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x0c08050f);
			break;
		case PCLK_DIV_12:
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x0000050c);
			break;
		case PCLK_DIV_10:
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x0000050a);
			break;
		case PCLK_DIV_14:
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x0000070e);
			break;
		default: 
			qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF, 0x00000202);
	}
}

void qfb_set_timing(struct device *dev, struct qfb_info *qfb_info)
{
	int is_en;

	// timing parameters in hardware are one less than the current qfb_setup. For example, 
	// if we give 1 for specific width, hardware will consider it as two inside its logic.
	uint32_t hort_timing = (qfb_info->panel.timing.hfp_width-1) << 16 | 
						   (qfb_info->panel.timing.hbp_width-1) << 8  | 
						   (qfb_info->panel.timing.hsy_width-1);

	uint32_t vert_timing = (qfb_info->panel.timing.vfp_width-1) << 16 | 
						   (qfb_info->panel.timing.vbp_width-1) << 8  | 
						   (qfb_info->panel.timing.vsy_width-1);

	// If qfb is running, we need to stop it first
	is_en = qfb_is_en(qfb_info);
	if (is_en) {
		qfb_shutdown(dev, qfb_info);
	}

	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_HORT_OFF, hort_timing);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_VERT_OFF, vert_timing);

	// If qfb was running, we need to start it now
	if (is_en) {
		qfb_start(dev, qfb_info);
	}
}

void qfb_setup(struct device *dev, struct qfb_info *qfb_info)
{
	uint32_t val = 0;
	struct qfb_devinfo_mm *info_mm = qfb_info->info_mm;
	struct qfb_devinfo *q_dev = info_mm->devinfo;

	int width     = q_dev->bi_prop.width;
	int height    = q_dev->bi_prop.height;
	int packed    = q_dev->bi_prop.packed;
	int bpp       = q_dev->bpp;

	/*
	 * Default values
	 */
	//const uint32_t tx_srgb_clk_pol   = 1;
	//const uint32_t pclk_pol          = 1;

	const uint32_t pll_src = 0x00000001;

	/*
	 * FIXME: will be removed after B0
	 * 
	 * Adjust DMA packet size and FFPW
	 * Note: Wrong packet size would affect performance. Small FFPW will
	 * cause timing issue on low PCLK freq
	 */
	val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_DBG1_OFF);
	val = (val & LCD_LCDCTRL_DBG1__PKTSIZE__INV_MASK & LCD_LCDCTRL_DBG1__FFPW__INV_MASK) |
		  (0x2 << LCD_LCDCTRL_DBG1__PKTSIZE__SHIFT)   | // adjust to 128bytes
		  (0x3f << LCD_LCDCTRL_DBG1__FFPW__SHIFT);      // Jay: default value is too small for 20MHz PCLK
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_DBG1_OFF, val);

	/**
	 * Initialize LCD controller
	 */
	val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
	val = (val & LCD_LCDCTRL_CSR__HVALIGN__INV_MASK) |
		  (qfb_info->panel.hv_align << LCD_LCDCTRL_CSR__HVALIGN__SHIFT); // HVALIGN
	val = (val & LCD_LCDCTRL_CSR__BPP__INV_MASK) | 
		  (qfb_bpp_to_regval(bpp) << LCD_LCDCTRL_CSR__BPP__SHIFT);  // BPP
	val = (val & LCD_LCDCTRL_CSR__PACKED__INV_MASK) |
		  (packed << LCD_LCDCTRL_CSR__PACKED__SHIFT);     // PACK
	val = (val & LCD_LCDCTRL_CSR__FRMT__INV_MASK) |
		  (q_dev->out_port << LCD_LCDCTRL_CSR__FRMT__SHIFT);     // FRMT
	//val = (val & LCD_LCDCTRL_CSR__PCLKPOL__INV_MASK) | 
	//	  (pclk_pol << LCD_LCDCTRL_CSR__PCLKPOL__SHIFT); // pclk polarity

	// XXX: the `en_pd` is only useful for test purpose, since duplication mode
	// is set by devicetree setting. See devicetree's `data-mapping` for Vx1
	val = (val & LCD_LCDCTRL_CSR__PIXDUP_EN__INV_MASK) |
		  (q_dev->en_pd << LCD_LCDCTRL_CSR__PIXDUP_EN__SHIFT);   // Pixel duplication

	if (bpp == 15 || q_dev->plt.bpp == 15)
	{
		val = (val & LCD_LCDCTRL_CSR__FRMT_555__INV_MASK) |
			  (1 << LCD_LCDCTRL_CSR__FRMT_555__SHIFT);          // FRMT_555
	}

	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF, val);

	/**
	  * Setup LCD horizontal & vertical timing
	  */
	qfb_set_timing(dev, qfb_info);

	/**
	  * Setup BI properties
	  */
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_BIBADDR_OFF, qfb_bi_get_dma(dev, 0));
	qfb_bi_flush_mem(dev, 0);

	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_BISIZE_OFF, ((height-1) << 16) | (width-1));

	if (q_dev->en_db) {
		val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
		QFB_CLRSET(val, LCD_LCDCTRL_CSR, DBEN, 1);

		qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF, val);
		qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_BI2BADDR_OFF, qfb_bi_get_dma(dev, 1));
		qfb_bi_flush_mem(dev, 1);
	} else {
		val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
		qfb_wreg32(qfb_info->base_ctl,
				LCD_LCDCTRL_CSR_OFF, val & LCD_LCDCTRL_CSR__DBEN__INV_MASK);
	}

	if (q_dev->en_fs) {
		qfb_setup_fs(dev, qfb_info);

		val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
		QFB_CLRSET(val, LCD_LCDCTRL_CSR, FS_EN, 1);
		qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF, val);
	} else {
		val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
		qfb_wreg32(qfb_info->base_ctl,
				LCD_LCDCTRL_CSR_OFF, val & LCD_LCDCTRL_CSR__FS_EN__INV_MASK);
	}

	/**
	 * Cache coherence
	 */
	if (q_dev->en_cc) {
		val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
		QFB_CLRSET(val, LCD_LCDCTRL_CSR, RCOH_EN, 1);
		qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF, val);
	} else {
		val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
		qfb_wreg32(qfb_info->base_ctl,
				LCD_LCDCTRL_CSR_OFF, val & LCD_LCDCTRL_CSR__RCOH_EN__INV_MASK);
	}

	/**
	 * Setup palette
	 */
	if (bpp <= 8)
	{
		int i;

		// integrity check: calculate total number of plt entries,
		// every 2 bytes contains one plt entry and the last address
		// contains 2 entries.
		int total_slots   = (LCD_LCDCTRL_PALT_END-LCD_LCDCTRL_PALT) / 2 + 2;
		int total_entries = q_dev->plt.size / 2;

		if (total_slots != total_entries)
		{
			dev_err(dev, "%s: PLT Mismatch Err: slots(%d) vs entries(%d)\n", 
					__func__, total_slots, total_entries);
		}

		dev_dbg(dev, "%s: Start to fill palette\n", __func__);
		for (i = 0; i < q_dev->plt.size; i+=4)
		{
			val = (q_dev->plt.bin[i+3] << 24) | (q_dev->plt.bin[i+2] << 16) |
				  (q_dev->plt.bin[i+1] << 8)  | q_dev->plt.bin[i];
			qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_PALT_OFF + i, val);
		}
	}

	/**
	 * Setup overlay
	 */
	qfb_setup_bi_ow(dev, qfb_info);

	/**
	  * Setup interrupt
	  */
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_INT_MASK_OFF, 0x00000000);            // Disable all interrupts
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_INT_CLR_OFF, 0xffffffff);             // Clear all interrupts
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_INT_MASK_OFF, QFB_INTR_MASKS);        // Enable specific interrupts

	/**
	  * Setup external clock & pixel clock
	  */
	qfb_wreg32(qfb_info->base_gpf, LCD_PCLK_CST_OFF, pll_src); // Select VX1 PLL source

	// Reset DIVCTRL and qfb_setup pos pwl start (later, main divider will be set)
	qfb_set_divider(dev, qfb_info);

	/**
	  * Release PCLK reset
	  */
	val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
	val = (val & LCD_LCDCTRL_CSR__PCLKRESETN__INV_MASK) | (1 << LCD_LCDCTRL_CSR__PCLKRESETN__SHIFT);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF, val);

	// make sure all the memory are visible to lcdctrl
	dma_sync_single_for_device(dev->parent, info_mm->dma_addr, sizeof(*info_mm->devinfo), DMA_TO_DEVICE);

	qfb_start(dev, qfb_info);
	
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_ID_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_gpf, LCD_LVDSVX1_CSR1_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_gpf, LCD_LVDSVX1_CSR2_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_BIBADDR_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_BISIZE_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_BI2BADDR_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_HORT_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_VERT_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_INT_MASK_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_gpf, LCD_PCLK_CST_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_gpf, LCD_PCLK_DIVCTRL_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_gpf, LCD_PCLK_DIVCTRL_TEST_OFF);
	QFB_REG_SHOW(dev, qfb_info->base_ctl, LCD_LCDCTRL_DBG1_OFF);
}

void qfb_reset_hw(struct device *dev, struct qfb_info *qfb_info)
{
	uint32_t val = 0;

	/**
	 * Stop LCD Controller & back to defaults
	 */
	val = QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__HVALIGN)      |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__PIXDUP_MODE)  |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__PIXDUP_EN)    |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__FRMT_555)     |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__FS_EN)        |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__RCOH_EN)      |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__DPWR)         |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__PCLKRESETN)   |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__DBSTAT)       |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__DBSEL)        |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__DBEN)         |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__FRMT)         |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__PCLKPOL)      |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__DEPOL)        |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__VSPOL)        |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__HSPOL)        |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__PACKED)       |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__ENDIAN)       |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__BPP)          |
          QFB_HW_DEFAULT(LCD_LCDCTRL_CSR__ENABLE);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF, val);

	/**
	 * Stop conveter & back to defaults
	 */
	val = QFB_HW_DEFAULT(LCD_LVDSVX1_CSR1__VX1TOLVDS_CONVEN)    |
		  QFB_HW_DEFAULT(LCD_LVDSVX1_CSR1__HTOL_MODE)           |
		  QFB_HW_DEFAULT(LCD_LVDSVX1_CSR1__LVCT4)               |
		  QFB_HW_DEFAULT(LCD_LVDSVX1_CSR1__LVDSPHY_PCLK)        |
		  QFB_HW_DEFAULT(LCD_LVDSVX1_CSR1__TX_SER_ORDER)        |
		  QFB_HW_DEFAULT(LCD_LVDSVX1_CSR1__TX_SRGB_ALT)         |
		  QFB_HW_DEFAULT(LCD_LVDSVX1_CSR1__TX_SRGB_G_ORDER)     |
		  QFB_HW_DEFAULT(LCD_LVDSVX1_CSR1__TX_SRGB_CLK_POL)     |
		  QFB_HW_DEFAULT(LCD_LVDSVX1_CSR1__SRGB_LCDDATA_EN)     |
		  QFB_HW_DEFAULT(LCD_LVDSVX1_CSR1__TX_MODE);
	qfb_wreg32(qfb_info->base_gpf, LCD_LVDSVX1_CSR1_OFF, val);

	/**
	 * Interrupt back to default
	 */
	val = QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__VSFDEL)          |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__OW0INFIFOU)      |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__OW1INFIFOU)      |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__RESPE)           |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__DTC)             |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__FSNM1)           |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__FSNM2)           |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__DIS_DONE)        |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__VSVFP)           |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__VSAF)            |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__VSVBP)           |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__VSVS)            |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__FIFOU)           |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__OUTFIFOU)        |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_INT_MASK__BIINFIFOU);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_INT_MASK_OFF, val);

	/**
	 * BI
	 */
	val = QFB_HW_DEFAULT(LCD_LCDCTRL_BIBADDR__ADDR);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_BIBADDR_OFF, val);

	val = QFB_HW_DEFAULT(LCD_LCDCTRL_BI2BADDR__ADDR);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_BI2BADDR_OFF, val);

	val = QFB_HW_DEFAULT(LCD_LCDCTRL_BISIZE__HPIX) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_BISIZE__VLINES);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_BISIZE_OFF, val);

	/**
	 * HORT & VERT
	 */
	val = QFB_HW_DEFAULT(LCD_LCDCTRL_HORT__HFPORCH) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_HORT__HBPORCH) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_HORT__HSYNC);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_HORT_OFF, val);

	val = QFB_HW_DEFAULT(LCD_LCDCTRL_VERT__VFPORCH) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_VERT__VBPORCH) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_VERT__VSYNC);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_VERT_OFF, val);

	/**
	 * OVERLAY CTRL
	 */
	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OWCSR__OWPRIO)     |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OWCSR__OW0ALPHA)   |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OWCSR__OW1ALPHA)   |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OWCSR__OW1AVSRC)   |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OWCSR__OW1ABEN)    |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OWCSR__OW1PACKED)  |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OWCSR__OW1EN)      |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OWCSR__OW0AVSRC)   |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OWCSR__OW0ABEN)    |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OWCSR__OW0PACKED)  |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OWCSR__OW0EN);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OWCSR_OFF, val);

	/**
	 * OVERLAY OW0
	 */
	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW0BADDR__ADDR);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW0BADDR_OFF, val);

	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW0SIZE__VLINES) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OW0SIZE__HPIX);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW0SIZE_OFF, val);

	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW0COORD__YSTART) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OW0COORD__XSTART);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW0COORD_OFF, val);

	/**
	 * OVERLAY OW1
	 */
	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW1BADDR__ADDR);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW1BADDR_OFF, val);

	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW1SIZE__VLINES) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OW1SIZE__HPIX);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW1SIZE_OFF, val);

	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW1COORD__YSTART) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OW1COORD__XSTART);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW1COORD_OFF, val);

	/**
	 * OVERLAY OW0 for BI2
	 */
	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW02BADDR__ADDR);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW02BADDR_OFF, val);

	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW02SIZE__VLINES) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OW02SIZE__HPIX);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW02SIZE_OFF, val);

	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW02COORD__YSTART) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OW02COORD__XSTART);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW02COORD_OFF, val);

	/**
	 * OVERLAY OW1 for BI2
	 */
	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW12BADDR__ADDR);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW12BADDR_OFF, val);

	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW12SIZE__VLINES) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OW12SIZE__HPIX);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW12SIZE_OFF, val);

	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW12COORD__YSTART) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OW12COORD__XSTART);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW12COORD_OFF, val);

	/**
	 * OVERLAY Visibility
	 */
	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OWVHSIZE__OW0_VIS_HPIX) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OWVHSIZE__OW1_VIS_HPIX);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OWVHSIZE_OFF, val);

	/**
	 * OVERLAY Visibility for BI2
	 */
	val = QFB_HW_DEFAULT(LCD_LCDCTRL_OW2VHSIZE__OW02_VIS_HPIX) |
		  QFB_HW_DEFAULT(LCD_LCDCTRL_OW2VHSIZE__OW12_VIS_HPIX); 
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_OW2VHSIZE_OFF, val);
}

void qfb_shutdown(struct device *dev, struct qfb_info *qfb_info)
{
	uint32_t val;
	int wait_cnt = 0;

	if (!qfb_is_en(qfb_info)) {
		dev_dbg(dev, "%s: lcdctrl not enabled, skip waiting\n", __func__);
		return;
	}

	// setup interrupt mask to monitor DIS_DONE
	qfb_wreg32(qfb_info->base_ctl, 
			LCD_LCDCTRL_INT_MASK_OFF, 
			LCD_LCDCTRL_INT_MASK__DIS_DONE__MASK);

	qfb_data_en(dev, qfb_info, 0);

	val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
	qfb_wreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF, 
			val & LCD_LCDCTRL_CSR__ENABLE__INV_MASK);

	// wait for the DIS_DONE event
	while (!(qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_INT_VECT_OFF) & 
				LCD_LCDCTRL_INT_VECT__DIS_DONE__MASK)) {
		wait_cnt += 1;
		if ((wait_cnt % 100) == 0) {
			dev_dbg(dev, "%s: waiting for shutdown (%d times)\n", __func__, wait_cnt);
		}

		if (wait_cnt > QFB_SHUTDOWN_RETRY_CNT) {
			dev_err(dev, 
					"%s: timeout waiting shutdown. May have trouble when relaunch\n",
					__func__);
			break;
		}
	}
	dev_dbg(dev, "%s: shutdown procedure leave\n", __func__);
}

uint32_t qfb_cur_dbid(struct device *dev, struct qfb_info *qfb_info)
{
	return do_qfb_cur_dbid(qfb_info);
}

void qfb_set_dbid(struct device *dev, struct qfb_info *qfb_info, uint32_t id)
{
	uint32_t val;

	qfb_setup_bi_ow_internal(dev, qfb_info, id);

	val = qfb_rreg32(qfb_info->base_ctl, LCD_LCDCTRL_CSR_OFF);
	QFB_CLRSET_WREG32(qfb_info->base_ctl, LCD_LCDCTRL_CSR, DBSEL, id);
}
