/* Kernel driver for DB9000 LCD controller found on some Quatro processors
 *
 *  Quasar LCD controller kernel driver
 * 
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
// =========================================================
//
//  $DateTime: 2022/03/31 10:38:25 $
//  $Change: 59512 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/fb.h>
#include <linux/wait.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/fb.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>
#include "evb.h"


#define IOREMAP(iB,iL) gmVOD_Remap.iSource=iB;gmVOD_Remap.piDestination=ioremap(iB,iL)
#define IOUNMAP() iounmap(gmVOD_Remap.piDestination)
#define WRREG_UINT32(iA,iV)	writel(iV,gmVOD_Remap.piDestination+(iA-gmVOD_Remap.iSource))
#define RDREG_UINT32(iA) readl(gmVOD_Remap.piDestination+(iA-gmVOD_Remap.iSource))

struct sVOD_Meta
{
	dma_addr_t iAddress;
	volatile u8 *piFB;
	unsigned long iSize;
	struct device *pDev;
};
static struct
{
	u8 __iomem *piDestination;
	unsigned long iSource;
}gmVOD_Remap;

static int rVOD_CV(struct fb_var_screeninfo *var, struct fb_info *info)
{
	return 0;
}
static int rVOD_SP(struct fb_info *info)
{
	return 0;
}
static int rVOD_MMAP(struct fb_info *pmFBI, struct vm_area_struct* vma)
{
	struct sVOD_Meta *pmMeta;
	int iSize,iRet;
	pmMeta=(struct sVOD_Meta*)pmFBI->par;
	iSize=vma->vm_end-vma->vm_start;
	iRet=dma_mmap_coherent(pmMeta->pDev,vma,(void*)pmMeta->piFB,pmMeta->iAddress,iSize);
	return 0;
}
static struct fb_ops gmVOD_Operations=
{
	.owner = THIS_MODULE,
	.fb_check_var = rVOD_CV,
	.fb_set_par = rVOD_SP,
	.fb_mmap = rVOD_MMAP
};
static int __init quasar_db9k_probe(struct platform_device *pdev)
{
	struct fb_info *pmFBI;
	struct sVOD_Meta *pmMeta;
	unsigned long iValue;

	//[OS Related]
	pmFBI=framebuffer_alloc(sizeof(struct sVOD_Meta),&pdev->dev);
	pmMeta=pmFBI->par;
	memset(pmMeta,0,sizeof(struct sVOD_Meta));
	pmMeta->iSize=roundup(1024*600*3,PAGE_SIZE);
	pmMeta->piFB=dma_alloc_coherent(&pdev->dev, pmMeta->iSize, &pmMeta->iAddress, GFP_KERNEL);
	memset((void *)pmMeta->piFB,0xFF,1024*600*3);
	platform_set_drvdata(pdev, pmFBI);
	device_init_wakeup(&pdev->dev,1);
	pmMeta->pDev=&pdev->dev;
	
#if V_LCD_PHY == V_DEF_LVDS_PHY
#include "lcd_lvds.c"
#elif V_LCD_PHY == V_DEF_VX1_PHY 
#include "lcd_vx1.c"
#else
#error Non-supported LCD PHY setup
#endif

	//[FB Setup]
	//	General
	pmFBI->pseudo_palette=NULL;
	pmFBI->screen_base=(void __iomem *)pmMeta->piFB;
	pmFBI->fbops=&gmVOD_Operations;
	pmFBI->flags=FBINFO_FLAG_DEFAULT | FBINFO_READS_FAST;
	//	Fix
	pmFBI->fix.smem_start=(unsigned long)pmMeta->iAddress;
	pmFBI->fix.smem_len=PAGE_ALIGN(pmMeta->iSize);
	strlcpy(pmFBI->fix.id,"db9k-quasar",sizeof(pmFBI->fix.id));
	pmFBI->fix.type=FB_TYPE_PACKED_PIXELS;
	pmFBI->fix.type_aux=0;
	pmFBI->fix.xpanstep=0;
	pmFBI->fix.ypanstep=0;
	pmFBI->fix.accel=FB_ACCEL_NONE;
	pmFBI->fix.visual=FB_VISUAL_TRUECOLOR;
	pmFBI->fix.line_length = 3*V_LCD_WIDTH;
	//	Var
	pmFBI->var.nonstd=0;
	pmFBI->var.activate=FB_ACTIVATE_NOW;
	pmFBI->var.height=1024;
	pmFBI->var.width=600;
	pmFBI->var.accel_flags=0;
	pmFBI->var.vmode=FB_VMODE_NONINTERLACED;
	pmFBI->var.sync=FB_SYNC_COMP_HIGH_ACT;
	pmFBI->var.xres=V_LCD_WIDTH;
	pmFBI->var.xres_virtual=V_LCD_WIDTH;
	pmFBI->var.yres=V_LCD_HEIGHT;
	pmFBI->var.yres_virtual=V_LCD_HEIGHT;
	pmFBI->var.bits_per_pixel=24;
	pmFBI->var.grayscale=0;
	pmFBI->var.blue.offset = 0;
	pmFBI->var.blue.length = 8;
	pmFBI->var.blue.msb_right = 0;
	pmFBI->var.green.offset = 8;
	pmFBI->var.green.length = 8;
	pmFBI->var.green.msb_right = 0;
	pmFBI->var.red.offset = 24;
	pmFBI->var.red.length = 8;
	pmFBI->var.red.msb_right = 0;
	pmFBI->var.transp.offset = 0;
	pmFBI->var.transp.length = 0;
	pmFBI->var.transp.msb_right = 0;
	register_framebuffer(pmFBI);
	return 0;
}

static int __exit quasar_db9k_remove(struct platform_device *pdev)
{
	struct fb_info *pmFBI;
	struct sVOD_Meta *pmMeta;

	pmFBI = platform_get_drvdata(pdev);
	if (!pmFBI)
	{
		return -EINVAL;
	}

	pmMeta = pmFBI->par;
	if (!pmMeta)
	{
		return -EINVAL;
	}

	dma_free_coherent(&pdev->dev, pmMeta->iSize, (void *)pmMeta->piFB, pmMeta->iAddress);
	return 0;
}

MODULE_ALIAS("platform:quasar-db9000");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-db9000" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver quasar_db9k_driver_ops = {
	.probe		= quasar_db9k_probe,
	.remove		= quasar_db9k_remove,
	.driver		= {
		.name	= "quasar-db9000",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init quasar_db9k_init(void)
{
	int ret;
	ret = platform_driver_register(&quasar_db9k_driver_ops);
	return ret;
}
module_init(quasar_db9k_init);

static void __exit quasar_db9k_exit(void)
{
	platform_driver_unregister(&quasar_db9k_driver_ops);
}
module_exit(quasar_db9k_exit);

MODULE_DESCRIPTION(" Quasar db9000 driver");
MODULE_LICENSE("Dual BSD/GPL");

