/*
 * A driver for CDIF found on QBit processors
 *
 * Quasar CDIF kernel driver
 *
 * Copyright (c) 2015, The Linux Foundation.
 * All rights reserved.
 *
 * Redistribution and use
 * in source and binary forms, with or without modification,
 * are permitted (subject to the limitations in the disclaimer
 * below) provided that the following conditions are met :
 *   *Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *   *Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and
 *    the following disclaimer
 *    in the documentation and/or other materials provided
 *    with the distribution.
 *
 *  NO EXPRESS OR IMPLIED LICENSES TO ANY PARTYS PATENT
 *  RIGHTS ARE GRANTED BY THIS LICENSE.
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS
 *  AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING,
 *  BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 *  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY,
 *  OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 *  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
 *  OR PROFITS;
 *  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 *  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 *  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 *  OF SUCH DAMAGE
 *
 */
 // =========================================================
//
//  $DateTime: 2022/04/08 12:12:26 $
//  $Change: 59748 $
//
// =========================================================
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#define QCDIF_DEVNAME	    "qcdif"
#define QBCD_BUSCFG_OFF     0x00000024
#define QBCD_IRQEN_OFF      0x00000028
#define QBCD_IRQENSET_OFF   0x00000030
#define QBCD_IRQSTAT_OFF    0x00000034

/* ---------------------------------------------------------------------
 * CDIF driver
 */
#define QCDIF_MAJOR		216

#define MAX_MAILBOX 64

struct cdif_quasar {
	int			ref;
	int			minor;
	struct cdev		cdev;
	volatile u8	__iomem	*regs;
	unsigned long		ioreg_start;
	unsigned long		ioreg_end;
	int			irq;

	wait_queue_head_t	mbq;	/* wait-queue for dma-doneinterrupt events */
	struct q_doneinfo	doneinfo[MAX_MAILBOX];
	int			di_head, di_tail;
	spinlock_t		lock;
	// bus config
	unsigned int		bus_cfg;
	// irq enabled
	unsigned int		irq_en;
};

static int qcdif_open(struct inode* inode, struct file* filp)
{
	struct cdif_quasar *cdif;
	
	cdif = container_of(inode->i_cdev, struct cdif_quasar, cdev);
	if(cdif->ref > 0)
		return -EBUSY;
	cdif->ref++;
	cdif->minor = iminor(inode);
	filp->private_data = cdif;
	cdif->di_head = cdif->di_tail = 0;
	
#if defined(Q6300) || defined(Q6600)
	{
		volatile u8 __iomem *reg;

		// CLK enabled
		reg = ioremap(IPM2CLKDISCTRL, 8);
		writel(readl(reg) & IPM2CLKDISCTRL__IPM2_CD__INV_MASK, reg);
		while (readl(reg + 4) & IPM2CLKDISSTAT__IPM2_CD__MASK) {;}
		iounmap(reg);
		// RSTGEN
		reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
		writel(readl(reg) & RSTGEN_SWRSTSTATIC3__IPM2_CD__INV_MASK, reg);
		iounmap(reg);
		reg = ioremap(RSTGEN_SWRSTPULSE3, 4);
		writel(RSTGEN_SWRSTPULSE3__IPM2_CD__MASK, reg);
		iounmap(reg);
	}
#endif
	return 0;
}

static int qcdif_release(struct inode* inode, struct file* filp)
{
	struct cdif_quasar *cdif;
	
	cdif = container_of(inode->i_cdev, struct cdif_quasar, cdev);
	if(cdif->ref <= 0)
		return -EFAULT;
	cdif->ref--;
	filp->private_data = NULL;
	return 0;
}

static ssize_t qcdif_read(struct file* filp, char __user *buffer, size_t length, loff_t* offset)
{
	struct cdif_quasar *cdif;	
	
	cdif = (struct cdif_quasar*)filp->private_data;
	return -EINVAL;
}

static ssize_t qcdif_write(struct file* filp, const char __user *buffer, size_t length, loff_t* offset)
{
	struct cdif_quasar *cdif;	
	
	cdif = (struct cdif_quasar*)filp->private_data;
	return -EINVAL;
}

static unsigned int qcdif_poll(struct file *filp, poll_table *wait)
{
	struct cdif_quasar* cdif;	
	unsigned int mask = 0;
	unsigned long flags;
	
	cdif = (struct cdif_quasar*)filp->private_data;

	poll_wait(filp, &cdif->mbq, wait);
	spin_lock_irqsave(&cdif->lock, flags);
	if(cdif->di_head != cdif->di_tail)
		mask |= POLLIN | POLLRDNORM; /* readable */
	spin_unlock_irqrestore(&cdif->lock, flags);
	return mask;
}

static long qcdif_ioctl(struct file* filp, unsigned int cmd, unsigned long arg)
{
	struct cdif_quasar *cdif;	
	unsigned long flags;
	int ret = 0;
	
	cdif = (struct cdif_quasar*)filp->private_data;
	
	switch(cmd)
	{
	case QGETDONE:
		if(cdif->di_head == cdif->di_tail)
			return -EAGAIN;
		copy_to_user((void*)arg,
			(void*)&cdif->doneinfo[cdif->di_tail], sizeof(struct q_doneinfo));
		spin_lock_irqsave(&cdif->lock, flags);
		cdif->di_tail++;
		if(cdif->di_tail >= MAX_MAILBOX)
			cdif->di_tail = 0;
		spin_unlock_irqrestore(&cdif->lock, flags);
		break;
	case QWAITDONE:
		while(cdif->di_head == cdif->di_tail) {
			/* TODO - check for dma in progress? */
			if (wait_event_interruptible(cdif->mbq,
				(cdif->di_head != cdif->di_tail)))
				return -ERESTARTSYS;
		}
		if(cdif->di_head != cdif->di_tail) {
			copy_to_user((void*)arg,
				(void*)&cdif->doneinfo[cdif->di_tail], sizeof(struct q_doneinfo));
			spin_lock_irqsave(&cdif->lock, flags);
			cdif->di_tail++;
			if(cdif->di_tail >= MAX_MAILBOX)
				cdif->di_tail = 0;
			spin_unlock_irqrestore(&cdif->lock, flags);
			ret = 0;
		}
		else {
			/* TODO - check for dma in progress? */
			ret = -EAGAIN;
		}
		break;
	default:
		printk("qcdif - bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}
	return ret;
}

static int qcdif_mmap(struct file* filp, struct vm_area_struct* vma)
{
	struct cdif_quasar *cdif;	
	int length, ret = 0;
	
	cdif = (struct cdif_quasar*)filp->private_data;	
	if(! cdif) return -ENODEV;

	/* !! mark pages as uncached for now !! */
	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	length = vma->vm_end - vma->vm_start;
	if(length > (cdif->ioreg_end - cdif->ioreg_start)) {
		printk(KERN_WARNING
				"CDIF VMA length %d truncated to io length %d\n",
				length,
				(int)(cdif->ioreg_end - cdif->ioreg_start));
		length = (cdif->ioreg_end - cdif->ioreg_start); 
	}
	ret = remap_pfn_range(
				vma,
				vma->vm_start,
				cdif->ioreg_start >> PAGE_SHIFT,
				length,
				vma->vm_page_prot
			);
	return ret;
}

static irqreturn_t quasar_cdif_interrupt(int irq, void *dev_id)
{
	struct cdif_quasar *cdif = (struct cdif_quasar *)dev_id; 
	struct timeval tv;
	int ret;
	unsigned val;
	unsigned long flags;
	
	/* clear interrupt bit in CDIF (or we keep getting them!)
	*/
	val = readl(cdif->regs + QBCD_IRQSTAT_OFF);
	writel(val, cdif->regs + QBCD_IRQSTAT_OFF);
	
	spin_lock_irqsave(&cdif->lock, flags);
	cdif->doneinfo[cdif->di_head].msg     = val;
	cdif->doneinfo[cdif->di_head].detail  = 0;
	do_gettimeofday(&tv);
	cdif->doneinfo[cdif->di_head].endtime = (unsigned long long)tv.tv_sec * 1000000 + (unsigned long long)tv.tv_usec;
	cdif->doneinfo[cdif->di_head].cycles  = 0;
	cdif->di_head++;
	if(cdif->di_head >= MAX_MAILBOX)
		cdif->di_head = 0;
	spin_unlock_irqrestore(&cdif->lock, flags);

	/* wakeup tasks waiting on dma done
	*/
	wake_up_interruptible(&cdif->mbq);

	ret = IRQ_HANDLED;
	return ret;
}

static struct file_operations quasar_cdif_ops = {
	.owner		= THIS_MODULE,
	.open		= qcdif_open,
	.release	= qcdif_release,
	.read		= qcdif_read,
	.write		= qcdif_write,
	.poll		= qcdif_poll,
	.mmap		= qcdif_mmap,
	.unlocked_ioctl	= qcdif_ioctl
};

static int __init quasar_cdif_probe(struct platform_device *pdev)
{
	struct cdif_quasar *cdif;
	struct resource	*regs;
	dev_t  cdifn;
	int    ret;
	static int ncdifs = 0;

	cdif = kzalloc(sizeof(struct cdif_quasar), GFP_KERNEL);
	if (!cdif) {
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	cdev_init(&cdif->cdev, &quasar_cdif_ops);
	cdifn = MKDEV(QCDIF_MAJOR, ncdifs);
	ret = cdev_add(&cdif->cdev, cdifn, 1);
	if (ret) {
		dev_dbg(&pdev->dev, "could not create char dev %d\n", ncdifs);
		goto out_err;
	}
	regs = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!regs) {
		dev_dbg(&pdev->dev, "no mmio reg resource defined\n");
		ret = -ENXIO;
		goto out_rerr;
	}
	cdif->ioreg_start = regs->start;
	cdif->ioreg_end   = regs->end;

	cdif->regs = ioremap(regs->start, regs->end - regs->start + 1);
	if(!cdif->regs) {
		ret = -ENOMEM;
		dev_dbg(&pdev->dev, "could not map reg I/O memory\n");
		goto out_ioerr;
	}

	cdif->irq = platform_get_irq(pdev, 0);
	if (cdif->irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		ret = -ENXIO;
		goto out_ioerr;
	}
	ret = request_irq(cdif->irq, quasar_cdif_interrupt,
		0, "cdifmail", cdif);
	if (ret) {
		dev_dbg(&pdev->dev, "could not request irq %d\n", cdif->irq);
		cdif->irq = -1;
		goto out_ioerr;
	}
	spin_lock_init(&cdif->lock);
	init_waitqueue_head(&cdif->mbq);

	ncdifs++;
	platform_set_drvdata(pdev, cdif);
	printk("QCDIF - mapped at %px, irq %d\n", cdif->regs, cdif->irq);
	device_init_wakeup(&pdev->dev, 1);
	return 0;
out_ioerr:
	iounmap(cdif->regs);
out_rerr:
	cdev_del(&cdif->cdev);		
out_err:
	kfree(cdif);
	return ret;
}

static int __exit quasar_cdif_remove(struct platform_device *pdev)
{
	struct cdif_quasar *cdif = platform_get_drvdata(pdev);

	if(cdif->irq > 0)
		free_irq(cdif->irq, cdif);
	iounmap(cdif->regs);
	cdev_del(&cdif->cdev);		
	kfree(cdif);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static int quasar_cdif_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct cdif_quasar *cdif = platform_get_drvdata(pdev);

    cdif->bus_cfg = readl(cdif->regs + QBCD_BUSCFG_OFF);
    cdif->irq_en  = readl(cdif->regs + QBCD_IRQEN_OFF);
#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u32 temp;

        reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
        temp = readl(reg);
        switch (cdif->minor)
        {
        default:
            break;
        case 0:
		    temp |= RSTGEN_SWRSTSTATIC3__IPM2_CD__MASK;
            break;
        }
        writel(temp, reg);
        temp = readl(reg);
		iounmap(reg);
    }
#endif
	return 0;
}

static int quasar_cdif_resume(struct platform_device *pdev)
{
	struct cdif_quasar *cdif = platform_get_drvdata(pdev);

#if defined(Q6300) || defined(Q6600)
	{
		u8 __iomem *reg;
		u32 temp;

        reg = ioremap(RSTGEN_SWRSTSTATIC3, 4);
        temp = readl(reg);
        switch (cdif->minor)
        {
        default:
            break;
        case 0:
            temp &= RSTGEN_SWRSTSTATIC3__IPM2_CD__INV_MASK;
            break;
        }
        writel(temp, reg);
        temp = readl(reg);
        iounmap(reg);
	}
#endif
    writel(cdif->bus_cfg, cdif->regs + QBCD_BUSCFG_OFF);
    writel(cdif->irq_en,  cdif->regs + QBCD_IRQENSET_OFF);
	return 0;
}

MODULE_ALIAS("platform:quasar-cdif");

static const struct of_device_id qbit_quasar_id_table[] = {
	{ .compatible = "qbit,quasar-cdif" },
	{}
};
MODULE_DEVICE_TABLE(of, qbit_quasar_id_table);

static struct platform_driver quasar_cdif_driver_ops = {
	.probe		= quasar_cdif_probe,
	.remove		= quasar_cdif_remove,
	.suspend	= quasar_cdif_suspend,
	.resume		= quasar_cdif_resume,
	.driver		= {
		.name	= "quasar-cdif",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(qbit_quasar_id_table),	
	},
};

static int __init quasar_cdif_init(void)
{
	int ret;
	
	ret = platform_driver_register(&quasar_cdif_driver_ops);
	return ret;
}
module_init(quasar_cdif_init);

static void __exit quasar_cdif_exit(void)
{
	platform_driver_unregister(&quasar_cdif_driver_ops);
}
module_exit(quasar_cdif_exit);

MODULE_DESCRIPTION("Quasar CDIF driver");
MODULE_LICENSE("Dual BSD/GPL");

