/*======================================================================
 *  Primax fax board driver
 */

/*======================================================================
 *  Include Files
 *----------------------------------------------------------------------*/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#include "pmxfaxboard.h"

#define DRIVER_VERSION		"FAX_BOARD: Ver0.03.00"
/*======================================================================*
 *  Types
 *----------------------------------------------------------------------*/
struct faxboard_info
{
	int          major;
	struct class *faxboard_class;
	struct cdev		myfaxboard;
	dev_t	 cdevno;
	int irq;
	volatile u32	__iomem *regISR;
	wait_queue_head_t faxboard_wait_queue;
};
/*======================================================================*
 *  Functions
 *----------------------------------------------------------------------*/
static int  __init faxboard_init(void);
static void __exit faxboard_exit(void);

static int faxboard_open(struct inode *inode, struct file *filep);
static int faxboard_close(struct inode *inode, struct file *filep);
static int faxboard_read(struct file *filep, char __user *buf, size_t len, loff_t *ptr);
static int faxboard_write(struct file *filep, const char __user *buf, size_t len, loff_t *ptr);
static long faxboard_ioctl(struct file *filep, unsigned int cmd, unsigned long arg);
static loff_t faxboard_seek(struct file *file, loff_t offset, int origin);
static unsigned int faxboard_poll(struct file *file, poll_table * wait);

static struct faxboard_info gFaxboard;

/*======================================================================*
 *  Variables
 *----------------------------------------------------------------------*/
int is_faxboard_int = 0;

//cmd: 1: enable interrupt, 0: disable interrupt.
static void faxboard_isr_status(struct faxboard_info *faxboard, int cmd)
{
	static int flag = 1;	//Disable is initial status.
		
	//Enable or disable interrupt bit.
	if ((cmd == 1) && (flag == 0))
	{
		enable_irq(faxboard->irq);
		flag = 1;
	}
	else if ((cmd == 0) && (flag == 1))
	{
		disable_irq(faxboard->irq);
		flag = 0;
	}
}

static void clear_gpio_isr(struct faxboard_info *myfax, unsigned int pin)
{
	u32 dat32;
	
	dat32 = readl(myfax->regISR);
	//printk("clear_dpio_isr_1, 0x%08x\n", dat32);
	dat32 &= 0xfffffffb;	//clear X2 interrupt status
	//printk("clear_dpio_isr_2, 0x%08x\n", dat32);
	writel(dat32, myfax->regISR);
}

/*======================================================================*
 *  open I/F
 *----------------------------------------------------------------------*/
static int faxboard_open(struct inode *inode, struct file *filep)
{
	struct faxboard_info *faxboard = &gFaxboard;

	faxboard = container_of(inode->i_cdev, struct faxboard_info, myfaxboard);
	printk("faxboard_open: 0x%08x\n", faxboard);
	faxboard_isr_status(faxboard, 0);
	filep->private_data = faxboard;

	//printk(KERN_DEBUG "faxboard: opened.\n");
	printk("faxboard: opened.\n");
	return 0;
}

/*======================================================================*
 *  close I/F
 *----------------------------------------------------------------------*/
static int faxboard_close(struct inode *inode, struct file *filep)
{
	return 0;
}

/*======================================================================*
 *  read I/F
 *----------------------------------------------------------------------*/
static int faxboard_read(struct file *filep, char __user *buf, size_t len, loff_t *ptr)
{
	int  retvalue = -EBUSY;
	
	printk(KERN_DEBUG "faxboard: read.\n");
	return retvalue;
}

/*======================================================================*
 *  write I/F
 *----------------------------------------------------------------------*/
static int faxboard_write(struct file *filep, const char __user *buf, size_t len, loff_t *ptr)
{
	int  retvalue;

	printk(KERN_DEBUG "faxboard: write.\n");
	retvalue = 0;
	return(retvalue);
}

/*======================================================================*
 *  ioctl I/F
 *----------------------------------------------------------------------*/
static long faxboard_ioctl(struct file *filep, unsigned int cmd, unsigned long arg)
{
	int  retvalue = 0;
	struct faxboard_info *faxboard = &gFaxboard;
	u32 dat32;
	
	
	switch (cmd) {
		case FAXBOARD_INTERRUPT_ENABLE:
			dat32 = readl(gFaxboard.regISR);
			printk("faxboard: FAXBOARD_INTERRUPT_ENABLE, status reg: 0x%x\n", dat32);
			if (dat32 == 0)
				is_faxboard_int = 0;
			else
			{
				clear_gpio_isr(&gFaxboard, 0);//*/
				is_faxboard_int = 1;
				wake_up_interruptible(&(gFaxboard.faxboard_wait_queue));
			}
			faxboard_isr_status(faxboard, 1);	//Enable fax board interrupt
			break;
		
		case FAXBOARD_INTERRUPT_DISABLE:
			printk("faxboard: FAXBOARD_INTERRUPT_DISABLE\n");
			is_faxboard_int = 0;
			faxboard_isr_status(faxboard, 0);	//Disable fax board interrupt
			break; 
		
		default:
			retvalue = -ENOTTY;
			break;
	}
	return(retvalue);
}

/*======================================================================*
 *  seek I/F
 *----------------------------------------------------------------------*/
static loff_t faxboard_seek(struct file *file, loff_t offset, int origin)
{
	loff_t retvalue = 0;

	printk(KERN_DEBUG "faxboard: seek\n");
	return(retvalue);
}

static unsigned int faxboard_poll(struct file *file, poll_table * wait)
{
	unsigned int retvalue = 0;
	struct faxboard_info *faxboard = &gFaxboard;

	//printk("==>faxboard_poll. 0x%08x<==\n", faxboard);
	poll_wait(file, &(faxboard->faxboard_wait_queue), wait);

	if (is_faxboard_int == 1) 
	{
		retvalue |= (POLLIN | POLLRDNORM);
	}

	is_faxboard_int = 0;
	//printk("[faxboard_poll] do poll. 0x%x.\n",retvalue);
	
	return (retvalue);
}

struct file_operations faxboard_fops =
{
	.owner          = THIS_MODULE,
	.read           = faxboard_read,
	.write          = faxboard_write,
	.unlocked_ioctl = faxboard_ioctl,
	.open           = faxboard_open,
	.release        = faxboard_close,
	.llseek         = faxboard_seek,
	.poll         	= faxboard_poll,
};

static irqreturn_t faxboard_isr(int irq, void *dev_id)
{
	struct faxboard_info *myfax = (struct faxboard_info *)dev_id;

	//faxboard_isr_status(myfax, 0);
	//quasar_clear_irq(gFaxboard.irq);	//Quasar 21.1 delete it.
	clear_gpio_isr(&gFaxboard, 0);//*/
	is_faxboard_int = 1;
	wake_up_interruptible(&(gFaxboard.faxboard_wait_queue));
	//printk("irq coming. %d. 0x%08x\n", irq, &gFaxboard);
	return IRQ_HANDLED;
}

void pmx_fax_init(void)
{
	unsigned int dat32;
	volatile u8 __iomem *reg;
	/* falling edge interrupt, M2
	 * SYS_PIOXMSK(0x0410D010): 10_xxxxB
	 * SYS_PIOINTSEL(0x0410D014): XX00XXXXH
	 */
	reg = ioremap(SYS_PIOXMSK, 4);
	dat32 = readl(reg);
	dat32 &= 0xffffffcf;
	dat32 |= 0x20;
	writel(dat32, reg);

	reg = ioremap(SYS_PIOINTSEL, 4);
	dat32 = readl(reg);
	dat32 &= 0xff00ffff;
	dat32 |= 0x00090000;	//FAX_IRQ is in SYSPIO09
	writel(dat32, reg);

}

static int __init primax_fax_probe(struct platform_device *pdev)
{
	struct faxboard_info *faxboard = &gFaxboard;
	int		retval;
	int		irq, error;
	static int noused;

	printk(KERN_ERR "faxboard: module_init. 15\n");

	printk("primax_fax_probe(%d): 0x%08x\n", __LINE__, faxboard);

	/* create class (/sys/class/faxboard) */
	faxboard->faxboard_class = class_create(THIS_MODULE, "faxboard");
	if(IS_ERR(faxboard->faxboard_class))
	{
		retval = PTR_ERR(faxboard->faxboard_class);
		printk(KERN_ERR "faxboard: can't create class\n");
	}

	retval = alloc_chrdev_region(&faxboard->cdevno, 0, 0, "faxboard");
	if(retval)
	{
		printk(KERN_ERR "faxboard: can't alloc character device.\n");
		return (retval);
	}
	faxboard->major = MAJOR(faxboard->cdevno);
	device_create(faxboard->faxboard_class, NULL, faxboard->cdevno, NULL, "faxboard");

	cdev_init(&faxboard->myfaxboard, &faxboard_fops);
	faxboard->myfaxboard.owner = THIS_MODULE;
	retval = cdev_add(&faxboard->myfaxboard, faxboard->cdevno, 1);
	if(retval < 0)
	{
		printk(KERN_ERR "faxboard: cdev_add() error(%d)\n", retval);
		return -1;
	}

	faxboard->regISR = ioremap(SYS_PIOINTSTS, 4);
	printk("primax_fax_probe: reg_addr: 0x%08x, reg: 0x%08x, val: 0x%08x.\n", 
						faxboard->regISR, SYS_PIOINTSTS, readl(faxboard->regISR));
	
	init_waitqueue_head(&(faxboard->faxboard_wait_queue));

	//set fax board interrupt interrupt
	pmx_fax_init();

	faxboard->irq = platform_get_irq(pdev, 0);
	if (faxboard->irq < 0) {
		dev_dbg(&pdev->dev, "could not get irq\n");
		retval = -ENXIO;
		return retval;
	}
	printk("==>fax irq: %d. 0x%08x<==\n", faxboard->irq, faxboard);//IRQF_TRIGGER_FALLING | IRQF_SHARED
	//error =  request_irq(faxboard->irq , faxboard_isr, IRQF_TRIGGER_NONE, "faxboard", &(faxboard->cdevno));
	//error =  request_irq(faxboard->irq , faxboard_isr, IRQF_TRIGGER_NONE, "faxboard", faxboard);
	error =  request_irq(faxboard->irq , faxboard_isr, IRQF_TRIGGER_NONE, "faxboard", &noused);
	if (error) {
		printk("faxboard:request_irq NG. irq: %d error: %d.\n", faxboard->irq, error);
	}

	printk("primax_fax_probe: 0x%08x\n", faxboard);
	return retval;
}

/*======================================================================*
 *  Do at Unloading
 *----------------------------------------------------------------------*/
static void __exit primax_fax_remove(struct platform_device *pdev)
{
	struct faxboard_info *faxboard = &gFaxboard;

	printk(KERN_ERR "faxboard: module_exit\n");

	/*if(faxboard->irq > 0)
	{
		free_irq(faxboard->irq, faxboard);
		faxboard->irq = 0;
	}//*/

	unregister_chrdev_region(faxboard->cdevno, 1);
	class_destroy(faxboard->faxboard_class);
	cdev_del(&faxboard->myfaxboard);
	//kfree(faxboard);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static const struct of_device_id pmx_quasar_id_table[] = {
	{ .compatible = "pmx,faxboard" },
	{}
};
MODULE_DEVICE_TABLE(of, pmx_quasar_id_table);

static struct platform_driver primax_fax_driver_ops = {
	.probe		= primax_fax_probe,
	.remove		= primax_fax_remove,
	.driver		= {
		.name	= "faxboard",
		.owner	= THIS_MODULE,
		.of_match_table = of_match_ptr(pmx_quasar_id_table),
	},
};

static int __init faxboard_init(void)
{
	int ret;
	
	ret = platform_driver_register(&primax_fax_driver_ops);
	return ret;
}
module_init(faxboard_init);

static void __exit faxboard_exit(void)
{
	platform_driver_unregister(&primax_fax_driver_ops);
}
module_exit(faxboard_exit);

MODULE_AUTHOR("PRIMAX BJ");
MODULE_DESCRIPTION("primax fax board");

MODULE_LICENSE("GPL");
MODULE_VERSION("2021_Oct_10 15");
