/*======================================================================
 *  Primax engine driver
 */

/*======================================================================
 *  Include Files
 *----------------------------------------------------------------------*/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/jiffies.h>
#include <linux/platform_device.h>
#include <linux/of_platform.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/mman.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/cdev.h>
#include <linux/poll.h>
#include <linux/mutex.h>
#include <asm/io.h>
#include <asm/mman.h>
#include <asm/uaccess.h>
#include <asm/dma.h>
#include <asm/pgalloc.h>
#include <quasar/qbsocregs.h>
#include <quasar/qioctl.h>

#include "../quasar/6300/qbsoc_prt_regs_B0.h"
#include "../vos/vos_if.h"
#include "../../../../../fw/CTL_FW/include/if_pmxengdrv.h"

#include "pmxengdrv.h"


#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE 1
#endif
#define pmxeng_assert(a)	if(!(a) )printk(KERN_WARNING "pmxeng: ASSERT here:%s @%d\n", __func__, __LINE__);

int pmxeng_deglogLevel=1;
#define ShowPmxengDbgMsg(level,...)  if(level <= pmxeng_deglogLevel) printk(__VA_ARGS__)


#define LPCHANNEL_NUM_CONTEXTS PMXENG_CH_COUNT

typedef struct _s_prtreg_map
{
	int nIndex;
	const u32 nPhyAddrBase;
	volatile u8 __iomem *pVirAddrBase;
	int nSize;
} S_PRTREG_MAP;

#define QLPDMA_DEVNAME "pmxengch"
/* "pmx_eng_channel"  */

#include "u32circle.c"


#define MAX_MAILBOX 128
#define MAX_BANDBUF_CNT	16
#define MAX_PAGE_BUF_CNT	1

typedef struct _s_bandImgInfo
{
	u32 nImgPhyAddr;
	u32 nImgSize;
	u32 nBytesPerLine;
	u32 nLines;
} S_BandImgInfo;

typedef struct _s_bandBufInfo
{
	u32 nMaxBandBufCnt; // maximum band buffer count, it should match MAX_BANDBUF_CNT
	u32 nPageHeight; // total lines of a page
	u32 nSetupImgHeight; // complete setup image lines, accumulate
	u32 nBandReadycnt;
	u32 nReadIdx;
	u32 nWriteIdx;
	S_BandImgInfo bandImgInfo[MAX_BANDBUF_CNT];
} S_BandBufInfo;

struct pmxengch_quasar
{
	int ref;   ///< reference count
	int minor; ///< device minor number
	struct cdev cdev;
	volatile u8 __iomem *regs;
	unsigned long ioreg_start;
	unsigned long ioreg_end;
	int irq;

	wait_queue_head_t mbq; ///< wait-queue for dma-done events
	// struct q_doneinfo doneinfo[MAX_MAILBOX];
	// int di_head;
	// int di_tail;
	int ints;		 ///< int count (debug only)
	spinlock_t lock; ///< lock for irq/ioctl access

	spinlock_t oplock; ///< lock for operations
	E_ENG_CH ech;		   ///< my ordinal
	int op;			   ///< 0=none, 1=decode, 2=encode
	u8 padded_short_stripe[4];
	u32 referenceLineBytes;
	HU32C  /* HVOSQUEUE */ hQueue;
	int nSetupPageIdx;
	int nPrintPageIdx;
	S_BandBufInfo bandBufInfo[MAX_PAGE_BUF_CNT];
};

struct pmxengch_quasar *s_pmxengch_context[LPCHANNEL_NUM_CONTEXTS];


typedef int (*PFNCallbackProc)(void *, ...);

static int pmxeng_start_banddma(struct pmxengch_quasar *pmxengch);
static int pmxeng_init_pagebandinfo(struct pmxengch_quasar *pmxengch, int nPageHeight);
static int pmxeng_reset_allpagebandinfo(struct pmxengch_quasar *pmxengch);

static volatile u8 __iomem *pmxeng_retrv_regaddress( u32 regPhyAddr )
{
	int n = 0;
	volatile u8 __iomem *p;
	static S_PRTREG_MAP asRegMap[] = 
	{
		{		0, 		PRIS_BASE, 		NULL,		0x1000		},
		{		1, 		PRI0_BASE, 		NULL,		0x1000		},
		{		2, 		PRI2_BASE, 		NULL,		0x1000		},
		{		3, 		PRI4_BASE, 		NULL,		0x1000		},
		{		4, 		PRI6_BASE, 		NULL,		0x1000		}
	};
#define RANGECHECK(pa, n)		(((pa) >= asRegMap[n].nPhyAddrBase) && ((pa) < (asRegMap[n].nPhyAddrBase+asRegMap[n].nSize)))
#define PHYADDRGET(n)		(asRegMap[n].nPhyAddrBase)
#define VIRADDRGET(n)		(asRegMap[n].pVirAddrBase)
#define VIRADDRSET(n, p)	(asRegMap[n].pVirAddrBase=(p))
#define ADDRSIZE(n)			(asRegMap[n].nSize)

	if(RANGECHECK(regPhyAddr, 0))				n = 0;
	else if(RANGECHECK(regPhyAddr, 1))			n = 1;
	else if(RANGECHECK(regPhyAddr, 2))			n = 2;
	else if(RANGECHECK(regPhyAddr, 3))			n = 3;
	else if(RANGECHECK(regPhyAddr, 4))			n = 4;
	else							return NULL;

	p = VIRADDRGET(n);
	if(NULL == p)
	{
		p = ioremap(PHYADDRGET(n), ADDRSIZE(n));
		if(NULL == p)
		{
			return NULL;
		}
		VIRADDRSET(n, p);
	}
	p += (regPhyAddr - PHYADDRGET(n));

	return p;

#undef RANGECHECK		// (n)		((regPhyAddr >= asRegMap[n].nPhyAddrBase) && (regPhyAddr < (asRegMap[n].nPhyAddrBase+asRegMap[n].nSize)))
#undef PHYADDRGET		// (n)		(asRegMap[n].nPhyAddrBase)
#undef VIRADDRGET		// (n)		(asRegMap[n].pVirAddrBase)
#undef VIRADDRSET		// (n, p)	(asRegMap[n].pVirAddrBase=(p))
#undef ADDRSIZE			// (n)			(asRegMap[n].nSize)

}

void pmxeng_write_reg(u32 pPhyAddr, u32 val)
{
	volatile u8 __iomem *p;
	p = pmxeng_retrv_regaddress(pPhyAddr);
	if(NULL == p)
	{
		printk("pmxeng: pmxeng_write_reg(0x%08X, %d) is not an vlid reg\n", pPhyAddr, val);
		return;
	}
	writel(val, p);
}

u32 pmxeng_read_reg(u32 pPhyAddr)
{
	volatile u8 __iomem *p;
	p = pmxeng_retrv_regaddress(pPhyAddr);
	if(NULL == p)
	{
		printk("pmxeng: pmxeng_read_reg(0x%08X) is not an vlid reg\n", pPhyAddr);
		return;
	}
	return readl(p);	
}




#define BIN32_REMASK(bgn, cnt)	(u32)(((bgn)<=0 ? 0x00000000 : (u32)(0xFFFFFFFF >> (32-(bgn)))) | \
										(((bgn)+(cnt))>=32 ? 0x00000000 : (u32)(0xFFFFFFFF << ((bgn)+(cnt)))))
#define BIN32_MASK(cnt)	(u32)((cnt)<=0 ? 0x00000000 : ((u32)0xFFFFFFFF >> (32 - (cnt))))

static int pmxeng_get_val32(u32 nOrgValue, int nBitBgn, int nBitCnt)
{
	int maskval;
	if (nBitBgn < 0)
	{
		pmxeng_assert(FALSE);
		return 0;
	}
	if ((nBitBgn + nBitCnt) > 32)
	{
		pmxeng_assert(FALSE);
		return 0;
	}
	if (nBitCnt <= 0)
	{
		pmxeng_assert(FALSE);
		return 0;
	}
	if (32 > nBitCnt)
	{
		// maskval = (std_bit(nBitCnt) - 1);
		maskval = BIN32_MASK(nBitCnt);
	}
	else
	if (32 == nBitCnt)
	{
		return nOrgValue;
	}
	else
	{
		pmxeng_assert(FALSE);
		return FALSE;
	}
	nOrgValue >>= nBitBgn;
	nOrgValue &= maskval;
	return nOrgValue;
}


static u32 pmxeng_set_val32(u32 nOrgValue, int nBitBgn, int nBitCnt, int val)
{

	u32 maskval = 0;
	u32 remaskval = 0;

	if (nBitBgn < 0)
	{
		pmxeng_assert(FALSE);
		return nOrgValue;
	}
	if ((nBitBgn + nBitCnt) > 32)
	{
		pmxeng_assert(FALSE);
		return nOrgValue;
	}
	if (nBitCnt <= 0)
	{
		pmxeng_assert(FALSE);
		return nOrgValue;
	}
	if (32 > nBitCnt)
	{
		remaskval = BIN32_REMASK(nBitBgn, nBitCnt);

	}
	else
	if (32 == nBitCnt)
	{
		remaskval = 0x00000000;
	}
	else
	{
		pmxeng_assert(FALSE);
		return nOrgValue;
	}
	maskval = BIN32_MASK(nBitCnt);
	nOrgValue &= remaskval;
	val &= maskval;
	nOrgValue |= (val << nBitBgn);
	return nOrgValue;
	// #undef BIN32_REMASK
}



static void pmxeng_set_reg_feild(int ch, u32 pAddr, int nBitBgn, int nBitCnt, int val)
{
	u32 nOrgVal = pmxeng_read_reg(pAddr);
	nOrgVal = pmxeng_set_val32(nOrgVal, nBitBgn, nBitCnt, val);
	pmxeng_write_reg(pAddr, nOrgVal);
}

static int pmxeng_get_reg_feild(int ch, u32 pAddr, int nBitBgn, int nBitCnt)
{
	u32 nOrgVal = pmxeng_read_reg(pAddr);
	return pmxeng_get_val32(nOrgVal, nBitBgn, nBitCnt);
}

static u32 g_aCh0Buffer[192];
static u32 g_aCh1Buffer[192];
static u32 g_aCh2Buffer[192];
static u32 g_aCh3Buffer[192];

static int pmxeng_init_msgqueue(struct pmxengch_quasar *pmxengch)
{
	int ech;
	static const char * astrName[4] = {	"Ch0_isr", 	"Ch1_isr", 	"Ch2_isr", 	"Ch3_isr"	};
	static const u32 *aulAddr[4] = {	g_aCh0Buffer, g_aCh1Buffer, g_aCh2Buffer, g_aCh3Buffer	};
	ech = pmxengch->ech;
	if(NULL != pmxengch->hQueue)
	{
		printk("pmxeng:ch%d: error: oepen the ch queue\r\n", pmxengch->ech);
		return 0;
	}
	// pmxengch->hQueue = vos_queueCreate(astrName[ech], MAX_MAILBOX, 8);
	pmxengch->hQueue = u32cCreate(astrName[ech], aulAddr[ech], sizeof(g_aCh0Buffer), 184);

	printk("pmxeng:ch%d: oepen the ch queue:%x\r\n", pmxengch->ech, pmxengch->hQueue);

	return 0;
}

static int pmxeng_channel_open(struct inode *inode, struct file *filp)
{
	struct pmxengch_quasar *pmxengch;
	// volatile u8 __iomem *reg;
	// volatile u8 __iomem *sts;

	pmxengch = container_of(inode->i_cdev, struct pmxengch_quasar, cdev);
	if (pmxengch->ref > 0)
	{
		printk("pmxeng: channel busy!!!!!!!!\n");
		return -EBUSY;
	}
	pmxengch->ref++;
	pmxengch->minor = iminor(inode);
	filp->private_data = pmxengch;
	// pmxengch->di_head = pmxengch->di_tail = 0;
	u32cResetData(pmxengch->hQueue);

	// pmxeng_create_msgqueue(pmxengch);

	return 0;
}

static int pmxeng_channel_release(struct inode *inode, struct file *filp)
{
	struct pmxengch_quasar *pmxengch;
	u32 msgData[2];

	pmxengch = container_of(inode->i_cdev, struct pmxengch_quasar, cdev);

	printk("pmxeng:ch%d: close the ref:%d queue:%x\r\n", pmxengch->ech, pmxengch->ref, pmxengch->hQueue);

	/*
	while(vos_queueGetMsgCnt(pmxengch->hQueue) > 0)
	{
		vos_queueRead(pmxengch->hQueue, msgData, 0);
	}
	*/
	u32cResetData(pmxengch->hQueue);

	if (pmxengch->ref <= 0)
		return -EFAULT;
	pmxengch->ref--;
	filp->private_data = NULL;

	return 0;
}

static ssize_t pmxeng_channel_read(
	struct file *filp, char __user *buffer, size_t length, loff_t *offset)
{
	int nRet = -EINVAL;
	u32 msgData[2] = {0,0};
	int cnt;
	struct pmxengch_quasar *pmxengch;
	unsigned long flags;


	pmxengch = (struct pmxengch_quasar *)filp->private_data;

	ShowPmxengDbgMsg(2, "read interrupt, nch:%d \n", pmxengch->ech);
	spin_lock_irqsave(&pmxengch->lock, flags);

	// cnt = vos_queueGetMsgCnt(pmxengch->hQueue);
	cnt = u32cGetDataCount(pmxengch->hQueue);

	if(cnt <= 0)
	{
		goto _endof_here;
	}
	// vos_queueRead(pmxengch->hQueue, msgData, 0);
	u32cPopupData(pmxengch->hQueue, msgData);

	copy_to_user(buffer, (void *)msgData, sizeof(msgData));

	nRet = sizeof(msgData);

	_endof_here:


	spin_unlock_irqrestore(&pmxengch->lock, flags);


	return nRet;
}

static ssize_t pmxeng_channel_write(
	struct file *filp, const char __user *buffer, size_t length,
	loff_t *offset)
{
	struct pmxengch_quasar *pmxengch;
	int ret = 0;
	unsigned long flags;
	S_BandImgInfo sBandImage;
	int nWriteIdx = 0;
	u32 nSetupPageIdx = 0;

	pmxengch = (struct pmxengch_quasar *)filp->private_data;
	if(16 != length)
	{
		ShowPmxengDbgMsg(0, "channel : %d, write length:%d is bad \n", pmxengch->ech, length);
		ret = -EINVAL;
	}

	spin_lock_irqsave(&pmxengch->lock, flags);
	nSetupPageIdx = pmxengch->nSetupPageIdx;
	if(pmxengch->bandBufInfo[nSetupPageIdx].nBandReadycnt < pmxengch->bandBufInfo[nSetupPageIdx].nMaxBandBufCnt)  /* has space to save the data */
	{
		copy_from_user(&sBandImage, buffer, 16);
		
		nWriteIdx = pmxengch->bandBufInfo[nSetupPageIdx].nWriteIdx;
		pmxengch->bandBufInfo[nSetupPageIdx].bandImgInfo[nWriteIdx].nImgPhyAddr = sBandImage.nImgPhyAddr;
		pmxengch->bandBufInfo[nSetupPageIdx].bandImgInfo[nWriteIdx].nImgSize = sBandImage.nImgSize;
		pmxengch->bandBufInfo[nSetupPageIdx].bandImgInfo[nWriteIdx].nBytesPerLine = sBandImage.nBytesPerLine;
		pmxengch->bandBufInfo[nSetupPageIdx].bandImgInfo[nWriteIdx].nLines = sBandImage.nLines;
		pmxengch->bandBufInfo[nSetupPageIdx].nBandReadycnt++;
		pmxengch->bandBufInfo[nSetupPageIdx].nWriteIdx++;
		if(pmxengch->bandBufInfo[nSetupPageIdx].nWriteIdx >= pmxengch->bandBufInfo[nSetupPageIdx].nMaxBandBufCnt)
			pmxengch->bandBufInfo[nSetupPageIdx].nWriteIdx = 0;
	}
	else 
	{
		ShowPmxengDbgMsg(0, "channel : %d, not space, nBandReadycnt:%d  \n", pmxengch->ech, pmxengch->bandBufInfo[nSetupPageIdx].nBandReadycnt);
		ret = -EINVAL;
	}
	spin_unlock_irqrestore(&pmxengch->lock, flags);
	
	return ret;
}

static unsigned int pmxeng_channel_poll(struct file *filp, poll_table *wait)
{
	struct pmxengch_quasar *pmxengch;
	unsigned int mask = 0;
	unsigned long flags;

	pmxengch = (struct pmxengch_quasar *)filp->private_data;

	poll_wait(filp, &pmxengch->mbq, wait);
	ShowPmxengDbgMsg(2, "get interrupt, nch:%d \n", pmxengch->ech);
	spin_lock_irqsave(&pmxengch->lock, flags);
	// if (pmxengch->di_head != pmxengch->di_tail)
	// 	mask |= POLLIN | POLLRDNORM; /* readable */

	if(u32cGetDataCount(pmxengch->hQueue) > 0)
	{
		mask |= POLLIN | POLLRDNORM;
	}

	spin_unlock_irqrestore(&pmxengch->lock, flags);
	return mask;
}

static long pmxeng_channel_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct pmxengch_quasar *pmxengch;
	unsigned long flags;
	int ret = 0;
	u32 nPrintPageIdx = 0;
	u32 nPageHeight = 0;

	pmxengch = (struct pmxengch_quasar *)filp->private_data;

	switch(cmd)
	{
	case PRDRV_PAGE_START: 
		spin_lock_irqsave(&pmxengch->lock, flags);
		copy_from_user(&nPageHeight, arg, 4);
		if(nPageHeight <= 0)
		{
			ShowPmxengDbgMsg(0, "pmxeng: channel(%d) - nPageHeight:%d is invalid \n", pmxengch->ech, nPageHeight);
			ret = -EINVAL;
			break;
		}

		pmxengch->nSetupPageIdx++;
		if(pmxengch->nSetupPageIdx >= MAX_PAGE_BUF_CNT)
		{
			pmxengch->nSetupPageIdx = 0;
		}

		ShowPmxengDbgMsg(2, "pmxeng: channel(%d), nSetupPageIdx:%d \n", pmxengch->ech, pmxengch->nSetupPageIdx);
		pmxeng_init_pagebandinfo(pmxengch, nPageHeight);
		spin_unlock_irqrestore(&pmxengch->lock, flags);
		break;
	case PRDRV_START_DMA: /* one page start, start to put band image address and size to SoC register */
		spin_lock_irqsave(&pmxengch->lock, flags);
#if 0		
		pmxengch->nPrintPageIdx++;
		if(pmxengch->nPrintPageIdx >= MAX_PAGE_BUF_CNT)
		{
			pmxengch->nPrintPageIdx = 0;
		}
#else
		pmxengch->nPrintPageIdx = pmxengch->nSetupPageIdx;
#endif
		nPrintPageIdx = pmxengch->nPrintPageIdx;
		if(pmxengch->bandBufInfo[nPrintPageIdx].nBandReadycnt >= 2)
		{
			pmxeng_start_banddma(pmxengch); // setup 1st dma block
			udelay(100); // delay 100us to let Soc operation last setup
			pmxeng_start_banddma(pmxengch); // setup 2nd dma block
		}
		else 
		{
			ret = -EINVAL;
			ShowPmxengDbgMsg(0, "pmxeng: channel(%d) - band ready cnt (%d) does not meet request \n", pmxengch->ech, pmxengch->bandBufInfo[nPrintPageIdx].nBandReadycnt);
		}
		spin_unlock_irqrestore(&pmxengch->lock, flags);
		break;
	case PRDRV_PAGE_FLUSH: /* clear band image information in kernel dirver side */
		spin_lock_irqsave(&pmxengch->lock, flags);
		pmxeng_reset_allpagebandinfo(pmxengch);
		spin_unlock_irqrestore(&pmxengch->lock, flags);
		break;
	default:
		printk(KERN_WARNING "pmxeng: channel - bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}
#if 0
	switch (cmd)
	{
	case QGETDONE:
		if (pmxengch->di_head == pmxengch->di_tail)
			return -EAGAIN;
		ret = copy_to_user(
			(void *)arg,
			(void *)&pmxengch->doneinfo[pmxengch->di_tail],
			sizeof(pmxengch->doneinfo[0]));
		spin_lock_irqsave(&pmxengch->lock, flags);
		pmxengch->di_tail++;
		if (pmxengch->di_tail >= MAX_MAILBOX)
			pmxengch->di_tail = 0;
		spin_unlock_irqrestore(&pmxengch->lock, flags);
		break;
	case QWAITDONE:
		while (pmxengch->di_head == pmxengch->di_tail)
		{
			/* TODO - check for dma in progress? */
			if (wait_event_interruptible(
					pmxengch->mbq, (pmxengch->di_head != pmxengch->di_tail)))
				return -ERESTARTSYS;
		}
		if (pmxengch->di_head != pmxengch->di_tail)
		{
			ret = copy_to_user(
				(void *)arg,
				(void *)&pmxengch->doneinfo[pmxengch->di_tail],
				sizeof(pmxengch->doneinfo[0]));
			spin_lock_irqsave(&pmxengch->lock, flags);
			pmxengch->di_tail++;
			if (pmxengch->di_tail >= MAX_MAILBOX)
				pmxengch->di_tail = 0;
			spin_unlock_irqrestore(&pmxengch->lock, flags);
			ret = 0;
		}
		else
		{
			/* TODO - check for dma in progress? */
			ret = -EAGAIN;
		}
		break;
	default:
		printk(KERN_WARNING "pmxeng: channel - bad ioctl %x\n", cmd);
		ret = -EINVAL;
	}
#endif 

	return ret;
}

static int pmxeng_channel_mmap(struct file *filp, struct vm_area_struct *vma)
{
	struct pmxengch_quasar *pmxengch;
	int length, ret = 0;

	pmxengch = (struct pmxengch_quasar *)filp->private_data;
	if (!pmxengch)
		return -ENODEV;

	/* !! mark pages as uncached for now !! */
	vma->vm_page_prot = pgprot_noncached(vma->vm_page_prot);
	length = vma->vm_end - vma->vm_start;
	if (length > (pmxengch->ioreg_end - pmxengch->ioreg_start + 1))
	{
		printk(KERN_WARNING "pmxeng ch-%d: VMA length %d > %d - %d + 1\n", pmxengch->ech, length , pmxengch->ioreg_end , pmxengch->ioreg_start);
		length = (pmxengch->ioreg_end - pmxengch->ioreg_start+1);
	}

	ret = remap_pfn_range(
		vma,
		vma->vm_start,
		pmxengch->ioreg_start >> PAGE_SHIFT,
		length,
		vma->vm_page_prot);
	printk("pmxeng ch-%d: remap_pfn_range ret %d\n", pmxengch->ech, ret);
	return ret;
}

static irqreturn_t pmxeng_ch_interrupt(int irq, void *dev_id, PFNCallbackProc pfnCallBack, void *pParam)
{
	struct pmxengch_quasar *pmxengch = (struct pmxengch_quasar *)dev_id;
//	struct timeval tv;
	int ret;
	unsigned long flags;

	//readl(pmxengch->regs + JBJSR);

	spin_lock_irqsave(&pmxengch->lock, flags);

	/*
	pmxengch->doneinfo[pmxengch->di_head].msg = QMSG_DONE;
	pmxengch->doneinfo[pmxengch->di_head].detail = 0;
	do_gettimeofday(&tv);
	pmxengch->doneinfo[pmxengch->di_head].endtime =
		(unsigned long long)tv.tv_sec * 1000000 +
		(unsigned long long)tv.tv_usec;
	pmxengch->doneinfo[pmxengch->di_head].cycles = 0;
	pmxengch->di_head++;
	if (pmxengch->di_head >= MAX_MAILBOX)
		pmxengch->di_head = 0;
		*/

	if(NULL != pfnCallBack)
	{
		(*pfnCallBack)(pParam);
	}


	spin_unlock_irqrestore(&pmxengch->lock, flags);

	/*printk("pmxengch int %d\n", pmxengch->ints++);*/

	/* wakeup tasks waiting on dma done */
	ShowPmxengDbgMsg(2, "wakeup interrupt, nch:%d, irq:%d \n", pmxengch->ech, irq);
	wake_up_interruptible(&pmxengch->mbq);

	ret = IRQ_HANDLED;
	return ret;
}

static int pmxeng_processing_dmaend_isr(struct pmxengch_quasar *pmxengch)
{
	u32 msgData[2] = { 0, 0 	};
	msgData[0] = PRTISR_TYPE_DMAEND;
	// vos_queueWrite(pmxengch->hQueue, msgData, 0);
	
	pmxeng_start_banddma(pmxengch);
	u32cPushData(pmxengch->hQueue, msgData[0]);
	return 0;
}

static irqreturn_t pmxeng_isr_dmaend(int irq, void *dev_id)
{
	int ret = IRQ_HANDLED;
	struct pmxengch_quasar *pmxengch = (struct pmxengch_quasar *)dev_id;


	ret = pmxeng_ch_interrupt(irq, dev_id, pmxeng_processing_dmaend_isr, pmxengch);

	return ret;
}

static int pmxeng_processing_dmaerr_isr(struct pmxengch_quasar *pmxengch)
{
	u32 msgData[2] = { 0, 0 	};
	msgData[0] = PRTISR_TYPE_DMAERR;
	// vos_queueWrite(pmxengch->hQueue, msgData, 0);
	u32cPushData(pmxengch->hQueue, msgData[0]);
	return 0;
}

static irqreturn_t pmxeng_isr_dmaerr(int irq, void *dev_id)
{

	int ret = IRQ_HANDLED;
	struct pmxengch_quasar *pmxengch = (struct pmxengch_quasar *)dev_id;

	ret = pmxeng_ch_interrupt(irq, dev_id, pmxeng_processing_dmaerr_isr, pmxengch);

	return ret;
}


static const u32 PRIxINTEN[] 			= 		{	PRI0INTEN,		PRI2INTEN,		PRI4INTEN,		PRI6INTEN	};
static const u32 PRIxLINESTARTADDRSW[]	= 		{	PRI0LINESTARTADDRSW, 	PRI2LINESTARTADDRSW, 	PRI4LINESTARTADDRSW, 	PRI6LINESTARTADDRSW	};
static const u32 PRIxBYTESPERLINE[]		=		{	PRI0BYTESPERLINE, 		PRI2BYTESPERLINE, 		PRI4BYTESPERLINE, 		PRI6BYTESPERLINE	};
static const u32 PRIxLINESPERBAND[]		=		{	PRI0LINESPERBAND, 		PRI2LINESPERBAND, 		PRI4LINESPERBAND, 		PRI6LINESPERBAND	};
static const u32 PRIxSTRIDE[]			= 		{	PRI0STRIDE, 			PRI2STRIDE, 			PRI4STRIDE,				PRI6STRIDE			};
static const u32 PRIxSTART[] 			= 		{	PRI0START, 				PRI2START, 				PRI4START,				PRI6START			};

static int pmxeng_reset_allpagebandinfo(struct pmxengch_quasar *pmxengch)
{
	int i;
	
	pmxengch->nPrintPageIdx = -1;
	pmxengch->nSetupPageIdx = -1;

	for(i=0; i<MAX_PAGE_BUF_CNT; i++)
	{
		pmxengch->bandBufInfo[i].nReadIdx = 0;
		pmxengch->bandBufInfo[i].nWriteIdx = 0;
		pmxengch->bandBufInfo[i].nBandReadycnt = 0;
		pmxengch->bandBufInfo[i].nPageHeight = 0;
		pmxengch->bandBufInfo[i].nSetupImgHeight = 0;
	}

	return 0;
}

static int pmxeng_init_pagebandinfo(struct pmxengch_quasar *pmxengch, int nPageHeight)
{
	u32 nSetupPageIdx;

	nSetupPageIdx = pmxengch->nSetupPageIdx;

	ShowPmxengDbgMsg(2, "pmxeng: %s, channel(%d), nSetupPageIdx:%d \n", __func__, pmxengch->ech, nSetupPageIdx);
	pmxengch->bandBufInfo[nSetupPageIdx].nReadIdx = 0;
	pmxengch->bandBufInfo[nSetupPageIdx].nWriteIdx = 0;
	pmxengch->bandBufInfo[nSetupPageIdx].nBandReadycnt = 0;
	pmxengch->bandBufInfo[nSetupPageIdx].nPageHeight = nPageHeight;
	pmxengch->bandBufInfo[nSetupPageIdx].nSetupImgHeight = 0;

	return 0;
}

static int pmxeng_start_banddma(struct pmxengch_quasar *pmxengch)
{
	u32 nch;
	u32 nPrintPageIdx;
	u32 nReadIdx;
	u32 nVal;
	
	nPrintPageIdx = pmxengch->nPrintPageIdx;

	ShowPmxengDbgMsg(2, "pmxeng: %s, channel(%d), nPrintPageIdx:%d \n", __func__, pmxengch->ech, nPrintPageIdx);
	if(pmxengch->bandBufInfo[nPrintPageIdx].nSetupImgHeight >= pmxengch->bandBufInfo[nPrintPageIdx].nPageHeight) /* all band put to SoC completely */
	{
		ShowPmxengDbgMsg(1, "channel : %d, setupimgheight : %d, pageheight : %d, skip DMA setup \n", pmxengch->ech, pmxengch->bandBufInfo[nPrintPageIdx].nSetupImgHeight, pmxengch->bandBufInfo[nPrintPageIdx].nPageHeight);
		return 0;
	}
	else if(pmxengch->bandBufInfo[nPrintPageIdx].nBandReadycnt < 1) /* There is not ready band for SoC setup */
	{
		ShowPmxengDbgMsg(0, "channel : %d, setupimgheight : %d, pageheight : %d, overrun occur \n", pmxengch->ech, pmxengch->bandBufInfo[nPrintPageIdx].nSetupImgHeight, pmxengch->bandBufInfo[nPrintPageIdx].nPageHeight);
		return -1;
	}
	
	nch = pmxengch->ech;
	nReadIdx = pmxengch->bandBufInfo[nPrintPageIdx].nReadIdx;
	pmxengch->bandBufInfo[nPrintPageIdx].nReadIdx++;
	pmxengch->bandBufInfo[nPrintPageIdx].nBandReadycnt--;
	pmxengch->bandBufInfo[nPrintPageIdx].nSetupImgHeight += pmxengch->bandBufInfo[nPrintPageIdx].bandImgInfo[nReadIdx].nLines;
	if(pmxengch->bandBufInfo[nPrintPageIdx].nReadIdx >= pmxengch->bandBufInfo[nPrintPageIdx].nMaxBandBufCnt)
	{
		pmxengch->bandBufInfo[nPrintPageIdx].nReadIdx = 0;
	}

	pmxeng_write_reg(PRIxLINESTARTADDRSW[nch], pmxengch->bandBufInfo[nPrintPageIdx].bandImgInfo[nReadIdx].nImgPhyAddr);
	pmxeng_write_reg(PRIxBYTESPERLINE[nch], pmxengch->bandBufInfo[nPrintPageIdx].bandImgInfo[nReadIdx].nBytesPerLine);
	pmxeng_write_reg(PRIxLINESPERBAND[nch], pmxengch->bandBufInfo[nPrintPageIdx].bandImgInfo[nReadIdx].nLines);
	pmxeng_write_reg(PRIxSTRIDE[nch], pmxengch->bandBufInfo[nPrintPageIdx].bandImgInfo[nReadIdx].nBytesPerLine);

	
	nVal = 0L;
	nVal = pmxeng_set_val32(nVal, PRI0START__XFREN__SHIFT, PRI0START__XFREN__WIDTH, 1);
	nVal=  pmxeng_set_val32(nVal, PRI0START__SASEL1__SHIFT, PRI0START__SASEL1__WIDTH, 0);
	nVal = pmxeng_set_val32(nVal, PRI0START__SASEL2TN__SHIFT, PRI0START__SASEL2TN__WIDTH, 1); 
	pmxeng_write_reg(PRIxSTART[nch], nVal);

	return 0;
}

static int pmxeng_processing_vsync_isr(struct pmxengch_quasar *pmxengch)
{
	u32 val, val0;
	int nch = pmxengch->ech;
	u32 msgData[2] = { 0x80004000 , 0 	};

	if(nch < 0 || nch > 3)
	{
		printk("pmxeng: error, pmxeng_processing_vsync_isr:idx:%d\n", nch);
		return -1;
	}


	val = val0= pmxeng_read_reg(PRIxINTEN[nch]);
	pmxeng_write_reg(PRIxINTEN[nch], val);
	if(pmxeng_get_val32(val, PRI0INTEN__VSDI__SHIFT, PRI0INTEN__VSDI__WIDTH))
	{
		msgData[0] |= 0x01;
		// val = pmxeng_set_val32(val, PRI0INTEN__VSDI__SHIFT, PRI0INTEN__VSDI__WIDTH, 1);
	}
// 	else
	if(pmxeng_get_val32(val, PRI0INTEN__VII__SHIFT, PRI0INTEN__VII__WIDTH))
	{
		msgData[0] |= 0x02;
//		val = pmxeng_set_val32(val, PRI0INTEN__VII__SHIFT, PRI0INTEN__VII__WIDTH, 1);
	}
//	else
	if(pmxeng_get_val32(val, PRI0INTEN__VAI__SHIFT, PRI0INTEN__VAI__WIDTH))
	{
		msgData[0] |= 0x04;
//		val = pmxeng_set_val32(val, PRI0INTEN__VAI__SHIFT, PRI0INTEN__VAI__WIDTH, 1);
	}
//	pmxeng_write_reg(PRIxINTEN[nch], val);

	// vos_queueWrite(pmxengch->hQueue, msgData, 0);
	printk("pmxeng_processing_vsync_isr(0x%08X, 0x%08X, 0x%08X)\n", pmxengch->hQueue, msgData[0], val0);
	u32cPushData(pmxengch->hQueue, msgData[0]);

	return sizeof(msgData);
}

static irqreturn_t pmxeng_isr_vsync(int irq, void *dev_id)
{
	int ret = IRQ_HANDLED;
	struct pmxengch_quasar *pmxengch = (struct pmxengch_quasar *)dev_id;

	// pmxeng_processing_vsync_isr(pmxengch);

	ret = pmxeng_ch_interrupt(irq, dev_id, pmxeng_processing_vsync_isr, pmxengch);


	return ret;
}

static void pmxeng_init_prtreg(struct pmxengch_quasar *pmxengch)
{
	u32 val;
	pmxeng_retrv_regaddress(PRIS_BASE);
	pmxeng_retrv_regaddress(PRI0_BASE);
	pmxeng_retrv_regaddress(PRI2_BASE);
	pmxeng_retrv_regaddress(PRI4_BASE);
	pmxeng_retrv_regaddress(PRI6_BASE);

	val = pmxeng_read_reg(PRIxINTEN[pmxengch->ech]);
	printk("pmxeng: prtisr_setup(ch:%d, val:0x%08X)\n", pmxengch->ech, val);
	val = pmxeng_set_val32(val, PRI0INTEN__EVSDI__SHIFT, PRI0INTEN__EVSDI__WIDTH, 0);		// PRI0INTEN.EVSDI	EVSDI=0
	val = pmxeng_set_val32(val, PRI0INTEN__EVII__SHIFT, PRI0INTEN__EVII__WIDTH, 0);		// PRI0INTEN.EVII	EVII=0
	val = pmxeng_set_val32(val, PRI0INTEN__EVAI__SHIFT, PRI0INTEN__EVAI__WIDTH, 0);		// PRI0INTEN.EVAI	EVAI=0
	val = pmxeng_set_val32(val, PRI0INTEN__VSDI__SHIFT, PRI0INTEN__VSDI__WIDTH, 1);		// PRI0INTEN.VSDI	VSDI=1
	val = pmxeng_set_val32(val, PRI0INTEN__VII__SHIFT, PRI0INTEN__VII__WIDTH, 1);		// PRI0INTEN.VII	VII=1
	val = pmxeng_set_val32(val, PRI0INTEN__VAI__SHIFT, PRI0INTEN__VAI__WIDTH, 1);		// PRI0INTEN.VAI	VAI=1
	pmxeng_write_reg(PRIxINTEN[pmxengch->ech], val);
}


static int pmxeng_prtisr_setup(struct platform_device *pdev, struct pmxengch_quasar *pmxengch)
{
	int ret = -1;
	int isrNo;
	static int n = 0;
	static struct 
	{
		void *pfnIsrProc;
		const char *strIsrName;
	} aisrinfo[12] = 
	{
		{	(void *)pmxeng_isr_dmaend,		"pmxeng_ch0_isrDmaDone\0"			},
		{	(void *)pmxeng_isr_dmaerr,		"pmxeng_ch0_isrDmaErr\0"			},
		{	(void *)pmxeng_isr_vsync,		"pmxeng_ch0_isrVSync\0"				},

		{	(void *)pmxeng_isr_dmaend,		"pmxeng_ch1_isrDmaDone\0"			},
		{	(void *)pmxeng_isr_dmaerr,		"pmxeng_ch1_isrDmaErr\0"			},
		{	(void *)pmxeng_isr_vsync,		"pmxeng_ch1_isrVSync\0"				},

		{	(void *)pmxeng_isr_dmaend,		"pmxeng_ch2_isrDmaDone\0"			},
		{	(void *)pmxeng_isr_dmaerr,		"pmxeng_ch2_isrDmaErr\0"			},
		{	(void *)pmxeng_isr_vsync,		"pmxeng_ch2_isrVSync\0"				},

		{	(void *)pmxeng_isr_dmaend,		"pmxeng_ch3_isrDmaDone\0"			},
		{	(void *)pmxeng_isr_dmaerr,		"pmxeng_ch3_isrDmaErr\0"			},
		{	(void *)pmxeng_isr_vsync,		"pmxeng_ch3_isrVSync\0"				}
	};
	// int ech;
	// ech = pmxengch->ech;


	if(n >= 12)
	{
		n = 0;
	}

	for (isrNo = 0; isrNo < 3; isrNo++,n++)
	{
		pmxengch->irq = platform_get_irq(pdev, isrNo);
		if (pmxengch->irq < 0)
		{
			dev_dbg(&pdev->dev, "could not get irq\n");
			ret = -ENXIO;
			return ret;
		}

		ret = request_irq(
			pmxengch->irq, 
			aisrinfo[n].pfnIsrProc, /* pmxeng_ch_interrupt */
			0x00010000, /* 0, IRQF_NO_THREAD */
			aisrinfo[n].strIsrName,  /* "pmxengch",  */
			pmxengch);
		if (ret)
		{
			dev_dbg(&pdev->dev, "could not request irq %d\n", pmxengch->irq);
			pmxengch->irq = -1;
			return ret;
		}
	}

	return 0;
}

static struct file_operations g_quasar_pmxengch_ops = {
	.owner = THIS_MODULE,
	.open = pmxeng_channel_open,
	.release = pmxeng_channel_release,
	.read = pmxeng_channel_read,
	.write = pmxeng_channel_write,
	.poll = pmxeng_channel_poll,
	.mmap = pmxeng_channel_mmap,
	.unlocked_ioctl = pmxeng_channel_ioctl};

static struct class *g_pmxengch_class = NULL;
static dev_t g_dev_num;

static int __init pmxeng_probe(struct platform_device *pdev)
{
	struct pmxengch_quasar *pmxengch;
	struct resource *regs;
	dev_t pmxengchn;
	int ret, isrNo;
	static int npmxengchs = 0;
	int i;

	pmxengch = kzalloc(sizeof(*pmxengch), GFP_KERNEL);
	if (!pmxengch)
	{
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	pmxengch->ech = npmxengchs;

	for(i=0; i<MAX_PAGE_BUF_CNT; i++)
	{
		pmxengch->bandBufInfo[i].nMaxBandBufCnt = MAX_BANDBUF_CNT;
		pmxengch->bandBufInfo[i].nBandReadycnt = 0;
		pmxengch->bandBufInfo[i].nReadIdx = 0;
		pmxengch->bandBufInfo[i].nWriteIdx = 0;
		pmxengch->bandBufInfo[i].nSetupImgHeight = 0;
	}
	pmxengch->nSetupPageIdx = -1;
	pmxengch->nPrintPageIdx = -1;
	// pmxengchn = MKDEV(LPDMA_MAJOR, npmxengchs); // g_dev_num; // MKDEV(LPDMA_MAJOR, npmxengchs);
	if (NULL == g_pmxengch_class)
	{
		alloc_chrdev_region(&g_dev_num, 0, LPCHANNEL_NUM_CONTEXTS, QLPDMA_DEVNAME);
		// g_dev_num = pmxengchn;
		// register_chrdev_region(g_dev_num, LPCHANNEL_NUM_CONTEXTS, QLPDMA_DEVNAME);
		g_pmxengch_class = class_create(THIS_MODULE, "pmxengch_class");
		printk("pmxengch: g_dev_num:%d, major:%d\n", g_dev_num, MAJOR(g_dev_num));
	}
	pmxengchn = MKDEV(MAJOR(g_dev_num), npmxengchs);
	device_create(g_pmxengch_class, NULL, /* no parent device */
				  pmxengchn, NULL,		/* no external data */
				  QLPDMA_DEVNAME "%d", npmxengchs);

	cdev_init(&pmxengch->cdev, &g_quasar_pmxengch_ops);
	// pmxengchn = g_dev_num; // MKDEV(LPDMA_MAJOR, npmxengchs);
	ret = cdev_add(&pmxengch->cdev, pmxengchn, 1);
	if (ret)
	{
		dev_dbg(&pdev->dev, "could not create char dev %d\n", npmxengchs);
		goto out_err;
	}

	regs = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!regs)
	{
		dev_dbg(&pdev->dev, "no mmio reg resource defined\n");
		ret = -ENXIO;
		goto out_rerr;
	}
	pmxengch->ioreg_start = regs->start;
	pmxengch->ioreg_end = regs->end;

	pmxengch->regs = ioremap(regs->start, regs->end - regs->start + 1);
	dev_dbg(&pdev->dev, "map reg I/O memory, regs->start=0x%x\n", regs->start);
	printk("pmxeng: 2-map reg I/O memory, regs->start=0x%x\n", regs->start);
	if (!pmxengch->regs)
	{
		ret = -ENOMEM;
		dev_dbg(&pdev->dev, "could not map reg I/O memory\n");
		goto out_ioerr;
	}

	pmxeng_init_prtreg(pmxengch);
	pmxeng_init_msgqueue(pmxengch);
	ret = pmxeng_prtisr_setup(pdev, pmxengch);
	if(0 != ret)
	{
		printk("pmxeng: error, printing isr setup failed\n");
		goto out_ioerr;		
	}

	spin_lock_init(&pmxengch->lock);
	init_waitqueue_head(&pmxengch->mbq);

	platform_set_drvdata(pdev, pmxengch);
	printk("pmxeng: channel %d - mapped at %p, irq %d\n",
		   npmxengchs, pmxengch->regs, pmxengch->irq);
	s_pmxengch_context[npmxengchs] = pmxengch;
	// pmxengch->ech = npmxengchs++;
	npmxengchs++;
	device_init_wakeup(&pdev->dev, 1);

	printk("pmxeng: dev name:%s\n", pdev->name);
	return 0;


out_ioerr:
	iounmap(pmxengch->regs);
out_rerr:
	cdev_del(&pmxengch->cdev);
out_err:
	kfree(pmxengch);
	return ret;
}

#if 0
static int __init pmxeng_probe__bob(struct platform_device *pdev)
{
	struct pmxengch_quasar *pmxengch;
	struct resource *regs;
	dev_t pmxengchn;
	int ret, isrNo;
	static int npmxengchs = 0;

	pmxengch = kzalloc(sizeof(*pmxengch), GFP_KERNEL);
	if (!pmxengch)
	{
		dev_dbg(&pdev->dev, "out of memory\n");
		return -ENOMEM;
	}
	cdev_init(&pmxengch->cdev, &g_quasar_pmxengch_ops);
	pmxengchn = MKDEV(LPDMA_MAJOR, npmxengchs);
	ret = cdev_add(&pmxengch->cdev, pmxengchn, 1);
	if (ret)
	{
		dev_dbg(&pdev->dev, "could not create char dev %d\n", npmxengchs);
		goto out_err;
	}
	regs = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!regs)
	{
		dev_dbg(&pdev->dev, "no mmio reg resource defined\n");
		ret = -ENXIO;
		goto out_rerr;
	}
	pmxengch->ioreg_start = regs->start;
	pmxengch->ioreg_end = regs->end;

	pmxengch->regs = ioremap(regs->start, regs->end - regs->start + 1);
	dev_dbg(&pdev->dev, "map reg I/O memory, regs->start=0x%x\n", regs->start);
	printk("2-map reg I/O memory, regs->start=0x%x\n", regs->start);
	if (!pmxengch->regs)
	{
		ret = -ENOMEM;
		dev_dbg(&pdev->dev, "could not map reg I/O memory\n");
		goto out_ioerr;
	}

	for (isrNo = 0; isrNo < 3; isrNo++)
	{
		pmxengch->irq = platform_get_irq(pdev, isrNo);
		if (pmxengch->irq < 0)
		{
			dev_dbg(&pdev->dev, "could not get irq\n");
			ret = -ENXIO;
			goto out_ioerr;
		}

		ret = request_irq(
			pmxengch->irq, pmxeng_ch_interrupt, 0, "pmxengch",
			pmxengch);
		if (ret)
		{
			dev_dbg(&pdev->dev, "could not request irq %d\n", pmxengch->irq);
			pmxengch->irq = -1;
			goto out_ioerr;
		}
	}

	spin_lock_init(&pmxengch->lock);
	init_waitqueue_head(&pmxengch->mbq);

	platform_set_drvdata(pdev, pmxengch);
	printk("pmxeng_channel %d - mapped at %p, irq %d\n",
		   npmxengchs, pmxengch->regs, pmxengch->irq);
	s_pmxengch_context[npmxengchs] = pmxengch;
	pmxengch->unit = npmxengchs++;
	device_init_wakeup(&pdev->dev, 1);
	printk("primax engine dev name:%s\n", pdev->name);
	return 0;
out_ioerr:
	iounmap(pmxengch->regs);
out_rerr:
	cdev_del(&pmxengch->cdev);
out_err:
	kfree(pmxengch);
	return ret;
}

#endif

static int __exit pmxeng_remove(struct platform_device *pdev)
{
	struct pmxengch_quasar *pmxengch = platform_get_drvdata(pdev);

	if (pmxengch->irq > 0)
		free_irq(pmxengch->irq, pmxengch);
	iounmap(pmxengch->regs);
	cdev_del(&pmxengch->cdev);
	kfree(pmxengch);
	platform_set_drvdata(pdev, NULL);
	return 0;
}

static int pmxeng_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct pmxengch_quasar *pmxengch = platform_get_drvdata(pdev);

	return 0;
}

static int pmxeng_resume(struct platform_device *pdev)
{
	struct pmxengch_quasar *pmxengch = platform_get_drvdata(pdev);

	return 0;
}

/* MODULE_ALIAS("pmx:engine-driver"); */
MODULE_ALIAS("platform:quasar-pmxengch");

static const struct of_device_id pmxeng_id_table[] = {
	{.compatible = "qbit,quasar-pmxengch"},
	{}};
MODULE_DEVICE_TABLE(of, pmxeng_id_table);

static struct platform_driver pmxeng_driver_ops = {
	.probe = pmxeng_probe,
	.remove = pmxeng_remove,
	.suspend = pmxeng_suspend,
	.resume = pmxeng_resume,
	.driver = {
		.name = QLPDMA_DEVNAME, /* "primax-engine", */
		.owner = THIS_MODULE,
		.of_match_table = of_match_ptr(pmxeng_id_table),
	},
};

static int __init pmx_engdrv_init(void)
{
	int ret;

	ret = platform_driver_register(&pmxeng_driver_ops);
	return ret;
}
module_init(pmx_engdrv_init);

static void __exit pmx_engdrv_exit(void)
{
	platform_driver_unregister(&pmxeng_driver_ops);
}
module_exit(pmx_engdrv_exit);

MODULE_DESCRIPTION("Primax engine driver");
MODULE_LICENSE("Dual BSD/GPL");
