/*=================================================================================================
Filename:    vos_task.c
Description: abstraction functions of Linux kernel 
Data:        Initial revision. Bob@2019-12-05
==================================================================================================*/
#ifdef __cplusplus 
    extern "C" { 
#endif

#include "vos_if.h"
#include "vos_innr.h"

//#define __msecs_to_jiffies msecs_to_jiffies
typedef void (*_TimeFunc)(struct timer_list *);


HVOSDLK g_hTimeDlk = 0;
void vos_timeFunc(struct timer_list * par);

//platform_sys\build-bundle\poky\build\tmp\work-shared\simva32\kernel-source\kernel\time\Timer.c

void vos_timerInit(void)
{
	g_hTimeDlk = vos_dlkCreate(sizeof(Timer_ITEM_S), 1, NULL);
	VOS_ASSERT(g_hTimeDlk != 0);

	//init_timers();
	return;
}

HVOSTIMER vos_timerCreate(char * Name, TimeFunc pfFunc, void *pParam, int nMilliSec, int nInterMilliSec, int autoAct)
{
	struct timer_list *pTimer=NULL;
	Timer_ITEM_S sTimerItem = {0,};
	Timer_ITEM_S *pTimerItem = &sTimerItem;
	HVOSTIMER vTimerId = 0;
	int nRet=0, i;

	pTimer = (struct timer_list *)vos_malloc(sizeof(struct timer_list));
	VOS_ASSERT(pTimer != NULL);	
	memset(pTimer, 0, sizeof(struct timer_list));
	//init_timer(pTimer);
	timer_setup(pTimer, NULL, 0);

	pTimer->expires = __msecs_to_jiffies(nMilliSec);
	pTimer->function = (_TimeFunc)pfFunc;
	
	pTimerItem->pTimer = pTimer;
	pTimerItem->nMilliSec = nMilliSec;
	pTimerItem->nInterMilliSec = nInterMilliSec;
	pTimerItem->pfFunc = pfFunc;
	pTimerItem->param = (unsigned long)pParam;
	pTimerItem->vosType = VOS_TIMER_TYPE;
	if(Name != NULL)
	{
		strncpy(pTimerItem->sName, Name, 11);
		pTimerItem->sName[11] = 0;
	}
	
	vTimerId = (HVOSTIMER)vos_dlkInsertData(g_hTimeDlk, (void *)pTimerItem);
	vosPrintf("vos_timerCreate: timerId=0x%x(0x%x)\r\n", pTimer, vTimerId);

	if(autoAct)
	{
		//pTimer->data = (unsigned long)vTimerId;
		pTimer->expires = jiffies + __msecs_to_jiffies(nMilliSec);

		add_timer(pTimer);
	}

	return vTimerId;
}

void vos_timeFunc(struct timer_list *par)
{
	struct timer_list *pTimer=NULL;
	Timer_ITEM_S *pTimerItem;
	HVOSTIMER vTimerId;

	vTimerId = (HVOSTIMER)par;
	pTimerItem = (Timer_ITEM_S *)vTimerId;
	VOS_ASSERT(pTimerItem->vosType == VOS_TIMER_TYPE);

	pTimer = pTimerItem->pTimer;
	VOS_ASSERT(NULL != pTimer);

	//vosPrintf("vos_timeFunc(0x%x): param=0x%x, nInterMilliSec=%u\n", vTimerId, pTimerItem->param, pTimerItem->nInterMilliSec);
	if(pTimerItem->pfFunc)
		pTimerItem->pfFunc(pTimerItem->param);

	if(pTimerItem->nInterMilliSec)
		mod_timer(pTimer, jiffies + __msecs_to_jiffies(pTimerItem->nMilliSec));

	return;
}

int vos_timerEnable(HVOSTIMER vTimerId)
{
	struct timer_list *pTimer=NULL;
	Timer_ITEM_S *pTimerItem;
	BOOL isRepeted = 1;

	pTimerItem = (Timer_ITEM_S *)vTimerId;
	VOS_ASSERT(pTimerItem->vosType == VOS_TIMER_TYPE);
	pTimer = pTimerItem->pTimer;
	VOS_ASSERT(NULL != pTimer);
	
	//add_timer(pTimer);
	//pTimer->data = (unsigned long)vTimerId;

	//mod_timer = del_timer + add_timer
	mod_timer(pTimer, jiffies + __msecs_to_jiffies(pTimerItem->nMilliSec));

	return VOS_OK;
}

int vos_timerDisble(HVOSTIMER vTimerId)
{
	struct timer_list *pTimer=NULL;
	Timer_ITEM_S *pTimerItem;
	int nRet;

	pTimerItem = (Timer_ITEM_S *)vTimerId;
	VOS_ASSERT(pTimerItem->vosType == VOS_TIMER_TYPE);
	pTimer = pTimerItem->pTimer;
	VOS_ASSERT(NULL != pTimer);

	nRet = del_timer(pTimer);

	return nRet;
}

int vos_timerDelete(HVOSTIMER vTimerId)
{
	struct timer_list *pTimer=NULL;
	Timer_ITEM_S *pTimerItem;
	int nRet;

	nRet = vos_timerDisble(vTimerId);
	if(nRet != 0)return nRet;

	pTimerItem = (Timer_ITEM_S *)vTimerId;
	VOS_ASSERT(pTimerItem->vosType == VOS_TIMER_TYPE);
	pTimer = pTimerItem->pTimer;
	VOS_ASSERT(NULL != pTimer);

	vos_free(pTimer);
	vos_dlkDelete(g_hTimeDlk, (DLK_NODE *)vTimerId-1);

	return VOS_OK;
}

int vos_timerGetRemainMs(HVOSTIMER vTimerId)
{
	struct timer_list *pTimer=NULL;
	Timer_ITEM_S *pTimerItem;
	int nMilliSec=0;
	int nRet, i;

	pTimerItem = (Timer_ITEM_S *)vTimerId;
	VOS_ASSERT(pTimerItem->vosType == VOS_TIMER_TYPE);
	pTimer = pTimerItem->pTimer;
	VOS_ASSERT(NULL != pTimer);

	//nMilliSec = its.it_value.tv_sec*1000 + (its.it_value.tv_nsec/1000000);
	return nMilliSec;
}

int vos_timerShowAll(void)
{
	int index = 0;
	struct timer_list *pTimer=NULL;
	Timer_ITEM_S *pTimerItem;
	HVOSTIMER vTimerId;
	DLK_NODE *pShow, *pHead;

	vosPrintf("%-4s %-8s   %-8s      %-12s %-10s %-8s  %-10s %-8s %-8s\r\n", \
		"No.", "TimerId", "vTid", "name", "Func", "MilliSec", "Param", "repeatMs", "expires");
    vosPrintf("--------------------------------------------------------------------------------------------\r\n");

	
	pHead = vos_dlkGetHead(g_hTimeDlk);
	DLK_WALK_LOOP_START(pShow, pHead)
	{
		pTimerItem = (Timer_ITEM_S *)(pShow + 1);
		pTimer = pTimerItem->pTimer;
		if(pTimer != 0)
		{
			vosPrintf("%-4d 0x%-8x 0x%-8x    %-12s 0x%-8x %-8d  0x%-8x %-8d %-8d\r\n", \
				index, pTimer, pTimerItem, pTimerItem->sName, pTimerItem->pfFunc, pTimerItem->nMilliSec, pTimerItem->param, pTimerItem->nInterMilliSec, pTimer->expires);
			index++;
		}
	}
	DLK_WALK_LOOP_END(pShow, pHead)
	
	return index;
}

void test_timeFunc(struct timer_list * pParam)
{
	static int nTimerCnt = 0;

	nTimerCnt++;
	vosPrintf("test_timeFunc, pParam=0x%x, nTimerCnt=%d\r\n", pParam, nTimerCnt);
	return;
}

int test_time(int nFlag, int nPar)
{
	HVOSTIMER hVtimer = (HVOSTIMER)nPar;
	int nMilliSec = nPar;
	int nParam=0x1234;
	int nRet=0;

	switch(nFlag)
	{
		case 0:
			nRet = vos_timerShowAll();
			break;
			
		case 1:
			if(nMilliSec == 0)nMilliSec=1000;
			hVtimer = vos_timerCreate("timer1", (TimeFunc)test_timeFunc, (void *)nParam, nMilliSec, 0, 1);
			vosPrintf("vos_timerCreate ret=0x%x\r\n", hVtimer);
			break;

		case 11:
			if(nMilliSec == 0)nMilliSec=1000;
			hVtimer = vos_timerCreate("timer2", (TimeFunc)test_timeFunc, (void *)(nParam+1), nMilliSec, nMilliSec, 0);
			vosPrintf("vos_timerCreate ret=0x%x\r\n", hVtimer);
			break;

		case 2:
			nRet = vos_timerEnable(hVtimer);
			break;

		case 3:
			nRet = vos_timerDisble(hVtimer);
			break;

		case 4:
			nRet = vos_timerDelete(hVtimer);
			break;

		default:
			break;
	}

    return nRet;
}


#ifdef __cplusplus 
    }
#endif

