/*=================================================================================================
Filename:    vos_task.c
Description: abstraction functions of Linux kernel 
Data:        Initial revision. Bob@2019-12-05
==================================================================================================*/
#ifdef __cplusplus 
    extern "C" { 
#endif

#include "vos_if.h"
#include "vos_innr.h"

HVOSDLK g_hTaskDlk = 0;

void vos_taskInit(void)
{
	g_hTaskDlk = vos_dlkCreate(sizeof(TASK_ITEM_S), 1, NULL);
	return;
}

//build-bundle\poky\build\tmp\work-shared\simva32\kernel-source\kernel\kthread.c
HVOSTASK vos_taskCreate(void *pfTask, void *pParam, char * szTskName)
{
	TASK_ITEM_S sTaskItem = {0, };
	HVOSTASK nVtaskID = 0;
	struct task_struct *pTask;
	
	if((szTskName == NULL) || (pfTask == NULL))return NULL;

	pTask = kthread_run(pfTask, pParam, szTskName);
	VOS_ASSERT(pTask != NULL);

	sTaskItem.nFunAddr = (unsigned long)pfTask;
	sTaskItem.nTaskID = pTask;
	strncpy(sTaskItem.sName, szTskName, 31);
	sTaskItem.vosType = VOS_TASK_TYPE;
	
	nVtaskID = (HVOSTASK)vos_dlkInsertData(g_hTaskDlk, (void *)&sTaskItem);
	vosPrintf("%s: nTaskID=0x%x(0x%x)\r\n", __func__, nVtaskID);
	
	return nVtaskID;
}

int vos_taskStop(HVOSTASK nVtaskID)
{
	int nRet = 0;
	struct task_struct *pTask;
	TASK_ITEM_S * pTaskItem;

	pTaskItem = (TASK_ITEM_S *)nVtaskID;
	VOS_ASSERT(pTaskItem->vosType == VOS_TASK_TYPE);
	
	pTask = pTaskItem->nTaskID;
	if(pTask == NULL)return 1;
	nRet = kthread_stop(pTask);
	return nRet;
}

int vos_taskCancel(HVOSTASK nVtaskID)
{
	struct task_struct *pTask;
	int nRet = 0;
	TASK_ITEM_S * pTaskItem;

	pTaskItem = (TASK_ITEM_S *)nVtaskID;
	VOS_ASSERT(pTaskItem->vosType == VOS_TASK_TYPE);

	pTask = pTaskItem->nTaskID;
	if(pTask == 0)
	{
		vosPrintf("vos_taskCancel: 0x%x is not exit\r\n", nVtaskID);
		return VOS_ERROR;
	}
	
	nRet = vos_taskStop(nVtaskID);
	VOS_ERROR_RPT(nRet);
	//if(nRet == 0)
	{
		memset(pTaskItem, 0, sizeof(TASK_ITEM_S));
	}
	
	return VOS_OK; //it return OK always.
}

int vos_taskShowTask(HVOSTASK nVtaskID)
{
	int nRet = 0;
	struct task_struct *pTask;
	TASK_ITEM_S * pTaskItem;

	pTaskItem = (TASK_ITEM_S *)nVtaskID;
	VOS_ASSERT(pTaskItem->vosType == VOS_TASK_TYPE);
	
	pTask = pTaskItem->nTaskID;
	if(pTask == NULL)return 1;

	vosPrintf("name=%s, id=0x%x(0x%x)\r\n", pTask->comm, nVtaskID, pTask);
	vosPrintf("stack=0x%x, usage=%d, flags=%d, ptrace=%d\r\n", pTask->stack, pTask->usage, pTask->flags, pTask->ptrace);
	vosPrintf("prio=%d, static_prio=%d, normal_prio=%d, rt_priority=%d\r\n", pTask->prio, pTask->static_prio, pTask->normal_prio, pTask->rt_priority);
	return 0;
}

int vos_taskShowAllTask(void)
{
	int nIndex=0;
	TASK_ITEM_S * pTaskItem;
	struct task_struct *pTask;
	DLK_NODE *pShow, *pHead;

	vosPrintf("%-4s %-12s   %-20s %-12s %-12s %-4s\r\n", \
			"No.", "vTaskId", "Name", "Func", "TaskId", "state");
	vosPrintf("-------------------------------------------------------------------------\r\n");

	pHead = vos_dlkGetHead(g_hTaskDlk);
	DLK_WALK_LOOP_START(pShow, pHead)
	{
		pTaskItem = (TASK_ITEM_S *)(pShow + 1);
		pTask = pTaskItem->nTaskID;
		if(pTask != 0)
		{
			nIndex++;
			vosPrintf("%-4d 0x%-12x %-20s 0x%-10x 0x%-10x\r\n", \
				nIndex, pTaskItem,pTaskItem->sName, pTaskItem->nFunAddr, pTaskItem->nTaskID);
		}
	}
	DLK_WALK_LOOP_END(pShow, pHead)
	return nIndex;
}


void vos_taskSleep(unsigned long ulMilliSeconds)
{
	msleep(ulMilliSeconds); //usleep(p1); nsleep(p1);
	return;
}

void usleep(unsigned int usecs)
{
	unsigned long timeout = usecs_to_jiffies(usecs) + 1;

	while (timeout)
		timeout = schedule_timeout_uninterruptible(timeout);
}

void vos_taskSleepUs(unsigned long ulUs)
{
	//usleep_range(ulUs, ulUs+2);
	usleep(ulUs);
	return;
}

void vos_taskDelay(unsigned long ulMilliSeconds)
{
	mdelay(ulMilliSeconds); //udelay(p1); ndelay(p1);
	return;
}

void vos_taskDelayUs(unsigned long ulUs)
{
	udelay(ulUs); //ndelay(p1);
	return;
}


EXPORT_SYMBOL(vos_taskCreate);
EXPORT_SYMBOL(vos_taskSleep);
EXPORT_SYMBOL(vos_taskSleepUs);
EXPORT_SYMBOL(vos_taskDelay);
EXPORT_SYMBOL(vos_taskDelayUs);

TASK_CK_S g_sTaskCallBack = {0,};
int vos_kthread( void *data )
{
	unsigned long ret, t1, t2;
	int index = (int)data;

	vosPrintf("vos_kthread: data=0x%x\r\n", (unsigned long)data);

	while(1)
	{
		vos_taskSleep(10);

		if(g_sTaskCallBack.pFunc != NULL)
		{
			vosPrintf("call(0x%x) at vos_kthread\r\n", g_sTaskCallBack.pFunc);
			t1 = vos_GetSysCntUs();
			ret = g_sTaskCallBack.pFunc(g_sTaskCallBack.par1, g_sTaskCallBack.par2, g_sTaskCallBack.par3);
			t2 = vos_GetSysCntUs();
			vosPrintf("return:0x%x, take %uus\r\n", ret, t2-t1);
			
			g_sTaskCallBack.pFunc = NULL; //only run 1 time
		}
	}
	return 0;
}

void vos_taskRun(pTaskFunc pFunc, int par1, int par2, int par3)
{
	g_sTaskCallBack.pFunc = pFunc;
	g_sTaskCallBack.par1 = par1;
	g_sTaskCallBack.par2 = par2;
	g_sTaskCallBack.par3 = par3;
}
EXPORT_SYMBOL(vos_taskRun);

int test_task(int type, int p1, int p2)
{
	int nRet = 0;
	struct task_struct *pTask;
	char taskName[16] = {0,};
	unsigned long ulVal = 0;

	switch(type)
	{
		case 0:
			vos_taskShowAllTask();
			break;
			
		case 1:
			pTask = vos_taskCreate((void *)vos_kthread, (void *)0, "vos_kthread_0");
			pTask = vos_taskCreate((void *)vos_kthread, (void *)1, "vos_kthread_1");
			printk( KERN_INFO "%s=%p\n", "vos_kthread", pTask );	
			break;

		case 2:
			nRet = vos_taskCancel((HVOSTASK)p1);
			printk( KERN_INFO "vos_taskCancel\n");	
			break;

		case 3:
			printk("vos_taskSleep: %u\r\n", vos_GetSysCntUs());
			vos_taskSleep(p1); 
			printk("vos_taskSleep: %u\r\n", vos_GetSysCntUs());
			break;
		case 4:
			printk("vos_taskDelay: %u\r\n", vos_GetSysCntUs());
			vos_taskDelay(p1);
			printk("vos_taskDelay: %u\r\n", vos_GetSysCntUs());
			break;
		default:
			printk("unknown: test_task(%u, %u, %u\r\n)", type, p1, p2);
			break;
	}
	return nRet;
}

#ifdef __cplusplus 
    }
#endif

