/*=================================================================================================
Filename:    vos_queue.c
Description: abstraction functions of Linux kernel 
Data:        Initial revision. Bob@2019-12-05
==================================================================================================*/
#ifdef __cplusplus 
    extern "C" { 
#endif

#include "vos_if.h"
#include "vos_innr.h"

HVOSDLK g_hQueDlk = 0;

void vos_queInit()
{
	g_hQueDlk = vos_dlkCreate(sizeof(Que_ITEM_S), 1, NULL);
	return;
}

//build-bundle\poky\build\tmp\work-shared\simva32\kernel-source\lib\kfifo.c
HVOSQUEUE vos_queueCreate(char * strQueName, int nMaxCounter, int nMaxItemSize)
{
	int i, nRet;
	Que_ITEM_S sQueItem = {0,};
	Que_ITEM_S *pQueItem = &sQueItem;
	HVOSQUEUE hVosQueId;
	char strName[12] = {'/', 0, };

	VOS_ASSERT(strQueName != NULL);
	strncpy(strName+1, strQueName, 10);
	strName[11] =0;

	strncpy(pQueItem->sName, strQueName, 11);
	pQueItem->nMaxCounter = (unsigned short)nMaxCounter;
	pQueItem->nMaxItemSize = (unsigned short)nMaxItemSize;
	pQueItem->vosType = VOS_MSGQUE_TYPE;

	nRet = kfifo_alloc(&pQueItem->fifo,  roundup_pow_of_two(nMaxCounter*nMaxItemSize), GFP_KERNEL );
    if( nRet != 0 )return 0;

	spin_lock_init( &pQueItem->lock);
	pQueItem->sem = vos_semCreate(strName, 0);
	VOS_ASSERT(pQueItem->sem != 0);
	
	hVosQueId = (HVOSQUEUE)vos_dlkInsertData(g_hQueDlk, (void *)pQueItem);
	vosPrintf("vos_queueCreate: QueId=0x%x\r\n", hVosQueId);

	return hVosQueId;
}

int vos_queueWrite(HVOSQUEUE hQueue, void *pMsg, int nWaitForMSec)
{
	Que_ITEM_S *pQueItem;
	int nRet, nWaitCnt = 0, nTimeOut = 0;;
	
	pQueItem = (Que_ITEM_S *)hQueue;
	VOS_ASSERT(pQueItem->vosType == VOS_MSGQUE_TYPE);
	VOS_ASSERT(pQueItem->sem != 0);

	vos_dbgPrint(1, "%s: hQueue=0x%x, nWaitForMSec=0x%x\r\n", __func__, hQueue, nWaitForMSec);
	while(vos_semGetCnt(pQueItem->sem) >= pQueItem->nMaxCounter)
	{
		vos_taskSleep(1);
		nWaitCnt++;
		if((nWaitForMSec != VOS_WAIT_FOREVER) && (nWaitCnt > nWaitForMSec))
		{
			return -1;
		}
	}
	
	nRet = kfifo_in_spinlocked(&pQueItem->fifo, pMsg, pQueItem->nMaxItemSize, &pQueItem->lock);
	//nRet = __kfifo_put(&pQueItem->fifo, pMsg, pQueItem->nMaxItemSize);
	vos_semGive(pQueItem->sem);
	
	return nRet;
}

int vos_queueRead(HVOSQUEUE hQueue, void *pMsg, int nWaitForMSec)
{
	Que_ITEM_S *pQueItem;
	int nRet;

	vos_dbgPrint(1, "%s: hQueue=0x%x, nWaitForMSec=0x%x\r\n", __func__, hQueue, nWaitForMSec);
	pQueItem = (Que_ITEM_S *)hQueue;
	VOS_ASSERT(pQueItem->vosType == VOS_MSGQUE_TYPE);
	VOS_ASSERT(pQueItem->sem != 0);

	nRet = vos_semTake(pQueItem->sem, nWaitForMSec);
	if(nRet != 0)return 0;

	nRet = kfifo_out_spinlocked(&pQueItem->fifo, pMsg, pQueItem->nMaxItemSize, &pQueItem->lock);
	//nRet = __kfifo_get(&pQueItem->fifo, pMsg, pQueItem->nMaxItemSize);
	return nRet;
}

int vos_queueClose(HVOSQUEUE hQueue)
{
	Que_ITEM_S *pQueItem;
	int nRet;
	
	pQueItem = (Que_ITEM_S *)hQueue;
	VOS_ASSERT(pQueItem->vosType == VOS_MSGQUE_TYPE);
	VOS_ASSERT(pQueItem->sem != 0);
	
	kfifo_free(&pQueItem->fifo);

	memset(pQueItem, 0, sizeof(Que_ITEM_S));
	vos_dlkDelete(g_hQueDlk, (DLK_NODE *)hQueue-1);
	
	return 0;
}

int vos_queueGetMsgCnt(HVOSQUEUE hQueue)
{
    Que_ITEM_S *pQueItem;
    int nMsgCnt = 0;
    int nRet = 0;

    vos_dbgPrint(1, "%s: hQueue=0x%x, nWaitForMSec=0x%x\r\n", __func__, hQueue);
	pQueItem = (Que_ITEM_S *)hQueue;
	VOS_ASSERT(pQueItem->vosType == VOS_MSGQUE_TYPE);
	VOS_ASSERT(pQueItem->sem != 0);
    
    nMsgCnt = vos_semGetCnt(pQueItem->sem);
    return nMsgCnt;
}

int vos_queueClear(HVOSQUEUE hQueue)
{
	int nMsgCnt = 0, nRet = 0;
	void * pBuff = NULL;

	pBuff = (void *)vos_malloc(0x10000);
	if(pBuff == NULL)return 2;
	
	nMsgCnt = vos_queueGetMsgCnt(hQueue);

	while(nMsgCnt-- > 0)
	{
		(void)vos_queueRead(hQueue, pBuff, 50);
	}

	vos_free(pBuff);

	return VOS_OK;
}

int vos_queueShowAll(void)
{
	Que_ITEM_S *pQueItem;
	int nIndex = 1;
	DLK_NODE *pShow, *pHead;

	vosPrintf("%-4s %-10s   %-12s %-12s %-10s %-8s\r\n", \
			"No.", "nVqueID", "Name", "MaxQueNum", "MsgSize", "MsgCnt");
	vosPrintf("----------------------------------------------------------------\r\n");

	pHead = vos_dlkGetHead(g_hQueDlk);
	DLK_WALK_LOOP_START(pShow, pHead)
	{
		pQueItem = (Que_ITEM_S *)(pShow + 1);
		if(pQueItem->sem != 0)
		{
			vosPrintf("%-4d 0x%-10x %-12s %-12d %-10d %-8d\r\n", \
				nIndex, pQueItem, pQueItem->sName, pQueItem->nMaxCounter, pQueItem->nMaxItemSize,vos_semGetCnt(pQueItem->sem));
			nIndex++;
		}
	}
	DLK_WALK_LOOP_END(pShow, pHead)
	return nIndex;
}

char * vos_queueGetName(HVOSQUEUE hQueue)
{
	Que_ITEM_S *pQueItem;

	pQueItem = (Que_ITEM_S *)hQueue;
	VOS_ASSERT(pQueItem->vosType == VOS_MSGQUE_TYPE);
	VOS_ASSERT(pQueItem->sem != 0);

	return pQueItem->sName;
}

EXPORT_SYMBOL(vos_queueCreate);
EXPORT_SYMBOL(vos_queueWrite);
EXPORT_SYMBOL(vos_queueRead);
EXPORT_SYMBOL(vos_queueClose);
EXPORT_SYMBOL(vos_queueClear);
EXPORT_SYMBOL(vos_queueGetMsgCnt);

static HVOSQUEUE g_ukQue[4] = {0,}; //queue between user space and kernel space
int vos_kqueRecv(void *pQueMsg, int index)
{
	char queName[32] = {0,};
	
	if(index >=4)return -1;
	if(g_ukQue[index] == 0)
	{
		sprintf(queName, "ukque_%d", index);
		g_ukQue[index] = vos_queueCreate(queName, 128, KQUE_FIXED_SIZE);
	}

	vos_queueRead(g_ukQue[index], pQueMsg, VOS_WAIT_FOREVER);
	return 0;
}

int vos_kqueSend(void *pQueMsg, int index)
{
	char queName[32] = {0,};

	if(index >=4)return -1;
	if(g_ukQue[index] == 0)
	{
		sprintf(queName, "ukque_%d", index);
		g_ukQue[index] = vos_queueCreate(queName, 128, KQUE_FIXED_SIZE);
	}

	vos_queueWrite(g_ukQue[index], pQueMsg, VOS_WAIT_FOREVER);
	return 0;
}
EXPORT_SYMBOL(vos_kqueRecv);
EXPORT_SYMBOL(vos_kqueSend);

HVOSQUEUE g_hQid1 = 0;
static int test_queTask( void *data )
{
	typedef int ( * PFunc)(int, int, int, int, int, int);
	int nRet, nCnt = 0;
	unsigned long t1,t2;
	VosMsg_S sMsg;
	PFunc pFunc = NULL;

	vosPrintf("vos_thread: data=0x%x\r\n", (unsigned long)data);
	g_hQid1 = vos_queueCreate("msgque1", 8, sizeof(VosMsg_S));

	while(1)
	{
		memset(&sMsg, 0, sizeof(VosMsg_S));
		nRet = vos_queueRead(g_hQid1, &sMsg, VOS_WAIT_FOREVER);
		nCnt++;
		vosPrintf("vos_queueRead%d: 0x%x, 0x%x, 0x%x. nRet=%d\r\n", nCnt, sMsg.nCmd1, sMsg.nCmd2, sMsg.nData[0], nRet);
		switch(sMsg.nCmd1)
		{
			case 1:
				nRet = (int)vos_malloc(sMsg.nCmd2<< 16);
				vosPrintf("malloc(%d), return=0x%x\r\n", sMsg.nCmd2<<16, nRet);
				break;
			case 2:
				nRet = (int)vos_kmalloc(sMsg.nCmd2<<16, __FILE__, __LINE__);
				vosPrintf("kmalloc(%d), return=0x%x\r\n", sMsg.nCmd2<<16, nRet);
				break;

			case 0x10:
				pFunc = (PFunc)sMsg.nCmd2;
				if(pFunc == 0)break;

				t1 = vos_GetSysCntUs();
				nRet = pFunc(sMsg.nData[0], sMsg.nData[1], sMsg.nData[2], sMsg.nData[3], sMsg.nData[4], sMsg.nData[5]);
				t2 = vos_GetSysCntUs();
				vosPrintf("Run (0x%x), return 0x%08x, take %uus\r\n", pFunc, nRet, t2-t1);
				break;
				
			default:
				break;
		}
	}
	return 0;
}

int test_que(int nFlag, HVOSQUEUE hQid, int nPar, int nPar2)
{
	HVOSQUEUE hQid1, hQid2;
	VosMsg_S sMsg;
	int nRet, nVal, nWaitForMSec;

	switch(nFlag)
	{
		case 0:
			nRet = vos_queueShowAll();
			break;

		case 1:
			hQid1 = vos_queueCreate("msgque1", 8, sizeof(VosMsg_S));
			hQid2 = vos_queueCreate("msgque2", 64, sizeof(VosMsg_S));
			break;

		case 2:
			nWaitForMSec = nPar;
			if(nPar == 0)nWaitForMSec = VOS_WAIT_FOREVER;
			sMsg.nCmd1 = nPar;
			sMsg.nCmd2 = nPar2;
			sMsg.nData[0] = 0x33445566;
			nVal = vos_GetSysCntMs();			
			nRet = vos_queueWrite(hQid, &sMsg, nWaitForMSec);
			vosPrintf("vos_queueWrite nRet=%d, take %dms\r\n", nRet, vos_GetSysCntMs()-nVal);
			break;

		case 3:
			nWaitForMSec = nPar;
			if(nPar == 0)nWaitForMSec = VOS_WAIT_FOREVER;
			memset(&sMsg, 0, sizeof(VosMsg_S));
			nVal = vos_GetSysCntMs();			
			nRet = vos_queueRead(hQid, &sMsg, nWaitForMSec);
			vosPrintf("vos_queueRead: 0x%x, 0x%x, 0x%x. nRet=%d, task %dms\r\n", sMsg.nCmd1, sMsg.nCmd2, sMsg.nData[0], nRet, vos_GetSysCntMs()-nVal);
			break;
	
		case 4:
			nRet = vos_queueClose(hQid);
			break;

		case 5:
			vos_queueClear(hQid);
			break;

		case 6:
			(void)vos_taskCreate((void *)test_queTask, (void *)nPar, "test_queTask");
			break;
			
		default:
			break;
	}
	
   return 0;
}

int vos_runFunc(int nFunAddr, int nPar1, int nPar2, int nPar3, int nPar4, int nPar5, int nPar6)
{
	int nRet;
	VosMsg_S sMsg;
	
	if(g_hQid1 == 0)
	{
		(void)vos_taskCreate((void *)test_queTask, 0, "kTask_run");
		while(g_hQid1 == 0)vos_taskSleep(10);
	}

	sMsg.nCmd1 = 0x10;
	sMsg.nCmd2 = nFunAddr;
	sMsg.nData[0] = nPar1;
	sMsg.nData[1] = nPar2;
	sMsg.nData[2] = nPar3;
	sMsg.nData[3] = nPar4;
	sMsg.nData[4] = nPar5;
	sMsg.nData[5] = nPar6;
	
	nRet = vos_queueWrite(g_hQid1, &sMsg, VOS_WAIT_FOREVER);
	return nRet;
}


#ifdef __cplusplus 
    }
#endif

