/*=================================================================================================
Filename:	 vos_mtmPool.c
Description: multi memory pool management 
Data:		  Initial revision. Bob@2022-03-28
==================================================================================================*/
#ifdef __cplusplus 
		extern "C" { 
#endif
	
#include "vos_if.h"
#include "vos_innr.h"

HVOSMTP vos_mtpCreate(char *pName, int type, int memType)
{
	MTM_POOL_S *pMtmPool = NULL;
	HVOSMPL pMplPool = NULL;
	void *pBaseAddr = NULL;
	char mplName[16];
	int i;

	pMtmPool = vos_malloc(sizeof(MTM_POOL_S));
	VOS_ASSERT(pMtmPool != NULL);
	memset(pMtmPool, 0, sizeof(MTM_POOL_S));

	pMtmPool->nMagicId = VOS_MTM_MAGIC_ID;
	if(pName != NULL)strncpy(pMtmPool->poolName, pName, 15);
	pMtmPool->hMtx = vos_mutexCreate("mtp", 0);
	VOS_ASSERT(pMtmPool->hMtx != 0);	
	pMtmPool->type = type;
	pMtmPool->memType = memType;
	pMtmPool->useList = vos_dlkCreate(sizeof(HVOSMPL), 0, NULL);
	VOS_ASSERT(pMtmPool->useList != 0);	
	pMtmPool->freeList = vos_dlkCreate(sizeof(HVOSMPL), 0, NULL);
	VOS_ASSERT(pMtmPool->freeList != 0);	
	return (HVOSMTP)pMtmPool;	
}

//pPoolBuf: used for type0&1; nPhyAddr:used for type1&2
void vos_mtpAddPool(HVOSMTP hMtmPool, int nPoolSize, void * pPoolBuf, unsigned int nPhyAddr)
{
	MTM_POOL_S *pMtmPool = (MTM_POOL_S *)hMtmPool;	
	HVOSMPL pMplPool = NULL;
	char poolName[32] ={0,};
	
	VOS_ASSERT(pMtmPool->nMagicId == VOS_MTM_MAGIC_ID);

	vos_mutexLock(pMtmPool->hMtx);

	pMtmPool->nPoolNum++;
	pMtmPool->nPoolSize   = nPoolSize;
	pMtmPool->nTotalSize += nPoolSize;

	sprintf(poolName, "%s_%d", pMtmPool->poolName, pMtmPool->nPoolNum);
	pMplPool = vos_mplCreatePoolPhy(poolName, pMtmPool->type, pPoolBuf, nPoolSize, nPhyAddr, 0);
	VOS_ASSERT(pMplPool != NULL); 

	vos_dlkInsert(pMtmPool->freeList, NULL, &pMplPool);	
	vos_mutexUnlock(pMtmPool->hMtx);

	return;	
}

DLK_NODE *vos_mtpGetNode(MTM_POOL_S *pMtmPool, PS_MPL_POOL pGetPool)
{	
	DLK_NODE *pTmp, *pHead;
	PS_MPL_POOL pPool;

	pHead = vos_dlkGetHead(pMtmPool->useList);
	DLK_WALK_LOOP_START(pTmp, pHead)
	{
		VOS_ASSERT(pTmp->nHeadId == VOS_DLK_HEAD_ID);
		pPool = *(PS_MPL_POOL *)(pTmp + 1);
		if(pPool == pGetPool)return pTmp;	
	}
	DLK_WALK_LOOP_END(pTmp, pHead)
	return NULL;
}

void *vos_mtpMalloc2(MTM_POOL_S *pMtmPool, int nSize, int align, const char * szFile, int nLine)
{	
	DLK_NODE *pTmp, *pHead;
	PS_MPL_POOL pPool;
	void *pRetAddr;
	
	if(nSize < pMtmPool->nPoolSize)
	{
		pHead = vos_dlkGetHead(pMtmPool->useList);
		DLK_WALK_LOOP_START(pTmp, pHead)
		{
			VOS_ASSERT(pTmp->nHeadId == VOS_DLK_HEAD_ID);
			pPool = *(PS_MPL_POOL *)(pTmp + 1);
		
			if(pPool->nRemainSize >= nSize)
			{
				pRetAddr = vos_mplAlloc((HVOSMPL)pPool, nSize, align,szFile, nLine);
				if(NULL != pRetAddr)return pRetAddr;
			}
		}
		DLK_WALK_LOOP_END(pTmp, pHead)
	}
	
	pHead = vos_dlkGetHead(pMtmPool->freeList);
	if(pHead == NULL)return NULL;

	pPool = *(PS_MPL_POOL *)(pHead + 1);
	pRetAddr = vos_mplAlloc((HVOSMPL)pPool, nSize, align,szFile, nLine);
	if(pRetAddr == NULL)return NULL;

	vos_dlkDelete(pMtmPool->freeList, pHead);
	vos_dlkInsertData(pMtmPool->useList, (void *)&pPool);
	pMtmPool->usedPoolNum++;
	
	return pRetAddr;	
}

void *vos_mtpMalloc(HVOSMTP hMtmPool, int nSize, int align, const char * szFile, int nLine)
{	
	MTM_POOL_S *pMtmPool = (MTM_POOL_S *)hMtmPool;
	void *pRet = NULL;
	
	VOS_ASSERT(pMtmPool->nMagicId == VOS_MTM_MAGIC_ID);
	VOS_ASSERT(nSize <= pMtmPool->nPoolSize);

	vos_mutexLock(pMtmPool->hMtx);
	pRet = vos_mtpMalloc2(pMtmPool, nSize, align, szFile, nLine);
	if(pRet != NULL)
	{
		pMtmPool->usedSize += nSize;
		if(pMtmPool->usedSize > pMtmPool->peakSize)pMtmPool->peakSize = pMtmPool->usedSize;
	}
	vos_mutexUnlock(pMtmPool->hMtx);
	return pRet;
}

void vos_mtpFree2(MTM_POOL_S *pMtmPool, void *pAddr)
{	
	PS_MPL_NODE pNode = NULL;
	PS_MPL_POOL pPool = NULL;
	DLK_NODE *pDlkNode;
	unsigned int nRemainSize;
	
	pNode = (PS_MPL_NODE)(pAddr)-1;
	VOS_ASSERT(pNode->nMagicId == MPL_MEM_MAGIC_ID);

	pPool = (PS_MPL_POOL)pNode->hPool;
	nRemainSize = pPool->nRemainSize;
	vos_mplFree((HVOSMPL)pPool, pAddr);

	pMtmPool->usedSize -= (pPool->nRemainSize - nRemainSize);

	if(pPool->pHeadNode->pNext == pPool->pTailNode)
	{
		pDlkNode = vos_mtpGetNode(pMtmPool, pPool);
		
		vos_dlkDelete(pMtmPool->useList, pDlkNode);
		vos_dlkInsertData(pMtmPool->freeList, (void *)&pPool);
		pMtmPool->usedPoolNum--;
	}
	
}

void vos_mtpFree(HVOSMTP hMtmPool, void *pAddr)
{
	MTM_POOL_S *pMtmPool = (MTM_POOL_S *)hMtmPool;
	
	VOS_ASSERT(pMtmPool->nMagicId == VOS_MTM_MAGIC_ID);
	
	vos_mutexLock(pMtmPool->hMtx);
	vos_mtpFree2(pMtmPool, pAddr);
	vos_mutexUnlock(pMtmPool->hMtx);
	return;
}

void vos_mtpFreeAll(HVOSMTP hMtmPool)
{		
	DLK_NODE *pTmp, *pHead;
	PS_MPL_POOL pPool;
	MTM_POOL_S *pMtmPool = (MTM_POOL_S *)hMtmPool;
	
	VOS_ASSERT(pMtmPool->nMagicId == VOS_MTM_MAGIC_ID);
	
	vos_mutexLock(pMtmPool->hMtx);	

	pHead = vos_dlkGetHead(pMtmPool->useList);
	DLK_WALK_LOOP_START(pTmp, pHead)
	{
		VOS_ASSERT(pTmp->nHeadId == VOS_DLK_HEAD_ID);
		pPool = *(PS_MPL_POOL *)(pTmp + 1);

		vos_mplFreeAll((HVOSMPL)pPool);
		vos_dlkInsertData(pMtmPool->freeList, (void *)&pPool);
	}
	DLK_WALK_LOOP_END(pTmp, pHead)

	vos_dlkClean(pMtmPool->useList);
	pMtmPool->usedPoolNum = 0;
	pMtmPool->usedSize = 0;
	pMtmPool->peakSize = 0;

	vos_mutexUnlock(pMtmPool->hMtx);
	
	return;
}

void vos_mtpShow(HVOSMTP hMtmPool, int nPrtFlag)
{
	MTM_POOL_S *pMtmPool = (MTM_POOL_S *)hMtmPool;
	DLK_NODE *pTmp, *pHead;
	PS_MPL_POOL pPool;
	int i = 1;
	
	VOS_ASSERT(pMtmPool->nMagicId == VOS_MTM_MAGIC_ID);

	vosPrintf("Total=0x%x(%d*0x%x), usedNum=%d, used=0x%x, peak=0x%x, free=0x%x\r\n", pMtmPool->nTotalSize, pMtmPool->nPoolNum, pMtmPool->nPoolSize, pMtmPool->usedPoolNum, pMtmPool->usedSize, pMtmPool->peakSize, (pMtmPool->nTotalSize - pMtmPool->usedSize));
	vosPrintf("%s: type=%d, memType=%d\r\n", pMtmPool->poolName, pMtmPool->type, pMtmPool->memType);
	vosPrintf("\r\n----used list:\r\n");
	pHead = vos_dlkGetHead(pMtmPool->useList);
	DLK_WALK_LOOP_START(pTmp, pHead)
	{
		VOS_ASSERT(pTmp->nHeadId == VOS_DLK_HEAD_ID);
		pPool = *(PS_MPL_POOL *)(pTmp + 1);
		vosPrintf("--%02d.usedList:\r\n", i);
		vos_mplInfo((HVOSMPL)pPool, NULL, nPrtFlag);
		i++;
	}
	DLK_WALK_LOOP_END(pTmp, pHead)

	vosPrintf("\r\n----free list:\r\n");
	i = 1;
	pHead = vos_dlkGetHead(pMtmPool->freeList);
	DLK_WALK_LOOP_START(pTmp, pHead)
	{
		VOS_ASSERT(pTmp->nHeadId == VOS_DLK_HEAD_ID);
		pPool = *(PS_MPL_POOL *)(pTmp + 1);
		vosPrintf("--%02d.freeList:\r\n", i);
		vos_mplInfo((HVOSMPL)pPool, NULL, nPrtFlag);
		i++;
	}
	DLK_WALK_LOOP_END(pTmp, pHead)
	return;
}


unsigned long vos_mtpGetUsedSize2(HVOSMTP hMtmPool)
{
	MTM_POOL_S *pMtmPool = (MTM_POOL_S *)hMtmPool;
	DLK_NODE *pTmp, *pHead;
	PS_MPL_POOL pPool;
	unsigned long usedSize = 0, totalSize = 0;
	
	VOS_ASSERT(pMtmPool->nMagicId == VOS_MTM_MAGIC_ID);

	vos_mutexLock(pMtmPool->hMtx);
	
	pHead = vos_dlkGetHead(pMtmPool->useList);
	DLK_WALK_LOOP_START(pTmp, pHead)
	{
		VOS_ASSERT(pTmp->nHeadId == VOS_DLK_HEAD_ID);
		pPool = *(PS_MPL_POOL *)(pTmp + 1);
		vos_mplGetSize((HVOSMPL)pPool, NULL, &usedSize, NULL);
		totalSize += usedSize;
	}
	DLK_WALK_LOOP_END(pTmp, pHead)

	vos_mutexUnlock(pMtmPool->hMtx);

	return totalSize;
}

unsigned long vos_mtpGetUsedSize(HVOSMTP hMtmPool)
{
	MTM_POOL_S *pMtmPool = (MTM_POOL_S *)hMtmPool;
	
	VOS_ASSERT(pMtmPool->nMagicId == VOS_MTM_MAGIC_ID);
	return pMtmPool->usedSize;
}

void * vos_mtpGetKvirAddr(HVOSMTP hMtmPool, void *pPhyAddr)
{
	MTM_POOL_S *pMtmPool = (MTM_POOL_S *)hMtmPool;
	DLK_NODE *pTmp, *pHead;
	PS_MPL_POOL pPool;
	void *pKvirAddr = NULL;
	
	VOS_ASSERT(pMtmPool->nMagicId == VOS_MTM_MAGIC_ID);

	vos_mutexLock(pMtmPool->hMtx);
	
	pHead = vos_dlkGetHead(pMtmPool->useList);
	DLK_WALK_LOOP_START(pTmp, pHead)
	{
		VOS_ASSERT(pTmp->nHeadId == VOS_DLK_HEAD_ID);
		pPool = *(PS_MPL_POOL *)(pTmp + 1);
		if(((unsigned int)pPhyAddr >= pPool->nPhyAddr) && ((unsigned int)pPhyAddr < (pPool->nPhyAddr+(unsigned int)pPool->nTotalSize)))
		{
			pKvirAddr = vos_mplGetKvirAddr((HVOSMPL)pPool, pPhyAddr);
			break;
		}
	}
	DLK_WALK_LOOP_END(pTmp, pHead)

	vos_mutexUnlock(pMtmPool->hMtx);

	return pKvirAddr;
}


//type: 0-no phyAddr, no USING_EXTERNAL_NODE; 1-have phyAddr, no USING_EXTERNAL_NODE; 2-have phyAddr, have USING_EXTERNAL_NODE
//memType:0-using vos_malloc; 1-using VOS_KMALLOC; 2-using vos_kmalloc_coherent
HVOSMTP vos_mtpCreatePool(char *pName, int type, int memType, int nPoolNum, int nPoolSize)
{	
	HVOSMTP hMtmPool = NULL;	
	void * pPoolBuf=NULL, *pPhyAddr=NULL, *pVirAddr=NULL;
	int i;

	hMtmPool = vos_mtpCreate(pName, type, memType);
	vosPrintf("hMtmPool(0x%x): type=%d, memType=%d, nPoolNum=%d, nPoolSize=0x%x\r\n", hMtmPool, type, memType, nPoolNum, nPoolSize);
	VOS_ASSERT(hMtmPool != NULL);

	for(i=0; i<nPoolNum; i++)
	{
		switch(type)
		{
			case 0:
				pPoolBuf = vos_malloc(nPoolSize);
				VOS_ASSERT(pPoolBuf != NULL);
				pPhyAddr = NULL;
				break;

			case 1:
			case 2:
				#if MPL_USING_IN_KENERL 
					if(1 == memType)
					{
						pVirAddr = VOS_KMALLOC(nPoolSize);
						pPhyAddr = (void *)vos_memVir2Phy((unsigned long)pVirAddr);
					}
					else
					{
						pVirAddr = vos_kmalloc_coherent(nPoolSize, (void *)&pPhyAddr);
					}
				#else
					pVirAddr = vos_malloc(nPoolSize);
					pPhyAddr = vos_malloc(nPoolSize);
				#endif

				VOS_ASSERT(pPhyAddr != NULL);
				pPoolBuf = pVirAddr;
				break;

			default:
				return NULL;
				
		}

		vos_mtpAddPool(hMtmPool, nPoolSize, pPoolBuf, (unsigned int)pPhyAddr);
	}

	return hMtmPool;
}

void vos_mtpDestroyPool(HVOSMTP hMtmPool)
{
	MTM_POOL_S *pMtmPool = (MTM_POOL_S *)hMtmPool;	
	DLK_NODE *pTmp, *pHead;
	PS_MPL_POOL pPool;
	
	VOS_ASSERT(pMtmPool->nMagicId == VOS_MTM_MAGIC_ID);

	vos_mtpFreeAll(hMtmPool);
	
	pHead = vos_dlkGetHead(pMtmPool->freeList);
	DLK_WALK_LOOP_START(pTmp, pHead)
	{
		VOS_ASSERT(pTmp->nHeadId == VOS_DLK_HEAD_ID);
		pPool = *(PS_MPL_POOL *)(pTmp + 1);

		switch(pMtmPool->memType)
		{
			case 0:
				if(pPool->nPoolAddrStart != 0)vos_free(pPool->nPoolAddrStart);
				if(pPool->nPhyAddrStart != 0)vos_free(pPool->nPhyAddrStart);
				break;
				
		#if MPL_USING_IN_KENERL 	
			case 1:
				vos_kfree(pPool->nPoolAddrStart);
				break;
			case 2:
				vos_kfree_coherent(pMtmPool->nPoolSize, (void *)pPool->nPoolAddrStart, (void *)pPool->nPhyAddrStart);
				break;
		#endif

			default:
				break;
		}

		vos_mplDestroy(pPool);
	}
	DLK_WALK_LOOP_END(pTmp, pHead)

	vos_dlkDestroy(pMtmPool->freeList);
	vos_dlkDestroy(pMtmPool->useList);
	vos_mutexClose(pMtmPool->hMtx);
	vos_free(pMtmPool);
	
	return;
}

unsigned int test_mtp(int nFlag, int nPar1, int nPar2, int nPar3)
{
	unsigned int i, nPoolNum, nPoolSize, ret = 0;	
	MTM_POOL_S *pMtmPool;
	static HVOSMTP hMtmPool = NULL;

	switch(nFlag)
	{
		case 0:
			if(NULL != hMtmPool)
				vos_mtpShow(hMtmPool, nPar1);
			break;
		
		case 1:
			hMtmPool = vos_mtpCreatePool("vos_test", 0, 0, nPar1, nPar2);
			VOS_ASSERT(hMtmPool != NULL);
			ret = (unsigned int)hMtmPool;
			break;

	#if MPL_USING_IN_KENERL 		
		case 10:
			hMtmPool = vos_mtpCreatePool("vos_test", 1, 1, nPar1, nPar2);
			VOS_ASSERT(hMtmPool != NULL);
			ret = (unsigned int)hMtmPool;
			break;
		
		case 11:
			hMtmPool = vos_mtpCreatePool("vos_test", 2, 1, nPar1, nPar2);
			VOS_ASSERT(hMtmPool != NULL);
			ret = (unsigned int)hMtmPool;
			break;
		
		case 12:
			hMtmPool = vos_mtpCreatePool("vos_test", 2, 2, nPar1, nPar2);
			VOS_ASSERT(hMtmPool != NULL);
			ret = (unsigned int)hMtmPool;
			break;
	#endif
	
		case 2:
			ret = (unsigned int)VOS_MTP_MALLOC(hMtmPool, nPar1);
			break;
		
		case 3:
			vos_mtpFree(hMtmPool, (void *)nPar1);
			break;
			
		case 4:
			vos_mtpFreeAll(hMtmPool);
			break;
			
		case 5:
			ret = (unsigned int)vos_mtpGetKvirAddr(hMtmPool, (void *)nPar1);
			break;
			
		case 6:
			pMtmPool = (MTM_POOL_S *)hMtmPool;
			ret = (unsigned int)vos_mtpGetUsedSize2(hMtmPool);
			vosPrintf("vos_mtpGetUsedSize:0x%x, pMtmPool->usedSize=0x%x\r\n", ret, pMtmPool->usedSize);
			break;
			
		case 7:
			vos_mtpDestroyPool(hMtmPool);
			break;
			
		default:
			vosPrintf("unkown command\r\n");
			break;		
	}

	return ret;
}

#if MPL_USING_IN_KENERL	
EXPORT_SYMBOL(vos_mtpFree);
EXPORT_SYMBOL(vos_mtpGetKvirAddr);
EXPORT_SYMBOL(vos_mtpFreeAll);
EXPORT_SYMBOL(vos_mtpGetUsedSize);
EXPORT_SYMBOL(vos_mtpMalloc);
EXPORT_SYMBOL(vos_mtpShow);
EXPORT_SYMBOL(vos_mtpCreatePool);
EXPORT_SYMBOL(vos_mtpDestroyPool);
EXPORT_SYMBOL(vos_mplGetPhyAddr);
#endif

#ifdef __cplusplus
}
#endif // __cplusplus


