/*=================================================================================================
Filename:    vos_misc.c
Description: abstraction functions of Linux kernel 
Data:        Initial revision. Bob@2019-12-05
==================================================================================================*/
#ifdef __cplusplus 
    extern "C" { 
#endif

#include "vos_if.h"
#include "vos_innr.h"
#include <linux/sched/clock.h>

//kernel symbols: /proc/kallsyms
//build-bundle/poky/build/tmp/work/simva32-poky-linux-gnueabi/linux-granite-upstream/4.2.8+gitAUTOINC+e5ca5d98fa-32bit/linux-simva32-standard-build/vmlinux

int  g_nVosDbgLev = 1;
void vos_showStack(void)
{
	vosPrintf("[task name]:%s\r\n", current->comm);	
	dump_stack();
}

void vos_assert(int bFlag, char *pMsg, char * pName, int nLines)
{
	int nCnt = 0;
	
	if(bFlag)return;

	vos_showStack();
	while(1)
	{
		vosPrintf("vos_assert(%s): %s, Line=%d\r\n", pMsg, pName, nLines);
		nCnt += 2;
		vos_taskSleep(1000*nCnt);
	}
	BUG();
	panic("BUG!");
}

void vos_miscInitAll(void)
{
	static int g_InitAllFlat = 0;

	if(g_InitAllFlat)return;
	g_InitAllFlat = 1;
	
	vos_mutexInit();
	vos_taskInit();	
	vos_semInit();
	vos_queInit();
	vos_timerInit();
	vos_hrtimerInit();
	vos_memInit();

	return;
}

void vos_Printf(const char* template, ...)
{
	static unsigned long g_printCnt = 0; 
	char taskName[20] = {0,};
	va_list args;
	char  pInfo[512] = {0,}; 
	char  *pHead = pInfo; 

	va_start(args, template);
	vsnprintf(pHead, 510, template, args);
	va_end(args);  

	printk("[%s]%s", current->comm, pInfo);

	return;
}

void vos_PrintfLong(const char* pStr)
{	
	int len, copyLen;
	char  pInfo[512] = {0,}; 
	char *pSource;

	len = strlen(pStr);
	if(len < 512)
	{
		printk("[%s]%s", current->comm, pStr);
		return;
	}

	pSource = pStr;
	copyLen = 511;
	while(len > 0)
	{
		memcpy(pInfo, pSource, copyLen);
		pInfo[copyLen] = 0;
		printk("[%s]%s", current->comm, pInfo);
		
		pSource += copyLen;
		len -= copyLen;
		copyLen = (len > 511)?511:len;
	}
	return;
}


void vos_Printf2(const char* template, ...)
{
	static unsigned long g_printCnt = 0; 
	char taskName[20] = {0,};
	va_list args;
	char  pInfo[512] = {0,}; 
	char  *pHead = pInfo; 

	va_start(args, template);
	vsnprintf(pHead, 510, template, args);
	va_end(args);  

	printk(KERN_EMERG "[%s]%s", current->comm, pInfo); //KERN_EMERG: output to serial_port

	return;
}

char vos_getAscii(char cVal)
{
	if((cVal < 33) || (cVal < 0))
	{
		return 46;
	}
	else
	{
		return cVal;
	}
}

void vos_dd(unsigned long ulAddr, unsigned long ulUnits, unsigned long ulWidth, char * pInfo1)
{
	unsigned long ulLines, ulRows, i, j;
	char pInfo2[256] = {0,};
	char * pInfo;

	pInfo = pInfo1;
	if(pInfo == 0)pInfo = pInfo2;
		
	ulUnits = (ulUnits > 1024)?1024:ulUnits;
    if((ulWidth != 1) && (ulWidth != 2) && (ulWidth != 4))
    {
        ulWidth = 4;
    }

    ulLines = (ulUnits*ulWidth - 1)/16 + 1;
    ulRows = 16/ulWidth;

    for(i=0; i<ulLines; i++)
    {
        pInfo += sprintf(pInfo, "0x%08x: ", ulAddr + 16*i);
        for(j=0; j<ulRows; j++)
        {
            if((i*j) > (ulUnits*ulWidth))
            {
                break;
            }

            /*֧��1/2/4�����ֽ���ʾ��ʽ*/
            if(1 == ulWidth)pInfo += sprintf(pInfo, "%02x ", *(unsigned char *)(ulAddr + 16*i + j*ulWidth));
            if(2 == ulWidth)pInfo += sprintf(pInfo, "%04x ", *(unsigned short *)(ulAddr + 16*i + j*ulWidth));
            if(4 == ulWidth)pInfo += sprintf(pInfo, "%08x ", *(unsigned long *)(ulAddr + 16*i + j*ulWidth));
        }

        /*��ʾASCII�ַ�*/
        pInfo += sprintf(pInfo, " *");
        for(j=0; j<16; j++)
        {
            if((i*j) > (ulUnits*ulWidth))
            {
                pInfo += sprintf(pInfo, "*\r\n");
                return;
            }

            pInfo += sprintf(pInfo, "%c", vos_getAscii(*(char *)(ulAddr + 16*i + j)));
        }
        pInfo += sprintf(pInfo, "*\r\n");

		if(pInfo1 == 0){vos_Printf("%s", pInfo2);pInfo2[0] = 0;pInfo=pInfo2;}
    }

    return;
}

void vos_ddPrint(unsigned long ulAddr, unsigned long ulUnits, unsigned long ulWidth)
{
	char pBuff[4*1024] = {0,};

	vos_dd(ulAddr, ulUnits, ulWidth, pBuff);
	vos_PrintfLong(pBuff);
	return;
}

void vos_mm(unsigned long ulAddr, unsigned long ulVal, unsigned long ulWidth, char * pInfo)
{
    switch(ulWidth)
    {
        case 1:
            *(unsigned char *)ulAddr = (unsigned char)ulVal;
            pInfo += sprintf(pInfo, "write addr(0x%08x) in %u(0x%x) with 1 byte\r\n", ulAddr, (unsigned char)ulVal, (unsigned char)ulVal);
            break;
            
        case 2:
            *(unsigned short *)ulAddr = (unsigned short)ulVal;
            pInfo += sprintf(pInfo, "write addr(0x%08x) in %u(0x%x) with 2 bytes\r\n", ulAddr, (unsigned short)ulVal, (unsigned short)ulVal);
            break;
            
        case 4:
            *(unsigned long *)ulAddr = (unsigned long)ulVal;
            pInfo += sprintf(pInfo, "write addr(0x%08x) in %u(0x%x) with 4 bytes\r\n", ulAddr, (unsigned long)ulVal, (unsigned long)ulVal);
            break;

        default:
            pInfo += sprintf(pInfo, "unknow parmeter:%u\r\n", ulWidth);
                break;                    
    }

    return;    
}

int vos_fileRead(const char * filename, char *pBuff, int size) 
{
	struct file *fd;
	loff_t pos = 0;

	fd = filp_open(filename, O_RDONLY, 0);
	if(fd == NULL)return 0;
	
	size = (int)kernel_read(fd, pBuff, size, &pos);
	filp_close(fd, NULL);
	return size;
} 

int vos_fileWrite(char* filename, char* data, int size)
{
	struct file *fd;
	loff_t pos = 0;

	fd = filp_open(filename, O_RDWR|O_CREAT, 0666);
	if(fd == NULL)return 0;

	size = (int)kernel_write(fd, data, size, &pos);
	filp_close(fd, NULL);
	return size;
}

int vos_fileAppend(char* filename, char* data, int size)
{
	struct file *fd;
	loff_t pos;

	fd = filp_open(filename, O_RDWR|O_CREAT|O_APPEND, 0666);
	if(fd == NULL)return 0;

	pos = fd->f_pos;
	//vfs_write(fd, pCmd, strlen(pCmd), &pos);
	size = (int)kernel_write(fd, data, size, &pos);
	fd->f_pos = pos;

	filp_close(fd, NULL);
	return size;
}

void vos_logFileOut(char *pFileName, int nPintFlag, const char* template, ...)
{
	static unsigned long g_nLogCnt = 0;
	va_list args;
	char  pInfo[512] = {0,}; 
	char pCmd[512] = {0,};

	va_start(args, template);
	vsnprintf(pInfo, 500, template, args);
	va_end(args);  

	g_nLogCnt++;
	sprintf(pCmd, "%u:[%u]%s\r\n", g_nLogCnt, vos_GetSysCntMs(), pInfo);
	vos_fileAppend(pFileName, pCmd, strlen(pCmd));
	return;
}

unsigned long vos_GetSysCntMs(void)
{
#if (VOS_CLOCK_FLAG==1)
    unsigned long long ts_nsec;
	unsigned long rem_nsec, ts_sec, ts_ms;
	
    ts_nsec = local_clock();
	rem_nsec = do_div(ts_nsec, 1000000000);
	ts_sec = (unsigned long)ts_nsec;
	ts_ms = ts_nsec*1000 + (rem_nsec/1000000);
	return ts_ms;    
#elif (VOS_CLOCK_FLAG==2)
    struct timespec ts;
    ts = current_kernel_time();
    return (ts.tv_sec * 1000 + ts.tv_nsec / 1000000);
#elif (VOS_CLOCK_FLAG==3)
		struct timespec ts;
		getrawmonotonic(&ts);
		return (ts.tv_sec * 1000 + ts.tv_nsec / 1000000);
#else
    return jiffies_to_msecs(jiffies);	
#endif
}

unsigned long vos_GetSysCntUs(void)
{
#if (VOS_CLOCK_FLAG==1)
    unsigned long long ts_nsec;
	unsigned long rem_nsec, ts_sec, ts_us;
	
    ts_nsec = local_clock();
	rem_nsec = do_div(ts_nsec, 1000000000);
	ts_sec = (unsigned long)ts_nsec;
	ts_us = ts_nsec*1000000 + (rem_nsec/1000);
	return ts_us;    
#elif (VOS_CLOCK_FLAG==2)
    struct timespec ts;
	unsigned long nUs; //max=4294(Sec)=71(Min)
	
    ts = current_kernel_time();
	nUs = ts.tv_sec * 1000000 + ts.tv_nsec / 1000;
    return nUs;
#elif (VOS_CLOCK_FLAG==3)
		struct timespec ts;
		unsigned long nUs; //max=4294(Sec)=71(Min)
		
		getrawmonotonic(&ts);
		nUs = ts.tv_sec * 1000000 + ts.tv_nsec / 1000;
		return nUs;
#else
    return jiffies_to_usecs(jiffies);
#endif
}

unsigned long vos_GetSysCntNs(void)
{
#if (VOS_CLOCK_FLAG==1)
    unsigned long long ts_nsec;
	unsigned long rem_nsec, ts_sec, ts_ns;
	
    ts_nsec = local_clock();
	rem_nsec = do_div(ts_nsec, 4000000000);
	ts_sec = (unsigned long)ts_nsec;
	//ts_ns = ts_sec*1000000000 + rem_nsec;
	ts_ns = rem_nsec;
	return ts_ns;    
#elif (VOS_CLOCK_FLAG==2)
    struct timespec ts;
    ts = current_kernel_time();
    return ts.tv_nsec;
#elif (VOS_CLOCK_FLAG==3)
		struct timespec ts;
		getrawmonotonic(&ts);
		return ts.tv_nsec;
#else
    return jiffies_to_nsecs(jiffies);	
#endif
}

static IRQ_ITEM_S g_asIntTbl[VOS_IRQ_NUM] = {0,};

void vos_intAttach( unsigned long int_num, unsigned long flags, void *handler, const char* name, void* context )
{
    int retval, i;
	IRQ_ITEM_S *pIntItem;

	for(i=0; i<VOS_IRQ_NUM; i++)
	{
		pIntItem = g_asIntTbl+i;
		if(pIntItem->status == 0)
		{
			pIntItem->status = VOS_IRQ_ATTACH;
			strncpy(pIntItem->name, name, 18);
			pIntItem->name[19] = 0;
			pIntItem->handler = handler;
			pIntItem->int_num = int_num;
			pIntItem->context = context;
			break;
		}
	}
	VOS_ASSERT(i<VOS_IRQ_NUM);
	
    //irq_set_status_flags(int_num, IRQ_NOAUTOEN);
    retval = request_irq(int_num, (irq_handler_t)handler, flags, name, context); //request_threaded_irq
    VOS_ASSERT(retval == 0);

	//arch\arm\kernel\irq.c: asm_do_IRQ - handle_IRQ
}

void vos_intDetach( unsigned long int_num, void* context)
{
	IRQ_ITEM_S *pIntItem;
	int i;

	for(i=0; i<VOS_IRQ_NUM; i++)
	{
		pIntItem = g_asIntTbl+i;
		if((pIntItem->status != 0) && (pIntItem->int_num == int_num))
		{
			break;
		}
	}
	VOS_ASSERT(i<VOS_IRQ_NUM);
	
    VOS_ASSERT(!in_interrupt());
    free_irq( int_num, context );
    memset(pIntItem, 0, sizeof(IRQ_ITEM_S));
}

void vos_intEnable( unsigned long int_num )
{
	IRQ_ITEM_S *pIntItem;
	int i;

	for(i=0; i<VOS_IRQ_NUM; i++)
	{
		pIntItem = g_asIntTbl+i;
		if((pIntItem->status != 0) && (pIntItem->int_num == int_num))
		{
			break;
		}
	}
	VOS_ASSERT(i<VOS_IRQ_NUM);
	
    VOS_ASSERT(!in_interrupt());

    if(pIntItem->status == VOS_IRQ_ENABLE)return;

    enable_irq( int_num );
    pIntItem->status = VOS_IRQ_ENABLE;
    return;
}

void vos_intDisable( unsigned long int_num )
{
	IRQ_ITEM_S *pIntItem;
	int i;

	for(i=0; i<VOS_IRQ_NUM; i++)
	{
		pIntItem = g_asIntTbl+i;
		if((pIntItem->status != 0) && (pIntItem->int_num == int_num))
		{
			break;
		}
	}
	VOS_ASSERT(i<VOS_IRQ_NUM);
	
    VOS_ASSERT(!in_interrupt());

    if(pIntItem->status == VOS_IRQ_DISABLE)return;

    disable_irq( int_num );
    pIntItem->status = VOS_IRQ_DISABLE;
    return;
}

int vos_intShowAllInt(void)
{
	int i, nRet, nIndex=0;
	IRQ_ITEM_S *pIntItem;

	vosPrintf("%-4s %-8s   %-20s %-12s %-12s %-4s\r\n", \
			"No.", "Int_num", "Name", "Func", "context", "state");
	vosPrintf("-------------------------------------------------------------------------\r\n");

	for(i=0; i<VOS_IRQ_NUM; i++)
	{
		pIntItem = g_asIntTbl+i;
		if(pIntItem->status != 0)
		{
			nIndex++;
			vosPrintf("%-4d %-8d %-20s 0x%-10x 0x%-10x %-4d\r\n", \
				nIndex, pIntItem->int_num, pIntItem->name, pIntItem->handler, pIntItem->context, pIntItem->status);
		}
	}
	return nIndex;
}

void vos_spinLockInit(spinlock_t* lock)
{
    spin_lock_init(lock);
}

void vos_spinLock(spinlock_t* lock)
{
    spin_lock(lock);
}

void vos_spinUnlock(spinlock_t* lock)
{
    spin_unlock(lock);
}

void vos_spinLockIsr(spinlock_t* lock, uint32_t* flags)
{
    unsigned long _flags;
    spin_lock_irqsave(lock, _flags);
    *flags = _flags;
}

void vos_spinUnlockIsr(spinlock_t* lock, uint32_t flags)
{
    spin_unlock_irqrestore(lock, flags);
}

void vos_atomicAdd(atomic_t *v, int val)
{
	atomic_add_return(val, v);
}

void vos_atomicSub(atomic_t *v, int val)
{
	atomic_sub_return(val, v);
}

int vos_atomicRead(atomic_t *v)
{
	//return ACCESS_ONCE(v->counter);
	return atomic_read(v);
}

void vos_atomicSet(atomic_t *v, int val)
{
	//v->counter = val;
	atomic_set(v, val);
}

#define TRACE_BUFF

HRBUF g_hTraceBuf = 0;
void DBG_traceInit(void)
{
	if(g_hTraceBuf != 0)return;
    g_hTraceBuf = vos_rbufCreate(DBG_TRACE_BUFF_SIZE, DBG_TRACE_NUM_MAX, 1);
    return;
}

void DBG_traceClear(void)
{
    if(g_hTraceBuf == 0)return;
	vos_rbufClean(g_hTraceBuf);
    return;
}

void DBG_tracePut(char *pStr, ...)
{
    va_list args;
    int     len;
	char buff[DBG_TRACE_BUFF_SIZE] = {0,};
	char *pTmp = buff;

	DBG_traceInit();
    if(NULL == pStr)return;

	pTmp += sprintf(pTmp, "[%u]", vos_GetSysCntMs());
    va_start( args, pStr );
    len = vsnprintf(pTmp, DBG_TRACE_BUFF_SIZE-1, pStr, args);
    va_end( args );

	(void)vos_rbufWriteTrace(g_hTraceBuf, (void *)buff);
}

int DBG_traceShow(void)
{
    if(g_hTraceBuf == 0)return 0;
	return vos_rbufShow(g_hTraceBuf, 1);
}
EXPORT_SYMBOL(DBG_tracePut);

#ifdef DTOL_SCT
/*************************************************************************************************/
void SHELL_codeLock(unsigned long ulAddr, unsigned long ulSize)
{
    unsigned long ulLen, ulOld = 0;

	ulLen = 0x1000;
	if((0x1000 - (ulAddr & 0xfff)) < ulSize)ulLen += 0x1000;
    //mprotect((void *)(ulAddr & 0xfffff000), ulLen, PROT_READ|PROT_EXEC);
    return;
}

void SHELL_codeUnlock(unsigned long ulAddr, unsigned long ulSize)
{
    unsigned long ulLen, ulOld = 0;

	ulLen = 0x1000;
	if((0x1000 - (ulAddr & 0xfff)) < ulSize)ulLen += 0x1000;
    //mprotect((void *)(ulAddr & 0xfffff000), ulLen, PROT_READ|PROT_WRITE|PROT_EXEC);
    return;
}

HVOSDLK g_hSctDlk = 0;
int sct_getKey(void *pData, unsigned long nKey)
{
	return *(unsigned long *)pData;
}

void sct_Init(void)
{
	if(0 == g_hSctDlk)
		g_hSctDlk = vos_dlkCreate(sizeof(SHELL_SC_NODE_S), 1, sct_getKey);
	VOS_ASSERT(g_hSctDlk != 0);
	return;
}

int sct_showAll(void)
{
	SHELL_SC_NODE_S *pSctItem;
	DLK_NODE *pShow, *pHead;

	sct_Init();
	vosPrintf("%-10s %-10s\r\n", "FuncAddr", "Return Val");
	vosPrintf("----------------------------------------------------\r\n");
	
	pHead = vos_dlkGetHead(g_hSctDlk);
	DLK_WALK_LOOP_START(pShow, pHead)
	{
		pSctItem = (Sem_ITEM_S *)(pShow + 1);
		vosPrintf("0x%-08x 0x%-08x\r\n", pSctItem->ulFunAddr, pSctItem->ulRetVal);		
	}
	DLK_WALK_LOOP_END(pShow, pHead)
	return 0;
}

void sct_LINARO_ARM(unsigned long ulFucAddr, unsigned long ulRetVal)
{
/* return:0x12345678
1fbbc:	 f245 6078	 movw	 r0, #22136  ; 0x5678
1fbc0:	 f2c1 2034	 movt	 r0, #4660	 ; 0x1234
1fbc4:	 4770		 bx  lr
1fbc6:	 bf00		 nop
*/
	unsigned short aSctText[SHELL_SC_SOURCE_CODE_SIZE/2] = {0xf245, 0x6078, 0xf2c1, 0x2034, 0x4770, 0xbf00};
	unsigned short *pFuncAddr = (unsigned short *)ulFucAddr;
	unsigned short val, i;

	for(i=0; i<(SHELL_SC_SOURCE_CODE_SIZE/2); i++)pFuncAddr[i] = aSctText[i];

	val = ulRetVal & 0xffff;
	pFuncAddr[0] = 0xf240 | ((val & 0xf000) >> 12) | ((val & 0x0800) >> 1);
	pFuncAddr[1] = ((val & 0x0700) << 4) | (val & 0x00ff);

	val = (ulRetVal >> 16);
	pFuncAddr[2] = 0xf2c0 | ((val & 0xf000) >> 12) | ((val & 0x0800) >> 1);
	pFuncAddr[3] = ((val & 0x0700) << 4) | (val & 0x00ff);
	
	return;
}

unsigned long scd(unsigned long ulFucAddr)
{
	SHELL_SC_NODE_S *pSctItem;
	DLK_NODE *pShow, *pHead;
	DLK_NODE *pDelList[32] = {0,};
	int i, listCnt = 0;

	sct_Init();
	
	pHead = vos_dlkGetHead(g_hSctDlk);
	DLK_WALK_LOOP_START(pShow, pHead)
	{
		pSctItem = (SHELL_SC_NODE_S *)(pShow + 1);
		if((ulFucAddr == 0) || (ulFucAddr == pSctItem->ulFunAddr))
		{
			pDelList[listCnt] = pShow;
			listCnt++;
		}
	}
	DLK_WALK_LOOP_END(pShow, pHead)

	for(i=0; i<listCnt; i++)
	{
		pSctItem = (SHELL_SC_NODE_S *)(pDelList[i] + 1);
		
		SHELL_codeUnlock(pSctItem->ulFunAddr, 0x20);
		memcpy((void *)pSctItem->ulFunAddr, (void *)&pSctItem->ulCodeData[0], SHELL_SC_SOURCE_CODE_SIZE);
		SHELL_codeLock(pSctItem->ulFunAddr, 0x20);
		
		vos_dlkDelete(g_hSctDlk, pDelList[i]);
	}
	
	return listCnt;
}

unsigned long sct(unsigned long ulFucAddr, unsigned long ulRetVal)
{
	DLK_NODE *pDlkNode;
    SHELL_SC_NODE_S stScNode = {0,};    

	sct_Init();
	ulFucAddr &= 0xfffffffe;
    if(0 == ulFucAddr)
    {
        sct_showAll();
        return 0;
    }
    
    pDlkNode = vos_dlkFind(g_hSctDlk, ulFucAddr);
    if(NULL != pDlkNode)
    {
        (void)scd(ulFucAddr);
    }

    /*��һ������ת��䣬��֧��*/
    if((*(unsigned long *)ulFucAddr & 0xff000000) == 0xea000000)return 3;

    stScNode.ulFunAddr = ulFucAddr;
    stScNode.ulRetVal = ulRetVal;
    memcpy((void *)&stScNode.ulCodeData[0], (void *)ulFucAddr, SHELL_SC_SOURCE_CODE_SIZE);
	vos_dlkInsertData(g_hSctDlk, (void *)&stScNode);

	SHELL_codeUnlock(stScNode.ulFunAddr, 0x20);
	sct_LINARO_ARM(ulFucAddr, ulRetVal);
	SHELL_codeLock(stScNode.ulFunAddr, 0x20);
	
	return 0;    
}
#endif

EXPORT_SYMBOL(vos_showStack);
EXPORT_SYMBOL(vos_assert);
EXPORT_SYMBOL(vos_Printf);
EXPORT_SYMBOL(vos_PrintfLong);
EXPORT_SYMBOL(vos_Printf2);
EXPORT_SYMBOL(vos_ddPrint);
EXPORT_SYMBOL(vos_logFileOut);
EXPORT_SYMBOL(vos_GetSysCntMs);
EXPORT_SYMBOL(vos_GetSysCntUs);
EXPORT_SYMBOL(vos_GetSysCntNs);
EXPORT_SYMBOL(vos_intAttach);
EXPORT_SYMBOL(vos_intDetach);
EXPORT_SYMBOL(vos_intEnable);
EXPORT_SYMBOL(vos_intDisable);
EXPORT_SYMBOL(vos_atomicAdd);
EXPORT_SYMBOL(vos_atomicSub);
EXPORT_SYMBOL(vos_atomicRead);
EXPORT_SYMBOL(vos_atomicSet);
EXPORT_SYMBOL(vos_fileRead);
EXPORT_SYMBOL(vos_fileWrite);
EXPORT_SYMBOL(vos_fileAppend);

int test_misc(int type, int p1, int p2)
{
	int i, nDely, nRet = 0;
	//struct timespec ts;
	unsigned long ulVal = 0;
	static spinlock_t g_spinlock;
    unsigned long long ts_nsec;
	unsigned long rem_nsec, ts_sec;
	char pBuff[1024] = {0,};
	static atomic_t vAtomic;
	atomic_t *pAtomic = &vAtomic;
	
	switch(type)
	{
	#if 0	
		case 0:
		    ts = current_kernel_time();
			vosPrintf("current_kernel_time: [%u.%u]\r\n", ts.tv_sec, ts.tv_nsec);

		    ts_nsec = local_clock();
			rem_nsec = do_div(ts_nsec, 1000000000);
			ts_sec = (unsigned long)ts_nsec;
			vosPrintf("local_clock_time: [%u.%u]\r\n", ts_sec, rem_nsec);
			break;
		case 1:
			for(i=0; i<p1; i++)
			{
			    ts_nsec = local_clock();
				rem_nsec = do_div(ts_nsec, 1000000000);
				ts_sec = (unsigned long)ts_nsec;
				DBG_tracePut("[%u.%u]ms:%u, us:%u, ns:%u", ts_sec, rem_nsec, vos_GetSysCntMs(), vos_GetSysCntUs(), vos_GetSysCntNs());
				nDely = p2;
				while(nDely){nDely--, ulVal=vos_GetSysCntMs() + vos_GetSysCntNs();};
			}
			break;
	#endif		
		case 2:
			printk("vos_taskDelay: %u\r\n", vos_GetSysCntUs());
			vos_taskDelay(p1);
			printk("vos_taskDelay: %u\r\n", vos_GetSysCntUs());
			break;
		case 3:
		    vosPrintf((char *)p1);
			break;
		case 4:
		    vos_intShowAllInt();
			break;
		case 5:
		    vos_spinLockInit(&g_spinlock);
			break;
		case 6:
		    nRet = vos_atomicRead(pAtomic);
			printk("vos_atomicRead(0x%x): %u\r\n", pAtomic, nRet);
			break;
		case 7:			
			nRet = vos_fileRead("/tmp/t1.log", pBuff, p1);
			//vos_ddPrint(pBuff, 32, 1);
			break;
		case 8:			
			for(i=0; i<p1; i++)pBuff[i] = i;
			nRet = vos_fileWrite("/tmp/t1.log", pBuff, p1);
			break;
		
		case 9:			
			vos_logFileOut("/tmp/t1.log", 0, "test123");
			break;
		
		default:
			printk("unknown: test_misc(%u, %u, %u\r\n)", type, p1, p2);
			break;
	}
	return nRet;
}

S_TS_INFO g_tsInfo = {0,};

//(prev->pid, prev->tgid, prev->comm, irqTime, irqNum)
void tsr_recTrace(unsigned int p1, unsigned int p2, unsigned int p3, unsigned int p4, unsigned int p5)
{
	S_TS_REC *pRec;
	struct timespec ts;

	if(g_tsInfo.index >= TS_MAX_NUM)return;

	pRec = g_tsInfo.psBuff + g_tsInfo.index;
	pRec->p1 = (short)p1;
	pRec->p2 = (short)p2;
	pRec->pName = (char *)p3;
	pRec->irqTime = p4; //from kernel/irq/halder.c:__handle_irq_event_percpu. bit0-15:cnt,bit16-bit31:#irq
	pRec->irqNum = p5; //bit0-7:cnt,bit8-bit31:#irq

	getrawmonotonic(&ts);
	pRec->sec = ts.tv_sec;
	pRec->ns = ts.tv_nsec;
	g_tsInfo.index++;
}

void tsr_recLoading(unsigned int p1, unsigned int p2, unsigned int p3, unsigned int p4, unsigned int p5)
{
	S_TS_LOAD *pLoad;
	char *pCurName = (char *)p3;
	unsigned long i, curUs;
	struct timespec ts;
	static unsigned long g_lastUs = 0;
	static char *pTask = "other";

	getrawmonotonic(&ts);
	curUs = ts.tv_sec * 1000000 + ts.tv_nsec / 1000;
	
	for(i=0; i<(TS_MAX_LOAD_TASKNUM-1); i++)
	{
		pLoad = &g_tsInfo.stLoadWhole.stLoad[i];
		if(pLoad->pName == NULL)
		{
			pLoad->pName = pCurName;
			pLoad->p1 = (short)p1;
			pLoad->p2 = (short)p2;
			if(i == 0)g_lastUs = curUs;
			break;
		}

		if(pLoad->pName == pCurName)break;
	}

	if(i == (TS_MAX_LOAD_TASKNUM-1))pLoad->pName = pTask;
		
	pLoad->totalTimeUs += (curUs - g_lastUs);
	g_lastUs = curUs;

	return;
}

void tsr_initSegLoading(void)
{
	unsigned int i;
	S_SEG_LOAD *pSegLod;
	S_TS_REC *pTmp;

	//for whole loading
	pSegLod = &g_tsInfo.stLoadWhole;
	pSegLod->startIndex = 0;
	pSegLod->endIndex = g_tsInfo.index-1;

	pTmp = g_tsInfo.psBuff + pSegLod->startIndex;
	pSegLod->startUs = pTmp->sec * 1000000 + (pTmp->ns / 1000);
	pTmp = g_tsInfo.psBuff + pSegLod->endIndex;
	pSegLod->endUs = pTmp->sec * 1000000 + (pTmp->ns / 1000);

	//for segments loading
	if(g_tsInfo.segments < 2)return;

	if(g_tsInfo.pSegLoad != NULL)return;
	
	g_tsInfo.pSegLoad = (S_SEG_LOAD *)vos_malloc(sizeof(S_SEG_LOAD) * g_tsInfo.segments);
	if(g_tsInfo.pSegLoad == NULL)
	{
		vos_Printf("g_tsInfo.pSegLoad malloc faile!\r\n");
		return;
	}

	memset(g_tsInfo.pSegLoad, 0, sizeof(S_SEG_LOAD) * g_tsInfo.segments);
	for(i=0; i<g_tsInfo.segments; i++)
	{
		pSegLod = g_tsInfo.pSegLoad + i;
		pSegLod->startIndex = g_tsInfo.segSartIndex[i];
		pSegLod->endIndex = g_tsInfo.segSartIndex[i+1]-1;
		
		pTmp = g_tsInfo.psBuff + pSegLod->startIndex;
		pSegLod->startUs = pTmp->sec * 1000000 + (pTmp->ns / 1000);
		pTmp = g_tsInfo.psBuff + pSegLod->endIndex;
		pSegLod->endUs = pTmp->sec * 1000000 + (pTmp->ns / 1000);
	}
	return;
}

void tsr_recLoadingitem(short p1, short p2 , char *pCurName, unsigned long takeUs, S_SEG_LOAD *pSegLoad)
{
	unsigned long i;
	S_TS_LOAD *pLoad = &pSegLoad->stLoad[0];
	static char *pTask = "other";

	for(i=0; i<(TS_MAX_LOAD_TASKNUM-1); i++)
	{
		if(pLoad->pName == NULL)
		{
			pLoad->pName = pCurName;
			pLoad->p1 = p1;
			pLoad->p2 = p2;
			break;
		}

		if(strcmp(pLoad->pName, pCurName) == 0)break;
		pLoad++;
	}

	if(i == (TS_MAX_LOAD_TASKNUM-1))pLoad->pName = pTask;
		
	pLoad->totalTimeUs += takeUs;
	pLoad->switchCnt++;

	return;
}

void tsr_recLoadingAll(short p1, short p2 , char *pCurName, unsigned long takeUs, unsigned long index)
{
	S_SEG_LOAD *pSegLoad = NULL;
	unsigned long i;
	static char *pTask = "other";

	//for whole loading
	pSegLoad = &g_tsInfo.stLoadWhole;
	tsr_recLoadingitem(p1, p2 , pCurName, takeUs, pSegLoad);

	//for segments loading
	if(g_tsInfo.segments > 1)
	{
		pSegLoad = NULL;
		for(i=0; i<g_tsInfo.segments; i++)
		{
			if((index >= g_tsInfo.segSartIndex[i]) && (index < g_tsInfo.segSartIndex[i+1]))
			{
				pSegLoad = g_tsInfo.pSegLoad + i;
				break;
			}
		}

		if(NULL == pSegLoad)return;

		tsr_recLoadingitem(p1, p2 , pCurName, takeUs, pSegLoad);
	}

	return;
}


int tsr_getLoading(char *pInfo, S_SEG_LOAD *pSegLoad)
{
	S_TS_LOAD *pLoad;
	int i, ret = 0;
	char *pTemp = pInfo;

	if(pInfo)
	{
		pTemp += sprintf(pTemp, "\r\n%-8s: %-8s %-8s %-16s %10s %10s %6s\r\n", "index", "pid", "tgid", "name", "time(us)", "switchCnt", "percent(\%)");
		pTemp += sprintf(pTemp, "----------------------------------------------------------------------------\r\n");
	}
	else
	{
		printk("%-8s: %-8s %-8s %-16s %10s %10s %6s\r\n", "index", "pid", "tgid", "name", "time(us)", "switchCnt", "percent(\%)");
		printk("----------------------------------------------------------------------------\r\n");
	}

	pLoad = &pSegLoad->stLoad[0];
	for(i=0; i<TS_MAX_LOAD_TASKNUM; i++)
	{
		if(pLoad->pName == NULL)break;

		if(pInfo)
			pTemp += sprintf(pTemp, "%-8d: %-8d %-8d %-16s %10u %10u %3u.%u\r\n", i, pLoad->p1, pLoad->p2, pLoad->pName, pLoad->totalTimeUs, pLoad->switchCnt, pLoad->perCent/10, pLoad->perCent%10);
		else
			printk("%-8d: %-8d %-8d %-16s %10u %10u %3u.%u\r\n", i, pLoad->p1, pLoad->p2, pLoad->pName, pLoad->totalTimeUs, pLoad->switchCnt, pLoad->perCent/10, pLoad->perCent%10);
		pLoad++;
	}	

	if(pInfo)
	{
		pTemp += sprintf(pTemp, "--------------------------------------------------------------------------------\r\n");
		pTemp += sprintf(pTemp, "--start=%u, end=%u, take:%ums, Index:(%u-%u), total counter:%u\r\n", pSegLoad->startUs, pSegLoad->endUs, (pSegLoad->endUs-pSegLoad->startUs)/1000, pSegLoad->startIndex, pSegLoad->endIndex, (pSegLoad->endIndex-pSegLoad->startIndex+1));
	}
	else
	{
		printk("--------------------------------------------------------------------------------\r\n");
		printk("--start=%u, end=%u, take:%ums, Index:(%u-%u), total counter:%u\r\n", pSegLoad->startUs, pSegLoad->endUs, (pSegLoad->endUs-pSegLoad->startUs)/1000, pSegLoad->startIndex, pSegLoad->endIndex, (pSegLoad->endIndex-pSegLoad->startIndex+1));
	}

	if(pInfo)ret = (int)(pTemp - pInfo);
	return ret;
}

int tsr_getLoadingAll(char *pInfo)
{	
	int i, ret;
	S_SEG_LOAD *pSegLoad = NULL;
	char *pTmp = pInfo;

	//for whole loading
	pSegLoad = &g_tsInfo.stLoadWhole;
	pTmp += tsr_getLoading(pTmp, pSegLoad);
	if(pTmp)
		pTmp += sprintf(pTmp, "--irqCnt:%u, irqTotalUs=%uus\r\n", g_tsInfo.irqCnt, g_tsInfo.irqTotalUs);
	else
		printk("--irqCnt:%u, irqTotalUs=%uus\r\n", g_tsInfo.irqCnt, g_tsInfo.irqTotalUs);
	if(pInfo == NULL)return 0;

	//for segments loading
	if(g_tsInfo.segments > 1)
	{		
		pTmp += sprintf(pTmp, "\r\n-------------below for segments loading:\r\n");
		for(i=0; i<g_tsInfo.segments; i++)
		{
			pSegLoad = (g_tsInfo.pSegLoad + i);
			pTmp += tsr_getLoading(pTmp, pSegLoad);
		}
	}

	vos_free(g_tsInfo.pSegLoad );
	ret = (int)(pTmp - pInfo);
	return ret;
}

int tsr_sortGetKey(void *pData, unsigned long nKey)
{
	S_TS_LOAD *pLoad = (S_TS_LOAD *)pData;
	return pLoad->totalTimeUs;
}

void tsr_loadingSort(S_TS_LOAD *pLoadInOut)
{
	S_TS_LOAD *pLoad, *pSort;
	unsigned long totalUs = 0, i, baseNum;
	HVOSDLK hSortDlk = 0;	
	DLK_NODE *pShow, *pHead;

	hSortDlk = vos_dlkCreate(sizeof(S_TS_LOAD), 0, NULL);
	vos_dlkSetSort(hSortDlk, tsr_sortGetKey, 2); //decent sort

	//make a sort data
	for(i=0; i<TS_MAX_LOAD_TASKNUM; i++)
	{
		pLoad = &pLoadInOut[i];
		if(pLoad->pName == NULL)break;
		
		totalUs += pLoad->totalTimeUs;
		vos_dlkInsertDataSort(hSortDlk, (void *)pLoad, pLoad->totalTimeUs);		
	}	

	pLoad = pLoadInOut;
	memset(pLoad, 0, sizeof(S_TS_LOAD)*TS_MAX_LOAD_TASKNUM);

	//copy sort data to pLoad
	pHead = vos_dlkGetHead(hSortDlk);
	DLK_WALK_LOOP_START(pShow, pHead)
	{
		pSort = (S_TS_LOAD *)(pShow + 1);
		
		if(pSort->totalTimeUs < (0xffffffff/1000))
			baseNum = 1;
		else if(pSort->totalTimeUs < (0xffffffff/100))	
			baseNum = 10;
		else if(pSort->totalTimeUs < (0xffffffff/10))	
			baseNum = 100;
		else
			baseNum = 1000;
			
		pSort->perCent = (pSort->totalTimeUs/baseNum*1000) / (totalUs/baseNum); //add percent*10
		memcpy(pLoad, pSort, sizeof(S_TS_LOAD));
		pLoad++;
	}
	DLK_WALK_LOOP_END(pShow, pHead)

	vos_dlkDestroy(hSortDlk);
	return;
}

void tsr_loadingSortAll(void)
{
	unsigned long i;
	S_TS_LOAD *pLoad;
	
	//sort whole
	pLoad = &g_tsInfo.stLoadWhole.stLoad[0];
	tsr_loadingSort(pLoad);

	//for segments loading
	if(g_tsInfo.segments < 2)return;
	
	for(i=0; i<g_tsInfo.segments; i++)
	{
		pLoad = &((g_tsInfo.pSegLoad+i)->stLoad[0]);
		tsr_loadingSort(pLoad);
	}
}

int tsr_saveRec(void)
{
	S_TS_REC *pTmp;
	S_SEG_LOAD *pSegLod = &g_tsInfo.stLoadWhole;
	unsigned long i, timeUs, takeUs, irqUs, bufSize;
	char taskName[] = "NULL";
	char *pName;
	char logName[32] = {0,};
	unsigned long lastUs = 0;
	char *pRecBuff = NULL, *pbuf;
	static char *pTask = "other";

	if(g_tsInfo.type < 2)
	{
		if(NULL == g_tsInfo.psBuff )return 0;

		bufSize = (g_tsInfo.index+4)*38 + ((TS_MAX_LOAD_TASKNUM+4)*88*g_tsInfo.segments);
		pRecBuff = vos_malloc(bufSize);
		if(NULL == pRecBuff)return 0;
		pbuf = pRecBuff;
	 
		pbuf += sprintf(pbuf, "%-16s\t%-9s\t%7s\n", "name", "time", "take(us)");
		pbuf += sprintf(pbuf, "----------------------------------------------\n");

		for(i=0; i<g_tsInfo.index; i++)
		{
			pTmp = g_tsInfo.psBuff + i;
			pName = (char *)pTmp->pName;
			if(pName == NULL)pName = taskName;

			timeUs = pTmp->sec * 1000000 + (pTmp->ns / 1000);
			if(i == 0)
			{	
				lastUs = timeUs;
				pTmp->irqNum = 0;
			}
			takeUs = (timeUs-lastUs) % 10000000; //7bytes
			lastUs = timeUs;
			if(pTmp->irqNum == 0)
			{
				pbuf += sprintf(pbuf, "%-16s\t%-9u\t%7u\n", pName, (timeUs % 1000000000), takeUs);
			}
			else
			{
				irqUs = pTmp->irqTime;
				irqUs = irqUs*100/takeUs;
				pbuf += sprintf(pbuf, "%-16s\t%-9u\t%7u-%u\n", pName, (timeUs % 1000000000), takeUs, irqUs);
				g_tsInfo.irqCnt += (pTmp->irqNum & 0xff);
				g_tsInfo.irqTotalUs += irqUs;
			}

			tsr_recLoadingAll(pTmp->p1, pTmp->p2, pName, takeUs, i);
		}
		pbuf += sprintf(pbuf, "----------------------------------------------\n");
		pbuf += sprintf(pbuf, "--start=%u, end=%u, take:%ums, irqCnt:%u, total counter:%u\n\n", pSegLod->startUs, pSegLod->endUs, (pSegLod->endUs-pSegLod->startUs)/1000, g_tsInfo.irqCnt, g_tsInfo.index);
	}
	
	tsr_loadingSortAll();
	if(NULL == pRecBuff)
	{		
		bufSize = (TS_MAX_LOAD_TASKNUM+4)*80;
		pRecBuff = vos_malloc(bufSize);
		if(NULL == pRecBuff)return 0;
		pbuf = pRecBuff;
	}
	pbuf += tsr_getLoadingAll(pbuf);

	sprintf(logName, "/tmp/tsr_%u.log", vos_GetSysCntMs());
	vos_fileWrite(logName, pRecBuff, pbuf-pRecBuff); //save to file
	vos_free(pRecBuff);
	
	return i;
}

int tsr_showTrace(int startIndex)
{
	S_TS_REC *pTmp;
	S_SEG_LOAD *pSegLod = &g_tsInfo.stLoadWhole;
	unsigned long i, timeUs;
	char taskName[] = "NULL";
	char *pName;
	static unsigned long g_lastUs = 0;

	if(NULL == g_tsInfo.psBuff )return 0;
	printk("%-8s: %-11s %-9s %10s %-16s %-11s %10s\r\n", "index", "pid/tgid", "irqTime", "irqNum", "name", "time", "take(us)");

	if(startIndex == 0)g_lastUs = pSegLod->startUs;
	for(i=startIndex; i<g_tsInfo.index; i++)
	{
		pTmp = g_tsInfo.psBuff + i;
		pName = (char *)pTmp->pName;
		if(pName == NULL)pName = taskName;

		timeUs = pTmp->sec * 1000000 + (pTmp->ns / 1000);
		if(i == 0)g_lastUs = timeUs;

		printk("%-8d: %5u/%-5u %-9u 0x%08x %-16s %-11u %10u\r\n", i, pTmp->p1, pTmp->p2, pTmp->irqTime, pTmp->irqNum, pName, timeUs, timeUs-g_lastUs);
		g_lastUs = timeUs;
		if((i-startIndex) == TS_NUM_PER_TIME)break;
	}
	
	return (i-startIndex);
}

void tsr_showLoading(void)
{
	tsr_getLoadingAll(NULL);
}

//(flag=1)start: type:0-record Trace with irq, 1-record Trace only, 2-record loading
//(flag=0)end:   type:0-end and save all record, 1-end only
//(flag=2)free memory
//(flag=3)add segments
void tsr_setFlag(int flag, int type) 
{
	if(flag == 2)
	{
		if(g_tsInfo.flag == 0) //only after end
		{
			if(g_tsInfo.psBuff != NULL)vos_free(g_tsInfo.psBuff);
			g_tsInfo.psBuff = NULL;
		}
		return;
	}
	
	if(flag == 3)
	{
		if(g_tsInfo.type < 2)
		{
			g_tsInfo.segments++;		
			g_tsInfo.segSartIndex[g_tsInfo.segments] = g_tsInfo.index;
		}
		return;
	}
	
	if(flag == 1)
	{
		g_tsInfo.flag = 1;
		g_tsInfo.type = type;
		g_tsInfo.index = 0;
		g_tsInfo.irqCnt = 0;
		g_tsInfo.irqTotalUs = 0;
		g_tsInfo.segments = 0;
		g_tsInfo.segSartIndex[0] = 0;
		g_tsInfo.pSegLoad = NULL;
		memset((void *)&g_tsInfo.stLoadWhole, 0, sizeof(S_SEG_LOAD));
		
		if(type < 2) //record switch
		{
			if(g_tsInfo.psBuff == NULL)g_tsInfo.psBuff = vos_malloc(sizeof(S_TS_REC)*TS_MAX_NUM);
			VOS_ASSERT(g_tsInfo.psBuff != NULL);	
			//memset(g_tsInfo.psBuff, 0, sizeof(S_TS_REC)*TS_MAX_NUM);

			if(type == 0)
				ts_regRecordFunc(tsr_recTrace, 1);
			else
				ts_regRecordFunc(tsr_recTrace, 0);
		}
		else //record loading
		{
			//Quasar\kernel\linux-4.19-quasar\kernel\sched\core.c __schedule
			ts_regRecordFunc(tsr_recLoading, 0);
		}
		return;
	}
	
	if(flag == 0)
	{
		if(g_tsInfo.flag == 1) //start to end
		{
			ts_regRecordFunc(NULL, 0);

			g_tsInfo.segments++;		
			g_tsInfo.segSartIndex[g_tsInfo.segments] = g_tsInfo.index;
			tsr_initSegLoading();
		}

		if(type == 1)
		{
			g_tsInfo.flag = 0; //only end
		}
		else if(g_tsInfo.flag != 0x10)
		{
			vos_taskSleep(1);
			tsr_saveRec();
			g_tsInfo.flag = 0x10;
		}
		return;
	}
		
}

int tsr_showRec(int flag) //flag: 0-loading, other-trace start block index(1-200)
{
	int startIndex, cnt, moreRec = 0;
	S_SEG_LOAD *pSegLod = &g_tsInfo.stLoadWhole;
	
	if(flag == 0)
	{
		tsr_showLoading();
		return 0;
	}

	if(NULL == g_tsInfo.psBuff)return 0;

	startIndex = (flag-1)*TS_NUM_PER_TIME;
	cnt = tsr_showTrace(startIndex);

	if(cnt >= TS_NUM_PER_TIME)moreRec = 1;
	if(0 == moreRec)
	{
		printk("--start=%u, end=%u, take:%ums, irqCnt:%u, total counter:%u\n\n", pSegLod->startUs, pSegLod->endUs, (pSegLod->endUs-pSegLod->startUs)/1000, g_tsInfo.irqCnt, g_tsInfo.index);
	}
	return moreRec;
}

#ifdef __cplusplus 
    }
#endif

