/*=================================================================================================
Filename:    vos_mem.c
Description: abstraction functions of Linux kernel 
Data:        Initial revision. Bob@2019-12-05
==================================================================================================*/
#ifdef __cplusplus 
    extern "C" { 
#endif

#include "vos_if.h"
#include "vos_innr.h"
#include "vos_usr_if.h"
#include <asm/dma-mapping.h>

extern struct device *g_pVosDev;

void vos_memInit(void)
{
	//don nothing
}

void * vos_malloc(unsigned long size)
{
	return vmalloc(size);
}

void vos_free(void *pVaddr)
{
	vfree(pVaddr);
	return;
}

void * vos_kmalloc(unsigned long size, const char* file, unsigned long line)
{
	return kmalloc(size, GFP_KERNEL|__GFP_NOWARN|__GFP_NORETRY | GFP_DMA );
}

void vos_kfree(void *pVaddr)
{
	kfree(pVaddr);
	return;
}

//arch\arm\include\asm\Memory.h
unsigned long vos_memPhy2vir(unsigned long nPhyAddr) //valid for kmalloc, not for vmalloc&ioremap
{
	unsigned long nVirAddr;
	
	nVirAddr = phys_to_virt(nPhyAddr);
	return nVirAddr;
}

unsigned long vos_memVir2Phy(unsigned long nVirAddr)
{
	unsigned long nPhyAddr;
	
	nPhyAddr = virt_to_phys(nVirAddr);
	return nPhyAddr;
}

unsigned long vos_phyAddrRd(unsigned long phyAddr, unsigned long len) //for IO read
{
	unsigned long vAddr, val=0xdeaddead;
	
	vAddr = ioremap(phyAddr, 32);
	switch(len)
	{
		case 1:
			val = (unsigned long)*(unsigned char *)vAddr;
			break;
			
		case 2:
			val = (unsigned long)*(unsigned short *)vAddr;
			break;
			
		case 4:
			val = (unsigned long)*(unsigned long *)vAddr;
			break;

		default:
			break;
	}
	
	iounmap(vAddr);
	return val;
}

unsigned long vos_phyAddrWt(unsigned long phyAddr, unsigned long val, unsigned long len) //for IO write
{
	unsigned long vAddr;
	
	vAddr = ioremap(phyAddr, 32);
	switch(len)
	{
		case 1:
			*(unsigned char *)vAddr = (unsigned char)val;
			break;
			
		case 2:
			*(unsigned short *)vAddr = (unsigned char)val;
			break;
			
		case 4:
			*(unsigned long *)vAddr = (unsigned long)val;
			break;

		default:
			len = 0;
			break;
	}
	
	iounmap(vAddr);
	return len;
}

unsigned long vos_phyAddrMap(unsigned long phyAddr, unsigned long size)
{
	unsigned long vAddr;
	
	vAddr = ioremap(phyAddr, size);
	return vAddr;
}

unsigned long vos_phyAddrUnmap(unsigned long vAddr)
{
	iounmap(vAddr);
	return vAddr;
}

unsigned long vos_kvAddrRd(unsigned long vAddr, unsigned long len)
{
	unsigned long val=0xdeaddead;
	
	switch(len)
	{
		case 1:
			val = (unsigned long)*(unsigned char *)vAddr;
			break;
			
		case 2:
			val = (unsigned long)*(unsigned short *)vAddr;
			break;
			
		case 4:
			val = (unsigned long)*(unsigned long *)vAddr;
			break;

		default:
			break;
	}
	
	return val;
}

unsigned long vos_kvAddrWt(unsigned long vAddr, unsigned long val, unsigned long len)
{
	switch(len)
	{
		case 1:
			*(unsigned char *)vAddr = (unsigned char)val;
			break;
			
		case 2:
			*(unsigned short *)vAddr = (unsigned char)val;
			break;
			
		case 4:
			*(unsigned long *)vAddr = (unsigned long)val;
			break;

		default:
			len = 0;
			break;
	}
	
	return len;
}

void *vos_kmalloc_coherent(unsigned long size, void *pPhyAddr)
{
	void *pKvirAddr = NULL;

	pKvirAddr = dma_alloc_coherent(g_pVosDev, size, pPhyAddr, GFP_DMA | GFP_KERNEL);
	return pKvirAddr;
}

void vos_kfree_coherent(unsigned long size, void *pKvirAddr, void *pPhyAddr)
{
	dma_free_coherent(g_pVosDev, size, pKvirAddr, pPhyAddr);
}

void * vos_dmalloc(int flag, unsigned long size, void *pPhyAddr)
{
	void *pKvirAddr = NULL;

	switch(flag)
	{
		case 0:
			pKvirAddr = vos_kmalloc(size, 0, 0);
			*(int *)pPhyAddr = vos_memVir2Phy(pKvirAddr);
			break;

		case 1:	
			pKvirAddr =	dma_alloc_coherent(g_pVosDev, size, pPhyAddr, GFP_DMA | GFP_KERNEL); //vosdev vosdev: coherent DMA mask is unset
			break;

		case 2:
		default:
			pKvirAddr =	dma_alloc_writecombine(g_pVosDev, size, pPhyAddr, GFP_DMA | GFP_KERNEL);
			break;
	}

	return pKvirAddr;
}

void vos_dfree(int flag, unsigned long size, void *pKvirAddr, void *pPhyAddr)
{	
	switch(flag)
	{
		case 0:
			vos_kfree(pKvirAddr);
			break;

		case 1:	
		case 2:
		default:
			dma_free_coherent(g_pVosDev, size, pKvirAddr, pPhyAddr);
			break;
	}
	return;
}

void *vos_dmemCoherence(unsigned long kv_addr, int len, int type)
{
	int direction = DMA_TO_DEVICE; //before device read: make cache write to phyMem
	void *hw_addr = NULL;

	if(type == VOSCMD_DMA_CACHE_MISS)direction = DMA_FROM_DEVICE; //after device write: make cache miss

	//vos_Printf("vos_dmemCoherence(0x%x, %d, %d)\r\n", kv_addr, len, type);
	hw_addr = dma_map_single(g_pVosDev, kv_addr, len,direction);
	dma_unmap_single(g_pVosDev, hw_addr, len,direction);

	return hw_addr;
}

void *vos_dmemSync(unsigned long kv_addr, int len, int type)
{
	int direction = DMA_TO_DEVICE; //before device read: make cache write to phyMem
	void *hw_addr = NULL;

	if(type == VOSCMD_DMA_CACHE_MISS)direction = DMA_FROM_DEVICE; //after device write: make cache miss

	vos_Printf("vos_dmemSync(0x%x, %d, %d)\r\n", kv_addr, len, type);
	//hw_addr = dma_cache_sync(kv_addr, len,direction);

	return hw_addr;
}


EXPORT_SYMBOL(vos_malloc);
EXPORT_SYMBOL(vos_free);
EXPORT_SYMBOL(vos_kmalloc);
EXPORT_SYMBOL(vos_kfree);
EXPORT_SYMBOL(vos_memPhy2vir);
EXPORT_SYMBOL(vos_memVir2Phy);
EXPORT_SYMBOL(vos_phyAddrRd);
EXPORT_SYMBOL(vos_phyAddrWt);
EXPORT_SYMBOL(vos_phyAddrMap);
EXPORT_SYMBOL(vos_phyAddrUnmap);

EXPORT_SYMBOL(vos_dmalloc);
EXPORT_SYMBOL(vos_dfree);
EXPORT_SYMBOL(vos_dmemCoherence);
EXPORT_SYMBOL(vos_dmemSync);
EXPORT_SYMBOL(vos_kmalloc_coherent);
EXPORT_SYMBOL(vos_kfree_coherent);

unsigned long test_mem(unsigned long type, unsigned long p1, unsigned long p2, unsigned long p3)
{
	unsigned long nRet = 0;
	unsigned long ulVal = 0;

	switch(type) //show_free_areas(0/1)
	{
		case 0:
		    nRet = (unsigned long)vos_kmalloc(p1, NULL, NULL);
			printk("vos_kmalloc(0x%x), result=0x%x\r\n", p1, nRet);
			break;
		case 1:
		    vos_kfree((void *)p1);
			printk("vos_kfree(0x%x)\r\n", p1);
			break;
		case 2:
		    nRet = vos_memPhy2vir(p1);
			printk("vos_memPhy2vir(0x%x), result=0x%x\r\n", p1, nRet);
			break;
		case 3:
		    vos_memVir2Phy(p1);
			printk("vos_memVir2Phy(0x%x), result=0x%x\r\n", p1, ulVal);
			break;
		case 4:
		    nRet = vos_phyAddrRd(p1, p2);
			printk("vos_phyAddrRd(0x%x, %u), result=0x%x\r\n", p1, p2, nRet);
			break;
		case 5:
		    nRet = vos_phyAddrWt(p1, p2, p3);
			printk("vos_phyAddrWt(0x%x, 0x%x, %u), result=0x%x\r\n", p1, p2, p3, nRet);
			break;
		case 6:
		    nRet = vos_phyAddrMap(p1, p2);
			printk("vos_phyAddrMap(0x%x, %u), result=0x%x\r\n", p1, p2, nRet);
			break;
		case 7:
		    nRet = vos_phyAddrUnmap(p1);
			printk("vos_phyAddrUnmap(0x%x), result=0x%x\r\n", p1, nRet);
			break;
		case 10:
		    nRet = vos_dmalloc(0, p1, (void *)&ulVal);
			printk("VOS_DMALLOC, kvirAddr=0x%x, phyAddr=0x%x\r\n", nRet, ulVal);
			break;
		case 11:
		    nRet = vos_dmalloc(1, p1, (void *)&ulVal);
			printk("VOS_DMALLOC_COHERENCE, kvirAddr=0x%x, phyAddr=0x%x\r\n", nRet, ulVal);
			break;
		case 12:
		    nRet = vos_dmemCoherence(p1, p2, VOSCMD_DMA_CACHE2PHY);
			printk("vos_dmemCoherence_CACHE2PHY, result=0x%x\r\n", nRet);
			break;
		case 13:
		    nRet = vos_dmemCoherence(p1, p2, VOSCMD_DMA_CACHE_MISS);
			printk("vos_dmemCoherence_CACHE_MISS, result=0x%x\r\n", nRet);
			break;
		default:
			printk("unknown: vos_mem_test(%u, %u, %u, %u\r\n)", type, p1, p2, p3);
			break;
	}
	return nRet;
}

#ifdef __cplusplus 
    }
#endif

