/*=================================================================================================
Filename:    vos_hrtimer.c
Description: abstraction functions of Linux kernel 
Data:        Initial revision. Bob@2020-7-15
==================================================================================================*/
#ifdef __cplusplus 
    extern "C" { 
#endif

#include "vos_if.h"
#include "vos_innr.h"

enum hrtimer_restart vos_hrtimeOut(struct hrtimer *pTimer);

static struct hrtimer *g_phrTimerTbl[VOS_TIMER_MAX_NUM] = {0,};

void vos_hrtimerInit(void)
{
	//do nothing
	return;
}

HVOSTIMER vos_hrtimerCreate(char * Name, TimeFunc pfFunc, void *pParam, int nanoSec, int autoAct)
{
	struct hrtimer *pTimer=NULL;
	hrTimer_ITEM_S *pTimerItem;
	HVOSTIMER vTimerId = 0;
	int nRet=0, i;
	ktime_t kt_periode;

	for(i=0; i<VOS_TIMER_MAX_NUM; i++)
	{
		if(g_phrTimerTbl[i] == 0)break;
	}
	if(i == VOS_TIMER_MAX_NUM)return 0;
	
	vTimerId = (HVOSTIMER)((VOS_TIMER_TYPE<<16) | i);
	pTimer = (struct hrtimer *)vos_malloc(sizeof(struct hrtimer) + sizeof(hrTimer_ITEM_S));
	VOS_ASSERT(pTimer != NULL);	
	memset((void *)pTimer, 0, (sizeof(struct hrtimer) + sizeof(hrTimer_ITEM_S)));

	g_phrTimerTbl[i] = pTimer;
	pTimerItem = (hrTimer_ITEM_S *)(pTimer + 1);
	pTimerItem->pTimer = pTimer;
	pTimerItem->nanoSec = nanoSec;
	pTimerItem->pfFunc = pfFunc;
	pTimerItem->param = (unsigned long)pParam;
	if(Name != NULL)
	{
		strncpy(pTimerItem->sName, Name, 11);
		pTimerItem->sName[11] = 0;
	}
	if(autoAct)pTimerItem->status = 1; //enable

    hrtimer_init(pTimer, CLOCK_REALTIME, HRTIMER_MODE_REL);
    pTimer->function = vos_hrtimeOut;
	vosPrintf("vos_hrtimerCreate: timerId=0x%x(0x%x)\r\n", pTimer, vTimerId);

	if(autoAct)
	{
		kt_periode = ktime_set(0, nanoSec);//(second, ns)
	    hrtimer_start(pTimer, kt_periode, HRTIMER_MODE_REL);
	}

	return vTimerId;
}

int vos_hrtimerEnable(HVOSTIMER vTimerId, int newNanoSec)
{
	struct hrtimer *pTimer=NULL;
	hrTimer_ITEM_S *pTimerItem;
	ktime_t kt_periode;

	VOS_ASSERT((vTimerId >> 16) == VOS_TIMER_TYPE);
	VOS_ASSERT( (vTimerId & 0xffff) < VOS_TIMER_MAX_NUM);

	pTimer = g_phrTimerTbl[vTimerId & 0xffff];
	if(pTimer == NULL)return 1;

	pTimerItem = (hrTimer_ITEM_S *)(pTimer + 1);
	VOS_ASSERT(pTimerItem->pTimer == pTimer);	

	if(newNanoSec)pTimerItem->nanoSec = newNanoSec;
	pTimerItem->status = 1; //enable
	
	kt_periode = ktime_set(0, pTimerItem->nanoSec);//(second, ns)
    hrtimer_start(pTimer, kt_periode, HRTIMER_MODE_REL);
	return VOS_OK;
}

int vos_hrtimerDisble(HVOSTIMER vTimerId)
{
	struct hrtimer *pTimer=NULL;
	hrTimer_ITEM_S *pTimerItem;

	VOS_ASSERT((vTimerId >> 16) == VOS_TIMER_TYPE);
	VOS_ASSERT( (vTimerId & 0xffff) < VOS_TIMER_MAX_NUM);

	pTimer = g_phrTimerTbl[vTimerId & 0xffff];
	if(pTimer == NULL)return 1;

	pTimerItem = (hrTimer_ITEM_S *)(pTimer + 1);
	VOS_ASSERT(pTimerItem->pTimer == pTimer);	

	pTimerItem->status = 0; //disable
	return VOS_OK;
}

enum hrtimer_restart vos_hrtimeOut(struct hrtimer * pTimer)
{
	hrTimer_ITEM_S *pTimerItem;
	ktime_t kt_periode;

	pTimerItem = (hrTimer_ITEM_S *)(pTimer + 1);
	VOS_ASSERT(pTimerItem->pTimer == pTimer);	

	pTimerItem->cnt++;
	if(pTimerItem->status == 0)return HRTIMER_NORESTART;
	
	if(pTimerItem->pfFunc)pTimerItem->pfFunc(pTimerItem->param);

	kt_periode = ktime_set(0, pTimerItem->nanoSec);//(second, ns)
	hrtimer_forward_now(pTimer, kt_periode);
	
	return HRTIMER_RESTART;
}

int vos_hrtimerShowAll(void)
{
	int i;
	struct hrtimer *pTimer=NULL;
	hrTimer_ITEM_S *pTimerItem;
	HVOSTIMER vTimerId;
	//char sFunc[24] = {0,};

	vosPrintf("%-4s %-8s   %-8s   %-12s %-8s   %-12s %-8s   %-8s %-12s\r\n", \
		"No.", "TimerId", "VId", "name", "Func", "nanoSec", "Param", "enable", "Counter");
    vosPrintf("--------------------------------------------------------------------------------------------\r\n");

	for(i=0; i<VOS_TIMER_MAX_NUM; i++)
	{
		pTimer = g_phrTimerTbl[i];
		if(pTimer == NULL)continue;
		
		pTimerItem = (hrTimer_ITEM_S *)(pTimer+1);
		VOS_ASSERT(pTimer == pTimerItem->pTimer);
		
		vosPrintf("%-4d 0x%-8x 0x%-8x %-12s 0x%-8x %-12u 0x%-8s %-8d %-12u\r\n", \
			i, pTimer, ((VOS_TIMER_TYPE<<16) | i), pTimerItem->sName, pTimerItem->pfFunc, pTimerItem->nanoSec, pTimerItem->param, pTimerItem->status, pTimerItem->cnt);
	}
	return 0;
}

void test_hrtimeFunc(void * pParam)
{
	static int nTimerCnt = 0;

	nTimerCnt++;
	DBG_tracePut("%u: pParam=0x%x", nTimerCnt, pParam);
	return;
}

int test_hrtime(int nFlag, int nPar)
{
	int nRet=0;

	switch(nFlag)
	{
		case 0:
			nRet = vos_hrtimerShowAll();
			break;
			
		case 1:
			DBG_tracePut("test_hrtime(%u)\r\n", nPar);
			nRet = (int)vos_hrtimerCreate("timer1", test_hrtimeFunc, (void *)0x11, nPar, 0);
			vosPrintf("vos_hrtimerCreate ret=0x%x\r\n", nRet);
			break;

		default:
			break;
	}

    return nRet;
}


#ifdef __cplusplus 
    }
#endif

